package MojoWiki::MojoDaemon::Page;
##
## HTTPリクエストを処理するクラス
## $Id: Page.pm 21 2011-08-02 01:39:58Z bobunderson $
##

use strict;
use warnings;

use Encode;
use FindBin;

use lib join( '/', $FindBin::Bin, '..', '..' );
use MojoWiki::Constants;
use MojoWiki::PageParts;
use MojoWiki::WikiFormatter;
use MojoWiki::YAML;

use base 'Mojolicious::Controller';

##
## '/' でアクセスされたら、'/index' にリダイレクトする
##
sub index {
    my ($self) = @_;

    $self->redirect_to( URL_PREFIX . 'index' );
}

##
## HTTPリクエストを処理する
##
sub page {
    my ($self) = @_;

    my $cmd = $self->tx->req->params->param('cmd') || 'view';

    if ( $cmd eq 'view' ) {
        &_view($self);
    }
    elsif ( $cmd eq 'edit' ) {
        &_edit($self);
    }
    elsif ( $cmd eq 'preview_or_save' ) {
        &_preview_or_save($self);
    }
    else {
        &_process_command_by_plugins( $self, $cmd );
    }
}

##
## 閲覧画面
##
sub _view {
    my ($self) = @_;

    my $path = $self->stash('path');

    my $page = $self->app->db->get_page($path);
    if ( !defined($page) ) {
        $page->{'Path'}         = $path;
        $page->{'Title'}        = '/' . $path;
        $page->{'Body'}         = '';
        $page->{'Revision'}     = 0;
        $page->{'CreatedTime'}  = 0;
        $page->{'ModofiedTime'} = 0;
    }

    my $formatter = new MojoWiki::WikiFormatter($self);
    my $body      = $formatter->format( $page->{'Body'} );

    my $common_parts       = new MojoWiki::PageParts($self);
    my $header_contents    = $common_parts->get_header();
    my $navigator_contents = $common_parts->get_navigator();
    my $footer_contents    = $common_parts->get_footer();

    $self->render(
        'wikiname' => $self->app->config->{'WikiName'},
        'title'    => Encode::encode_utf8( $page->{'Title'} ),

        'contents' => Encode::encode_utf8($body),

        'header_contents'    => Encode::encode_utf8($header_contents),
        'navigator_contents' => Encode::encode_utf8($navigator_contents),
        'footer_contents'    => Encode::encode_utf8($footer_contents),

        'template' => 'page/view',
        'handler'  => 'tt',
        'format'   => 'html'
    );

    $self->rendered();
}

##
## 編集画面
##
sub _edit {
    my ($self) = @_;

    my $path = $self->stash('path');

    my $page = $self->app->db->get_page($path);
    if ( !defined($page) ) {
        $page->{'Path'}         = $path;
        $page->{'Title'}        = '/' . $path;
        $page->{'Body'}         = '';
        $page->{'Revision'}     = 0;
        $page->{'CreatedTime'}  = 0;
        $page->{'ModifiedTime'} = 0;
    }

    my $common_parts       = new MojoWiki::PageParts($self);
    my $header_contents    = $common_parts->get_header();
    my $navigator_contents = $common_parts->get_navigator();
    my $footer_contents
        = $common_parts->get_footer( { 'revision' => $page->{'Revision'} } );

    $self->render(
        'wikiname' => $self->app->config->{'WikiName'},
        'title'    => Encode::encode_utf8( $page->{'Title'} ),

        'raw_contents' => Encode::encode_utf8( $page->{'Body'} ),
        'revision'     => $page->{'Revision'},

        'header_contents'    => Encode::encode_utf8($header_contents),
        'navigator_contents' => Encode::encode_utf8($navigator_contents),
        'footer_contents'    => Encode::encode_utf8($footer_contents),

        'template' => 'page/edit',
        'handler'  => 'tt',
        'format'   => 'html'
    );
}

##
## パラメータによってプレビューか保存か判断する
##
sub _preview_or_save {
    my ($self) = @_;

    my $preview = $self->tx->req->params->param('preview');
    my $save    = $self->tx->req->params->param('save');

    if ( defined($save) && $save ne '' ) {
        &_save($self);
    }
    else {
        &_preview($self);
    }
}

##
## プレビュー画面
##
sub _preview {
    my ($self) = @_;

    my $title        = $self->tx->req->params->param('title');
    my $raw_contents = $self->tx->req->params->param('contents');
    my $revision     = $self->tx->req->params->param('revision');

    my $formatter = new MojoWiki::WikiFormatter($self);

    # 見出しにIDをつける
    $raw_contents = $formatter->head_index($raw_contents);

    my $contents = $formatter->format($raw_contents);

    my $common_parts       = new MojoWiki::PageParts($self);
    my $header_contents    = $common_parts->get_header();
    my $navigator_contents = $common_parts->get_navigator();
    my $footer_contents
        = $common_parts->get_footer( { 'revision' => $revision } );

    $self->render(
        'wikiname' => $self->app->config->{'WikiName'},
        'title'    => Encode::encode_utf8($title),

        'raw_contents' => Encode::encode_utf8($raw_contents),
        'contents'     => Encode::encode_utf8($contents),
        'revision'     => $revision,

        'header_contents'    => Encode::encode_utf8($header_contents),
        'navigator_contents' => Encode::encode_utf8($navigator_contents),
        'footer_contents'    => Encode::encode_utf8($footer_contents),

        'template' => 'page/preview',
        'handler'  => 'tt',
        'format'   => 'html'
    );
}

##
## ページへの変更を保存
##
sub _save {
    my ($self) = @_;

    my $path = $self->stash('path');

    my $title    = $self->tx->req->params->param('title');
    my $raw_body = $self->tx->req->params->param('contents');
    my $revision = $self->tx->req->params->param('revision');

    # revisionのチェック
    my $page_on_db = $self->app->db->get_page($path);
    if ( !defined($page_on_db) ) {
        $page_on_db->{'Revision'} = 0;
    }

    if ( $revision != $page_on_db->{'Revision'} ) {
        &_collision($self);

        return;
    }

    if ( $raw_body eq '' ) {

        # ページ本文が空文字列なら、ページを削除
        $self->_delete();
    }
    else {

        # 見出しにIDをつける
        my $formatter = new MojoWiki::WikiFormatter($self);
        $raw_body = $formatter->head_index($raw_body);

        my $page;
        $page->{'Title'}    = $title;
        $page->{'Body'}     = $raw_body;
        $page->{'Revision'} = $revision;

        $self->app->db->set_page( $path, $page );

        $self->redirect_to( '/' . $path );
    }
}

##
## ページを削除
##
sub _delete {
    my ($self) = @_;

    my $path = $self->stash('path');

    # DBからページを削除
    $self->app->db->delete_page($path);

    # 各プラグインにデータの後始末をさせる
    my @block_modules   = MojoWiki::Plugins::get_block_plugins($self);
    my @command_modules = MojoWiki::Plugins::get_command_plugins($self);
    my @inline_modules  = MojoWiki::Plugins::get_inline_plugins($self);
    foreach my $module ( @block_modules, @command_modules, @inline_modules ) {
        $module->delete_page($path);
    }

    my $common_parts    = new MojoWiki::PageParts($self);
    my $header_contents = $common_parts->get_header(
        { 'title' => 'ページを削除しました' } );
    my $navigator_contents = $common_parts->get_navigator();
    my $footer_contents    = $common_parts->get_footer();

    my $message = $path . ' を削除しました';

    $self->render(
        'wikiname' => $self->app->config->{'WikiName'},
        'title'    => 'ページを削除しました',

        'message' => $message,

        'header_contents'    => Encode::encode_utf8($header_contents),
        'navigator_contents' => Encode::encode_utf8($navigator_contents),
        'footer_contents'    => Encode::encode_utf8($footer_contents),

        'template' => 'page/deleted',
        'handler'  => 'tt',
        'format'   => 'html'
    );
    $self->rendered;
}

##
## 編集衝突時のエラー画面
##
sub _collision {
    my ($self) = @_;

    my $path = $self->stash('path');

    my $title    = $self->tx->req->params->param('title');
    my $raw_body = $self->tx->req->params->param('contents');

    my $formatter = new MojoWiki::WikiFormatter($self);
    my $contents  = $formatter->format($raw_body);

    my $common_parts       = new MojoWiki::PageParts($self);
    my $header_contents    = $common_parts->get_header();
    my $navigator_contents = $common_parts->get_navigator();
    my $footer_contents    = $common_parts->get_footer();

    my $page = $self->app->db->get_page($path);
    if ( !defined($page) ) {
        $page->{'Revision'} = 0;
    }
    my $revision = $page->{'Revision'};

    $self->render(
        'wikiname' => $self->app->config->{'WikiName'},
        'title'    => Encode::encode_utf8($title),

        'raw_contents' => Encode::encode_utf8($raw_body),
        'contents'     => Encode::encode_utf8($contents),
        'revision'     => $revision,

        'header_contents'    => Encode::encode_utf8($header_contents),
        'navigator_contents' => Encode::encode_utf8($navigator_contents),
        'footer_contents'    => Encode::encode_utf8($footer_contents),

        'template' => 'page/collision',
        'handler'  => 'tt',
        'format'   => 'html'
    );
}

##
## URLで渡されたコマンドを処理する
##
sub _process_command_by_plugins {
    my ( $self, $cmd ) = @_;

    my @plugins = MojoWiki::Plugins::get_command_plugins($self);

    foreach my $plugin (@plugins) {
        if ( $plugin->is_processable_command($cmd) ) {

       # 当該コマンドを処理可能なプラグインに処理させる
            $plugin->process_command( $self, $cmd );
        }
    }

    &error( $self,
        'コマンドを処理可能なプラグインが見つかりませんでした'
    );
}

##
## エラー画面
##
sub error {
    my ( $self, $message ) = @_;

    my $common_parts = new MojoWiki::PageParts($self);
    my $header_contents
        = $common_parts->get_header( { 'title' => 'エラー' } );
    my $navigator_contents = $common_parts->get_navigator();
    my $footer_contents    = $common_parts->get_footer();

    $self->render(
        'wikiname' => $self->app->config->{'WikiName'},
        'title'    => Encode::encode_utf8('エラー'),

        'message' => $message,

        'header_contents'    => Encode::encode_utf8($header_contents),
        'navigator_contents' => Encode::encode_utf8($navigator_contents),
        'footer_contents'    => Encode::encode_utf8($footer_contents),

        'template' => 'page/error',
        'handler'  => 'tt',
        'format'   => 'html'
    );
    $self->rendered;
}

1;
