/*
 * Copyright 2023 Syntarou YOSHIDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.synthtarou.midimixer.mx30controller;

import java.util.ArrayList;
import javax.swing.JOptionPane;
import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMessageTemplate;
import jp.synthtarou.midimixer.libs.midi.MXMidi;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MX32MixerData {
    MX32MixerProcess _process;

    private ArrayList<MGSlider>[] _matrixSliderComponent;
    private ArrayList<MGCircle>[] _matrixCircleComponent;
    private ArrayList<MGPad>[] _matrixDrumComponent;

    private ArrayList<MGStatus>[] _matrixSliderStatus;
    private ArrayList<MGStatus>[] _matrixCircleStatus;
    private ArrayList<MGStatus>[] _matrixDrumStatus;

    public boolean ready() {
        return _matrixDrumComponent != null;
    }
    
    ArrayList<MGStatus>[][] _cachedControlChange;
    ArrayList<MGStatus>[][] _cachedChannelMessage;
    ArrayList<MGStatus>[][] _cachedNoteMessage;
    ArrayList<MGStatus>_cachedSystemMessage;
    ArrayList<MGStatus>_cachedDataentry;
    
    protected MX32MixerData(MX32MixerProcess process) {
        _process = process;
        initVolumeMixer();
    }
    
    public static final int INIT_TYPE_ZERO = 0;
    public static final int INIT_TYPE_MIXER = 1;
    public static final int INIT_TYPE_GMTOME = 2;
    public static final int INIT_TYPE_DAW = 3;
    public static final int INIT_TYPE_SOUDMODULE = 4;
    
    public boolean initializeData(int initType) {
        switch(initType) {
            case INIT_TYPE_MIXER:
                initVolumeMixer();
                break;
            case INIT_TYPE_DAW:
                initDaw();
                break;
            case INIT_TYPE_ZERO:
                initZero();
                break;
            case INIT_TYPE_GMTOME:
                initGMTone();
                break;
            case INIT_TYPE_SOUDMODULE:
                initSoundModule();
                break;
            default:
                JOptionPane.showMessageDialog(_process._view, "Not ready", "Sorry", JOptionPane.ERROR_MESSAGE);
                return false;
        }
        _cachedControlChange = null;
        _cachedChannelMessage = null;
        _cachedNoteMessage = null;
        _cachedSystemMessage = null;
        _cachedDataentry = null;
        return true;
    }
    
    private void initVolumeMixer() {
        MX32MixerProcess process = _process;
        ArrayList<MGStatus>[] circleMatrix = new ArrayList[MXStatic.CIRCLE_ROW_COUNT];
        ArrayList<MGStatus>[] sliderMatrix = new ArrayList[MXStatic.SLIDER_ROW_COUNT];

        int port = process._port;
        int column;
        
        circleMatrix[0] = new ArrayList();
        circleMatrix[1] = new ArrayList();
        circleMatrix[2] = new ArrayList();
        circleMatrix[3] = new ArrayList();

        sliderMatrix[0] = new ArrayList();
        
        MGStatus status;
        MXMessage message;

        for(int row = 0; row < sliderMatrix.length; ++ row) {
            ArrayList<MGStatus> slider = sliderMatrix[row];

            while (slider.size() < MXStatic.SLIDER_COLUMN_COUNT) {
                String text;
                column = slider.size();
                if (column >= 16) {
                    text = "F0h, 7Fh, 00h, 04h, 01h, #VL, #VH, F7h";
                    status = new MGStatus(process._port, MGStatus.TYPE_SLIDER, row, column);

                    MXMessageTemplate template = MXMessageFactory.fromDtext(text, 0);
                    message = template.bind(null, _process._port, 0, 128 * 128-1);
                    status.setMonitoringTarget(text, 0, 0, 128 * 128 - 1);
                }else {
                    status = new MGStatus(process._port, MGStatus.TYPE_SLIDER, row, column);
                    message = MXMessageFactory.fromShortMessage(port, null, MXMidi.COMMAND_CONTROLCHANGE + column, MXMidi.DATA1_CC_CHANNEL_VOLUME, 128 -1);
                    status.setMonitoringTarget(message.toDText(), message.getChannel(), message.getGate(), message.getValue());
                }
                slider.add(status);
            }
            sliderMatrix[row] = slider;
        }

        for (int row = 0; row < circleMatrix.length; ++ row) {
            ArrayList<MGStatus> circle = new ArrayList();
            column = 0;
            int[] ccCode = new int[] { 
                MXMidi.DATA1_CC_EFFECT3_CHORUS,
                MXMidi.DATA1_CC_EFFECT1_REVERVE, 
                MXMidi.DATA1_CC_EXPRESSION,
                MXMidi.DATA1_CC_PANPOT
            };
            while (circle.size() < MXStatic.SLIDER_COLUMN_COUNT) {
                if (column >= 16) {
                    status = new MGStatus(process._port, MGStatus.TYPE_CIRCLE, row, column);
                    String text = "F0h, 7Fh, 0gh, 04h, 01h, #VL, #VH, F7h";

                    MXMessageTemplate template = MXMessageFactory.fromDtext(text, 0);
                    message = template.bind(null, _process._port, 0, 128 * 128-1);
                    status.setMonitoringTarget(text, 0, 0, 128 * 128 - 1);
                    circle.add(status);
                    column ++;
                }else {
                    status = new MGStatus(process._port, MGStatus.TYPE_CIRCLE, row, column);
                    message = MXMessageFactory.fromShortMessage(port, null, MXMidi.COMMAND_CONTROLCHANGE + column, ccCode[row], 64);
                    status.setMonitoringTarget(message.toDText(), column, message.getGate(), message.getValue());
                    circle.add(status);
                    
                    column ++;
                }
            }
            circleMatrix[row] = circle;
        }
 
        _matrixSliderStatus = sliderMatrix;
        _matrixCircleStatus = circleMatrix;
        _cachedControlChange = null;
        initDrumMinMidleMax();
    }
    
    private void initDaw() {
        initZero();
        
        MX32MixerData data =  _process._data;

        MGStatus status;
        MXMessage message;
        int port = _process._port;
        
        int[] cclist = {
            73, 75, 79, 72, 80, 81, 82, 83, 85
        };
        int[] cclist2 = {
            74, 71, 76, 77, 93, 18, 19, 16, 17
        };
        
        for(int row = 0; row < MXStatic.SLIDER_ROW_COUNT; ++ row) {
            for(int col = 0; col <MXStatic.SLIDER_COLUMN_COUNT; ++ col) {
                status = data.getSliderStatus(row, col);
                if (col < cclist.length) {
                    message = MXMessageFactory.fromShortMessage(port, null, MXMidi.COMMAND_CONTROLCHANGE + row, cclist[col], 128 -1);
                }else {
                    message = MXMessageFactory.createDummy();
                }
                status.setMonitoringTarget(message.toDText(), 0, message.getGate(), message.getValue());
            }
        }

        for (int row = 0; row < MXStatic.CIRCLE_ROW_COUNT; ++ row) {
            for(int col = 0; col <MXStatic.SLIDER_COLUMN_COUNT; ++ col) {
                status = data.getCircleStatus(row, col);
                if (col < cclist2.length) {
                    message = MXMessageFactory.fromShortMessage(port, null, MXMidi.COMMAND_CONTROLCHANGE + row, cclist2[col], 128 -1);
                    status.setMonitoringTarget(message.toDText(), row, message.getGate(), message.getValue());
                }
            }
        }

        _cachedControlChange = null;
        initDrumMinMidleMax();
    }

    private void initZero() {
        MX32MixerProcess process = _process;
        ArrayList<MGStatus>[] circleMatrix = new ArrayList[MXStatic.CIRCLE_ROW_COUNT];
        ArrayList<MGStatus>[] sliderMatrix = new ArrayList[MXStatic.SLIDER_ROW_COUNT];

        circleMatrix[0] = new ArrayList();
        circleMatrix[1] = new ArrayList();
        circleMatrix[2] = new ArrayList();
        circleMatrix[3] = new ArrayList();

        sliderMatrix[0] = new ArrayList();
        
        int port = process._port;
        int column = 0;
        MGStatus status;
        MXMessage message;
        
        for(int row = 0; row < sliderMatrix.length; ++ row) {
            ArrayList<MGStatus> slider = new ArrayList();

            while (slider.size() < MXStatic.SLIDER_COLUMN_COUNT) {
                status = new MGStatus(process._port, MGStatus.TYPE_SLIDER, row, column);
                slider.add(status);
                column ++;
            }
            sliderMatrix[row] = slider;
        }
        column = 0;
        for (int row = 0; row < circleMatrix.length; ++ row) {
            ArrayList<MGStatus> circle = new ArrayList();
            while (circle.size() < MXStatic.SLIDER_COLUMN_COUNT) {
                status = new MGStatus(process._port, MGStatus.TYPE_CIRCLE, row, column);
                circle.add(status);
                column ++;
            }
            circleMatrix[row] = circle;
        }

        _matrixSliderStatus = sliderMatrix;
        _matrixCircleStatus = circleMatrix;
        _cachedControlChange = null;
        initDrumMinMidleMax();
    }

    private void initSoundModule() {
        initZero();

        MX32MixerProcess process = _process;

        int port = process._port;
        MGStatus status;
        MXMessage message;
        
        int[] cclist = {
            114, 18, 19, 16, 17, 91, 79, 72
        };
        int[] cclist2 = {
            112, 74, 71, 76, 77, 93, 73, 75
        };
        
        MX32MixerData data =  _process._data;

        for(int row = 0; row < MXStatic.SLIDER_ROW_COUNT; ++ row) {
            for(int col = 0; col <MXStatic.SLIDER_COLUMN_COUNT; ++ col) {
                status = data.getSliderStatus(row, col);
                if (col < cclist.length) {
                    message = MXMessageFactory.fromShortMessage(port, null, MXMidi.COMMAND_CONTROLCHANGE + row, cclist[col], 128 -1);
                    status.setMonitoringTarget(message.toDText(), 0, message.getGate(), message.getValue());
                }else {
                    message = MXMessageFactory.createDummy();
                    status.setMonitoringTarget(message.toDText(), 0, message.getGate(), message.getValue());
                }
            }
        }

        for(int row = 0; row < MXStatic.CIRCLE_ROW_COUNT; ++ row) {
            for(int col = 0; col <MXStatic.SLIDER_COLUMN_COUNT; ++ col) {
                status = data.getCircleStatus(row, col);
                if (col < cclist2.length) {
                    message = MXMessageFactory.fromShortMessage(port, null, MXMidi.COMMAND_CONTROLCHANGE + row, cclist2[col], 128 -1);
                    status.setMonitoringTarget(message.toDText(), 0, message.getGate(), message.getValue());
                }else {
                    message = MXMessageFactory.createDummy();
                    status.setMonitoringTarget(message.toDText(), 0, message.getGate(), message.getValue());
                }
            }
        }
 
        _cachedControlChange = null;
        initDrumMinMidleMax();
    }

    private void initGMTone() {
        initZero();

        MX32MixerProcess process = _process;

        int port = process._port;
        MXMessage message = null;
        MGStatus status = null;
        MX32MixerData data =  _process._data;

        for(int row = 0; row < MXStatic.SLIDER_ROW_COUNT; ++ row) {
            for(int col = 0; col <MXStatic.SLIDER_COLUMN_COUNT; ++ col) {
                status = data.getSliderStatus(row, col);
                if (col >= 16) {
                    String text = "F0h, 7Fh, 0gh, 04h, 01h, #VL, #VH, F7h";

                    MXMessageTemplate template = MXMessageFactory.fromDtext(text, 0);
                    message = template.bind(null, _process._port, 0, 128 * 128-1);
                    status.setMonitoringTarget(text, 0, 0, 0);
                }else {
                    message = MXMessageFactory.fromShortMessage(port, null, MXMidi.COMMAND_CONTROLCHANGE + col, MXMidi.DATA1_CC_EXPRESSION, 128 -1);
                    status.setMonitoringTarget(message.toDText(), col, message.getGate(), message.getValue());
                }
            }
        }
        
        int[] ccCode = new int[] { 
            MXMidi.DATA1_CC_SOUND_ATTACKTIME,
            MXMidi.DATA1_CC_SOUND_DECAYTIME, 
            MXMidi.DATA1_CC_SOUND_RELEASETIME, 
            MXMidi.DATA1_CC_SOUND_BLIGHTNESS,
        };
        for(int row = 0; row < MXStatic.CIRCLE_ROW_COUNT; ++ row) {
            for(int col = 0; col <MXStatic.SLIDER_COLUMN_COUNT; ++ col) {
                status = data.getSliderStatus(row, col);
                if (col >= 16) {
                    String text = "F0h, 7Fh, 0gh, 04h, 01h, #VL, #VH, F7h";
                    MXMessageTemplate template = MXMessageFactory.fromDtext(text, 0);
                    message = template.bind(null, _process._port, 0, 128 * 128-1);
                    status.setMonitoringTarget(text, 0, 0, 128 * 128 + 1);
                }else {
                    message = MXMessageFactory.fromShortMessage(port, null, MXMidi.COMMAND_CONTROLCHANGE + col, ccCode[row], 64);
                    status.setMonitoringTarget(message.toDText(), col, message.getGate(), message.getValue());
                }
            }
        }
        
        int[] proglist = {
            -1, 0, 2, 5, 8, 10, 13, 27, 36, 40, 56, 65, 72, 82, 96, 106, 120
        };

        for(int row = 0; row < MXStatic.DRUM_ROW_COUNT; ++ row) {
            for(int col = 0; col <MXStatic.SLIDER_COLUMN_COUNT; ++ col) {
                status = data.getSliderStatus(row, col);
                int prog = proglist[col];
                switch(row) {
                    case 0:
                        if (prog < 0) {
                            message = MXMessageFactory.fromShortMessage(process._port, null, MXMidi.COMMAND_CONTROLCHANGE + 0, MXMidi.DATA1_CC_MODULATION, 127);
                        }else {
                            message = MXMessageFactory.fromShortMessage(process._port, null, MXMidi.COMMAND_PROGRAMCHANGE + 0, prog, 0);
                        }
                        break;
                    case 1:
                        if (prog < 0) {
                            message = MXMessageFactory.fromShortMessage(process._port, null, MXMidi.COMMAND_CONTROLCHANGE + 0, MXMidi.DATA1_CC_MODULATION, 64);
                        }else {
                            message = MXMessageFactory.fromShortMessage(process._port, null, MXMidi.COMMAND_PROGRAMCHANGE + 1, prog, 0);
                        }
                        break;
                    case 2:
                        if (prog < 0) {
                            message = MXMessageFactory.fromShortMessage(process._port, null, MXMidi.COMMAND_CONTROLCHANGE + 0, MXMidi.DATA1_CC_MODULATION, 0);
                        }else {
                            message = MXMessageFactory.fromShortMessage(process._port, null, MXMidi.COMMAND_PROGRAMCHANGE + 2, prog, 0);
                        }
                        break;
                }
                status.setMonitoringTarget(message.toDText(), message.getChannel(), message.getGate(), message.getValue());
            }
        }
    }
    
    public void fillMaxOfSlider(MGStatus status, int column) {
        MGStatus sliderStatus = _matrixSliderStatus[0].get(column);
        
        int max = sliderStatus.getRangeMax();
        int min = sliderStatus.getRangeMin();
        MXMessage message = sliderStatus.toMXMessage(null);
        
        status.setMonitoringTarget(message.toDText(), message.getChannel(), message.getGate(), max);
        status.setSwitchType(MGStatus.SWITCH_TYPE_ON); // 1回のみで
        status.setRangeMin(max);
        status.setRangeMax(max);
        status.setSwitchOutOnTypeOfValue(MGStatus.SWITCH_OUT_ON_VALUE_FIXED);
        status.setSwitchOutOnValueFixed(max);
    }

    public void fillMiddleOfSlider(MGStatus status, int column) {
        MGStatus sliderStatus = _matrixSliderStatus[0].get(column);
        
        int max = sliderStatus.getRangeMax();
        int min = sliderStatus.getRangeMin();
        if (((max - min) % 2) != 0) {
            max ++;
        }
        int middle = (max + min) / 2;
        MXMessage message = sliderStatus.toMXMessage(null);
        
        status.setMonitoringTarget(message.toDText(), message.getChannel(), message.getGate(), max);
        status.setSwitchType(MGStatus.SWITCH_TYPE_ON); // 1回のみで
        status.setRangeMin(middle);
        status.setRangeMax(middle);
        status.setSwitchOutOnTypeOfValue(MGStatus.SWITCH_OUT_ON_VALUE_FIXED);
        status.setSwitchOutOnValueFixed(middle);
    }
    
    public void fillMinOfSlider(MGStatus status, int column) {
        MGStatus sliderStatus = _matrixSliderStatus[0].get(column);
        
        int max = sliderStatus.getRangeMax();
        int min = sliderStatus.getRangeMin();
        MXMessage message = sliderStatus.toMXMessage(null);
        
        status.setMonitoringTarget(message.toDText(), message.getChannel(), message.getGate(), max);
        status.setSwitchType(MGStatus.SWITCH_TYPE_ON); // 1回のみで
        status.setRangeMin(min);
        status.setRangeMax(min);
        status.setSwitchOutOnTypeOfValue(MGStatus.SWITCH_OUT_ON_VALUE_FIXED);
        status.setSwitchOutOnValueFixed(min);
    }
    
    public void initDrumMinMidleMax() {
        MX32MixerProcess process = _process;

        ArrayList<MGStatus>[] sliderMatrix = _matrixSliderStatus;
        ArrayList<MGStatus>[] padMatrix = new ArrayList[MXStatic.DRUM_ROW_COUNT];

        padMatrix[0] = new ArrayList();
        padMatrix[1] = new ArrayList();
        padMatrix[2] = new ArrayList();

        int column = 0;
        while (padMatrix[0].size() < MXStatic.SLIDER_COLUMN_COUNT) {
            MGStatus status;
            status = new MGStatus(process._port, MGStatus.TYPE_DRUMPAD, 0, column);
            fillMaxOfSlider(status, column);
            padMatrix[0].add(status);

            status = new MGStatus(process._port, MGStatus.TYPE_DRUMPAD, 1, column);
            fillMiddleOfSlider(status, column);
            padMatrix[1].add(status);

            status = new MGStatus(process._port, MGStatus.TYPE_DRUMPAD, 2, column);
            fillMinOfSlider(status, column);
            padMatrix[2].add(status);
            column ++;
        }
        
        _matrixDrumStatus = padMatrix;
    }
    
    public  MGSlider getSlider(int row, int column) {
        if (_matrixSliderComponent == null) {
            return null;
        }
        if (row >= MXStatic.SLIDER_ROW_COUNT || column >= MXStatic.SLIDER_COLUMN_COUNT) {
            throw new IllegalArgumentException("row " + row + " , column = " + column);
        }
        return _matrixSliderComponent[row].get(column);
    }

    public  MGCircle getCircle(int row, int column) {
        if (_matrixCircleComponent == null) {
            return null;
        }
        if (row >= MXStatic.CIRCLE_ROW_COUNT || column >= MXStatic.SLIDER_COLUMN_COUNT) {
            throw new IllegalArgumentException("row " + row + " , column = " + column);
        }
        return _matrixCircleComponent[row].get(column);
    }

    public  MGPad getDrumPad(int row, int column) {
        if (_matrixDrumComponent == null) {
            return null;
        }
        if (row >= MXStatic.DRUM_ROW_COUNT || column >= MXStatic.SLIDER_COLUMN_COUNT) {
            throw new IllegalArgumentException("row " + row + " , column = " + column);
        }
        return _matrixDrumComponent[row].get(column);
    }

    public  MGStatus getSliderStatus(int row, int column) {
        return _matrixSliderStatus[row].get(column);
    }

    public  MGStatus getCircleStatus(int row, int column) {
        return _matrixCircleStatus[row].get(column);
    }

    public  MGStatus getDrumPadStatus(int row, int column) {
        return _matrixDrumStatus[row].get(column);
    }

    public  void  setSliderStatus(int row, int column, MGStatus status) {
        _matrixSliderStatus[row].set(column, status);
    }

    public  void  setCircleStatus(int row, int column, MGStatus status) {
        _matrixCircleStatus[row].set(column, status);
    }

    public  void  setDrumPadStatus(int row, int column, MGStatus status) {
        _matrixDrumStatus[row].set(column, status);
    }

    public void setEveryComponents(ArrayList<MGSlider>[] slider, ArrayList<MGCircle>[] circle, ArrayList<MGPad>[] drum) {
        _matrixSliderComponent = slider; 
        _matrixCircleComponent = circle;
        _matrixDrumComponent = drum;
        _cachedControlChange = null;
    }
}
