/*
 * Copyright 2023 Syntarou YOSHIDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.synthtarou.midimixer.mx30controller;

import java.awt.Color;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.util.ArrayList;
import javax.swing.SwingUtilities;
import jp.synthtarou.midimixer.libs.MXDebugLines;
import jp.synthtarou.midimixer.libs.MXUtil;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXTraceNumber;
import jp.synthtarou.midimixer.libs.swing.MXFocusAble;
import jp.synthtarou.midimixer.libs.swing.MXFocusGroupElement;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MGCircle extends javax.swing.JPanel implements MXFocusAble, MouseWheelListener {
    private static final MXDebugLines _debug = new MXDebugLines(MGCircle.class);

    MX32MixerProcess _process;
    int _row, _column;
    Color foreground = null;
    boolean _disconnectMoment = false;
    
    public MGStatus getStatus() {
        if (_process == null) return null;
        return _process._data.getCircleStatus(_row, _column);
    }
    
    public void setStatus(MGStatus status) {
        _process._data.setCircleStatus(_row, _column, status);
    }

    public void setInverted(boolean invert) {
        jCircleValue.setInverted(invert);
    }

    public MGCircle(MX32MixerProcess process, int row, int column) {
        _row = row;
        _column = column;
        _process = process;

        initComponents();

        updateUI();
        addMouseWheelListener(this);
    }
    
    public void updateUI() {
        MGStatus status = getStatus();
        if (status != null) {
            status.fixRangedValue();
            jCircleValue.setRange(status.getRangeMin(), status.getRangeMax());
            jCircleValue.setInverted(status.isUiValueInvert());
            jCircleValue.setValue(status.getValue());

            if (status.getName() == null || status.getName().length() == 0) {
                MXMessage message = status.toMXMessage(new MXTraceNumber());
                if (message == null) {
                    jLabel1.setText("?");
                }else {
                    jLabel1.setText(message.toShortString());
                }
            }else {
                jLabel1.setText(status.getName());
            }
            focusStatusChanged(false);
        }
        super.updateUI();
    }

    ArrayList<MouseListener> backupedListener = new ArrayList();
    
    public void setValueChangeable(boolean usable) {
        if (usable == false) {            
            for (MouseListener l : jCircleValue.getMouseListeners()) {
                if (l instanceof MXFocusGroupElement.Listen) {
                    continue;
                }else {
                    backupedListener.add(l);
                }
            }
            for (MouseListener l : backupedListener) {
                jCircleValue.removeMouseListener(l);
            }
        }else {
            for (MouseListener l : backupedListener) {
                jCircleValue.addMouseListener(l);
            }
            backupedListener.clear();
        }
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">                          
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        jCircleValue = new CurvedSlider(45);
        jLabel1 = new javax.swing.JLabel();

        setLayout(new java.awt.GridBagLayout());

        //jCircleValue.setOrientation(javax.swing.JSlider.VERTICAL);
        jCircleValue.addChangeListener(new javax.swing.event.ChangeListener() {
            public void stateChanged(javax.swing.event.ChangeEvent evt) {
                jCircleValueStateChanged(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weighty = 1.0;
        add(jCircleValue, gridBagConstraints);

        jLabel1.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        jLabel1.setText("-");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        add(jLabel1, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.weightx = 1.0;
    }// </editor-fold>                        

    boolean _ignoreEvent = false;

    private void jCircleValueStateChanged(javax.swing.event.ChangeEvent evt) {                                          
        int newValue = jCircleValue.getValue();
        if (getStatus().getValue() == newValue) {
            return;
        }
        if (_ignoreEvent) {
            return;
        }
        MXTraceNumber track = new MXTraceNumber();
        _process.catchedValue(getStatus(), track, newValue, null);
    }                                         

    MXTraceNumber _trackNumer;
    
    public void changeUIOnly(MXTraceNumber traceNumber, int newValue) {
        if (jCircleValue.getValue() != newValue) {
            getStatus().setValue(newValue);

            if (SwingUtilities.isEventDispatchThread() == false) {
                //new Throwable().printStackTrace();
                SwingUtilities.invokeLater(new Runnable() {
                    public void run() {
                        synchronized(this) {
                            _ignoreEvent = true;
                            jCircleValue.setValue(newValue);
                            _ignoreEvent = false;
                        }
                    }
                });
            }else {
                synchronized(this) {
                    _ignoreEvent = true;
                    jCircleValue.setValue(newValue);
                    _ignoreEvent = false;
                }
            }
        }
    }
    
    // Variables declaration - do not modify                     
    private javax.swing.JLabel jLabel1;
    private CurvedSlider jCircleValue;
    // End of variables declaration                   

    @Override
    public void focusedMousePressed(MouseEvent e) {
        if (SwingUtilities.isRightMouseButton(e) || _process._parent._editingControl) {
            _process._parent.enterEditMode(false);
            focusStatusChanged(false);
            editContoller();
        }
    }

    @Override
    public void focusStatusChanged(boolean flag) {
        if (flag) {
            _process._parent.showTextForFocus(MGStatus.TYPE_CIRCLE, _process._port, _row, _column);
        }       
    }

    @Override
    public void focusChangingValue() {
        _process._parent.showTextForFocus(MGStatus.TYPE_CIRCLE, _process._port, _row, _column);
    }

    public void increment(MXTraceNumber traceNumebr) {
        MGStatus status = getStatus();
        int newValue = status.getValue()+ 1;
        if (status.isUiValueInvert()) {
            newValue = status.getValue() - 1;
        }
        if (newValue > status.getRangeMax()) {
            newValue = status.getRangeMax();
        }
        if (newValue < status.getRangeMin()) {
            newValue = status.getRangeMin();
        }
        if (newValue != status.getValue()) {
            _process.catchedValue(status, traceNumebr, newValue, null);
        }
    }
    
    public void decriment(MXTraceNumber traceNumber) {
        MGStatus status = getStatus();
        int newValue = status.getValue() - 1;
        if (status.isUiValueInvert()) {
            newValue = status.getValue() + 1;
        }
        if (newValue > status.getRangeMax()) {
            newValue = status.getRangeMax();
        }
        if (newValue < status.getRangeMin()) {
            newValue = status.getRangeMin();
        }
        if (newValue != status.getValue()) {
            _process.catchedValue(status, traceNumber, newValue, null);
        }
    }

    public void editContoller() { 
        _process._parent.enterEditMode(false);
        MGStatusConfig config = new MGStatusConfig(_process, getStatus());
        MXUtil.showAsDialog(this, config, "Enter Edit Circle {row:" + _row + ", column:" + _column + "}");
        if (config._okOption) {
            setStatus(config._status);
            jLabel1.setText(config._status.getName());
            _process.notifyCacheBroken();
            updateUI();
        }
    }

    @Override
    public void mouseWheelMoved(MouseWheelEvent e) {
        int d = e.getUnitsToScroll();
        if (d > 0) {
            this.decriment(new MXTraceNumber());
        }else {
            this.increment(new MXTraceNumber());
        }
    }
}
