/*
 * Copyright 2023 Syntarou YOSHIDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.synthtarou.midimixer.mx10input;

import javax.swing.JComponent;
import javax.swing.JPanel;
import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXReceiver;
import jp.synthtarou.midimixer.libs.settings.MXSetting;
import jp.synthtarou.midimixer.libs.settings.MXSettingTarget;
import jp.synthtarou.midimixer.mx12masterkeys.MX12Process;
import jp.synthtarou.midimixer.mx20patchbay.MX20Process;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MX10Process extends MXReceiver implements MXSettingTarget {
    MX10Data _data;
    MX10View _view;

    MX20Process _patchBay;
    MX12Process _masterKey;
    MXSetting _setting;

    public MX10Process() {
        _data = new MX10Data();
        _patchBay = new MX20Process(true);
        _view = new MX10View(this);
        _setting = new MXSetting("InputSkip");
        _setting.setTarget(this);
        _patchBay.setNextReceiver(new MXReceiver()  {
            @Override
            public String getReceiverName() {
                return "dummy";
            }

            @Override
            public JComponent getReceiverView() {
                return null;
            }

            @Override
            public void processMXMessage(MXMessage message) {
                if (MX10Process.this.isUsingThisRecipe()) {
                    if (_data.isMessageToSkip(message)) {
                        return;
                    }
                }
                MX10Process.this.sendToNext(message);
            }
            
        });
    }
    
    public void readSettings() {
        _setting.readFile();
        _patchBay.readSettings();
    }
    
    public void setMasterKeyProcess(MX12Process master) {
        _masterKey = master;
    }

    public void setUseMesssageFilter(boolean log) {
        setUsingThisRecipe(log);
    }
    
    public boolean isUseMessageFilter() {
        return isUsingThisRecipe();
    }
    
    public void setUsePortFowarding(boolean pf) {
        _patchBay.setUsingThisRecipe(pf);
    }
    
    public boolean isUsePortFowarding() {
        if (_patchBay == null) {
            return true;
        }
        return _patchBay.isUsingThisRecipe();
    }
    
    @Override
    public void processMXMessage(MXMessage message) {
        _patchBay.processMXMessage(message);
    }

    @Override
    public String getReceiverName() {
        return "Input Receiver";
    }

    @Override
    public JPanel getReceiverView() {
        return _view;
    }

    @Override
    public void prepareSettingFields(MXSetting setting) {
        String prefix = "Setting[].";
        for (String text : MX10Data.typeNames) {
            setting.register(prefix + text);
        }
    }

    @Override
    public void afterReadSettingFile(MXSetting setting) {
        for (int port = 0; port < MXStatic.TOTAL_PORT_COUNT; ++ port) {
            String prefix = "Setting[" + port + "].";
            StringBuffer str = new StringBuffer();
            for (int j = 0; j <_data.countOfTypes(); ++ j) {
                String name = _data.typeNames[j];
                boolean set = setting.getSettingAsBoolean(prefix + name, false);
                _data.setSkip(port, j, set);
            }
        }
        _view.resetTableModel();
    }

    @Override
    public void beforeWriteSettingFile(MXSetting setting) {
        for (int port = 0; port < MXStatic.TOTAL_PORT_COUNT; ++ port) {
            String prefix = "Setting[" + port + "].";
            StringBuffer str = new StringBuffer();
            for (int j = 0; j <_data.countOfTypes(); ++ j) {
                boolean set = _data.isSkip(port, j);
                String name = _data.typeNames[j];
                setting.setSetting(prefix + name, set);
            }
        }
    }
}
