/*
 * Copyright 2023 Syntarou YOSHIDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.synthtarou.midimixer.mx00playlist;

import java.io.File;
import java.util.List;
import javax.swing.DefaultListModel;
import javax.swing.JComponent;
import jp.synthtarou.midimixer.libs.midi.MXReceiver;
import jp.synthtarou.midimixer.libs.MXDebugLines;
import jp.synthtarou.midimixer.libs.settings.MXSetting;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.settings.MXSettingNode;
import jp.synthtarou.midimixer.libs.settings.MXSettingTarget;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MX00Process extends MXReceiver implements MXSettingTarget {
    private static final MXDebugLines _debug = new MXDebugLines(MX00Process.class);

    public MX00Process() {
        _playListModel = new DefaultListModel();
        _setting = new MXSetting("PlayList");
        _setting.setTarget(this);

        _view = new MX00View(this);
    }

    public void readSettings() {
        _setting.readFile();

        if (_playListModel.size() == 0) {
            _playListModel.addElement(new FileWithId(new File("SynthTAROU000.mid")));
            _playListModel.addElement(new FileWithId(new File("SynthTAROU001.mid")));
            _playListModel.addElement(new FileWithId(new File("SynthTAROU002.mid")));
        }

        _view.settingUpdated();
    }
    
    MX00View _view;
    FileWithId _file;
    DefaultListModel _playListModel;
    MXSetting _setting;
    boolean _playAsChained;
    boolean _playAsRepeated;
    
    @Override
    public String getReceiverName() {
        return "SMF Player";
    }

    @Override
    public JComponent getReceiverView() {
        return _view;
    }
    @Override
    public void prepareSettingFields(jp.synthtarou.midimixer.libs.settings.MXSetting config) {
        config.register("playAsLooped");
        config.register("playAsChained");
        config.register("song[]");
    }

    @Override
    public void afterReadSettingFile(jp.synthtarou.midimixer.libs.settings.MXSetting config) {
        _playListModel.clear();

        _playAsRepeated = config.getSettingAsBoolean("playAsLooped", false);
        _playAsChained = config.getSettingAsBoolean("playAsChained", false);
        
        List<MXSettingNode> nodeList  = _setting.findByPath("song[]");
        int min = 100000;
        int max = -1;
        for (MXSettingNode node : nodeList) {
            String name = node.getName();
            try {
                int x = Integer.parseInt(name);
                if (x < min) min = x;
                if (x > max) max = x;
            }catch(NumberFormatException e) {
                _debug.printStackTrace(e);
            }
        }
        for (int x = min; x <= max; ++ x) {
            String value = config.getSetting("song[" + x + "]");
            if (value != null && value.length() > 0) {
                FileWithId file = new FileWithId(new File(value));
                _playListModel.addElement(file);
            }
        }
    }

    @Override
    public void beforeWriteSettingFile(MXSetting config) {
        config.clearValue();
        config.setSetting("playAsLooped", _playAsRepeated);
        config.setSetting("playAsChained", _playAsChained);

        for (int i = 0; i < _playListModel.getSize(); ++ i) {
            FileWithId f = (FileWithId)_playListModel.get(i);
            config.setSetting("song[" + (i + 1) + "]", f._file.getPath());
        }
    }

    @Override
    public void processMXMessage(MXMessage message) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }
}
