/*
 * Copyright 2023 Syntarou YOSHIDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.synthtarou.midimixer.libs.midi.smf;


import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import javax.sound.midi.InvalidMidiDataException;
import jp.synthtarou.midimixer.libs.midi.MXMidi;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class SMFPlayer {

    /**
     * @return the _parser
     */
    public SMFParser getParser() {
        return _parser;
    }

    /**
     * @param _parser the _parser to set
     */
    public void setParser(SMFParser _parser) {
        this._parser = _parser;
    }

    /**
     * @return the _info
     */
    public SMFFileInfo getInfo() {
        return _info;
    }
    
    public void open(File file) throws IOException {
        setParser(new SMFParser(file));
        _info = getParser().getFileInfo();
    }

    // MIDI 繝�繝ｼ繧ｿ
    private SMFParser _parser;
    private SMFFileInfo _info;
    private int _position;

    public SMFPlayer(SMFParser listMessage) {
        _parser = listMessage;
        _info = _parser.getFileInfo();
    }

    public SMFPlayer(File file) throws IOException {
        _parser = new SMFParser(file);
        _info = _parser.getFileInfo();
    }

    protected void playWithSMPTE(int smpteFormat, int fileResolution) throws InvalidMidiDataException {
        double frameRate = smpteFormat;
        if (smpteFormat == 29) {
            frameRate = 29.97;
        }
        
        double tickPerMilliSecond = frameRate * fileResolution / 1000;
        
        int pos = _position;
        System.out.println("playWithSMPTE from " + pos);
        boolean didProgChange = false;
        ArrayList<SMFMessage> list = getParser().getMessageList().listAll();

        while (pos < list.size()) {
            SMFMessage smf = list.get(pos);
            int command = smf.getStatus() & 0xf0;
            if (command == MXMidi.COMMAND_PROGRAMCHANGE) {
                callback.playingSMFNote(smf);
                didProgChange = true;
                pos ++;
            }else if (command == MXMidi.COMMAND_CONTROLCHANGE && (smf.getData1() == 0 || smf.getData1() == 32)) {
                callback.playingSMFNote(smf);
                didProgChange = true;
                pos ++;
            }else if (smf.getStatus() == 0xff) {
                pos ++;
                continue;
            }else {
                break;
            }
        }
        if (didProgChange) {
            System.out.println("did prog change");
            try {
                Thread.sleep(500);
            }catch(Exception e) {
                
            }
        }

        long stopWatch = System.currentTimeMillis();
        long alreadySpent = (long)(list.get(pos)._tick * tickPerMilliSecond) / 1000;
        stopWatch -= alreadySpent;

        while (pos < list.size() && _continueRun) {
            // 邨碁℃譎る俣 (milliseconds)
            long elapsed = (System.currentTimeMillis() - stopWatch);
            // 邨碁℃譎る俣 (Ticks)
            long elapsedTicks = (long)(elapsed * tickPerMilliSecond);

            //  elapsedTicks 縺瑚ｲ�縺ｮ蝣ｴ蜷医�ｯ繝舌げ縺倶ｽ輔°�ｼ昴Ν繝ｼ繝怜屓驕ｿ
            if (elapsedTicks < 0) {
                break;
            }

            while (list.get(pos).getTick() <= elapsedTicks && _continueRun) {
                _position = pos;
            
                // 蜀咲函縺輔ｌ繧九∋縺肴凾蛻ｻ(AbsoluteTime) 繧帝℃縺弱※縺�繧後�ｰ蜀咲函縺吶ｋ
                SMFMessage currentEvent = list.get(pos++);

                
                // 繧､繝吶Φ繝医�ｮ騾夂衍
                callback.playingSMFNote(currentEvent);
                if (pos >= list.size()) {
                    break;
                }
            }
            synchronized(this) {
                try {
                    if (pos >= list.size()) {
                        break;
                    }
                    long nextTick = list.get(pos).getTick();
                    int nextMillis = (int)(nextTick / tickPerMilliSecond - System.currentTimeMillis());
                    // 1ms 縺ｮ繝�繧｣繝ｬ繧､繧貞�･繧後ｋ
                    if (nextMillis >= 10) {
                        wait(nextMillis - 5);
                    }else {
                        //wait(1);
                    }
                }catch(InterruptedException e) {
                    return;
                }
            }
        }
    }

    protected void playWithTempo() throws InvalidMidiDataException {
        SMFTempoTable tempo = new SMFTempoTable(getParser(), getInfo().getResolution());

        // 縺薙％縺九ｉ譖ｲ縺ｮ蜀咲函繧帝幕蟋九☆繧�

        int pos = _position;
        System.out.println("playWithTempo from " + pos);
        boolean didProgChange = false;
        
        List<SMFMessage> list = getParser().getMessageList().listAll();

        while (pos < list.size()) {
            SMFMessage smf = list.get(pos);
            int command = smf.getStatus() & 0xf0;
            if (command == MXMidi.COMMAND_PROGRAMCHANGE) {
                callback.playingSMFNote(smf);
                didProgChange = true;
                pos ++;
            }else if (command == MXMidi.COMMAND_CONTROLCHANGE && (smf.getData1() == 0 || smf.getData1() == 32)) {
                callback.playingSMFNote(smf);
                didProgChange = true;
                pos ++;
            }else if (smf.getStatus() == 0xff) {
                pos ++;
                continue;
            }else {
                break;
            }
        }
        if (didProgChange) {
            try {
                Thread.sleep(500);
            }catch(Exception e) {
                
            }
        }

        long stopWatch = System.currentTimeMillis();
        long alreadySpent = tempo.TicksToMicroseconds(list.get(pos)._tick) / 1000;
        stopWatch -= alreadySpent;
        while (pos < list.size() && _continueRun) {
            // 邨碁℃譎る俣 (microseconds)
            long elapsed = (System.currentTimeMillis() - stopWatch) * 1000L;
            // 邨碁℃譎る俣 (Ticks)
            long elapsedTicks = tempo.MicrosecondsToTicks(elapsed);
            
            //System.out.println("listtick " + list.get(pos).getTick() + " elapsed " + elapsedTicks +" (" + elapsed);
            //  elapsedTicks 縺瑚ｲ�縺ｮ蝣ｴ蜷医�ｯ繝舌げ縺倶ｽ輔°�ｼ昴Ν繝ｼ繝怜屓驕ｿ
            if (elapsedTicks < 0) {
                break;
            }

            while (list.get(pos).getTick() <= elapsedTicks && _continueRun) {
                _position = pos;

                // 蜀咲函縺輔ｌ繧九∋縺肴凾蛻ｻ(AbsoluteTime) 繧帝℃縺弱※縺�繧後�ｰ蜀咲函縺吶ｋ
                SMFMessage currentEvent = list.get(pos++);
                
                // 繧､繝吶Φ繝医�ｮ騾夂衍
                if (currentEvent.isBinaryMessage() || currentEvent.isMetaMessage()) {
                    callback.playingSMFNote(currentEvent);
                    if (pos >= list.size()) {
                        break;
                    }
                }else if (currentEvent.getStatus() >= 0x80) {
                    callback.playingSMFNote(currentEvent);
                    if (pos >= list.size()) {
                        break;
                    }
                }
            }
            synchronized(this) {
                try {
                    if (pos >= list.size()) {
                        break;
                    }
                    long nextTick = list.get(pos).getTick();
                    long nextMillis = tempo.TicksToMicroseconds(nextTick)/ 1000;
                    long differ = nextMillis - (System.currentTimeMillis() - stopWatch);;
                    // 1ms 縺ｮ繝�繧｣繝ｬ繧､繧貞�･繧後ｋ
                    if (differ >= 10) {
                        wait(differ - 5);
                    }else {
                        //wait(1);
                    }
                }catch(InterruptedException e) {
                    return;
                }
            }
        }
    }

    /**
     * @return the callback
     */
    public SMFPlayerCallback getCallback() {
        return callback;
    }

    /**
     * @param callback the callback to set
     */
    public void setCallback(SMFPlayerCallback callback) {
        this.callback = callback;
    }
    
    boolean _continueRun;
    
    public synchronized void start() throws InvalidMidiDataException {
        stop();
        _continueRun = true;
        try {
            SMFMessage message;
            if (getPosition() == 0) {
                for (int i = 0; i < 16; ++ i) {
                    message = new SMFMessage(0, MXMidi.COMMAND_CONTROLCHANGE + i, MXMidi.DATA1_CC_PEDAL, 0);
                    callback.playingSMFNote(message);
                    message = new SMFMessage(0, MXMidi.COMMAND_CONTROLCHANGE + i, MXMidi.DATA1_CC_ALLNOTEOFF, 127);
                    callback.playingSMFNote(message);
                    message = new SMFMessage(0, MXMidi.COMMAND_CONTROLCHANGE + i, MXMidi.DATA1_CC_EXPRESSION, 127);
                    callback.playingSMFNote(message);
                    message = new SMFMessage(0, MXMidi.COMMAND_CONTROLCHANGE + i, MXMidi.DATA1_CC_CHANNEL_VOLUME, 127);
                    callback.playingSMFNote(message);
                }
            }
            if (getInfo().getSmpteFormat() > 0) {
                playWithSMPTE(getInfo().getSmpteFormat(), getInfo().getResolution());
            }else {
                playWithTempo();
            }
        }catch(Exception e) {
            e.printStackTrace();
        }finally {
            _continueRun = false;
            synchronized(this) {
                notifyAll();
            }
        }
        stop();
    }
    
    public synchronized  void stop() {
        _continueRun = false;
        for (int i = 0; i < 16; ++ i) {
            SMFMessage message;
            message = new SMFMessage(0, MXMidi.COMMAND_CONTROLCHANGE + i, MXMidi.DATA1_CC_PEDAL, 0);
            callback.playingSMFNote(message);
            message = new SMFMessage(0, MXMidi.COMMAND_CONTROLCHANGE + i, MXMidi.DATA1_CC_ALLNOTEOFF, 127);
            callback.playingSMFNote(message);
            message = new SMFMessage(0, MXMidi.COMMAND_CONTROLCHANGE + i, MXMidi.DATA1_CC_ALLSOUNDOFF, 127);
            callback.playingSMFNote(message);
        }
        notifyAll();
    }
    
    public synchronized  boolean isRunning() {
        return _continueRun;
    }

    public int getPosition() {
        return _position;
    }

    public void setPosition(int pos) {
        if (pos > getLength()) {
            throw new IllegalArgumentException("timeing error pos = " + pos + " length " + getLength());
        }
        _position = pos;
    }

    public int getLength() {
        return getParser().getMessageList().size();
    }
    
    private SMFPlayerCallback callback;
}
