package jp.synthtarou.midimixer.libs.midi.port;
 
import java.io.IOException;
import java.util.ArrayList;
import javax.sound.midi.InvalidMidiDataException;
import javax.sound.midi.MidiUnavailableException;
import javax.sound.midi.ShortMessage;
import jdk.nashorn.api.tree.ContinueTree;
import jp.synthtarou.midimixer.MXMain;
import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.libs.MXDebugLines;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXException;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.libs.midi.MXTraceNumber;
import jp.synthtarou.midimixer.libs.midi.smf.SMFMessage;
import jp.synthtarou.midimixer.libs.midi.smf.SMFMessageList;
import jp.synthtarou.midimixer.libs.midi.smf.SMFPlayer;
import jp.synthtarou.midimixer.libs.midi.smf.SMFPlayerCallback;
import jp.synthtarou.midimixer.mx00playlist.FileWithId;
import jp.synthtarou.midimixer.mx70console.ConsoleElement;
 
/**
 *
 * @author YOSHIDA Shintarou
 */
public class MXMIDIInForPlayer extends MXMIDIIn{
    private static final MXDebugLines _debug = new MXDebugLines(MXMIDIInForPlayer.class);
 
    FileWithId _file;
    private Callback _callback;
    
    public static interface Callback {
        public void scoreStart(FileWithId File);                
        public void scoreProgress(int pos, int max);
        public void scoreDone(FileWithId file);
 
        public void setSongName(String name);
        public void updatePianoKeys(int port, int dword);
        public void createPianoControls(int lowNote, int octaveRange, boolean[] aciveChannels, int[] listPrograms, int[] drumProgs);
    }
 
    public MXMIDIInForPlayer() {
        super("<PlayList>", null);
    }
    
    public void setOutPort(int outPort) {
        open();
        _assigned = outPort;
    }

    long whenShow = 0;
    
    public void openFile(FileWithId file) throws IOException, MidiUnavailableException, InvalidMidiDataException {
        int _noteLowest;
        int _noteHighest;
 
        _file = file;
        String fileName = file.toString();
 
        if (_sequencer != null) {
           _sequencer.stop();;
           _sequencer = null;
        }
 
        _sequencer = new SMFPlayer(file._file);
 
        _noteLowest = 200;
        _noteHighest = 0;
 
        if (getCallback() != null) {
            getCallback().setSongName(fileName);
        }
        
        _sequencer.setCallback(new SMFPlayerCallback() {
            @Override
            public void playingSMFNote(SMFMessage smf) {
                try {
                    int port = assigned();
                    
                    long timing = System.currentTimeMillis();
                    if (_callback != null) {
                        if (whenShow + 1000 < timing) {
                            _callback.scoreProgress(_sequencer.getPosition(), _sequencer.getLength());
                            whenShow = timing;
                        }
                    }
                    
                    if (port >= 0) {
                        if (smf.isBinaryMessage()) {
                            startMainPath(0, smf.getBinary());
                        }else {
                            int dword = smf.toDwordMessage();
                            if (_skipAnother) {
                                boolean skip = true;

                                int status = (dword >> 16) & 0xff;
                                int data1 = (dword >> 8) & 0xff;
                                int data2 = (dword) & 0xff;

                                if (_forceSingleChannel >= 0) {
                                    if (status >= 0x80 && status <= 0xef) { //(message.isMessageTypeChannel()
                                        status = status & 0xf0 | _forceSingleChannel;
                                    }
                                }
                                
                                switch(status & 0xf0) {
                                    case MXMidi.COMMAND_NOTEON:
                                    case MXMidi.COMMAND_NOTEOFF:
                                    case MXMidi.COMMAND_PITCHWHEEL:
                                        skip = false;
                                        break;
                                    case MXMidi.COMMAND_CONTROLCHANGE:
                                        if (data1 == MXMidi.DATA1_CC_PEDAL
                                        || data1 == MXMidi.DATA1_CC_MODULATION
                                        || data1 == MXMidi.DATA1_CC_ALLNOTEOFF
                                        || data1 == MXMidi.DATA1_CC_ALLSOUNDOFF) {
                                            skip = false;
                                        }
                                        break;
                                }
                                if (skip) {
                                    return;
                                }
                            }
                            if (getCallback() != null) {
                                getCallback().updatePianoKeys(port, dword);
                            }
                            startMainPath(smf.toDwordMessage(), null);
                        }
                    }else {
                    }
                }catch(Throwable e) {
                    e.printStackTrace();
                }
            }
        });
 
        boolean[] existNoteChannel = new boolean[16];
        int[] programList = new int[16];
        ArrayList<Integer> drums = new ArrayList<Integer>();
        int firstNotePos = -1;
        
        ArrayList<SMFMessage> list = _sequencer.getParser().getMessageList().listAll();
        for (int i = 0; i < list.size(); ++ i) {
            int msg = 0;
            
            if (list.get(i).isBinaryMessage()) {
                continue;
            }
            try {
                msg = list.get(i).toDwordMessage();
            }catch(MXException e) {
                //Not happens
                continue;
            }

            int status = (msg >> 8) >> 8;
            int command = status & 0xf0;
            int ch = status & 0x0f;
            int data1 = (msg >> 8) & 0xff;
            int data2 = msg & 0xff;

            if (command == MXMidi.COMMAND_NOTEON) {
                if (firstNotePos < 0) {
                    firstNotePos = i;
                }
                existNoteChannel[ch] = true;
                if (ch == MXStatic.DRUM_CH && data2 >= 1) {
                    drums.add(data1);
                }else {
                    if (data1 < _noteLowest) _noteLowest = data1;
                    if (data1 > _noteHighest) _noteHighest = data1;
                }
            }
            else if (command == ShortMessage.PROGRAM_CHANGE) {
                if (programList[ch]  < 0) {
                    programList[ch] = data1;
                }
            }
        }
 
        int lo2 = _noteLowest;
        lo2 = lo2 / 12;
        lo2 = lo2 * 12;
 
        int hi2 = lo2;
 
        while(hi2 < _noteHighest) {
            hi2 += 12;
        }
 
        int x = lo2 / 12;
        int width = hi2 / 12;
 
        width -= x;
        x *= 12;
 
        if (width <= 2) {
            if (x >= 12) {
                x -= 12;
                width += 1;
            }
            if (width <= 2) {
                width += 1;
            }
        }
 
        int rows = 0;
 
        int[] drumProgs = new int[drums.size()];
        for (int id = 0; id < drums.size(); ++ id) {
            drumProgs[id] = drums.get(id);
        }
        if (getCallback() != null) {
            getCallback().createPianoControls(lo2, width, existNoteChannel, programList, drumProgs);
        }
        _firstNotePos = firstNotePos;
    }
 
    public static String[] readFileInfo(FileWithId file) {
        int _noteLowest;
        int _noteHighest;

        ArrayList<String> ret = new ArrayList();
        String fileName = file.toString();
 
        try {
            SMFPlayer player = new SMFPlayer(file._file);
            SMFMessageList list = player.getParser().getMessageList();
            ArrayList<SMFMessage> listMessage = list.listAll();

            for (SMFMessage message : listMessage) {
                if (message.getStatus() != 0xff) {
                    continue;
                }

                int type = message.getDataType();
                byte[] data = message.getBinary();
                String text = null;
                try {
                    text = new String(data, "ASCII");
                    text = new String(data);
                    text = new String(data, "SJIS");
                }catch(Exception e) {
                    e.printStackTrace();
                }
                int number = 0;

                switch(type) {
                    case 0:
                        if (data.length >= 2) {                                
                            number = data[0] * 128 + data[1];
                            ret.add("Sequence Number : " + number);
                        }
                        break;
                    case 1:
                       ret.add("Text : " + text);
                       break;
                    case 2:
                        ret.add("Copyright : " + text);
                        break;  
                    case 3:
                        ret.add("Track Name : " + text);
                        break;  
                };
            }
        } catch (Exception e) {
            //e.printStackTrace();
            ret.add("Can't open [" + file.toString() + "]");
            ret.add(e.toString());
        }
        String[] list = new String[ret.size()];
        ret.toArray(list);
        return list;
    }
    
    private SMFPlayer _sequencer = null;
    private Thread _threadSeq = null;
    private boolean _gotBreak = false;
    boolean _skipAnother = false;
    int _forceSingleChannel = -1;
    int _firstNotePos = -1;
    
    public long getLength() {
        return _sequencer.getLength();
    }
    
    public int getFirstNotePos() {
        if (_firstNotePos < 0) {
            ArrayList<SMFMessage> list = _sequencer.getParser().getMessageList().listAll();
            int pos = 0;
            for (SMFMessage smf : list) {
                int command = smf.getStatus(); 
                if ((command & 0xf0) == MXMidi.COMMAND_NOTEON) {
                    _firstNotePos = pos;
                    break;
                }
                pos ++;
            }
            if (_firstNotePos < 0) {
                _firstNotePos = 0;
            }
        }
        return (_firstNotePos >= 0) ? _firstNotePos : 0;
    }
    
    public void setOnlyNotePitchWheel(boolean skipAnother) {
        _skipAnother = skipAnother;
    }
 
    public void setPlayAsSingleChannel(int channel) {
        _forceSingleChannel = channel;
    }
 
    public synchronized void startSequencer() {
        stopSequencer();
 
        _gotBreak = false;
 
        _threadSeq = new Thread(new Runnable() {
            public void run() {
                try {
                    if (getCallback() != null) {
                        getCallback().scoreStart(_file);
                        getCallback().scoreProgress(0, _sequencer.getLength());
                    }
                    _sequencer.start();
 
                    try {
                        _sequencer.stop();
                    }catch(Throwable e) {
                    }
                    if (_gotBreak == false && getCallback() != null) {
                        getCallback().scoreDone(_file);
                        getCallback().scoreProgress(_sequencer.getLength(), _sequencer.getLength());
                    }
                }catch(Throwable e) {
                    e.printStackTrace();
                }
            }
        });
        _threadSeq.start();
    }
 
    public FileWithId getCurrent() {
        return _file;
    }
 
    public synchronized boolean isSequencerPlaying() {
        if (_threadSeq != null && _threadSeq.isAlive()) {
            return true;
        }
        return false;
    }
    
    public synchronized void stopSequencer() {
        if (_threadSeq != null && _threadSeq.isAlive()) {
            try {
                _gotBreak = true;
                _threadSeq.interrupt();
            }catch(Throwable e) {
                e.printStackTrace();
            }
        }
    }

    public void waitSequencerDone() {
        while (_threadSeq != null && _threadSeq.isAlive()) {
            try {
                _gotBreak = true;
                wait(500);
            }catch(InterruptedException e) {
                e.printStackTrace();
            }
        }
    }
    
    public boolean hasGotBreak() {
        return _gotBreak;
    }

    public void setCurrentPos(int pos) {
        _sequencer.setPosition(pos);
    }

    /**
     * @return the _callback
     */
    public Callback getCallback() {
        return _callback;
    }

    /**
     * @param _callback the _callback to set
     */
    public void setCallback(Callback _callback) {
        this._callback = _callback;
    }
}