/*
 * Copyright 2023 Syntarou YOSHIDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.synthtarou.midimixer.mx30controller;

import java.awt.Color;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.util.ArrayList;
import javax.swing.JLabel;
import javax.swing.SwingUtilities;
import jp.synthtarou.midimixer.libs.MXUtil;
import jp.synthtarou.midimixer.libs.MXDebugLines;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXTraceNumber;
import jp.synthtarou.midimixer.libs.swing.MXSliderUIForTablet;
import jp.synthtarou.midimixer.libs.swing.MXFocusAble;
import jp.synthtarou.midimixer.libs.swing.MXFocusGroupElement;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MGSlider extends javax.swing.JPanel implements MXFocusAble, MouseWheelListener {
    private static final MXDebugLines _debug = new MXDebugLines(MGSlider.class);

    MX32MixerProcess _process;
    int _row, _column;

    public MGStatus getStatus() {
        if (_process == null) return null;
        return _process._data.getSliderStatus(_row, _column);
    }
    
    public void setStatus(MGStatus status) {
        _process._data.setSliderStatus(_row, _column, status);
    }

    public MGSlider(MX32MixerProcess process, int row, int column) {
        _row = row;
        _column = column;
        _process = process;
        initComponents();

        updateUI();

        new MXSliderUIForTablet(jSliderValue);
        addMouseWheelListener(this);
    }

    public void updateUI() {
        super.updateUI();
        MGStatus status = getStatus();
        if (status != null) {
            jSliderValue.setMinimum(status.getRangeMin());
            jSliderValue.setMaximum(status.getRangeMax());
            jSliderValue.setInverted(status.isUiValueInvert());
            jSliderValue.setPaintLabels(true);
            jSliderValue.setValue(status.getValue());
            jLabelValue.setText(String.valueOf(status.getValue()));
            status.fixRangedValue();
            if (status.getName() == null || status.getName().length() == 0) {
                MXMessage message = status.toMXMessage(new MXTraceNumber());
                if (message == null) {
                    jLabel1.setText("?");
                }else {
                    jLabel1.setText(message.toShortString());
                }
            }else {
                jLabel1.setText(status.getName());
            }
            focusStatusChanged(false);
        }
    }

    ArrayList<MouseListener> backupedListener = new ArrayList();
    
    public void setValueChangeable(boolean usable) {
        if (usable == false) {            
            for (MouseListener l : jSliderValue.getMouseListeners()) {
                if (l instanceof MXFocusGroupElement.Listen) {
                    continue;
                }else {
                    backupedListener.add(l);
                }
            }
            for (MouseListener l : backupedListener) {
                jSliderValue.removeMouseListener(l);
            }
        }else {
            for (MouseListener l : backupedListener) {
                jSliderValue.addMouseListener(l);
            }
            backupedListener.clear();
        }
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        jSliderValue = new javax.swing.JSlider();
        jLabel1 = new javax.swing.JLabel();
        jLabelValue = new javax.swing.JLabel();

        setLayout(new java.awt.GridBagLayout());

        jSliderValue.setOrientation(javax.swing.JSlider.VERTICAL);
        jSliderValue.addChangeListener(new javax.swing.event.ChangeListener() {
            public void stateChanged(javax.swing.event.ChangeEvent evt) {
                jSliderValueStateChanged(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        add(jSliderValue, gridBagConstraints);

        jLabel1.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        jLabel1.setText("-");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        add(jLabel1, gridBagConstraints);

        jLabelValue.setFont(new java.awt.Font("Yu Gothic UI", 1, 12)); // NOI18N
        jLabelValue.setForeground(MXUtil.mixedColor(Color.red, Color.yellow, 30));
        jLabelValue.setText("0");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        add(jLabelValue, gridBagConstraints);
    }// </editor-fold>//GEN-END:initComponents

    boolean _ignoreEvent = false;
    
    private void jSliderValueStateChanged(javax.swing.event.ChangeEvent evt) {//GEN-FIRST:event_jSliderValueStateChanged
        int newValue = jSliderValue.getValue();
        if (getStatus().getValue() == newValue) {
            return;
        }
        jLabelValue.setText(String.valueOf(newValue));
        if (_ignoreEvent) {
            return;
        }
        MXTraceNumber track = new MXTraceNumber();
        _process.catchedValue(getStatus(), track, newValue, null);
    }//GEN-LAST:event_jSliderValueStateChanged

    public void updateUIOnly(MXTraceNumber traceNumber, int newValue) {
        if (jSliderValue.getValue() != newValue) {
            jLabelValue.setText(String.valueOf(newValue));
            getStatus().setValue(newValue);

            if (SwingUtilities.isEventDispatchThread() == false) {
                //new Throwable().printStackTrace();
                SwingUtilities.invokeLater(new Runnable() {
                    public void run() {
                        synchronized(this) {
                            _ignoreEvent = true;
                            jSliderValue.setValue(newValue);
                            _ignoreEvent = false;
                        }
                    }
                });
            }else {
                synchronized(this) {
                    _ignoreEvent = true;
                    jSliderValue.setValue(newValue);
                    _ignoreEvent = false;
                }
            }
        }
    }

    public JLabel labelFor(int num, int max) {
        String name = "";
        if (max >= 256) {
            int msb = num / 128;
            int cut = msb * 128;
            int lsb = num - cut;
            name = MXUtil.toHexFF(msb) + ":" + MXUtil.toHexFF(lsb);
        }else {
            name = MXUtil.toHexFF(num);
        }
        JLabel label = new JLabel(name);
        //label.setFont(new Font(Font.SANS_SERIF, Font.ITALIC, 10));
        return label;
    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabelValue;
    private javax.swing.JSlider jSliderValue;
    // End of variables declaration//GEN-END:variables

    @Override
    public void focusStatusChanged(boolean flag) {
        if (flag) {
            _process._parent.showTextForFocus(MGStatus.TYPE_SLIDER,  _process._port, _row, _column);
        }
    }

    @Override
    public void focusedMousePressed(MouseEvent e) {
        if (SwingUtilities.isRightMouseButton(e) || _process._parent._editingControl) {
            _process._parent.enterEditMode(false);
            focusStatusChanged(false);
            editContoller();
        }
    }

    @Override
    public void focusChangingValue() {
        _process._parent.showTextForFocus(MGStatus.TYPE_SLIDER,  _process._port, _row, _column);
    }

    public void increment(MXTraceNumber traceNumber) {
        MGStatus status = getStatus();
        int newValue = status.getValue() + 1;
        if (status.isUiValueInvert()) {
            newValue = status.getValue() - 1;
        }
        if (newValue > status.getRangeMax()) {
            newValue = status.getRangeMax();
        }
        if (newValue < status.getRangeMin()) {
            newValue = status.getRangeMin();
        }
        if (newValue != status.getValue()) {
            _process.catchedValue(status, traceNumber, newValue, null);
        }
    }
    
    public void decriment(MXTraceNumber traceNumber) {
        MGStatus status = getStatus();
        int newValue = status.getValue() - 1;
        if (status.isUiValueInvert()) {
            newValue = status.getValue() + 1;
        }
        if (newValue > status.getRangeMax()) {
            newValue = status.getRangeMax();
        }
        if (newValue < status.getRangeMin()) {
            newValue = status.getRangeMin();
        }
        if (newValue != status.getValue()) {
            _process.catchedValue(status, traceNumber, newValue, null);
        }
    }
    
    public void beHomePosition() {
        final MGStatus status = getStatus();
        final int current = status.getValue();
        final int value = status.getValueHome();
        Thread t = new Thread(new Runnable() {
            public void run() {
                try {
                    for (int i = 0; i < 5; ++ i) {
                        int x = current * (5 - i) + value * i;
                        x /= 5;
                        Thread.sleep(70);
                        status.setValue(x);
                        updateUIOnly(null, x);
                        if (x == value) { 
                            break;
                        }
                    }
                }catch(Exception e) {
                    e.printStackTrace();
                }finally {
                    status.setValue(value);
                    updateUIOnly(null, value);
                }
            }
        });
        t.start();
    }

    public void editContoller() {
        _process._parent.enterEditMode(false);
        MGStatus status = (MGStatus)getStatus().clone();
        MGStatusConfig config = new MGStatusConfig(_process, status);
        MXUtil.showAsDialog(this, config, "Enter Edit Slider {row:" + _row + ", column:" + _column + "}");
        if (config._okOption) {
            setStatus(config._status);
            jLabel1.setText(config._status.getName());
            _process.notifyCacheBroken();
            updateUI();
        }
    }

    @Override
    public void mouseWheelMoved(MouseWheelEvent e) {
        int d = e.getUnitsToScroll();
        if (d > 0) {
            this.decriment(new MXTraceNumber());
        }else {
            this.increment(new MXTraceNumber());
        }
    }
}
