/*
 * Copyright 2023 Syntarou YOSHIDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.synthtarou.midimixer.libs.midi.port;

import jp.synthtarou.midimixer.libs.midi.driver.MXMIDIDevice;
import java.io.File;
import jp.synthtarou.midimixer.MXMain;
import jp.synthtarou.midimixer.libs.MXDebugLines;
import jp.synthtarou.midimixer.libs.MultiThreadQueue;
import jp.synthtarou.midimixer.libs.domino.DTextXML;
import jp.synthtarou.midimixer.libs.midi.programlist.database.PDFileManager;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.libs.midi.MXTraceNumber;
import jp.synthtarou.midimixer.libs.midi.programlist.database.PDFile;
import jp.synthtarou.midimixer.mx70console.ConsoleElement;
import org.xml.sax.SAXException;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MXMIDIOut {
    private static final MXDebugLines _debug = new MXDebugLines(MXMIDIOut.class);
    public static final MXMIDIOut OUTPUT_NONE = new MXMIDIOut("none", null);

    private MXMIDIDevice _device;
    private String _name;
    private boolean _isopen;
    int _assigned;
    private File _DXMLFile;
    private MXVisitant16 _visitant16 = new MXVisitant16();

    boolean _useThread = false; //trueだと、処理時間は軽くなるけど、起動とスレッド管理が重い
    MultiThreadQueue<MXMessage> _queue;
    Thread _thread;

    public int assignedPort() {
        return _assigned;
    }
    
    public boolean isOpen() {
        return _isopen;
    }
    
    protected MXMIDIOut(String name, MXMIDIDevice device) {
        this._assigned = -1;
        this._isopen = false;
        this._device = device;
        this._name = name;
        if (_useThread) {
            _thread = new Thread() {
                public void run() {
                    System.out.println("Thread Launcher");
                    while(true) {
                        MXMessage message = _queue.pop();
                        if (message == null) {
                            break;
                        }
                        try {
                            MXMIDIOut.this.processMidiOutInternal(message);
                        }catch(Throwable e) {
                            e.printStackTrace();
                        }
                    }
                }
            };
            _queue= new MultiThreadQueue<MXMessage>();
            _thread.start();
        }
    }

    public String getName() {
        return _name;
    }

    public File getDXMLFile() {
        return _DXMLFile;
    }
    
    public void setDXMLFile(File file) throws SAXException {
        try {
            PDFileManager manager = PDFileManager.getManager();
            if (_DXMLFile != null) {
                manager.unregist(_DXMLFile.getName());
            }
            PDFile parser = DTextXML.fromFile(file);
            _DXMLFile = file;
            manager.register(parser);
        }catch(SAXException e) {
            throw e;
        }catch(Exception e) {
             e.printStackTrace();
        }
    }
    
    public int hashCode() {
        return _name.hashCode();
    }

    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        MXMIDIOut out = (MXMIDIOut) o;
        if (out._name == this._name) {
            return true;
        }
        return false;
    }

    public String toString() {
        return _name;
    }

    long startTime = System.currentTimeMillis();
    
    public long getTimestamp() {
        try {
            return System.currentTimeMillis() - startTime;
        }catch(Exception e) {
            return -1;
        }
    }
   
    public void processMidiOut(MXMessage message) {
        try {
            if (_useThread) {
                _queue.push(message);
            }else {
                processMidiOutInternal(message);
            }
        }catch(Throwable e) {
            e.printStackTrace();
        }
    }

    private void processMidiOutInternal(MXMessage message) {
        if (_isopen == false || _assigned < 0) {
            return;
        }
        
        if (_device.isOpen() == false) {
            _isopen = false;
            return;
        }
        
        MXMain.addInsideOutput(message);
        /*

        if (message.isDataentry()) {
            if (message.getVisitant().getDataentryValue14() == 0) {
                message._trace.printStackTrace();
            }
        }
*/

        try {
            if (MXVisitant.isMesssageHaveVisitant(message)) {
                _visitant16.get(message.getChannel()).updatevisitantChannel(message);
            }

            long timeStamp = getTimestamp();
            long recTime = 0;

            if (message.isMessageTypeChannel()) {
                MXVisitant msgVisitant = message.getVisitant();
                MXVisitant visitant = _visitant16.get(message.getChannel());
                int command = message.getCommand();
                int channel = message.getChannel();
                int gate = message.getGate();
                if (command != MXMidi.COMMAND_PROGRAMCHANGE) {
                    if (msgVisitant != null && msgVisitant.isHavingProgram()) {
                        if (visitant.isHavingProgram() == false || visitant.getProgram() != message.getVisitant().getProgram()) {
                            visitant.mergeNew(message.getVisitant());
                            MXMessage newMessage = MXMessageFactory.fromShortMessage(_assigned, message._traceNumber, MXMidi.COMMAND_PROGRAMCHANGE + channel, visitant.getProgram(),0);
                            newMessage.setVisitant(visitant.getSnapShot());
                            processMidiOutInternal(newMessage);
                        }
                    }
                }
                if (command != MXMidi.COMMAND_CONTROLCHANGE || (gate != MXMidi.DATA1_CC_BANKSELECT && gate != MXMidi.DATA1_CC_BANKSELECT + 32)) {
                    //0と32が連続でくる間は、この処理は実行されないので、半端を送ることもない
                    if (msgVisitant != null && msgVisitant.isHavingBank()) {
                        if (visitant.isHavingBank() == false || visitant.getBankMSB() != msgVisitant.getBankMSB() || visitant.getBankLSB() != msgVisitant.getBankLSB()) {
                            visitant.mergeNew(message.getVisitant());
                            
                            MXMessage newMessage1 = MXMessageFactory.fromShortMessage(_assigned, message._traceNumber, MXMidi.COMMAND_CONTROLCHANGE + channel, MXMidi.DATA1_CC_BANKSELECT ,visitant._bankMSB);
                            MXMessage newMessage2 = MXMessageFactory.fromShortMessage(_assigned, message._traceNumber, MXMidi.COMMAND_CONTROLCHANGE + channel, MXMidi.DATA1_CC_BANKSELECT + 32 ,visitant._bankLSB);

                            processMidiOutInternal(newMessage1);
                            processMidiOutInternal(newMessage2);
                        }
                    }
                }
                
                if (command != MXMidi.COMMAND_CONTROLCHANGE || (gate != MXMidi.DATA1_CC_CHANNEL_VOLUME)) {
                    if (msgVisitant != null && msgVisitant.isHavingVolume()) {
                        if (visitant.isHavingVolume() == false || visitant.getInfoVolume()!= msgVisitant.getInfoVolume()) {
                            visitant.mergeNew(message.getVisitant());
                            
                            MXMessage newMessage = MXMessageFactory.fromShortMessage(_assigned, message._traceNumber, MXMidi.COMMAND_CONTROLCHANGE + channel, MXMidi.DATA1_CC_CHANNEL_VOLUME ,visitant.getInfoVolume());

                            processMidiOutInternal(newMessage);
                        }
                    }
                }

                if (command != MXMidi.COMMAND_CONTROLCHANGE || (gate != MXMidi.DATA1_CC_EXPRESSION)) {
                    if (msgVisitant != null && msgVisitant.isHavingExpression()) {
                        if (visitant.isHavingExpression() == false || visitant.getInfoExpression()!= msgVisitant.getInfoExpression()) {
                            visitant.mergeNew(message.getVisitant());
                            
                            MXMessage newMessage = MXMessageFactory.fromShortMessage(_assigned, message._traceNumber, MXMidi.COMMAND_CONTROLCHANGE + channel, MXMidi.DATA1_CC_EXPRESSION,visitant.getInfoExpression());

                            processMidiOutInternal(newMessage);
                        }
                    }
                }

                if (command != MXMidi.COMMAND_CONTROLCHANGE || (gate != MXMidi.DATA1_CC_PANPOT)) {
                    if (msgVisitant != null && msgVisitant.isHavingExpression()) {
                        if (visitant.isHavingPan()== false || visitant.getInfoPan()!= msgVisitant.getInfoPan()) {
                            visitant.mergeNew(message.getVisitant());
                            
                            MXMessage newMessage = MXMessageFactory.fromShortMessage(_assigned, message._traceNumber, MXMidi.COMMAND_CONTROLCHANGE + channel, MXMidi.DATA1_CC_PANPOT,visitant.getInfoPan());

                            processMidiOutInternal(newMessage);
                        }
                    }
                }
            }
            
            int col = message.getDwordCount();
            if (col == 0) {
                byte[] data = message.getDataBytes();
                try {
                    _device.OutputLongMessage(data);
                    MXMain.addOutsideOutput(new ConsoleElement(message._traceNumber, message.getPort(), data));
                }catch(NullPointerException e) {
                    //
                }catch(Throwable e)  {
                    e.printStackTrace();
                }
            }else if (col > 0) {
                for (int i = 0; i < col; ++ i) {
                    int dword = message.getAsDword(i);
                    try {
                        if (dword == 0) {
                            break;
                        }
                        _device.OutputShortMessage(dword);
                        MXMain.addOutsideOutput(new ConsoleElement(message._traceNumber, message.getPort(), dword));
                    }catch(Throwable e) {
                    }
                }
            }else {
                if (message.isDataentry()) {
                    if (message.getVisitant() == null) {
                        System.out.println("DATAENTRY no set VISITANT");
                        return;
                    }
                    if (message.getVisitant().isHaveDataentryRPN() == false
                     && message.getVisitant().isHaveDataentryNRPN() == false) {
                        System.out.println("DATAENTRY no set RPN / NRPN");
                        return;
                    }
                }
                //Nothing
                System.out.println("nothing to send : " + message);
            }
        }catch(Throwable e) {
            _debug.printStackTrace(e);
        }
        
    }

    public void allNoteOff(MXTraceNumber traceNumber) {
        for (int ch = 0; ch < 16; ++ ch) {
            try {
                MXMessage msg = MXMessageFactory.fromShortMessage(_assigned, traceNumber, MXMidi.COMMAND_CONTROLCHANGE + ch, 123, 127);
                processMidiOutInternal(msg);
            }catch(Throwable e) {
                
            }
        }
    }

    public boolean open() {
        MXMIDIOutManager manager = MXMIDIOutManager.getManager();
        synchronized(manager) {
            manager.clearMIDIOutCache();
            if (_assigned < 0) {
                return false;
            }
            if (_isopen) {
                return true;
            }
            _visitant16 = new MXVisitant16();
            if (_device != null) {
                if (_device.isOpen() == false) {
                    _device.open();
                }
                _isopen = true;
                return true;
            }
            return false;
        }
    }
    
    public void close() {
        MXMIDIOutManager manager = MXMIDIOutManager.getManager();
        synchronized(manager) {
            if (_isopen) {
                allNoteOff(new MXTraceNumber());
                manager.onClose(this);
                _isopen = false;
                if(_name.equals("Gervill")) {
                    
                }else if (_device != null) {
                    _device.close();
                }
            }
        }
    }
}
