/*
 * Copyright 2023 Syntarou YOSHIDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.synthtarou.midimixer.libs.midi.port;

import jp.synthtarou.midimixer.libs.midi.driver.MXMIDIDriver;
import jp.synthtarou.midimixer.libs.midi.driver.MXMIDIDevice;
import java.util.TreeMap;
import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.libs.MXDebugLines;
import jp.synthtarou.midimixer.libs.MXWrapList;
import jp.synthtarou.midimixer.libs.midi.driver.MXMIDIDriverJava;
import jp.synthtarou.midimixer.libs.midi.driver.MXMIDIDriverUWP;
import jp.synthtarou.midimixer.libs.settings.MXSetting;
import jp.synthtarou.midimixer.libs.settings.MXSettingTarget;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MXMIDIInManager implements MXSettingTarget {
    private static final MXDebugLines _debug = new MXDebugLines(MXMIDIInManager.class);
    private static final MXMIDIInManager _instance = new MXMIDIInManager();
    
    public static MXMIDIInManager getManager() {
        return _instance;
    }

    public void reloadDeviceList() {
        //TODO Java not support?
    }

    protected MXMIDIInManager() {
    }
    

    private MXSetting _setting;
    
    public void initWithSetting() {
        if (_setting == null) {
            _setting = new MXSetting("MIDIInput");
            _setting.setTarget(this);
            listAllInput();
            _setting.readFile();

            if (MXMIDIIn.INTERNAL_PLAYER.assigned() < 0) {
                int found = -1;
                for (int i = 0; i < MXStatic.TOTAL_PORT_COUNT; ++ i) {
                    boolean retry = false;
                    for (MXMIDIIn in : listAllInput().valueList()) {
                        if (in.assigned() == i) {
                            retry = true;
                            break;
                        }
                    }
                    if (!retry) {
                        found = i;
                        break;
                    }
                }
                if (found >= 0) {
                    reserveInput(MXMIDIIn.INTERNAL_PLAYER, found);
                    MXMIDIIn.INTERNAL_PLAYER.open();
                }
            }
        }
    }

    protected MXWrapList<MXMIDIIn> _listAllInput;
    protected MXWrapList<MXMIDIIn> _listUsingInput;
    protected MXMIDIIn[] _cache;

    public synchronized MXWrapList<MXMIDIIn> listAllInput() {
        if (_listAllInput != null) {
            return _listAllInput;
        }

        MXWrapList<MXMIDIIn> temp = new MXWrapList<MXMIDIIn>();
        
        MXMIDIDriver java = MXMIDIDriverJava._instance;
        
        MXMIDIIn sequencer = MXMIDIIn.INTERNAL_PLAYER;
        temp.addNameAndValue(sequencer.getName(), sequencer);

        for (int i = 0; i < java.InputDevicesRoomSize(); i++) {
            MXMIDIDevice device = new MXMIDIDevice(java, true, i);
            try {
                String name = device.getName();
                if (name.equals("Real Time Sequencer")) {
                    continue;
                }
                temp.addNameAndValue(name, new MXMIDIIn(name, device));
            } catch (Exception e) {
            }
        }
        
        MXMIDIDriver uwp = MXMIDIDriverUWP._instance;
        for (int i = 0; i < uwp.InputDevicesRoomSize(); i++) {
            MXMIDIDevice device = new MXMIDIDevice(uwp, true, i);
            String name = device.getName();
            if (temp.indexOfName(name) >= 0) {
                continue;
            }
            name = name +" (UWP)";
            temp.addNameAndValue(name, new MXMIDIIn(name, device));
        }

        _listAllInput = temp;
        return _listAllInput;
    }

    public MXMIDIIn findMIDIInput(String deviceName) {
        MXWrapList<MXMIDIIn> model = listAllInput();
        return model.valueOfName(deviceName);
    }

    public MXMIDIIn findMIDIInput(int assigned) {
        listSelectedInput();
        if (assigned >= 0) {
            return _cache[assigned];
        }else {
            return null;
        }
    }

    synchronized void onClose(MXMIDIIn input) {
        clearMIDIInCache();
    }
    
    protected synchronized void clearMIDIInCache() {
        _listUsingInput = null;
        _cache = null;        
    }

    public synchronized MXWrapList<MXMIDIIn>listSelectedInput() {
        if (_listUsingInput != null) {
            return _listUsingInput;
        }
        _cache = new MXMIDIIn[MXStatic.TOTAL_PORT_COUNT];
        MXWrapList<MXMIDIIn> newInput = new MXWrapList();
        for (MXMIDIIn midi : listAllInput().valueList()) {
            if (midi.assigned() >= 0) {
                newInput.addNameAndValue(midi.toString(), midi);
                _cache[midi.assigned()] = midi;
            }
        }
        _listUsingInput = newInput;
        return newInput;
    }

    public synchronized void closeAll() {
        for( MXMIDIIn input : listSelectedInput().valueList()) {
            input.close();
        }
    }
    
    public synchronized boolean reserveInput(MXMIDIIn input, int assignnew) {
        MXWrapList<MXMIDIIn> list = listAllInput();
        if (input.assigned() >= 0) {
            if (input.assigned() == assignnew) {
                return true;
            }
        }
        if (assignnew < 0) {
            input.close();
            clearMIDIInCache();
            input._assigned = assignnew;
            return true;
        }else {
            for (int i = 0; i < list.size(); ++ i) {
                MXMIDIIn x = list.valueOfIndex(i);
                if (x.assigned() == assignnew) {
                    x._assigned = -1;
                    x.close();
                }
            }
            clearMIDIInCache();
            input._assigned = assignnew;
            return true;
        }
    }

    @Override
    public void prepareSettingFields(MXSetting setting) {
        setting.register("device[].name");
        setting.register("device[].open");
        setting.register("device[].toMaster");
    }

    @Override
    public void afterReadSettingFile(MXSetting setting) {
        TreeMap<String, MXMIDIIn> dummy = new TreeMap();
        
        for (int x = 0; x < MXStatic.TOTAL_PORT_COUNT; ++ x) {
            String deviceName = setting.getSetting("device[" + x + "].name");
            String deviceOpen = setting.getSetting("device[" + x + "].open");
            String deviceMaster = setting.getSetting("device[" + x + "].toMaster");
            
            if (deviceName == null) {
                continue;
            }
            if (deviceOpen == null) {
                deviceOpen = "0";
            }
            
            MXWrapList<MXMIDIIn> detected = listAllInput();
            MXMIDIIn in = detected.valueOfName(deviceName);
            if (in != null) {
                reserveInput(in, x);
                if (deviceOpen.equals("1")) {
                    in.open();
                }
            }else {
                in  = new MXMIDIIn(deviceName, null);
                reserveInput(in, x);
                dummy.put(deviceName, in);
            }
            in.setMasterList(deviceMaster);
        }
        for (MXMIDIIn in : dummy.values()) {
            _listAllInput.addNameAndValue(in.getName(),in);
        }
        clearMIDIInCache();
    }

    @Override
    public void beforeWriteSettingFile(MXSetting setting) {
        MXWrapList<MXMIDIIn> all = listAllInput();
        for (MXMIDIIn e : all.valueList()) {
            int x = e.assigned();
            if (x >= 0) {
                setting.setSetting("device[" + x + "].name", e.getName());
                setting.setSetting("device[" + x + "].open", e.isOpen() ? "1" : "0");
                setting.setSetting("device[" + x + "].toMaster", e.getMasterList());
            }
        }
    }
}
