/*
 * Copyright 2023 Syntarou YOSHIDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.synthtarou.midimixer.libs.midi.port;

import jp.synthtarou.midimixer.libs.midi.driver.MXMIDIDevice;
import jp.synthtarou.midimixer.libs.midi.driver.MXMIDIDriverReceiver;
import java.util.ArrayList;
import jp.synthtarou.midimixer.MXMain;
import jp.synthtarou.midimixer.libs.MXDebugLines;
import jp.synthtarou.midimixer.libs.MXUtil;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMessageTemplate;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.libs.midi.MXNoteOffWatcher;
import jp.synthtarou.midimixer.libs.midi.MXTraceNumber;
import jp.synthtarou.midimixer.mx70console.ConsoleElement;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MXMIDIIn {
    private static final MXDebugLines _debug = new MXDebugLines(MXMIDIIn.class);

    public static final MXMIDIInForPlayer INTERNAL_PLAYER = new MXMIDIInForPlayer();
    
    private boolean _isopen;
    private String _name;
    private MXMIDIDevice _device;
    public final MXNoteOffWatcher _noteOff = new MXNoteOffWatcher();
    protected int _assigned;
    private boolean[] _isToMaster = new boolean[16];
    private MXVisitant16 _visitant16 = new MXVisitant16();
    
    public boolean isOpen() {
        return _isopen;
    }
    
    public MXMIDIIn(String name, MXMIDIDevice device) {
        _isopen = false;
        _name = name;
        _assigned = -1;
        _device = device;
    }
    
    public int assigned() {
        return _assigned;
    }
    
    public boolean isToMaster(int channel) {
        return _isToMaster[channel];
    }
    
    public void setToMaster(int channel, boolean toMaster) {
        _isToMaster[channel] = toMaster;
    }

    public String getMasterList() {
        StringBuffer str = new StringBuffer();
        for (int ch = 0; ch < 16; ++ ch) {
            if (isToMaster(ch)) {
                if (str.length() == 0) {
                    str.append(Integer.toString(ch));
                }else {
                    str.append(",");
                    str.append(Integer.toString(ch));
                }
            }
        }
        return str.toString();
    }
    
    public void setMasterList(String text) {
        if (text == null) {
            return;
        }
        ArrayList<String> list = new ArrayList();
        MXUtil.split(text, list, ',');
        _isToMaster = new boolean[16];
        for (String x : list) {
            int ch = MXUtil.parseTextForNumber(x);
            if (ch >= 0) {
                _isToMaster[ch] = true;
            }
        }
    }
    
    /**
     * 
     * @return 
     */
    public String getName() {
        return _name;
    }

    /**
     * 
     * @return 
     */
    public String toString() {
        return _name;
    }

    public void close()  {
        MXMIDIInManager manager = MXMIDIInManager.getManager();
        synchronized(manager) {
            allNoteOff(new MXTraceNumber());
            if (this instanceof MXMIDIInForPlayer) {
                _isopen = false;
                return;
            }
            if (_isopen) {
                MXMIDIInManager.getManager().onClose(this);
                if (_device != null) {
                    _device.close();
                }
                _isopen = false;
            }
        }
    }

    
    public boolean open() {
        if (_assigned < 0) {
            return false;
        }
        MXMIDIInManager manager = MXMIDIInManager.getManager();
        synchronized(manager) {
            if (_isopen) {
                return true;
            }
            if (this instanceof MXMIDIInForPlayer) {
                //SEQUENCER
                _isopen = true;
                return true;
            }
            _device.open();
            _isopen = true;
            _device.setInputReceiver(new MXMIDIDriverReceiver() {
                @Override
                public boolean DoShortMessage(int port, MXTraceNumber traceNumber, int dword) {
                    int code = (dword >> 16) & 0xff;
                    if (code == 0xf8 || code == 0xfe) {
                        return false;
                    }
                    startMainPath(dword, null);
                    return true;
                }

                @Override
                public boolean DoLongMessage(int port, MXTraceNumber traceNumber, byte[] data) {
                    startMainPath(0, data);
                    return true;
                }
            });
            return true;
        }
    }

    public int hashCode() {
        return _name.hashCode();
    }

    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        MXMIDIIn in = (MXMIDIIn) o;
        if (in._name == this._name) {
            return true;
        }
        return false;
    }
    
    public void allNoteOff(MXTraceNumber  trace) {
        _noteOff.allNoteOff(trace);
    }

    public String textForMasterChannel() {
        StringBuffer masterMark = new StringBuffer();
        for (int ch = 0; ch < 16; ++ ch) {
            if (isToMaster(ch)) {
                if (masterMark.length() != 0) {
                    masterMark.append(", ");
                }
                masterMark.append(Integer.valueOf(ch + 1));
            }
        }
        return masterMark.toString();
    }
    
    public int parseMasteredText(String text){ 
        if (text == null) {
            return 0;
        }
        String[] array = text.split("[ ,]+");
        int hit = 0;
        
        for (int i = 0; i < array.length; ++ i) {
            String parts = array[i];
            if (parts.length() >= 1) {
                int ch1 = parts.charAt(0) - '0';
                if (ch1 < 0 || ch1 > 9) {
                    continue;
                }
                if (parts.length() >= 2) {
                    int ch2 = parts.charAt(2) - '0';
                    if (ch2 < 0 || ch2 > 9) {
                        continue;
                    }
                    ch1 = ch1 * 10 + ch2;
                }
                if (ch1 >= 1 && ch1 <= 16) {
                    setToMaster(ch1 - 1, true);
                }
                hit ++;
            }
        }
        return hit;
    }

    public void startMainPath(int dword, byte[] data) {
        MXMessage message;

        int port = assigned();
        int status = (dword >> 16) & 0xff;
        int data1 = (dword >> 8) & 0xff;
        int data2 = (dword) & 0xff;

        if (data == null) {
            if (status == 0xf8 || status == 0xfe) {
                return;
            }
        }

        //どうせ、NoteOffWAtcherがかきかえる
        //if ((status & 0xf0) == MXMidi.COMMAND_NOTEON && data2 == 0) {
        //    status = MXMidi.COMMAND_NOTEOFF + (status & 0x0f);
        //}

        synchronized(MXMain.getMain()) {
            if (data == null) {
                message = MXMessageFactory.fromShortMessage(port, new MXTraceNumber(), status, data1, data2);
                MXMain.addOutsideInput(new ConsoleElement(message._traceNumber, _assigned, dword));
            }else {
                message = MXMessageFactory.fromBinary(port, new MXTraceNumber(), data);
                if (message == null) {
                    return;
                }
                MXMain.addOutsideInput(new ConsoleElement(message._traceNumber, _assigned, data));
            }

            if (status >= 0x80 && status <= 0xef) { //(message.isMessageTypeChannel()
                boolean worked = false;
                int ch = message.getChannel();
                int command = message.getCommand();
                int gate = message.getGate();

                MXVisitant visit = _visitant16.get(ch);

                visit.updatevisitantChannel(message);
                message.setVisitant(visit.getSnapShot());

                if (command == MXMidi.COMMAND_CONTROLCHANGE) {
                    if (gate == MXMidi.DATA1_CC_BANKSELECT || gate == (MXMidi.DATA1_CC_BANKSELECT + 0x20)) {
                        if (visit.isHavingBank()) {
                            message = MXMessageFactory.fromShortMessage(port, message._traceNumber, command + ch, MXMidi.DATA1_CC_BANKSELECT, 0);
                            message.setVisitant(visit.getSnapShot());
                            dispatchMainPath(message);
                        }
                        return;
                    }
                    if (gate == MXMidi.DATA1_CC_DATAENTRY || gate == MXMidi.DATA1_CC_DATAENTRY + 0x20) {
                        if (visit.isHaveDataentryRPN()) {
                            int[] temp0 = { MXMessageTemplate.DTEXT_RPN, 0, 0, 0 };
                            MXMessageTemplate template = new MXMessageTemplate(temp0);
                            message = template.bind(message._traceNumber, port, 0, 0);
                            message.setVisitant(visit.getSnapShot());
                            dispatchMainPath(message);
                            return;
                        }else if (visit.isHaveDataentryNRPN()) {
                            int[] temp0 = { MXMessageTemplate.DTEXT_NRPN, 0, 0, 0};
                            MXMessageTemplate template = new MXMessageTemplate(temp0);
                            message = template.bind(message._traceNumber, port, 0, 0);
                            message.setVisitant(visit.getSnapShot());
                            dispatchMainPath(message);
                            return;
                        }else {
                        }
                        return;
                    }
                    if (gate == MXMidi.DATA1_CC_RPN_LSB || gate == MXMidi.DATA1_CC_RPN_MSB 
                    ||gate == MXMidi.DATA1_CC_NRPN_LSB || gate == MXMidi.DATA1_CC_NRPN_MSB) {
                        return;
                    }
                }

                if (visit.isIncompleteBankInfo()) {
                    if (command == MXMidi.COMMAND_CONTROLCHANGE && (gate == MXMidi.DATA1_CC_BANKSELECT || gate == (MXMidi.DATA1_CC_BANKSELECT + 0x20))) {
                        ;
                    }else {
                        visit.forceCompleteBankInfo();
                    }
                }

                if (visit.isIncomplemteDataentry()) {
                    if (command == MXMidi.COMMAND_CONTROLCHANGE && (gate == MXMidi.DATA1_CC_DATAENTRY || (gate == MXMidi.DATA1_CC_DATAENTRY + 0x20))) { 
                        ;
                    }else {
                        visit.forceCompleteBankDataentry();
                    }
                }

                if (visit.isIncomplemteDataroom()) {
                    if (command == MXMidi.COMMAND_CONTROLCHANGE
                      && (gate == MXMidi.DATA1_CC_RPN_LSB || gate == MXMidi.DATA1_CC_RPN_MSB
                        ||gate == MXMidi.DATA1_CC_NRPN_LSB || gate == MXMidi.DATA1_CC_NRPN_MSB)) {
                        ;
                    }else {
                        visit.forceCompleteBankDataroom();
                    }
                }
            }

            if (message.isMessageTypeChannel()) {
                final int channel = message.getChannel();
                if (message.getCommand() == MXMidi.COMMAND_NOTEOFF) {
                    _noteOff.notifyNoteOffEvent(assigned(), message._traceNumber, channel,  message.getGate(), "@7");
                    return;
                }
                else if (message.getCommand() == MXMidi.COMMAND_NOTEON && message.getValue() == 0) {
                    _noteOff.notifyNoteOffEvent(assigned(), message._traceNumber, channel,  message.getGate(), "@6");
                    return;
                }
                dispatchMainPath(message);

                if (message.getCommand() == MXMidi.COMMAND_NOTEON) {
                    int note = message.getGate();
                    _noteOff.addListener(message, null, new MXNoteOffWatcher.Handler() {
                        @Override
                        public void onNoteOffEvent(MXMessage target, MXTraceNumber trace) {
                            target._traceNumber = trace;
                            dispatchMainPath(target);
                        }
                    });
                }
                else if (message.getCommand()== MXMidi.COMMAND_CONTROLCHANGE && message.getGate() == MXMidi.DATA1_CC_ALLNOTEOFF) {
                    _noteOff.allNoteOff(message._traceNumber);
                }
                else if (message.getCommand() == MXMidi.COMMAND_CONTROLCHANGE && message.getGate() == MXMidi.DATA1_CC_ALLSOUNDOFF) {
                    _noteOff.allNoteOff(message._traceNumber);
                }
            }else {
                dispatchMainPath(message);
            }
        }
    }
    
    Thread lastSent = null;
    
    public void dispatchMainPath(MXMessage message) {
        MXMain main = MXMain.getMain();
        if (main.getMasterKeys()  != null && main.getMasterKeys() .isAvail()) {
            if (message.isMessageTypeChannel()) {
                boolean force = false;
                if (main.getMasterKeys().isAcceptInputPanelSignal()) {
                    if (isToMaster(message.getChannel())) {
                        main.getMasterKeys().processMXMessage(message);
                        force = true;
                    }
                }
                if (!force && main.getMasterKeys().isAcceptThisPageSignal()) {
                    if (message.getPort() == main.getMasterKeys().getMousePort()
                     && message.getChannel() == main.getMasterKeys().getMouseChannel()) {
                        main.getMasterKeys().processMXMessage(message);
                        force = true;
                    }
                }
                if (force) {
                    return;
                }
            }
        }
        if (lastSent != Thread.currentThread()) {
            lastSent = Thread.currentThread();
            Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
            System.out.println("Thread MAX");
        }
        MXMain.getMain().dispatchMainPath(message);
    }
}
