/*
 * Copyright 2023 user0001.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.synthtarou.midimixer.libs.midi.driver;

import java.util.ArrayList;
import javax.sound.midi.InvalidMidiDataException;
import javax.sound.midi.MidiDevice;
import javax.sound.midi.MidiMessage;
import javax.sound.midi.MidiSystem;
import javax.sound.midi.MidiUnavailableException;
import javax.sound.midi.Receiver;
import javax.sound.midi.ShortMessage;
import javax.sound.midi.SysexMessage;
import jp.synthtarou.midimixer.libs.MXUtil;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.libs.midi.MXTraceNumber;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MXMIDIDriverJava implements MXMIDIDriver {
    public static final MXMIDIDriverJava _instance = new MXMIDIDriverJava();
    
    public boolean isUsable() {
        return true;
    }

    public void StartLibrary() {
        listAllInput();
        listAllOutput();
    }

    ArrayList<MidiDevice> _listInput;
    ArrayList<MidiDevice> _listOutput;
    MXMIDIDriverReceiver _inputReceiver;
    
    public void InputSetReceiver(MXMIDIDriverReceiver receiver) {
        _inputReceiver = receiver;
    }

    public synchronized void listAllInput() {
        if (_listInput != null) {
            return;
        }

        ArrayList<MidiDevice> newList = new ArrayList<MidiDevice>();
        MidiDevice.Info[] infoList = MidiSystem.getMidiDeviceInfo();

        for (int i = 0; i < infoList.length; i++) {
            MidiDevice device = null;
            try {
                device = MidiSystem.getMidiDevice(infoList[i]);
            }catch(MidiUnavailableException e) {
                e.printStackTrace();
                continue;
            }

            if (device.getMaxTransmitters() != 0) {
                try {
                    String name = device.getDeviceInfo().getName();
                    if (name.equals("Real Time Sequencer")) {
                        continue;
                    }
                    newList.add(device);
                } catch (Exception e) {
                }
            }
        }

        _listInput = newList;
    }
    
    public synchronized void listAllOutput() {
        if (_listOutput != null) {
            return;
        }
        
        ArrayList<MidiDevice> newList = new ArrayList<MidiDevice>();
        MidiDevice.Info[] infoList = MidiSystem.getMidiDeviceInfo();

        for (int i = 0; i < infoList.length; i++) {
            MidiDevice device = null;
            try {
                device = MidiSystem.getMidiDevice(infoList[i]);
            }catch(MidiUnavailableException e) {
                e.printStackTrace();
                continue;
            }

            if (device.getMaxReceivers() != 0) {
                String name = device.getDeviceInfo().getName();
                if (name.equals("Real Time Sequencer")) {
                    continue;
                }
                newList.add(device);
            }
        }

        _listOutput = newList;
    }


    public int InputDevicesRoomSize() {
        listAllInput();
        return _listInput.size();
    }
    
    public String InputDeviceName(int x) {
        listAllInput();
        MidiDevice device = _listInput.get(x);
        return device.getDeviceInfo().getName();
    }
    
    public String InputDeviceId(int x) {
        listAllInput();
        return InputDeviceName(x);
    }

    public boolean InputDeviceIsOpen(int x) {
        listAllInput();
        return _listInput.get(x).isOpen();
    }

    class InputReceiver implements Receiver {
        int _port;
        
        public InputReceiver(int port) {
            _port = port;
        }
                
        
        @Override
        public void send(MidiMessage msg, long timestamp) {
            if (msg instanceof ShortMessage) {
                ShortMessage shortMsg = (ShortMessage)msg;
                int status = shortMsg.getStatus() & 0xff;
                int data1 = shortMsg.getData1() & 0xff;
                int data2 = shortMsg.getData2() & 0xff;
                if (status == MXMidi.STATUS_RESET || status == MXMidi.STATUS_ACTIVESENSING) {
                    return;
                }
                
                int x = (((status << 8) | data1) << 8) | data2;
                MXTraceNumber traceNumber = new MXTraceNumber();
                _inputReceiver.DoShortMessage(_port, traceNumber, x);
            }else {
                byte[] data = msg.getMessage();
                MXTraceNumber traceNUmber = new MXTraceNumber();
                _inputReceiver.DoLongMessage(_port, traceNUmber, data);
            }
        }

        @Override
        public void close() {
            InputDeviceClose(_port);
        }
    }
    
    public boolean InputDeviceOpen(int x) {
        listAllInput();
        if (!_listInput.get(x).isOpen()) {
            try {
                _listInput.get(x).open();
                _listInput.get(x).getTransmitter().setReceiver(new InputReceiver(x));
            }catch(MidiUnavailableException e) {
                return false;
            }
        }
        return _listInput.get(x).isOpen();
    }

    public boolean InputDeviceQueryOpen(int x) {
        listAllInput();
        InputDeviceOpen(x);
        return true;
    }
    
    public boolean InputDeviceWaitDoneOpening(int device, long timeout) {
        listAllInput();
        return false;
    }
    
    public boolean InputDeviceClose(int x) {
        listAllInput();
        if (!_listInput.get(x).isOpen()) {
            return false;
        }
        _listInput.get(x).close();
        return true;
    }

    public int OutputDevicesRoomSize() {
        listAllOutput();
        return _listOutput.size();
    }
    
    public String OutputDeviceName(int x) {
        listAllOutput();
        return _listOutput.get(x).getDeviceInfo().getName();
    }
    
    public String OutputDeviceId(int x) {
        listAllOutput();
        return OutputDeviceName(x);
    }

    public boolean OutputDeviceIsOpen(int x) {
        listAllOutput();
        return _listOutput.get(x).isOpen();
    }

    public boolean OutputDeviceOpen(int x) {
        listAllOutput();
        if (!_listOutput.get(x).isOpen()) {
            try {
                _listOutput.get(x).open();
            }catch(MidiUnavailableException e) {
                return false;
            }
        }
        return _listOutput.get(x).isOpen();
    }

    public boolean OutputDeviceQueryOpen(int x) {
        listAllOutput();
        return OutputDeviceOpen(x);
    }
    
    public boolean OutputDeviceWaitDoneOpening(int x, long timeout) {
        listAllOutput();
        return false;
    }
    
    public boolean OutputDeviceClose(int x) {
        listAllOutput();
        if (!_listOutput.get(x).isOpen()) {
            return false;
        }
        _listOutput.get(x).close();
        return true;
    }
    
    public boolean OutputShortMessage(int x, int message) {
        listAllOutput();

        int status = ((message >> 8) >> 8) & 0xff;
        int data1 = (message >> 8) & 0xff;
        int data2 = (message) & 0xff;
        if (_listOutput.get(x).isOpen()) {
            try {
                ShortMessage msg = new ShortMessage(status, data1, data2);
                _listOutput.get(x).getReceiver().send(msg, 0);
                return true;
            }catch(Throwable e) {
                System.out.println("Unknown Message: " + MXUtil.toHexFF(status)  + "  " + MXUtil.toHexFF(data1) + " " + MXUtil.toHexFF(data2) );
                e.printStackTrace();
            }
        }
        return false;
    }
    
    public boolean OutputLongMessage(int x, byte[] data) {
        listAllOutput();

        if (data == null || data.length == 0) {
            return true;
        }
        
        try {
            int status = data[0] & 0xff;
            switch(status) {
                case 0xf0:
                case 0xf7:
                    byte[] newData = new byte[data.length -1];
                    for (int i = 1; i < data.length; ++ i) {
                        newData[i-1] = data[i];
                    }
                    try {
                        SysexMessage msg = new SysexMessage(data[0] & 0xff, newData, newData.length);
                        _listOutput.get(x).getReceiver().send(msg, 0);
                    }catch(InvalidMidiDataException e) {
                        System.out.println("Bug Message " + MXUtil.dumpHexFF(data));
                    }
                    break;
                case 0xff:
                    break;
                default:
                    if (data.length <= 3) {
                        int data1 = (data.length >= 2) ? data[1] : 0;
                        int data2 = (data.length >= 3) ? data[2] : 0;
                        if (_listOutput.get(x).isOpen()) {
                            try {
                                ShortMessage msg3 = new ShortMessage(status & 0xff, data1 & 0xff, data2 & 0xff);
                                _listOutput.get(x).getReceiver().send(msg3, 0);
                                return true;
                            }catch(InvalidMidiDataException e) {
                            }catch(Throwable e) {
                                e.printStackTrace();
                            }
                        }
                    }
                    break;
            }
        }catch(Throwable e) {
            e.printStackTrace();
        }
        return false;
    }
}
