#include "common.h"

extern "C"
{
#include <dirent.h>
#include <libgen.h>
#include <termios.h>
#include <time.h>
#include <sys/ioctl.h>

#if defined(__CYGWIN__) && defined(MBCNVT)
#include "mbcnvt.h"
#endif  /* mbcnvt.h */

#include <fnmatch.h>
}

static list_obj* gDirStack;

void cDirWnd::Init()
{
    gDirStack = list_new();
}

void cDirWnd::Final()
{
    for(list_it* i = list_begin(gDirStack); i; i = list_next(i)) {
        FREE(list_item(i));
    }
    list_delete(gDirStack);
}

void cDirWnd::PushFrontDir()
{
    list_push_front(gDirStack, STRDUP(mPath));
}

void cDirWnd::PushBackDir()
{
    list_push_back(gDirStack, STRDUP(mPath));
}

void cDirWnd::PopDir()
{
    if(list_size(gDirStack) > 0) {
        char* dir = (char*)list_pop_front(gDirStack);
        Move((char*)dir);
        FREE(dir);
    }
}

cDirWnd* ActiveDir()
{
    if(gLDir->mActive)
        return gLDir;
    else
        return gRDir;
}

cDirWnd* SleepDir()
{
    if(!gLDir->mActive)
        return gLDir;
    else
        return gRDir;
}

cDirWnd::eSortKind cDirWnd::gSortKind = cDirWnd::kName;
cDirWnd::eViewOption cDirWnd::gViewOption = cDirWnd::kAll;

cDirWnd::cDirWnd(char* path, bool active)
{
TBEGIN();

    mFiles = vector_new(30);
    
    mDirStack = list_new();

    strcpy(mPath, path);
    
    strcpy(mMask, "*");

    read();

    mActive = active;
    mScrollTop = 0; 
    if(vector_size(mFiles) <= 2)
        mCursor = 1;
    else
        mCursor = 2;
        
    Sort();

TEND();
}

cDirWnd::~cDirWnd()
{
    for(int i=0; i<vector_size(mFiles); i++) {
        delete (sFile*)vector_item(mFiles, i);
    }
    vector_delete(mFiles);

    for(list_it* i=list_begin(mDirStack);
        i;
        i=list_next(i))
        {
            char* item = (char*)list_item(i);
            FREE(item);
        }
    list_delete(mDirStack);
}

BOOL cDirWnd::sort_name(void* left, void* right)
{
    sFile* l = (sFile*) left;
    sFile* r = (sFile*) right;

    if(strcmp(l->mName, ".") == 0) return TRUE;
    if(strcmp(l->mName, "..") == 0) {
        if(strcmp(r->mName, ".") == 0) return FALSE;

        return TRUE;          
    }
    if(strcmp(r->mName, ".") == 0) return FALSE;
    if(strcmp(r->mName, "..") == 0) return FALSE;
    
    
    if(!S_ISDIR(l->mStat.st_mode) && !S_ISDIR(r->mStat.st_mode)
        || S_ISDIR(l->mStat.st_mode) && S_ISDIR(r->mStat.st_mode) )
        {
        return strcmp(l->mName, r->mName) < 0;
        }
          
    if(S_ISDIR(l->mStat.st_mode)) return TRUE;
    return FALSE;
}
    
BOOL cDirWnd::sort_ext(void* left, void* right)
{
     sFile* l = (sFile*) left;
     sFile* r = (sFile*) right;

     if(strcmp(l->mName, ".") == 0) return TRUE;
     if(strcmp(l->mName, "..") == 0) {
          if(strcmp(r->mName, ".") == 0) return FALSE;

          return TRUE;          
     }
     if(strcmp(r->mName, ".") == 0) return FALSE;
     if(strcmp(r->mName, "..") == 0) return FALSE;
     
     char* lext = extname(l->mName);
     char* rext = extname(r->mName);
    
     bool result;
     if(S_ISDIR(l->mStat.st_mode)) {
          if(S_ISDIR(r->mStat.st_mode))
                result= strcmp(l->mName, r->mName) <0;
          else
                result = TRUE;
     }
     else if(S_ISDIR(r->mStat.st_mode)) {
          result = FALSE;
     }
     else {
          const int cmp = strcmp(lext, rext);
          if(cmp == 0) {
                result = strcmp(l->mName, r->mName) < 0;
          }
          else {
                result = cmp < 0;
          }
     }
    
     FREE(lext);
     FREE(rext);
    
     return result;
}
  
BOOL cDirWnd::sort_size(void* left, void* right)
{
     sFile* l = (sFile*) left;
     sFile* r = (sFile*) right;

     if(strcmp(l->mName, ".") == 0) return TRUE;
     if(strcmp(l->mName, "..") == 0) {
          if(strcmp(r->mName, ".") == 0) return FALSE;

          return TRUE;          
     }
     if(strcmp(r->mName, ".") == 0) return FALSE;
     if(strcmp(r->mName, "..") == 0) return FALSE;

     if(S_ISDIR(l->mStat.st_mode)) {
          if(S_ISDIR(r->mStat.st_mode)) return strcmp(l->mName, r->mName) < 0;
          return TRUE;
     }
     if(S_ISDIR(r->mStat.st_mode)) return FALSE;

     const int lsize = S_ISDIR(l->mStat.st_mode) ? 0 : l->mStat.st_size; 
     const int rsize = S_ISDIR(r->mStat.st_mode) ? 0 : r->mStat.st_size;
    
     return lsize < rsize;
}
    
BOOL cDirWnd::sort_time(void* left, void* right)
{ 
     sFile* l = (sFile*) left;
     sFile* r = (sFile*) right;

     if(strcmp(l->mName, ".") == 0) return TRUE;
     if(strcmp(l->mName, "..") == 0) {
          if(strcmp(r->mName, ".") == 0) return FALSE;

          return TRUE;          
     }
     if(strcmp(r->mName, ".") == 0) return FALSE;
     if(strcmp(r->mName, "..") == 0) return FALSE;
     
     return difftime(l->mStat.st_mtime, r->mStat.st_mtime) > 0;
}

void cDirWnd::Sort()
{
TBEGIN();

M(("vector %d", vector_size(mFiles)));

    switch(gSortKind)
    {
    case kName:
        vector_sort(mFiles, sort_name);
        break;

    case kExt:
        vector_sort(mFiles, sort_ext);
        break;

    case kSize:
        vector_sort(mFiles, sort_size);
        break;

    case kTime:
        vector_sort(mFiles, sort_time);
        break;
    }

TEND();     
}

void cDirWnd::read()
{
TBEGIN();

    if(access(Path(), F_OK) != 0) Move("/");

    DIR* dir = opendir(mPath);
    if(dir == NULL) {
        err_msg("cDirWnd::read(): opendir err");
        Move("/");
    }

    for(int i=0; i<vector_size(mFiles); i++) {
        delete (sFile*)vector_item(mFiles, i);
    }
    vector_clear(mFiles);
    
    struct dirent* entry;
    while(entry = readdir(dir)) {
        char path[PATH_MAX];
        strcpy(path, mPath);
        strcat(path, entry->d_name);

        struct stat stat_;
        memset(&stat_, 0, sizeof(struct stat));
        stat(path, &stat_);
          
        struct stat lstat_;
        memset(&lstat_, 0, sizeof(struct stat));
        lstat(path, &lstat_);

        if(strcmp(entry->d_name, ".") == 0 || strcmp(entry->d_name, "..") == 0
            ||fnmatch(mMask, entry->d_name, 0) == 0)
        {
            vector_add(mFiles
                          , new sFile(entry->d_name, &stat_, &lstat_, false));
        }
    }
    closedir(dir);

M(("all_files"));
for(int i=0; i<vector_size(mFiles); i++) {
    sFile* file = (sFile*)vector_item(mFiles, i);
    M((file->mName));
}
M((""));

TEND();     
}

void cDirWnd::Reread()
{
TBEGIN();

    read();
    MoveCursorIndex(mCursor);
    Sort();

TEND();
}

void cDirWnd::MoveCursor(int value)
{
TBEGIN();

    mCursor += value;

    if(mCursor < 0) mCursor = 0;
    if(mCursor >= vector_size(mFiles)) mCursor = vector_size(mFiles)-1;

    struct winsize ws;
    ioctl(STDOUT_FILENO,TIOCGWINSZ,&ws);

    const int maxy = ws.ws_row;
    
    if(mCursor > mScrollTop + maxy-kMaxYMinus-(list_size(gDirStack)>0?1:0)
                                          -(vector_size(gTty)>0?1:0)-1)
        mScrollTop += mCursor-mScrollTop - maxy+kMaxYMinus
                            +(list_size(gDirStack)>0?1:0)+(vector_size(gTty)>0?1:0)+1;
    if(mCursor < mScrollTop) 
        mScrollTop = mCursor;

TEND();          
}

void cDirWnd::MoveCursorIndex(int index)
{
TBEGIN();

    mCursor = index;
     
    if(mCursor < 0) mCursor = 0;
    if(mCursor >= vector_size(mFiles)) mCursor = vector_size(mFiles)-1;
     
    struct winsize ws;
    ioctl(STDOUT_FILENO,TIOCGWINSZ,&ws);

    const int maxy = ws.ws_row;

    if(mCursor > mScrollTop + maxy-kMaxYMinus-(list_size(gDirStack)>0?1:0)
                                                 -(vector_size(gTty)>0?1:0)-1)
        mScrollTop += mCursor-mScrollTop - maxy+kMaxYMinus
                    +(list_size(gDirStack)>0?1:0)+(vector_size(gTty)>0?1:0)+1;
    if(mCursor < mScrollTop) mScrollTop = mCursor;

TEND();     
}

void cDirWnd::Activate(cDirWnd* current)
{
TBEGIN();

    current->mActive = false;
    mActive = true;

    if(!gIndividualCursor)
        MoveCursorIndex(mScrollTop + current->mCursor - current->mScrollTop);

    chdir(mPath);

TEND();     
}

sFile* cDirWnd::CursorFile()
{
    return (sFile*)vector_item(mFiles, mCursor);
}

void cDirWnd::Mark()
{
TBEGIN();

    sFile* cursor = (sFile*)vector_item(mFiles, mCursor);
    cursor->mMark = !cursor->mMark;

TEND();     
}

void cDirWnd::MarkAll()
{
TBEGIN();

    for(int i=0; i<vector_size(mFiles); i++) {
        sFile* file = (sFile*)vector_item(mFiles, i);
        if(strcmp(file->mName, ".") != 0 && strcmp(file->mName, "..") != 0) {
             file->mMark = !file->mMark;
        }
    }

TEND();     
}

void cDirWnd::MarkAllFiles()
{
TBEGIN();

    for(int i=0; i<vector_size(mFiles); i++) {
        sFile* file = (sFile*)vector_item(mFiles, i);
        if(!S_ISDIR(file->mStat.st_mode)) {
             file->mMark = !file->mMark;
        }
    }

TEND();     
}

bool cDirWnd::Marking()
{
    for(int i=0; i<vector_size(mFiles); i++) {
        sFile* file = (sFile*)vector_item(mFiles, i);
        if(file->mMark) return true;
    }

    return false;
}

VALUE cDirWnd::MarkFiles()
{
TBEGIN();

    VALUE result = rb_ary_new();

    if(Marking()) {
        for(int i=0; i<vector_size(mFiles); i++) {
            sFile* file = (sFile*)vector_item(mFiles, i);
            if(file->mMark) {
                rb_ary_push(result, rb_str_new2(file->mName));
            }
        }
    }
    else {
        if(ActiveDir() == this) {
            rb_ary_push(result, rb_str_new2(ActiveDir()->CursorFile()->mName));
        }
    }

    return result;

TEND();     
}

void cDirWnd::MarkFilesSQuote(char* result)
{
TBEGIN();

    strcpy(result, "");

    if(Marking()) {
        for(int i=0; i<vector_size(mFiles); i++) {
            sFile* file = (sFile*)vector_item(mFiles, i);
            if(file->mMark) {
                strcat(result, "'");
                strcat(result, file->mName);
                strcat(result, "'");
                strcat(result, " ");
            }
        }
    }
    else {
        if(ActiveDir() == this) {
            strcat(result, "'");
            strcat(result, ActiveDir()->CursorFile()->mName);
            strcat(result, "'");
        }
    }

TEND();     
}

bool cDirWnd::check_path(char* new_path)
{
    bool result = false;

    DIR* dir = opendir(new_path);
    if(dir) {
        result = true;
        closedir(dir);
    }

    return result;   
}

void cDirWnd::Move(char* path)
{
TBEGIN();

    char buf[1024];

    if(strcmp(path, "..") == 0){
        if(strcmp(mPath, "/") != 0) {
             char* parentpath = parentname(mPath);

             if(check_path(parentpath)) {
                 list_push_back(mDirStack, STRDUP(mPath));
                 if(list_size(mDirStack) > kDirStackSize) {
                     FREE(list_item(list_begin(mDirStack)));
                     list_pop_front(mDirStack);
                 }
                 
                 char* tmp = STRDUP(mPath);
                 char* fname = STRDUP(basename(tmp));
                 FREE(tmp);
                 
                 strcpy(mPath, parentpath); 
             
                 read();
                 Sort();

                 bool found = false;                
                 mScrollTop = 0;
                 for(int i=0; i<vector_size(mFiles); i++) {
                     sFile* file = (sFile*)vector_item(mFiles, i);
                     if(strcmp(file->mName, fname) == 0) {
                         MoveCursorIndex(i);
                         found = true;
                     }
                 }

                 if(!found) MoveCursorIndex(2);

                 FREE(fname);
             
                if(this == ActiveDir())  chdir(mPath);
            }
            
            FREE(parentpath);
        }
    }
    else if(path[0] == '/') {
        if(access(path, F_OK) == 0) {
             char path2[PATH_MAX];

             strcpy(path2, path);
             if(path[strlen(path) -1] != '/') {
                  strcat(path2, "/");
             }

             if(check_path(path2)) {
                 list_push_back(mDirStack, STRDUP(mPath));
                 if(list_size(mDirStack) > kDirStackSize) {
                     FREE(list_item(list_begin(mDirStack)));
                     list_pop_front(mDirStack);
                 }
    
                 strcpy(mPath, path2);
                 read(); 
                 mScrollTop = 0;
                 MoveCursorIndex(2);
                 Sort();
                 
                if(this == ActiveDir())  chdir(mPath);
             }
        }
    }
    else if(strcmp(path, ".") != 0) {
        char path2[PATH_MAX];
        strcpy(path2, mPath);
        strcat(path2, path);
        strcat(path2, "/");
        
        if(access(path2, F_OK) == 0 && check_path(path2)) { 
             list_push_back(mDirStack, STRDUP(mPath));
             if(list_size(mDirStack) > kDirStackSize) {
                 FREE(list_item(list_begin(mDirStack)));
                 list_pop_front(mDirStack);
             }
        
             strcpy(mPath, path2);
             read();
             Sort();
             mScrollTop = 0;
             MoveCursorIndex(2);

            if(this == ActiveDir())  chdir(mPath);
        }
    }

TEND();    
}

void cDirWnd::MoveBack()
{
TBEGIN();

    if(list_size(mDirStack) > 0) { 
        char* path = (char*)list_pop_back(mDirStack);

        if(access(path, F_OK) == 0) {
             strcpy(mPath, path);
             read();

             Sort();
    
             mScrollTop = 0;
             MoveCursorIndex(2);
        }

        FREE(path);
    }

    if(ActiveDir() == this) {
        chdir(mPath);
    }

TEND();    
}

bool cDirWnd::ChangeMask(char* mask)
{
    strcpy(mMask, mask);

    Reread();

    return true;
}

void cDirWnd::View()
{
TBEGIN();

#if defined(__CYGWIN__) && defined(MBCNVT)
	char    str_src[MAX_MBSTRBUF + 1];
	char    str_dst[MAX_MBSTRBUF + 1];
#endif

    const int maxx = getmaxx(stdscr);
    const int maxy = getmaxy(stdscr);

    const int dstack_exist = list_size(gDirStack) > 0 ? 1 : 0;
    const int ptty_exist = vector_size(gTty) > 0 ? 1:0;

    /// view one dir ///
    if(cDirWnd::gViewOption == kOneDir) {
        if(mActive) {
            mybox(0, dstack_exist, maxx-1, maxy-2-dstack_exist-ptty_exist);

            char path[256];
            strcpy(path, mPath);
            strcat(path, mMask);

            char buf[256];

            int i;
            bool kanji = false;

#if defined(__CYGWIN__) && defined(MBCNVT)
			/* multi byte code conversion */
			strcpy(str_src, (const char *)path);
			run_iconv(str_dst, str_src, DEF_TGT_CODE, DEF_FRM_CODE);
			strcpy((char *)path, str_dst);
#endif

            for(i=0; i<maxx-4 && path[i]!=0; i++) {
                if(kanji) {
                    kanji = false;
                }
                else if(is_kanji(path[i])) {
                    kanji = true;
                }

                buf[i] = path[i];
            }
            if(kanji) {
                buf[i-1] = ' ';
            }
            buf[i] = 0;

            attron(A_BOLD);
            mvprintw(0 + dstack_exist, 2, buf);
            attroff(A_BOLD);

#if defined(__CYGWIN__) && defined(MBCNVT)
			/* revert mPath string */
//			strcpy(mPath, str_src);
#endif

            for(int i=mScrollTop;
                i-mScrollTop<maxy-kMaxYMinus -dstack_exist -ptty_exist
                            && i<vector_size(mFiles);
                i++)
            {
                int attr = 0;
                sFile* file = (sFile*)vector_item(mFiles, i);
                 
                if(mCursor == i && mActive) {
                     attr |= A_REVERSE;
                }

                char fname[256];
                if(file->mMark) {
                     strcpy(fname, "*");

                     if(gColor)
                        attr |= COLOR_PAIR(kRed);
                     else 
                        attr |= A_BOLD;
                }
                else {
                     strcpy(fname, " ");
                }
        
                strcat(fname, file->mName);

                if(S_ISDIR(file->mStat.st_mode)) {
                     strcat(fname, "/");

                     if(gColor)
                         attr |= COLOR_PAIR(kBlue) | A_BOLD;
                     else
                         attr |= A_BOLD;
                }
                else if(file->mStat.st_mode&S_IXUSR
                            || file->mStat.st_mode&S_IXGRP
                            || file->mStat.st_mode&S_IXGRP)
                {
                     strcat(fname, "*");
             
                     if(gColor)
                         attr |= COLOR_PAIR(kGreen);
                     else
                         attr |= A_BOLD;
                }

                char permission[12];
                strcpy(permission, "----------");
                if(S_ISDIR(file->mStat.st_mode)) permission[0] = 'd';
                if(S_ISLNK(file->mStat.st_mode)) permission[0] = 'l';
            
                mode_t smode = file->mStat.st_mode;
                if(smode & S_IRUSR) permission[1] = 'r';
                if(smode & S_IWUSR) permission[2] = 'w';
                if(smode & S_IXUSR) permission[3] = 'x';
                if(smode & S_IRGRP) permission[4] = 'r';
                if(smode & S_IWGRP) permission[5] = 'w';
                if(smode & S_IXGRP) permission[6] = 'x';
                if(smode & S_IROTH) permission[7] = 'r';
                if(smode & S_IWOTH) permission[8] = 'w';
                if(smode & S_IXOTH) permission[9] = 'x';
        
                time_t t = file->mStat.st_mtime;
                struct tm* tm_ = (struct tm*)localtime(&t);
             
                cut(fname, buf,maxx-62-3);

                char owner[256];
                char* tmp = mygetpwuid(&file->mStat);
                if(tmp)
                    cut(tmp, owner, 9);
                else
                    sprintf(owner, "%d", file->mStat.st_uid);

                char group[256];
                tmp = mygetgrgid(&file->mStat);
                if(tmp)
                    cut(tmp, group, 10);
                else
                    sprintf(group, "%d", file->mStat.st_gid);


                int year = tm_->tm_year-100;
                if(year < 0) year+=100;
                while(year > 100) year-=100;
                
                sprintf(buf + strlen(buf)
                       , " %s %3d %-9s %-10s%10lld %02d-%02d-%02d %02d:%02d "
                       , permission, file->mStat.st_nlink
                       , owner, group
                       
                       , file->mStat.st_size
                       
                       , year, tm_->tm_mon+1
                       , tm_->tm_mday, tm_->tm_hour, tm_->tm_min
                );

#if defined(__CYGWIN__) && defined(MBCNVT)
				/* multi byte code conversion */
				strcpy(str_src, buf);
				run_iconv(str_dst, str_src, DEF_TGT_CODE, DEF_FRM_CODE);
				strcpy(buf, str_dst);
#endif
        
                attron(attr);
                mvprintw(i-mScrollTop + 1 + dstack_exist, 1, buf);
                attroff(attr);
            }
        }
        
        mvprintw(maxy - kMaxYMinus+1 -ptty_exist, 2
                                    , "%dfiles", vector_size(mFiles));
    }
    
    /// view_nameonly view_all ///
    else {
        int x;
        if(this == gLDir)
            x = 0;
        else
            x = maxx / 2; 

        int line = 0;
        mybox(x, dstack_exist, maxx/2-1, maxy-2-dstack_exist-ptty_exist);

        char path[256];
        strcpy(path, mPath);
        strcat(path, mMask);
        
        char buf[256];

        int i;
        bool kanji = false;

#if defined(__CYGWIN__) && defined(MBCNVT)
		/* multi byte code conversion */
		strcpy(str_src, (const char *)path);
		run_iconv(str_dst, str_src, DEF_TGT_CODE, DEF_FRM_CODE);
		strcpy((char *)path, str_dst);
#endif

        for(i=0; i<maxx/2-5 && path[i]!=0; i++) {
            if(kanji) {
                kanji = false;
            }
            else if(is_kanji(path[i])) {
                kanji = true;
            }

            buf[i] = path[i];
        }
        if(kanji) {
            buf[i-1] = ' ';
        }
        buf[i] = 0;

        if(mActive) attron(A_BOLD);
        mvprintw(dstack_exist, 2+x, buf);
        if(mActive) attroff(A_BOLD);

#if defined(__CYGWIN__) && defined(MBCNVT)
		/* revert mPath string */
//		strcpy(mPath, str_src);
#endif

        for(int i=mScrollTop;
                i-mScrollTop<maxy-kMaxYMinus-dstack_exist-ptty_exist
                        && i<vector_size(mFiles);
                i++)
            {
            int attr = 0;
            sFile* file = (sFile*)vector_item(mFiles, i);
                 
            if(mCursor == i && mActive) attr |= A_REVERSE;

            char fname[256];
            if(file->mMark) {
                strcpy(fname, "*");
                if(gColor)
                    attr |= COLOR_PAIR(kRed);
                else
                    attr |= A_BOLD;
            }
            else {
                strcpy(fname, " ");
            }
        
            strcat(fname, file->mName);
        
            if(S_ISDIR(file->mStat.st_mode)) {
                strcat(fname, "/");

                if(gColor)
                    attr |= COLOR_PAIR(kBlue) | A_BOLD;
                else
                    attr |= A_BOLD;
            }
            else if(file->mStat.st_mode&S_IXUSR
                    || file->mStat.st_mode&S_IXGRP
                    || file->mStat.st_mode&S_IXGRP)
            {
                strcat(fname, "*");
             
                if(gColor) 
                    attr |= COLOR_PAIR(kGreen);
                else
                    attr |= A_BOLD;
            }
            else if(S_ISLNK(file->mStat.st_mode)) {
                strcat(fname, "@");
                if(gColor) attr |= COLOR_PAIR(kCyan);
            }

            /// view_all ///
            if(gViewOption == kAll) {
                char buf[256];

                char name[256];
                cut(fname, name, maxx/2-28);

                time_t t = file->mStat.st_mtime;
                struct tm* tm_ = (struct tm*)localtime(&t);

                int year = tm_->tm_year-100;
                if(year < 0) year+=100;
                while(year > 100) year-=100;
                
                sprintf(buf, "%s%9lld %02d-%02d-%02d %02d:%02d "
                         , name, file->mStat.st_size, year
                         , tm_->tm_mon+1, tm_->tm_mday, tm_->tm_hour
                         , tm_->tm_min);

#if defined(__CYGWIN__) && defined(MBCNVT)
				/* multi byte code conversion */
				strcpy(str_src, buf);
				run_iconv(str_dst, str_src, DEF_TGT_CODE, DEF_FRM_CODE);
                strcpy(buf, str_dst);
#endif
                                          
                attron(attr);
                mvprintw(i-mScrollTop + 1 + dstack_exist, x + 1, buf);
                attroff(attr);
            }
            /// view nameonly ///
            else {
                char name[256];
                cut(fname, name, maxx/2-3);

#if defined(__CYGWIN__) && defined(MBCNVT)
				/* multi byte code conversion */
				strcpy(str_src, name);
				run_iconv(str_dst, str_src, DEF_TGT_CODE, DEF_FRM_CODE);
                strcpy(name, str_dst);
#endif
                 
                attron(attr);
                mvprintw(i-mScrollTop + 1 + dstack_exist, x + 1, name);
                attroff(attr);
            }
        }
          
        mvprintw(maxy - kMaxYMinus+1 -ptty_exist, x + 2
                                    , "%dfiles", vector_size(mFiles));
    }

#if defined(__CYGWIN__) && defined(MBCNVT)
	/* we need this line to avoid improper kanji indication of curses */
	/* side efffect: display drawing speed will slow down */
	clearok(stdscr, true);
#endif

    /// draw gDirStack ///
    if(dstack_exist) {
        const int width = maxx / 3; //(gDirStack.size()+1);
        int j = 0;
        for(list_it* i = list_begin(gDirStack); i; i=list_next(i))
            {
            char* item = (char*)list_item(i);
            
            if(j > 2) break;
            
            char buf[256];
            buf[0] = '[';
            buf[1] = '%';
            sprintf(buf + 2, "-%ds]", width-2);

            char buf2[256];
            const int len = strlen(item);
            bool kanji = false;
            bool kanji_ato = false;
            int c = 0;

            for(int k=0; k<len; k++) {
                if(kanji) {
                    kanji = false;
                    kanji_ato = true;
                }
                else if(is_kanji(item[k])) {
                    kanji = true;
                    kanji_ato = false;
                }
                else {
                    kanji_ato = false;
                }

                if(len-k <= (width-2)) {
                    if(kanji_ato && c==0) {
                        buf2[c++] = ' ';
                    }
                    else {
                        buf2[c++] = item[k];
                    }
                }
            }
            buf2[c] = 0;
M(("[%s]",buf2));            

#if defined(__CYGWIN__) && defined(MBCNVT)
			/* multi byte code conversion */
			strcpy(str_src, buf2);
			run_iconv(str_dst, str_src, DEF_TGT_CODE, DEF_FRM_CODE);
			strcpy(buf2, str_dst);
#endif
            
            mvprintw(0, width * j, buf, buf2);
            j++;                        
            }
    }

TEND(); 
}
