#include "common.h"

extern "C" {
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/stat.h>
#include <signal.h>
}

const int kKeyMetaFirst = 128;
const int kKeyEsc = 27;

cDirWnd* gLDir;
cDirWnd* gRDir;
bool gMainLoop;
map<string, VALUE> gKeyCommand[2][KEY_MAX];

bool gISearchNormalChar = true;
bool gISearchShiftChar = true;
bool gISearchSignChar = true;
bool gColor = false;
bool gIndividualCursor = false;
bool gCheckDeleteFile = false;

static void read_rc_file();
static void input(int meta, int key);
static void view();

void sig_exit(int signal)
{
M(("signal %d", signal));
    gMainLoop = false;
}

int main(int argc, char* argv[])
{
CHECKML_BEGIN();
LOG_BEGIN();

    /// is terminal ///
    if(!isatty(0) || !isatty(1)) {
        fprintf(stderr, "standard input is not a tty\n");
        return 1;
    }

    /// signal ///
    signal(SIGINT, sig_exit);
    signal(SIGQUIT, sig_exit);
    signal(SIGABRT, sig_exit);
    signal(SIGKILL, sig_exit);
    signal(SIGPIPE, sig_exit);
    signal(SIGALRM, sig_exit);
    signal(SIGTERM, sig_exit);
    signal(SIGHUP, sig_exit);

//    signal(SIGWINCH, sig_winch);

    /// send socket message ///
    if(argc >= 3) {
        int pid = atoi(argv[1]);
        if(pid == 0) {
            fprintf(stderr, "illegal pid");
            return 1;
        }

        char soc_name[256];
        sprintf(soc_name, "/tmp/mfiler%d", pid);

        if(access(soc_name, F_OK) != 0) {
            fprintf(stderr, "illegal pid.");
            return 1;
        }

        int soc = socket(AF_UNIX, SOCK_DGRAM, 0);
        if(soc < 0) {
            perror("socket");
            return 1;
        }

        struct sockaddr_un addr;
        addr.sun_family = AF_UNIX;
        strcpy(addr.sun_path, soc_name);
        if(connect(soc, (struct sockaddr *)&addr, sizeof(addr)) == -1) {
            perror("bind");
            return 1;
        }

        write(soc, argv[2], strlen(argv[2]) + 1);
        close(soc);
    
        return 0;
    }

    /// init ruby interpreter ///
M(("init ruby itepreter"));
    
    ruby_init();
    
    rb_define_global_const("KEY_UP", INT2NUM(KEY_UP));
    rb_define_global_const("KEY_RIGHT", INT2NUM(KEY_RIGHT));
    rb_define_global_const("KEY_DOWN", INT2NUM(KEY_DOWN));
    rb_define_global_const("KEY_LEFT", INT2NUM(KEY_LEFT));
    rb_define_global_const("KEY_INSERT", INT2NUM(KEY_IC));
    rb_define_global_const("KEY_DELETE", INT2NUM(KEY_DC));
    rb_define_global_const("KEY_HOME", INT2NUM(KEY_HOME));
    rb_define_global_const("KEY_END", INT2NUM(KEY_END));
    rb_define_global_const("KEY_PAGEUP", INT2NUM(KEY_PPAGE));
    rb_define_global_const("KEY_PAGEDOWN", INT2NUM(KEY_NPAGE));
    
    rb_define_global_const("KEY_ENTER", INT2NUM(10));
    rb_define_global_const("KEY_BACKSPACE", INT2NUM(KEY_BACKSPACE));
    rb_define_global_const("KEY_SPACE", INT2NUM(32));
    
    rb_define_global_const("KEY_F1", INT2NUM(KEY_F(1)));
    rb_define_global_const("KEY_F2", INT2NUM(KEY_F(2)));
    rb_define_global_const("KEY_F3", INT2NUM(KEY_F(3)));
    rb_define_global_const("KEY_F4", INT2NUM(KEY_F(4)));
    rb_define_global_const("KEY_F5", INT2NUM(KEY_F(5)));
    rb_define_global_const("KEY_F6", INT2NUM(KEY_F(6)));
    rb_define_global_const("KEY_F7", INT2NUM(KEY_F(7)));
    rb_define_global_const("KEY_F8", INT2NUM(KEY_F(8)));
    rb_define_global_const("KEY_F9", INT2NUM(KEY_F(9)));
    rb_define_global_const("KEY_F10", INT2NUM(KEY_F(10)));
    rb_define_global_const("KEY_F11", INT2NUM(KEY_F(11)));
    rb_define_global_const("KEY_F12", INT2NUM(KEY_F(12)));
    
    rb_define_global_const("KEY_a", INT2NUM('a'));
    rb_define_global_const("KEY_b", INT2NUM('b'));
    rb_define_global_const("KEY_c", INT2NUM('c'));
    rb_define_global_const("KEY_d", INT2NUM('d'));
    rb_define_global_const("KEY_e", INT2NUM('e'));
    rb_define_global_const("KEY_f", INT2NUM('f'));
    rb_define_global_const("KEY_g", INT2NUM('g'));
    rb_define_global_const("KEY_h", INT2NUM('h'));
    rb_define_global_const("KEY_i", INT2NUM('i'));
    rb_define_global_const("KEY_j", INT2NUM('j'));
    rb_define_global_const("KEY_k", INT2NUM('k'));
    rb_define_global_const("KEY_l", INT2NUM('l'));
    rb_define_global_const("KEY_m", INT2NUM('m'));
    rb_define_global_const("KEY_n", INT2NUM('n'));
    rb_define_global_const("KEY_o", INT2NUM('o'));
    rb_define_global_const("KEY_p", INT2NUM('p'));
    rb_define_global_const("KEY_q", INT2NUM('q'));
    rb_define_global_const("KEY_r", INT2NUM('r'));
    rb_define_global_const("KEY_s", INT2NUM('s'));
    rb_define_global_const("KEY_t", INT2NUM('t'));
    rb_define_global_const("KEY_u", INT2NUM('u'));
    rb_define_global_const("KEY_v", INT2NUM('v'));
    rb_define_global_const("KEY_w", INT2NUM('w'));
    rb_define_global_const("KEY_x", INT2NUM('x'));
    rb_define_global_const("KEY_y", INT2NUM('y'));
    rb_define_global_const("KEY_z", INT2NUM('z'));
    
    rb_define_global_const("KEY_CTRL_SPACE", INT2NUM(0));
    rb_define_global_const("KEY_CTRL_A", INT2NUM(1));
    rb_define_global_const("KEY_CTRL_B", INT2NUM(2));
    rb_define_global_const("KEY_CTRL_C", INT2NUM(3));
    rb_define_global_const("KEY_CTRL_D", INT2NUM(4));
    rb_define_global_const("KEY_CTRL_E", INT2NUM(5));
    rb_define_global_const("KEY_CTRL_F", INT2NUM(6));
    rb_define_global_const("KEY_CTRL_G", INT2NUM(7));
    rb_define_global_const("KEY_CTRL_H", INT2NUM(8));
    rb_define_global_const("KEY_CTRL_I", INT2NUM(9));
    rb_define_global_const("KEY_CTRL_J", INT2NUM(10));
    rb_define_global_const("KEY_CTRL_K", INT2NUM(11));
    rb_define_global_const("KEY_CTRL_L", INT2NUM(12));
    rb_define_global_const("KEY_CTRL_M", INT2NUM(13));
    rb_define_global_const("KEY_CTRL_N", INT2NUM(14));
    rb_define_global_const("KEY_CTRL_O", INT2NUM(15));
    rb_define_global_const("KEY_CTRL_P", INT2NUM(16));
    rb_define_global_const("KEY_CTRL_Q", INT2NUM(17));
    rb_define_global_const("KEY_CTRL_R", INT2NUM(18));
    rb_define_global_const("KEY_CTRL_S", INT2NUM(19));
    rb_define_global_const("KEY_CTRL_T", INT2NUM(20));
    rb_define_global_const("KEY_CTRL_U", INT2NUM(21));
    rb_define_global_const("KEY_CTRL_V", INT2NUM(22));
    rb_define_global_const("KEY_CTRL_W", INT2NUM(23));
    rb_define_global_const("KEY_CTRL_X", INT2NUM(24));
    rb_define_global_const("KEY_CTRL_Y", INT2NUM(25));
    rb_define_global_const("KEY_CTRL_Z", INT2NUM(26));
    
    rb_define_global_const("KEY_A", INT2NUM('A'));
    rb_define_global_const("KEY_B", INT2NUM('B'));
    rb_define_global_const("KEY_C", INT2NUM('C'));
    rb_define_global_const("KEY_D", INT2NUM('D'));
    rb_define_global_const("KEY_E", INT2NUM('E'));
    rb_define_global_const("KEY_F", INT2NUM('F'));
    rb_define_global_const("KEY_G", INT2NUM('G'));
    rb_define_global_const("KEY_H", INT2NUM('H'));
    rb_define_global_const("KEY_I", INT2NUM('I'));
    rb_define_global_const("KEY_J", INT2NUM('J'));
    rb_define_global_const("KEY_K", INT2NUM('K'));
    rb_define_global_const("KEY_L", INT2NUM('L'));
    rb_define_global_const("KEY_M", INT2NUM('M'));
    rb_define_global_const("KEY_N", INT2NUM('N'));
    rb_define_global_const("KEY_O", INT2NUM('O'));
    rb_define_global_const("KEY_P", INT2NUM('P'));
    rb_define_global_const("KEY_Q", INT2NUM('Q'));
    rb_define_global_const("KEY_R", INT2NUM('R'));
    rb_define_global_const("KEY_S", INT2NUM('S'));
    rb_define_global_const("KEY_T", INT2NUM('T'));
    rb_define_global_const("KEY_U", INT2NUM('U'));
    rb_define_global_const("KEY_V", INT2NUM('V'));
    rb_define_global_const("KEY_W", INT2NUM('W'));
    rb_define_global_const("KEY_X", INT2NUM('X'));
    rb_define_global_const("KEY_Y", INT2NUM('Y'));
    rb_define_global_const("KEY_Z", INT2NUM('Z'));

    rb_define_global_const("KEY_0", INT2NUM('0'));
    rb_define_global_const("KEY_1", INT2NUM('1'));
    rb_define_global_const("KEY_2", INT2NUM('2'));
    rb_define_global_const("KEY_3", INT2NUM('3'));
    rb_define_global_const("KEY_4", INT2NUM('4'));
    rb_define_global_const("KEY_5", INT2NUM('5'));
    rb_define_global_const("KEY_6", INT2NUM('6'));
    rb_define_global_const("KEY_7", INT2NUM('7'));
    rb_define_global_const("KEY_8", INT2NUM('8'));
    rb_define_global_const("KEY_9", INT2NUM('9'));
        
    rb_define_global_const("KEY_EXCLAM", INT2NUM('!'));
    rb_define_global_const("KEY_DQUOTE", INT2NUM('"'));
    rb_define_global_const("KEY_SHARP", INT2NUM('#'));
    rb_define_global_const("KEY_DOLLAR", INT2NUM('$'));
    rb_define_global_const("KEY_PERCENT", INT2NUM('%'));
    rb_define_global_const("KEY_AND", INT2NUM('&'));
    rb_define_global_const("KEY_SQUOTE", INT2NUM('\''));
    rb_define_global_const("KEY_LPAREN", INT2NUM('('));
    rb_define_global_const("KEY_RPAREN", INT2NUM(')'));
    rb_define_global_const("KEY_TILDA", INT2NUM('~'));
    rb_define_global_const("KEY_EQUAL", INT2NUM('='));
    rb_define_global_const("KEY_MINUS", INT2NUM('-'));
    rb_define_global_const("KEY_CUP", INT2NUM('^'));
    rb_define_global_const("KEY_VBAR", INT2NUM('|'));
    rb_define_global_const("KEY_BACKSLASH", INT2NUM('\\'));
    rb_define_global_const("KEY_ATMARK", INT2NUM('@'));
    rb_define_global_const("KEY_BAPOSTROPHE", INT2NUM('`'));
    rb_define_global_const("KEY_LCURLY", INT2NUM('{'));
    rb_define_global_const("KEY_LBRACK", INT2NUM('['));
    rb_define_global_const("KEY_PLUS", INT2NUM('+'));
    rb_define_global_const("KEY_SEMICOLON", INT2NUM(';'));
    rb_define_global_const("KEY_STAR", INT2NUM('*'));
    rb_define_global_const("KEY_COLON", INT2NUM(':'));
    rb_define_global_const("KEY_RCURLY", INT2NUM('}'));
    rb_define_global_const("KEY_RBRACK", INT2NUM(']'));
    rb_define_global_const("KEY_LSS", INT2NUM('<'));
    rb_define_global_const("KEY_COMMA", INT2NUM(','));
    rb_define_global_const("KEY_GTR", INT2NUM('>'));
    rb_define_global_const("KEY_DOT", INT2NUM('.'));
    rb_define_global_const("KEY_SLASH", INT2NUM('/'));
    rb_define_global_const("KEY_QMARK", INT2NUM('?'));
    rb_define_global_const("KEY_UNDERBAR", INT2NUM('_'));
    
    rb_define_global_const("NOMETA", INT2NUM(0));
    rb_define_global_const("META", INT2NUM(1));

    /// init global var ///
M(("init global var"));
    
    char* cwd = mygetcwd();

    char tmp[PATH_MAX];
    strcpy(tmp, cwd);
    if(strcmp(tmp, "/") != 0) strcat(tmp,"/");
    
    gLDir = new cDirWnd(tmp, true);
    gRDir = new cDirWnd(tmp, false);

    FREE(cwd);

    gKeyCommand[0][3]["*"] = rb_str_new2("mf_exit();");    // Control-C
    
    /// init modules ///
M(("init modules"));
    
    command_init();
    cmdline_init();

    /// init socket ///
M(("init socket"));
    
    char soc_name[256];
    sprintf(soc_name, "/tmp/mfiler%d", getpid());
    
    int soc = socket(AF_UNIX, SOCK_DGRAM, 0);
    if(soc < 0) {
        perror("socket");
        return 1;
    }
    
    struct sockaddr_un addr;
    addr.sun_family = AF_UNIX;
    strcpy(addr.sun_path, soc_name);
    if(bind(soc, (struct sockaddr *)&addr, sizeof(addr)) == -1) {
        perror("bind");
        return 1;
    }
    
    /// read rc file ///
M(("read rc file"));
    
    read_rc_file();
    
    /// start curses ///
    initscr();
    noecho();
    keypad(stdscr, 1);
    raw();

    if(has_colors() && gColor) {
        start_color();
        gColor = true;
        init_pair(kWhite, COLOR_WHITE, COLOR_BLACK);
        init_pair(kBlue, COLOR_BLUE, COLOR_BLACK);
        init_pair(kCyan, COLOR_CYAN, COLOR_BLACK);
        init_pair(kGreen, COLOR_GREEN, COLOR_BLACK);
        init_pair(kYellow, COLOR_YELLOW, COLOR_BLACK);
        init_pair(kMagenta, COLOR_MAGENTA, COLOR_BLACK);
        init_pair(kRed, COLOR_RED, COLOR_BLACK);
    }
    else {
        gColor = false;
    }

    clear();
    refresh();

    /// main loop ///
M(("main loop start"));

    gMainLoop = true;
    fd_set mask;
    fd_set read_ok;

    FD_ZERO(&mask);
    FD_SET(0, &mask);
    FD_SET(soc, &mask);

    while(gMainLoop) {
M(("Main Loop"));    
        /// view ///
        view();

        /// get event ///
M(("select"));        
        read_ok = mask;
        select(soc + 1, &read_ok, NULL, NULL, NULL);
M(("select end"));        

        /// key event ///
        if(FD_ISSET(0, &read_ok)) {
M(("key event"));
            int key = getch();
           
            /// esc ///
            if(key == -1) {
            }
            else if(key == KEY_RESIZE) {
            }
            else if(key == kKeyEsc) {
                int key2 = getch();

                input(1, key2);
            }
            
            /// meta key || kanji ///
            else if(key >= kKeyMetaFirst && key <= kKeyMetaFirst+127) {
                if(gCmdLineActive) {    // for Kanji input
                    input(0, key);
                }
                else {
                    input(1, key - kKeyMetaFirst);
                }
            }

            /// normal key ///
            else {
               input(0, key);
            }
        }

        /// socket event ///
        else if(FD_ISSET(soc, &read_ok)) {
M(("socket event"));        
            char buf[256];
            
            struct sockaddr_un caddr;
            socklen_t caddr_len;
            int len = recvfrom(soc, buf, 256, 0, (struct sockaddr *)&caddr, &caddr_len);

            rb_eval_string(buf);
        }
M(("Main Loop end"));        
    }

    /// finish curses ///
    endwin();
    
    /// finalize modules ///
    command_final();
    cMenu::Final();
    
    /// release global var ///    
    unlink(soc_name);

    delete gLDir;
    delete gRDir;

CHECKML_END();
LOG_END();

    return 0;
}
/*
static void read_rc_file()
{
    char* home = getenv("HOME");
    if(home == NULL) {
        fprintf(stderr, "$HOME is NULL");
        exit(1);
    }

    char rc_fname[256];
    sprintf(rc_fname, "%s/.mfiler", home);
    
    if(access(rc_fname, F_OK) == 0) {
        rb_load(rb_str_new2(rc_fname), 0);
    }
}
*/
static void read_rc_file()
{
    char* home = getenv("HOME");
    if(home == NULL) {
        fprintf(stderr, "$HOME is NULL");
        exit(1);
    }

    char rc_fname[256];
    sprintf(rc_fname, "%s/.mfiler", home);
    
    if(access(rc_fname, F_OK) == 0) {
        /// get file size ///
        struct stat sbuf;
        stat(rc_fname, &sbuf);
        const int size = sbuf.st_size;

        /// read to buffer ///
        char* buf = (char*)MALLOC(size + 1);
        
        FILE* rcfile = fopen(rc_fname, "r");
        fread(buf, sizeof(char), size, rcfile);
        buf[size] = 0;
        fclose(rcfile);

        /// eval ///
        rb_eval_string(buf);
        
        FREE(buf);
    }
}

string gInputFileName;

static bool match()
{
    bool same = true;
    if(gInputFileName.size() > 1) {
        for(int i=1; i<gInputFileName.size(); i++)
            if(gInputFileName[i] != gInputFileName[i-1]) {
                same = false;
            }
    }

    if(same) {
        vector<int> mfiles;
        
        for(int i=0; i<ActiveDir()->Files()->size(); i++) {
            sFile file = (*ActiveDir()->Files())[i];
            if(tolower(file.mName[0]) == tolower(gInputFileName[0])) {
                mfiles.push_back(i);
            }
        }

        if(mfiles.size() > 0) {
            ActiveDir()->MoveCursorIndex(mfiles[(gInputFileName.size()-1) % mfiles.size()] );
            return true;
        }
    }
    else {
        for(int i=0; i<ActiveDir()->Files()->size(); i++) {
            sFile file = (*ActiveDir()->Files())[i];

/*            
            if(strstr(file.mName, gInputFileName.c_str()) == file.mName) {
                ActiveDir()->MoveCursorIndex(i);

                return true;
            }
*/            

            bool equal = true;
            const int len = gInputFileName.size();
            for(int j=0; j<len; j++) {
                if(tolower(gInputFileName[j]) != tolower(file.mName[j])) {
                    equal = false;
                }
            }

            if(equal) {
                ActiveDir()->MoveCursorIndex(i);
                
                return true;
            }
        }
    }

    return false;
}

static void input(int meta, int key)
{
TBEGIN();

    if(gCmdLineActive) {
        cmdline_input(meta, key);
    }
    else if(gActiveMenu) {
        gActiveMenu->Input(meta, key);
    }
    else {
M(("main input"));    
        sFile cursor = ActiveDir()->CursorFile();
        const char* name = cursor.mName;

        char extension[256];
        char* tmp = extname(cursor.mName);
        sprintf(extension, ".%s", tmp);
        FREE(tmp);
    
        char* command;
//        int error;

        if(ActiveDir()->Marking()
            && gKeyCommand[meta][key].find(".mark")
                        != gKeyCommand[meta][key].end())
        {
M(("key mark eval"));
            char* tmp = RSTRING(gKeyCommand[meta][key][".mark"])->ptr;
            if(tmp) {
                rb_eval_string(tmp);
            }
            
            gInputFileName = "";
        }
        else if(gKeyCommand[meta][key].find(name)
                    != gKeyCommand[meta][key].end())
           {
M(("key name eval"));
            char* tmp = RSTRING(gKeyCommand[meta][key][name])->ptr;
            if(tmp) {
                rb_eval_string(tmp);
            }

            gInputFileName = "";
        }
        else if(gKeyCommand[meta][key].find(extension)
                != gKeyCommand[meta][key].end())
            {
M(("key extension eval"));
            char* tmp = RSTRING(gKeyCommand[meta][key][extension])->ptr;
            if(tmp) { 
                rb_eval_string(tmp);
            }

            gInputFileName = "";
        }
        else if(S_ISDIR(cursor.mStat.st_mode)
            && gKeyCommand[meta][key].find(".dir")
                    != gKeyCommand[meta][key].end())
        {
M(("key dir eval"));
            char* tmp = RSTRING(gKeyCommand[meta][key][".dir"])->ptr;
            if(tmp) {
                rb_eval_string(tmp);
            }
            
            gInputFileName = "";
        }
        else if((cursor.mStat.st_mode&S_IXUSR
             || cursor.mStat.st_mode&S_IXGRP || cursor.mStat.st_mode&S_IXOTH)
            && gKeyCommand[meta][key].find(".exe")
                    != gKeyCommand[meta][key].end())
        {
M(("key exe eval"));
            char* tmp = RSTRING(gKeyCommand[meta][key][".exe"])->ptr;
            if(tmp) {
                rb_eval_string(tmp);
            }
            
            gInputFileName = "";
        }
        else if(gKeyCommand[meta][key].find("*")
                != gKeyCommand[meta][key].end())
        {
M(("key file * eval"));
            char* tmp = RSTRING(gKeyCommand[meta][key]["*"])->ptr;
            if(tmp) {
                rb_eval_string(tmp);
            }
            
            gInputFileName = "";
        }
        else if(meta == 0
                &&((key>='!' && key<='@' && gISearchSignChar)
                || (key>='[' && key<='_' && gISearchSignChar)
                || (key>='{' && key<='}' && gISearchSignChar)
                || (key>='a' && key<='z' && gISearchNormalChar)
                || (key>='A' && key<='Z' && gISearchShiftChar)))
            {
M(("key incremental search"));
            gInputFileName.push_back((char)key);

            if(!match()) {
                gInputFileName = "";
                gInputFileName.push_back((char)key);

                if(!match()) {
                    gInputFileName = "";
                }
            }
        }
        else {
            gInputFileName = "";
        }
    }

TEND();    
}

static void view()
{
TBEGIN();

    if(gActiveMenu)
        gActiveMenu->View();
    else {
        myclear();

        if(gCCandidate.size()) {
            cmdline_completion_view();
        }
        else {
            gLDir->View();
            gRDir->View();
        }
    
        if(gCmdLineActive)
            cmdline_view();
        else {
            const int maxy = getmaxy(stdscr);
            const int maxx = getmaxx(stdscr);

            mvprintw(maxy -2, 0, "the Minnu's Filer2 version 1.12 %s", gInputFileName.c_str());
    
            sFile file = ActiveDir()->CursorFile();
        
            char permission[12];
            strcpy(permission, "----------");
            if(S_ISDIR(file.mStat.st_mode)) permission[0] = 'd';
            if(S_ISLNK(file.mStat.st_mode)) permission[0] = 'l';
            
            mode_t smode = file.mStat.st_mode;
            if(smode & S_IRUSR) permission[1] = 'r';
            if(smode & S_IWUSR) permission[2] = 'w';
            if(smode & S_IXUSR) permission[3] = 'x';
            if(smode & S_IRGRP) permission[4] = 'r';
            if(smode & S_IWGRP) permission[5] = 'w';
            if(smode & S_IXGRP) permission[6] = 'x';
            if(smode & S_IROTH) permission[7] = 'r';
            if(smode & S_IWOTH) permission[8] = 'w';
            if(smode & S_IXOTH) permission[9] = 'x';
            permission[10] = 0;
        
            time_t t = file.mStat.st_mtime;
            struct tm* tm_ = (struct tm*)localtime(&t);
             
            char buf[1024];

            char owner[256];
            char* tmp = mygetpwuid(&file.mStat);
            if(tmp)
                cut(tmp, owner, 8);
            else
                sprintf(owner, "%d", file.mStat.st_uid);

            char group[256];
            tmp = mygetgrgid(&file.mStat);
            if(tmp)
                cut(tmp, group, 7);
            else
                sprintf(group, "%d", file.mStat.st_gid);

            if(!S_ISLNK(file.mLStat.st_mode)) {
                sprintf(buf
                    , "%s %3d %-8s %-7s%10lld %02d-%02d-%02d %02d:%02d %s"
                    , permission, file.mStat.st_nlink
                    , owner, group
                                        
                    , file.mStat.st_size
                    
                    , tm_->tm_year-100, tm_->tm_mon+1, tm_->tm_mday
                    , tm_->tm_hour, tm_->tm_min
                    , file.mName);

                const int len = strlen(buf);
                if(len > maxx) {
                    bool kanji = false;
                    int i;
                    for(i=0; i<maxx; i++) {
                        if(kanji)
                            kanji = false;
                        else if(is_kanji(buf[i]))
                            kanji = true;
                    }

                    if(kanji)
                        buf[i-1] = 0;
                    else 
                        buf[i] = 0;
                }
                
                mvprintw(maxy-1, 0, buf);
            }
            else {
                char link[PATH_MAX + 1];
                char path[PATH_MAX];
                strcpy(path, ActiveDir()->Path());
                strcat(path, file.mName);                 
                 
                int bytes = readlink(path, link, PATH_MAX);
                link[bytes] = 0;
                 
                sprintf(buf
                   , "%s %3d %s%s%10lld %02d-%02d-%02d %02d:%02d %s -> %s"
                   , permission, file.mStat.st_nlink
                   , owner, group
                                        
                   , file.mStat.st_size

                   , tm_->tm_year-100, tm_->tm_mon+1, tm_->tm_mday
                   , tm_->tm_hour, tm_->tm_min
                   , file.mName
                   , link);

                const int len = strlen(buf);
                if(len > maxx) {
                    bool kanji = false;
                    int i;
                    for(i=0; i<maxx; i++) {
                        if(kanji)
                            kanji = false;
                        else if(is_kanji(buf[i]))
                            kanji = true;
                    }

                    if(kanji) {
                        buf[i-1] = 0;
                    }
                    else {
                        buf[i] = 0;
                    }
                }
                     
                mvprintw(maxy-1, 0, buf);
            }
        }

        refresh();
    }

TEND();    
}

bool is_kanji(unsigned char c)
{
    return c >= 161 && c <= 254;
}

void string_erase(string& s, int n)
{
    int j = 0;
    for(string::iterator i=s.begin();
        i != s.end();
        i++)
    {
        if(j == n) {
            s.erase(i);
            break;
        }
        j++;
    }
}

void cut(char* src, char* dest, int len)
{
    int n;
    bool tmpend = false;
    bool kanji = false;
    for(n=0; n<len; n++) {
        if(kanji)
            kanji = false;
        else if(is_kanji(src[n])) {
            kanji = true;
        }

        if(src[n] == 0) tmpend = true;
                    
        if(tmpend)
            dest[n] = ' ';
        else
            dest[n] = src[n];
    }
    
    if(kanji) dest[n-1] = ' ';
    dest[n] = 0;
}

void err_msg(char* msg, ...)
{
    char buf[BUFSIZ];

    va_list args;
    va_start(args, msg);
    vsprintf(buf, msg, args);
    va_end(args);

    myclear_online(getmaxy(stdscr)-1);
    myclear_online(getmaxy(stdscr)-2);
    mvprintw(getmaxy(stdscr)-2, 0, buf);
    refresh();
    getch();
/*    
    int key = getch();

    
    /// for file_copy, file_remove ///
    if(key == 3 || key == 7) {
        gCopyOverride = kCancel;
    }
*/    
}

void msg(char* msg)
{
    myclear_online(getmaxy(stdscr)-1);
    myclear_online(getmaxy(stdscr)-2);
    mvprintw(getmaxy(stdscr)-2, 0, msg);
    refresh();
}

int select_str(char* msg, char* str[], int len, int cancel)
{
    int cursor = 0;

    while(1) {
        /// viwe ///
        myclear_online(getmaxy(stdscr)-1);
        myclear_online(getmaxy(stdscr)-2);

        move(getmaxy(stdscr)-2, 0);
        printw(msg);
        
        for(int i=0; i< len; i++) {
            printw(" ");
            if(cursor == i) {
                attron(A_REVERSE);
                printw(str[i]);
                attroff(A_REVERSE);
            }
            else {
                printw(str[i]);
            }
        }

        refresh();

        /// input ///
        int key = getch();
        if(key == 10 || key == 13) {
            break;
        }
        else if(key == 6 || key == KEY_RIGHT) {
            cursor++;

            if(cursor >= len) cursor = len-1;
        }
        else if(key == 2 || key == KEY_LEFT) {
            cursor--;

            if(cursor < 0) cursor = 0;
        }
        else if(key == 3 || key == 7) {    // CTRL-C -G
            cursor = cancel;
            break;
        }
    }

    return cursor;
}
