/*
 * messasy
 *
 * Copyright (C) 2006,2007,2008 DesigNET, INC.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/*
 * $RCSfile: msy_config.c,v $
 * $Revision: 1.21 $
 * $Date: 2009/01/23 07:05:52 $
 */

#include <stdio.h>
#include <errno.h>
#include <limits.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/stat.h>
#include <syslog.h>
#include <pthread.h>

#define _MSY_CONFIG_C_
#include "msy_config.h"

#include "log.h"

static pthread_mutex_t config_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t config_ref_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t config_ref_old_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t config_reload_lock = PTHREAD_MUTEX_INITIALIZER;
static int config_ref = 0;
static int config_ref_old = 0;

struct config *cur_cfg = NULL;
struct config *new_cfg = NULL;
struct config *old_cfg = NULL;

/*
 * setup_config()
 *
 * Setup config structure.
 *
 * args: (void)
 *
 * Return value:
 *  struct config *	Success
 *  NULL		error
 */
static struct config *
setup_config()
{
    struct config *cf = NULL;

    // allocate memory
    cf = (struct config *)malloc(sizeof(struct config));
    if (cf == NULL) {
	log(ERR_MEMORY_ALLOCATE, "setup_config", "cf", strerror(errno));
	return NULL;
    }

    // set all members to NULL
    memset(cf, '\0', sizeof(struct config));
    return cf;
}

/*
 * is_port()
 *
 * Check if the number is port number.
 *
 * Args:
 *  int value			number
 *
 * Return value:
 *  0				Success
 *  -1				Error
 */
static int
is_port(int value)
{
     if(value > USHRT_MAX) {
          return -1;
     }
     return 0;
}

/*
 * is_syslogfacility()
 *
 * Check if the string is valid syslog facility name
 *
 * Args:
 *  char *str			facility name
 *
 * Return value:
 *  0			Success
 *  -1			Error
 */
static int
is_syslogfacility(char *str)
{
    if (syslog_facility(str) < 0) {
	return -1;
    }
    return 0;
}


/*
 * is_writable_directory()
 *
 * Check if the str is a directory and is writable.
 *
 * Args:
 *  char *str			Directory name
 *
 * Return value:
 *  NULL			Success
 *  char *			Error
 */
static int
is_writable_directory(char *str)
{
//    static char errbuf[MAX_CONFIG_LINE];
    struct stat st;

    if(stat(str, &st) < 0) {
//          sprintf(errbuf, "%s: %s", str, strerror(errno));
//          return errbuf;
	   return -1;
    }

    if(!S_ISDIR(st.st_mode)) {
//         errno = ENOTDIR;
//         sprintf(errbuf, "%s: %s", str, strerror(errno));
//         return errbuf;
	   return -1;
    }

    if(access(str, W_OK) != 0) {
//          sprintf(errbuf, "%s: %s", str, strerror(errno));
//          return errbuf;
	   return -1;
    }
//    return NULL;
    return 0;
}

/*
 * is_readable()
 *
 * Check if the file is readable.
 *
 * Args:
 *  char *str			File name
 *
 * Return value:
 *  NULL			Success
 *  char *			Error
 */
static int
is_readable(char *str)
{
//    static char errbuf[MAX_CONFIG_LINE];

    struct stat st;

    if(stat(str, &st) < 0) {
//	sprintf(errbuf, "%s: %s", str, strerror(errno));
//	return errbuf;
	   return -1;
    }

    if(S_ISDIR(st.st_mode)) {
//	errno = EISDIR;
//	sprintf(errbuf, "%s: %s", str, strerror(errno));
	//return errbuf;
	   return -1;
    }

    if(access(str, R_OK) != 0) {
	//sprintf(errbuf, "%s: %s", str, strerror(errno));
	//return errbuf;
	   return -1;
    }
    //return NULL;
    return 0;
}

static struct filter *
split_filter(char *str)
{
    struct filter *f = NULL, *nf = NULL;
    char *p, *q;
    int num = 0;

    if (str == NULL) {
	return NULL;
    }

    for (p = str; (q = strchr(p, ',')) != NULL; ) {
	*q = '\0';
	if (*p == '\0') {
	    /* skip if no column value found */
	    p = q + 1;
	    continue;
	}
	num++;
	nf = (struct filter *)realloc(f, sizeof(struct filter) * (num + 1));
	if (nf == NULL) {
	    free_filter(f);
	    return NULL;
	}
	f = nf;
	f[num - 1].fl_string = strdup(p);
	if (f[num - 1].fl_string == NULL) {
	    free_filter(f);
	    return NULL;
	}
	p = q + 1;
	f[num].fl_string = NULL;
    }

    num++;
    if (*p == '\0') {
	/* skip if no column value found */
	return f;
    }
    nf = (struct filter *)realloc(f, sizeof(struct filter) * (num + 1));
    if (nf == NULL) {
	free_filter(f);
	return NULL;
    }
    f = nf;
    f[num - 1].fl_string = strdup(p);
    if (f[num - 1].fl_string == NULL) {
	free_filter(f);
	return NULL;
    }
    f[num].fl_string = NULL;

    return f;
}

/*
 * read_config()
 *
 * Read configuration from the file.
 *
 * Args:
 *  char *file			File name of the configuration file.
 *  struct cfentry *cfg_gen	Configuration definision structure.
 *  int count			Number of configuration keywords.
 *  void *config		Configuration structure.
 *
 * Return value:
 *  0				Success
 *  >0				Config file error
 *  CFG_NG			System error
 */
int
read_config(char *file, struct cfentry *cfg_gen, int count, void *config)
{
    int  i;
    int  len;
    char line[MAX_CONFIG_LINE + 1];
    FILE *fp;
    char *ptr;
    int   nline;
    int   error = 0;
    char *str;
    int  *check;

    // check if the file is readable 
    if ((is_readable(file)) != 0) {
	log(ERR_CONFIG_FILE_OPEN, "read_config", file, strerror(errno));
	return -1;
    }

    // open the file
    fp = fopen(file, "ro");
    if (fp == NULL) {
	log(ERR_CONFIG_FILE_OPEN, "read_config", file, strerror(errno));
	return CFG_NG;
    }

    // prepare the check matrix
    check = (int *)malloc((sizeof(int)) * count);
    if (check == NULL) {
	log(ERR_MEMORY_ALLOCATE, "read_config", "check", strerror(errno));
	fclose(fp);
	return CFG_NG;
    }
    memset(check, 0, sizeof(int) * count);

    // read file 
    for (nline = 1; fgets(line, MAX_CONFIG_LINE + 1, fp) != NULL; nline ++) {
	if (strchr(line, '\n') == NULL) {
	    // seems too long line
	    log(ERR_CONFIG_LINE_LONG, "read_config", file, nline);
	    for (;;) {
		// skip this line
		fgets(line, MAX_CONFIG_LINE + 1, fp);
		if (strchr(line, '\n') != NULL) {
		    break;
		}
	    }
	    error++;
	    continue;
	}
	if ((line[0] == '#') || (line[0] == '\n')) {
	    // #(comment) or empty line
	    continue;
	}

	// search config keyword
	for (i = 0; i < count; i ++) {
	    len = strlen(cfg_gen[i].cf_name);

	    if ((strncasecmp(line, cfg_gen[i].cf_name, len) == 0)
		&& (line[len] == '=') ) {
		// find the keyword

		// check duplicated
		if (check[i]) {
		    log(ERR_CONFIG_DUPLICATED, "read_config",
			cfg_gen[i].cf_name, nline, file);
		    error ++;
		    break;
		}

		str = &line[len + 1];
		check[i] = 1;

		if (isspace((int)line[len + 1])) {
                    if ((strcmp(cfg_gen[i].cf_name, DOT_DELIMITER) != 0) &&
                        (strcmp(cfg_gen[i].cf_name, SLASH_DELIMITER) != 0)) {
			log(ERR_CONFIG_SYNTAX,
			    "read_config", cfg_gen[i].cf_name, nline, file);
			error++;
			break;
		    }
		}
                
		switch(cfg_gen[i].cf_type) {
		case CF_INTEGER:
		case CF_PLUS:
		{
		    unsigned long val;
		    char *tptr;

		    val = strtoul(str, &tptr, 10);
		    if (*tptr != '\n') {
			// invalid line
			log(ERR_CONFIG_SYNTAX,
			    "read_config", cfg_gen[i].cf_name, nline, file);
			error ++;
			break;
		    } else {
			if((val == ULONG_MAX) && (errno == ERANGE)) {
			    // out of range
			    log(ERR_CONFIG_OUT_OF_RANGE,
				"read_config", cfg_gen[i].cf_name, nline, file);
			    error ++;
			    break;
			}

			if (cfg_gen[i].cf_type == CF_PLUS) {
			    if ((val == 0) || (val > LONG_MAX)) {
				// seems negative number
			        log(ERR_CONFIG_VALUE_MINUS,
				    "read_config", cfg_gen[i].cf_name, nline, file);
				    error ++;
			 	    break;
			    }
			}

			// call check function
			if (cfg_gen[i].cf_check != 0) {
			    if (((*cfg_gen[i].cf_check)(val)) != 0 ) {
				log(ERR_CONFIG_SYNTAX, "read_config",
				    cfg_gen[i].cf_name, nline, file);
				error ++;
				break;
			    }
			}
		    }

		    *((unsigned int *)(config + cfg_gen[i].cf_dataoffset)) = val;
		    break;
		}

		case CF_STRING:
		case CF_MULTI:
		{

		    char *vstr = NULL;

		    ptr = strchr(str, '\n');

		    // trim LF chr.
		    *ptr = '\0';

		    // call check function
		    if (cfg_gen[i].cf_check != 0) {
			if (((*cfg_gen[i].cf_check)(str)) != 0) {
			    log(ERR_CONFIG_SYNTAX, "read_config",
				cfg_gen[i].cf_name, nline, file);
			    error ++;
			    break;
			}
		    }

		    if (cfg_gen[i].cf_type == CF_MULTI) {
			struct filter *f = NULL;
			f = split_filter(str);
			if (f == NULL) {
			    log(ERR_MEMORY_ALLOCATE, "read_config",
				"f", strerror(errno));
			    error ++;
			} else {
			    *((struct filter **)(config + cfg_gen[i].cf_dataoffset)) = f;
			}
		    } else {
			// copy value to heap area
			vstr = strdup(str);
			if (vstr == NULL) {
			    log(ERR_MEMORY_ALLOCATE, "read_config",
				"vstr", strerror(errno));
			    error ++;
			} else {
			    *((char **)(config + cfg_gen[i].cf_dataoffset)) = vstr;
			}
		    }
		    break;

		}

		default:
		    // unknown data type (not reached)
		    log(ERR_CONFIG_SYNTAX, "read_config",
			cfg_gen[i].cf_name, nline, file);
		    error ++;
		    break;
		}
		break;
	    }
	}
	if (i == count) {
	    // invalid keyword
	    log(ERR_CONFIG_UNKNOWN_NAME, "read_config", file, nline);
	    error ++;
	}
    }
    fclose(fp);

    for(i = 0; i < count; i++) {
	if(!check[i]) {
	    // set default value if allowed
	    if(cfg_gen[i].cf_default) {
		switch(cfg_gen[i].cf_type) {
		case CF_INTEGER:
		case CF_PLUS:
		    *((unsigned int *)(config + cfg_gen[i].cf_dataoffset)) =
                              atoi(cfg_gen[i].cf_default);
		    break;
		case CF_STRING:
		    str = strdup(cfg_gen[i].cf_default);
		    if(str == NULL) {
			log(ERR_MEMORY_ALLOCATE, "read_config",
			    "str", strerror(errno));
			error++;
		    } else {
			*((char **)(config + cfg_gen[i].cf_dataoffset)) = str;
		    }
		    break;
		}
		continue;
	    } else {
		log(ERR_CONFIG_MUST_BE_SET, "read_config",
		    cfg_gen[i].cf_name, file);
		error++;
	    }
	}
    }
    free(check);
    return error;
}

void
free_filter(struct filter *fl)
{
    int i = 0;
    if (fl == NULL) {
	return;
    }
    for (i = 0; fl[i].fl_string != NULL; i ++) {
	free(fl[i].fl_string);
    }
    free(fl);
}

struct filter *
dup_filter(struct filter *fl)
{
    struct filter *f = NULL;
    int i;

    if (fl == NULL) {
	return NULL;
    }

    for (i = 0; fl[i].fl_string != NULL; i++) {
	f = (struct filter *) realloc(f, sizeof(struct filter) * (i + 2));
	if (f == NULL) {
	    log(ERR_MEMORY_ALLOCATE, "dup_filter", "f", strerror(errno));
	    free_filter(f);
	    return NULL;
	}
	f[i].fl_string = strdup(fl[i].fl_string);
	if (f[i].fl_string == NULL) {
	    log(ERR_MEMORY_ALLOCATE, "dup_filter",
		"f[i].fl_string", strerror(errno));
	    free_filter(f);
	    return NULL;
	}
	f[i + 1].fl_string = NULL;
    }

    return f;
}

static void
free_config(struct config *cfg)
{
    if (cfg == NULL) {
	return;
    }
    if (cfg->cf_maildir != NULL) {
	free(cfg->cf_maildir);
    }
    if (cfg->cf_mailfolder != NULL) {
	free(cfg->cf_mailfolder);
    }
    if (cfg->cf_savemailaddress != NULL) {
	free_filter(cfg->cf_savemailaddress);
    }
    if (cfg->cf_syslogfacility != NULL) {
	free(cfg->cf_syslogfacility);
    }
    if (cfg->cf_listenip != NULL) {
	free(cfg->cf_listenip);
    }
    if (cfg->cf_mydomain != NULL) {
	free_filter(cfg->cf_mydomain);
    }
    if (cfg->cf_dotdelimiter != NULL) {
	free(cfg->cf_dotdelimiter);
    }
    if (cfg->cf_slashdelimiter != NULL) {
	free(cfg->cf_slashdelimiter);
    }
    free(cfg);
    return;
}

static void *
cfree_handler(void *arg)
{
    int ret = 0;
    pthread_mutex_lock(&config_reload_lock);
    free_config(old_cfg);
    old_cfg = NULL;
    pthread_mutex_unlock(&config_reload_lock);
    pthread_exit(&ret);
    return NULL;
}

/*
 * config_init()
 *
 * Get current config structure pointer,
 * and countup the reference counter.
 *
 * Args: (void)
 *
 * Return value:
 *  struct config *		config structure pointer
 */
struct config *
config_init()
{
    struct config *ret_ptr;

    // countup reference counter
    pthread_mutex_lock(&config_lock);
    pthread_mutex_lock(&config_ref_lock);
    config_ref ++;

    // get config structure pointer
    ret_ptr = cur_cfg;

    pthread_mutex_unlock(&config_ref_lock);
    pthread_mutex_unlock(&config_lock);
    return ret_ptr;
}

/*
 * config_release()
 * 
 * Countdown config reference counter.
 *
 * Args:
 *  struct config *cfg		To release pointer.
 *
 * Return value:
 *  (void)
 */
void
config_release(struct config *cfg)
{
    pthread_mutex_lock(&config_lock);
    if (old_cfg == cfg) {
	// case to release old config
	pthread_mutex_lock(&config_ref_old_lock);
	config_ref_old --;
	if (config_ref_old == 0) {
	    pthread_mutex_unlock(&config_reload_lock);
	}
	pthread_mutex_unlock(&config_ref_old_lock);
    } else {
	// case to release cur config
	pthread_mutex_lock(&config_ref_lock);
	config_ref --;
	pthread_mutex_unlock(&config_ref_lock);
    }
    pthread_mutex_unlock(&config_lock);
}

/*
 * reload_config()
 *
 * Reload configuration file
 *
 * Args:
 *  char *file		Configuration file name
 *
 * Return value:
 *  0			Success
 *  CFG_NG		System error
 *  1			Temporaly error (during reloading)
 *  2			Temporaly error (because of config file)
 */
int
reload_config(char *file)
{
    struct config *cfg = NULL;
    int ret;
    pthread_t cfree;
    unsigned int *ccfp, *cfp;
    int i;

    pthread_mutex_lock(&config_ref_old_lock);
    ret = config_ref_old;
    pthread_mutex_unlock(&config_ref_old_lock);
    if (ret > 0) {
	// case reloading
	log(ERR_CONFIG_RELOADING, "reload_config", file);
	return 1;
    }

    // setup config structure
    cfg = setup_config();
    if (cfg == NULL) {
	return -1;
    }

    // read config file
    ret = read_config(file, cfe, NCONFIG, cfg);

    if (ret > 0) {
	// case config file error
	free_config(cfg);
	return 2;
    }
    if (ret == CFG_NG) {
	// case system error
	free_config(cfg);
	return CFG_NG;
    }

    // change config structure
    pthread_mutex_lock(&config_lock);
    pthread_mutex_lock(&config_ref_lock);
    pthread_mutex_lock(&config_ref_old_lock);
    config_ref_old = config_ref;
    config_ref = 0;
    if (config_ref_old > 0) {
	pthread_mutex_lock(&config_reload_lock);
    }
    pthread_mutex_unlock(&config_ref_old_lock);
    pthread_mutex_unlock(&config_ref_lock);

    // replace ignored items in reloading
    if (cur_cfg != NULL) {
	for (i = 0; i < NCONFIG; i++) {
	    if (cfe[i].cf_reload == RELOAD) {
		continue;
	    }
	    switch (cfe[i].cf_type) {
		case CF_STRING:
		{
		    ccfp = (void *)cur_cfg + cfe[i].cf_dataoffset;
		    cfp = (void *)cfg + cfe[i].cf_dataoffset;
		    free(*((char **)cfp));

		    *((char **)cfp) = strdup(*((char **)ccfp));
		    if (*((char **)cfp) == NULL) {
			free_config(cfg);
			pthread_mutex_unlock(&config_lock);
			log(ERR_MEMORY_ALLOCATE, "reload_config",
			    "*((char **)cfp)", strerror(errno));
			return CFG_NG;
		    }
		    break;
		}
		case CF_PLUS:
		{
		    ccfp = (void *)cur_cfg + cfe[i].cf_dataoffset;
		    cfp = (void *)cfg + cfe[i].cf_dataoffset;
		    *((unsigned int *)cfp) = *((unsigned int*)ccfp);
		    break;
		}
	    }
	}
    }

    // create config free thread
    ret = pthread_create(&cfree, NULL, cfree_handler, NULL);
    if (ret != 0) {
	free_config(cfg);
	pthread_mutex_unlock(&config_lock);
	log(ERR_THREAD_CREATE, "reload_config",
	    "cfree_handler", strerror(errno));
	return CFG_NG;
    }
    ret = pthread_detach(cfree);
    if (ret != 0) {
	free_config(cfg);
	pthread_mutex_unlock(&config_lock);
	log(ERR_THREAD_DETACH, "reload_config",
	    "cfree_handler", strerror(errno));
	return CFG_NG;
    }

    // switch pointer
    old_cfg = cur_cfg;
    cur_cfg = cfg;
    switch_log(cfg->cf_syslogfacility);
    pthread_mutex_unlock(&config_lock);

    return 0;
}

/*
 * is_ip()
 *
 * Check if the string is valid ip address.
 *
 * Args:
 *  char *str			string
 *
 * Return value:
 *  0				Success
 *  <0				Error
 */
static int
is_ip(char *str)
{
    char data[DOT_ADDRESS + 1];
    char *p, *q;
    int i;
    int len;
    int num;
    int count;
    int dot_addr_cnt = 0;

    p = str;
    count = strlen(str) + 1;

    // character check
    for (i = 0; i < count; i++) {

	// number check
        if ((isdigit(str[i]) != 0)) {
            continue;
        }

	// delimiter check
        if ((str[i] != '.') && (str[i] != '\0')){
            return -1;
        }

	// length check
	len = str + i - p;
	if (len == 0 || len > 3) {
	    return -1;
	}

        q = strncpy(data ,p ,len);
	*(q + len) = '\0';
	// value check
        num = atoi(q);
        if (num < 0 || num > 255) {
            return -1;
        }

        p = str + i + 1;
        dot_addr_cnt++;
    }

    // xxx.xxx.xxx.xxx
    if (dot_addr_cnt != 4) {
        return -1;
    }
    return 0;
}

/*
 * is_mailfolder()
 *
 * Check if the string is valid character.
 *
 * Args:
 *  char *str			string
 *
 * Return value:
 *  0				Success
 *  -1				Error
 */
static int
is_mailfolder(char *str)
{
    char string[] = CHAR_MAILFOLDER;
    int i, j;

    if (str == NULL) {
	return -1;
    }

    // the first character check
    if (str[0] != '.') {
	return -1;
    }

    // character check
    for (i = 0; str[i] != '\0'; i++) {

        // exclusion ".."
	if ((str[i] == '.') && (str[i + 1] == '.')) {
	    return -1;
	}

        for (j = 0; string[j] != '\0'; j++) {
            if (str[i] == string[j]){
                break;
            }
        }
        if (string[j] == '\0') {
            return -1;
        }
    }
    if (str[i - 1] == '.') {
	return -1;
    }
    return 0;
}

/*
 * is_dotdelimiter()
 *
 * Check if the string is valid character.
 *
 * Args:
 *  char *str			string
 *
 * Return value:
 *  0				Success
 *  -1				Error
 */
static int
is_dotdelimiter(char *str)
{
    char string[] = CHAR_DOT_DELIMITER;
    int i;

    if (str == NULL) {
	return -1;
    }

    // not cne character
    if (str[1] != '\0') {
        return -1;
    }

    // character check
    for (i = 0; string[i] != '\0'; i++) {
        if (str[0] == string[i]){
            break;
        }
    }
    if (string[i] == '\0' ) {
        return -1;
    }
    return 0;
}

/*
 * is_slashdelimiter()
 *
 * Check if the string is valid character.
 *
 * Args:
 *  char *str			string
 *
 * Return value:
 *  0				Success
 *  -1				Error
 */
static int
is_slashdelimiter(char *str)
{
    char string[] = CHAR_SLASH_DELIMITER;
    int i;

    if (str == NULL) {
	return -1;
    }

    // not one character
    if (str[1] != '\0') {
        return -1;
    }

    // character check
    for (i = 0; string[i] != '\0'; i++) {
        if (str[0] == string[i]){
            break;
        }
    }
    if (string[i] == '\0' ) {
        return -1;
    }
    return 0;
}

/*
 * is_savemailaddress()
 *
 * Check if the string is valid character.
 *
 * Args:
 *  char *str			string
 *
 * Return value:
 *  0				Success
 *  -1				Error
 */
static int
is_savemailaddress(char *str)
{
    if (strchr(str, SPACE) != NULL) {
	return -1;
    }
    return 0;
}
