/*
 * messasy
 *
 * Copyright (C) 2006,2007,2008 DesigNET, INC.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/*
 * $RCSfile: messasy.c,v $
 * $Revision: 1.21 $
 * $Date: 2009/04/08 01:53:44 $
 */

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <pthread.h>

#include <libmilter/mfapi.h>
#include "log.h"
#include "msy_config.h"
#include "maildrop.h"
#include "messasy.h"
#include "client_side.h"
#include "filter.h"

#define MLFIPRIV        ((struct mlfiPriv *) smfi_getpriv(ctx))

#define NOTSAVED 0
#define SAVED 1
#define WRITING 2

/*
 *  sfsistat mlfi_envfrom
 *  env
 */
sfsistat
mlfi_envfrom(SMFICTX *ctx, char **envfrom)
{
    struct mlfiPriv *priv;
    int filtercn;
    int len;
    char *fromaddr;

    /* mail save */
    priv = malloc(sizeof *priv);
    if(priv == NULL) {
	log(ERR_MEMORY_ALLOCATE, "mlfi_envfrom", "priv", strerror(errno));
        return SMFIS_ACCEPT;
    }

    memset(priv, '\0', sizeof *priv);

    /* read config */
    priv->mlfi_conf = config_init();

    len = strlen(*envfrom);

    fromaddr = (char *)malloc(len + 1);
    if(fromaddr == NULL) {
	log(ERR_MEMORY_ALLOCATE, "mlfi_envfrom", "fromaddr", strerror(errno));
        return SMFIS_ACCEPT;
    }

    /* get mail addr */
    if (filter_getaddress(*envfrom, fromaddr, len) != 0) {
        free(fromaddr);
	log(ERR_MAIL_FIND_ADDRESS, "mlfi_envfrom");
        return SMFIS_ACCEPT;
    }

    /* save from addr temporally */
    priv->mlfi_savefrom = fromaddr;

    /* mail adder filter */
    filtercn = filter_address_search(fromaddr,
		      (struct filter *) priv->mlfi_conf->cf_savemailaddress);
    /* filter error */
    if (filtercn == -2) {
	log(ERR_MAIL_FIND_ADDRESS, "mlfi_envfrom");
        mlfi_cleanup(ctx, false);
        return SMFIS_ACCEPT;
    }

    /* if matched */
    if (filtercn >= 0) {
	/* set save flag */
	priv->mlfi_mailsave = SAVED;

	priv->mlfi_maildrop = maildrop_open(priv->mlfi_conf,
					    priv->mlfi_savefrom);
	if(priv->mlfi_maildrop == NULL) {
	    mlfi_cleanup(ctx, false);
	    return SMFIS_ACCEPT;
	}
    }

    /* copy buffer */
    smfi_setpriv(ctx, priv);

    return SMFIS_CONTINUE;
}

/*
 *  sfsistat mlfi_envrcpt
 *  env
 */
sfsistat
mlfi_envrcpt(SMFICTX *ctx, char **rcptto)
{
    int filtercn;
    int len;
    char *rcptaddr;

    len = strlen(*rcptto);

    rcptaddr = (char *)malloc(len + 1);
    if(rcptto == NULL) {
	log(ERR_MEMORY_ALLOCATE, "mlfi_envrcpt", "rcptto", strerror(errno));
	mlfi_cleanup(ctx, false);
	return SMFIS_ACCEPT;
    }

    /* get mail addr */
    if (filter_getaddress(*rcptto, rcptaddr, len) != 0) {
	log(ERR_MAIL_FIND_ADDRESS, "mlfi_envrcpt");
	mlfi_cleanup(ctx, false);
	free(rcptaddr);
	return SMFIS_ACCEPT;
    }

    /* push rcpt addr */
    if (push_rcptlist(&(MLFIPRIV->mlfi_savercpt), rcptaddr) != 0) {
	mlfi_cleanup(ctx, false);
	free(rcptaddr);
	return SMFIS_ACCEPT;
    }

    /* do nothing when already matched */
    if ((MLFIPRIV->mlfi_mailsave) == SAVED) {
	free(rcptaddr);
	return SMFIS_CONTINUE;
    }

    /* mail adder filter */
    filtercn = filter_address_search(rcptaddr,
		     (struct filter *)MLFIPRIV->mlfi_conf->cf_savemailaddress);
    free(rcptaddr);
    /* filter error */
    if (filtercn == -2) {
	log(ERR_MAIL_FIND_ADDRESS, "mlfi_envrcpt");
	mlfi_cleanup(ctx, false);
	return SMFIS_ACCEPT;
    }

    /* none matched */
    if (filtercn == -1) {
	return SMFIS_CONTINUE;
    }

    /* set save flag */
    MLFIPRIV->mlfi_mailsave = SAVED;

    /* mail drop */
    MLFIPRIV->mlfi_maildrop = maildrop_open(MLFIPRIV->mlfi_conf,
					    MLFIPRIV->mlfi_savefrom);
    if(MLFIPRIV->mlfi_maildrop == NULL) {
        mlfi_cleanup(ctx, false);
        return SMFIS_ACCEPT;
    }

    return SMFIS_CONTINUE;
}

/*
 * sfsistat mlfi_header
 * header
 */
sfsistat
mlfi_header(SMFICTX *ctx, char *headerf, char *headerv)
{
    char *line;
    int len, i;

    /* nothing to do */
    if ((MLFIPRIV->mlfi_mailsave) == NOTSAVED) {
        mlfi_cleanup(ctx, true);
	return SMFIS_ACCEPT;
    }

    /* write custom header if not written yet */
    if ((MLFIPRIV->mlfi_mailsave) == SAVED) {
	MLFIPRIV->mlfi_mailsave = WRITING;

	/* write from */
	/* [X-Messasy-From: xxx@yyyy.zzz\n\0] */
	line = malloc(sizeof(HEADER_FROM)
		      + strlen(MLFIPRIV->mlfi_savefrom)
		      + 4);
	if (line == NULL) {
	    log(ERR_MEMORY_ALLOCATE, "mlfi_eoh", "line", strerror(errno));
	    mlfi_cleanup(ctx, false);
	    return SMFIS_ACCEPT;
	}
	len = sprintf(line, "%s: %s\n", HEADER_FROM, MLFIPRIV->mlfi_savefrom);
	if (maildrop_write_body(MLFIPRIV->mlfi_maildrop, line, len) == -1 ) {
	    mlfi_cleanup(ctx, false);
	    free(line);
	    return SMFIS_ACCEPT;
	}
	free(line);

	/* write to */
	for (i = 0; (MLFIPRIV->mlfi_savercpt + i)->rcpt_addr != NULL; i++) {
	    /* [X-Messasy-To: xxx@yyyy.zzz\n\0] */
	    line = malloc(sizeof(HEADER_TO)
			  + (MLFIPRIV->mlfi_savercpt + i)->rcpt_addr_len
			  + 4);
	    if (line == NULL) {
		log(ERR_MEMORY_ALLOCATE, "mlfi_eoh", "line", strerror(errno));
		mlfi_cleanup(ctx, false);
		return SMFIS_ACCEPT;
	    }
	    len = sprintf(line, "%s: %s\n", HEADER_TO,
			  (MLFIPRIV->mlfi_savercpt + i)->rcpt_addr);
	    if (maildrop_write_body(MLFIPRIV->mlfi_maildrop,
						line, len) == -1 ) {
		mlfi_cleanup(ctx, false);
		free(line);
		return SMFIS_ACCEPT;
	    }
	    free(line);
	}
    }

    /* write header */
    if (maildrop_write_header(MLFIPRIV->mlfi_maildrop,
                                              headerf, headerv) == -1 ) {
        /* error free maildrop */
        mlfi_cleanup(ctx, false);
        return SMFIS_ACCEPT;
    }

    return SMFIS_CONTINUE;
}

/*
 * mlfi_eoh
 * header & body
 */
sfsistat
mlfi_eoh(SMFICTX *ctx)
{
    /* write new line between header and body */
    if (maildrop_write_body(MLFIPRIV->mlfi_maildrop,
                                                (u_char *) "\n", 1) == -1 ) {
        /* error free maildrop */
        mlfi_cleanup(ctx, false);
        return SMFIS_ACCEPT;
    }

    return SMFIS_CONTINUE;
}

/*
 * mlfi_body
 * write body
 */
sfsistat
mlfi_body(SMFICTX *ctx, u_char *bodyp, size_t bodylen)
{
    /* write body */
    if (maildrop_write_body(MLFIPRIV->mlfi_maildrop, bodyp, bodylen) == -1 ) {
        /* error free maildrop */
        mlfi_cleanup(ctx, false);
        return SMFIS_ACCEPT;
    }

    return SMFIS_CONTINUE;
}

/*
 * mlfi_eom
 * end
 */
sfsistat
mlfi_eom(SMFICTX *ctx)
{
    /* close */
    maildrop_close(MLFIPRIV->mlfi_maildrop);
    return mlfi_cleanup(ctx, true);
}

/*
 * mlfi_close
 * close
 */
sfsistat
mlfi_close(SMFICTX *ctx)
{
    return SMFIS_ACCEPT;
}

/*
 * mlfi_abort
 * abort
 */
sfsistat
mlfi_abort(SMFICTX *ctx)
{
    return mlfi_cleanup(ctx, false);
}

/*
 * mlfi_cleanup
 * cleanup
 */
sfsistat
mlfi_cleanup(SMFICTX *ctx, bool ok)
{
    sfsistat rstat = SMFIS_CONTINUE;
    struct mlfiPriv *priv = MLFIPRIV;

    /* null data return */
    if(priv == NULL) {
        return rstat;
    }

    /* config relase */
    if (priv->mlfi_conf != NULL) {
        config_release(priv->mlfi_conf);
    }

    if (ok == false) {
        maildrop_abort(priv->mlfi_maildrop);
    }

    /* priv release */
    if (priv->mlfi_savefrom != NULL) {
	free(priv->mlfi_savefrom);
    }
    if (priv->mlfi_savercpt != NULL) {
	free_rcptlist(priv->mlfi_savercpt);
    }
    free(priv);

    /* set NULL */
    smfi_setpriv(ctx, NULL);

    return rstat;
}

static void *
call_command(void *arg)
{
    struct thread_control *tcl;

    tcl = (struct thread_control *)arg;

    accept_command(tcl->configname , tcl->addr, tcl->port);

    return (NULL);  /* thread exit */
}

/*
 * usage
 * 
 */
void
usage(char *arg)
{
    log("usage: %s [-t timeout] [config file]", arg);
}

int
main(int argc, char *argv[])
{

    char  *args = "t:";
    char  *confname;
    char  *tmp;
    char   c;
    char  oconn[OCONN_LENGTH];
    char  defaultconf[MAX_CONFIG_LINE];
    int    timeout = DEFAULT_TIMEOUT;
    unsigned long tmp_val;
    struct thread_control tcl;
    struct config *cfg;

    /* set error output to stderr */
    init_log();

    /* arg */
    while ((c = getopt(argc, argv, args)) != -1) {

        switch (c) {

        /* arg time out */
        case 't':
            if((optarg == NULL) || (optarg[0] == '\0')) {
                usage(argv[0]);
                exit(1);
            }

            tmp_val = strtoul(optarg, &tmp, 10);
            if((*tmp != '\0') ||
               ((tmp_val == ULONG_MAX) && (errno == ERANGE)) ||
	       (tmp_val > INT_MAX)) {
                usage(argv[0]);
                exit(1);
            }

            timeout = (int) tmp_val;
            break;

        /* error */
        default:
            usage(argv[0]);
            exit(1);
        }
    }

    /* check arg */
    if(optind + 1 != argc) {

	/* set default config name */
	sprintf(defaultconf, "%s/messasy.conf", DEFAULT_CONFDIR);
	confname = defaultconf;

    } else {

	/* set config name */
	confname = argv[optind];
    }

    /* read config file */
    if (reload_config(confname) != 0) {
        exit(2);
    }

    cfg = config_init();

    /* set thread data */
    tcl.configname = confname;
    tcl.port = cfg->cf_commandport;
    tcl.addr = strdup(cfg->cf_listenip);
    if (tcl.addr == NULL) {
	log(ERR_MEMORY_ALLOCATE, "main", "tcl.addr", strerror(errno));
	exit(2);
    }

    /* prepare socket string */
    sprintf(oconn, OCONN, cfg->cf_listenport, cfg->cf_listenip);

    /* config relase */
    config_release(cfg);

    /* command line thread */
    pthread_create(&child, NULL, call_command, &tcl);

    /* set socket */
    if(smfi_setconn(oconn) == MI_FAILURE) {
        log(ERR_MILTER_SET_SOCKET, "main", strerror(errno));
        exit(3);
    }

    /* set time out */
    if(smfi_settimeout(timeout) == MI_FAILURE) {
        log(ERR_MILTER_SET_TIMEOUT, "main", strerror(errno));
        exit(3);
    }

    /* set register */
    if(smfi_register(smfilter) == MI_FAILURE) {
        log(ERR_MILTER_REGISTER, "main", strerror(errno));
        exit(4);
    }

    /* run main */
    if(smfi_main() == MI_FAILURE) {
        log(ERR_MILTER_START, "main", strerror(errno));
        exit(5);
    }

    exit(0);
}
