/*
 * messasy
 *
 * Copyright (C) 2006,2007,2008 DesigNET, INC.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/*
 * $RCSfile: maildrop.c,v $
 * $Revision: 1.16 $
 * $Date: 2009/01/27 06:24:10 $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/utsname.h>
#include <iconv.h>
#include <errno.h>
#include <ctype.h>

#include "msy_config.h"
#include "maildrop.h"
#include "filter.h"
#include "log.h"

int  md_makemaildir(char *, int);
int  md_makesubdir(char *);
int  md_movefile(struct maildrop *);
void md_free(struct maildrop *);
char *md_makefolder(struct maildrop *);
int  md_makesavefile(struct maildrop *, char *);

/*
 * maildrop_open
 *
 * Function
 *	Initialize maildrop function.
 *
 * Argument
 *	struct config *config	Pointer to config structure.
 *
 * Return value
 *	Pointer to maildrop structure
 */
struct maildrop *
maildrop_open(struct config *config, char *envfrom)
{
    struct maildrop *md;
    struct utsname utsname;
    mode_t old_umask;
    int ret;

    if (config == NULL) {
	log(ERR_CONFIG_NULL, "maildrop_open");
        return(NULL);
    }
    if (config->cf_maildir == NULL) {
	log(ERR_CONFIG_MAILDIR_NULL, "maildrop_open");
        return(NULL);
    }
    if (config->cf_mailfolder == NULL) {
	log(ERR_CONFIG_MAILFOLDER_NULL, "maildrop_open");
        return(NULL);
    }

    /* Allocate maildrop structure */
    md = (struct maildrop *) malloc(sizeof(struct maildrop));
    if (md == NULL) {
	log(ERR_MEMORY_ALLOCATE, "maildrop_open", "md", strerror(errno));
        return(NULL);
    }

    md->md_fd = -1;
    md->md_maildir = NULL;
    md->md_maildir_len = 0;
    md->md_mailfolder = NULL;
    md->md_mailfolder_len = 0;
    md->md_hostname = NULL;
    md->md_hostname_len = 0;
    md->md_from = NULL;
    md->md_from_len = 0;
    md->md_replaceaddr = NULL;
    md->md_replaceaddr_len = 0;
    md->md_curtime = time(NULL);
    md->md_tempfile = NULL;
    md->md_cr = 0;
    md->md_dotdelimiter = '\0';
    md->md_slashdelimiter = '\0';
    md->md_mydomain = NULL;

    /* Get dot delimiter */
    md->md_dotdelimiter = config->cf_dotdelimiter[0];

    /* Get slash delimiter */
    md->md_slashdelimiter = config->cf_slashdelimiter[0];

    /* Get maildir base directory */
    md->md_maildir = strdup(config->cf_maildir);
    if (md->md_maildir == NULL) {
        md_free(md);
	log(ERR_MEMORY_ALLOCATE,
	    "maildrop_open", "md->md_maildir", strerror(errno));
        return(NULL);
    }
    md->md_maildir_len = strlen(md->md_maildir);

    /* Get maildir folder */
    md->md_mailfolder = strdup(config->cf_mailfolder);
    if (md->md_mailfolder == NULL) {
        md_free(md);
	log(ERR_MEMORY_ALLOCATE,
	    "maildrop_open", "md->md_mailfolder", strerror(errno));
        return(NULL);
    }
    md->md_mailfolder_len = strlen(md->md_mailfolder);

    /* Save envelope from */
    md->md_from = strdup(envfrom);
    if (md->md_from == NULL) {
	md_free(md);
	log(ERR_MEMORY_ALLOCATE,
	    "maildrop_open", "md->md_from", strerror(errno));
	return(NULL);
    }
    md->md_from_len = strlen(md->md_from);

    /* Get domain filter */
    md->md_mydomain = dup_filter(config->cf_mydomain);
    if (md->md_mydomain == NULL) {
	md_free(md);
	log(ERR_MEMORY_ALLOCATE,
	    "maildrop_open", "md->md_mydomain", strerror(errno));
	return(NULL);
    }

    ret = filter_domain_search(envfrom, 
			(struct filter *) md->md_mydomain);

    /* filter error */
    if (ret == -2) {
	md_free(md);
        log(ERR_MAIL_FIND_ADDRESS, "maildrop_open");
	return(NULL);
    }

    /* none matched */
    if (ret == -1) {
	md->md_replaceaddr = strdup(OTHER);
	if (md->md_replaceaddr == NULL) {
	    md_free(md);
	    log(ERR_MEMORY_ALLOCATE,
	        "maildrop_open", "md->md_replaceaddr", strerror(errno));
            return(NULL);
	}
	md->md_replaceaddr_len = sizeof(OTHER);
    }

    /* Get hostname */
    if (uname(&utsname) < 0) {
        md_free(md);
	log(ERR_KERNEL_GET_INFO, "maildrop_open", strerror(errno));
        return(NULL);
    }
    md->md_hostname = strdup(utsname.nodename);
    if (md->md_hostname == NULL) {
        md_free(md);
	log(ERR_MEMORY_ALLOCATE,
	    "maildrop_open", "md->md_hostname", strerror(errno));
        return(NULL);
    }
    md->md_hostname_len = strlen(md->md_hostname);

    /* Make maildir base directory */
    if (md_makemaildir(md->md_maildir, md->md_maildir_len) < 0) {
        md_free(md);
        return(NULL);
    }

    /* Open temporary file name */
    md->md_tempfile = (char *) malloc(md->md_maildir_len +
                                      md->md_hostname_len + TMPFILE_LEN);
    if (md->md_tempfile == NULL) {
        md_free(md);
	log(ERR_MEMORY_ALLOCATE,
	    "maildrop_open", "md->md_tempfile", strerror(errno));
        return(NULL);
    }
    sprintf(md->md_tempfile, TMPFILE_NAME,
            md->md_maildir, md->md_curtime, md->md_hostname);
    old_umask = umask(0077);
    md->md_fd = mkstemp(md->md_tempfile);
    umask(old_umask);
    if (md->md_fd < 0) {
        md_free(md);
	log(ERR_FILE_CREATE_TMPFILE,
	    "maildrop_open", md->md_tempfile, strerror(errno));
        return(NULL);
    }

    return(md);
}

/*
 * maildrop_close
 *
 * Function
 *	End maildrop function. (normal)
 *	1. Close temporary file.
 *	2. Move to save file from temporary file.
 *
 * Argument
 *	struct maildrop *md	Pointer to maildrop structure.
 *
 * Return value
 *	0	Normal end.
 *	-1	Abormal end.
 */
int
maildrop_close(struct maildrop *md)
{
    if (md->md_fd > 0) {
        if (md->md_cr) {
            if (write(md->md_fd, "\r", 1) < 0) {
		log(ERR_IO_WRITE, "maildrop_close", strerror(errno));
            }
            md->md_cr = 0;
        }
        close(md->md_fd);
    }

    if (md_movefile(md) < 0) {
        md_free(md);
        return(-1);
    }

    md_free(md);

    return(0);
}

/*
 * maildrop_abort
 *
 * Function
 *	End maildrop function. (abnormal)
 *	1. Close temporary file.
 *	2. Delete temporary file.
 *
 * Argument
 *	struct maildrop *md	Pointer to maildrop structure.
 *
 * Return value
 *	0	Normal end.
 */
int
maildrop_abort(struct maildrop *md)
{
    if (md == NULL) {
	return(0);
    }
    if (md->md_fd > 0) {
        close(md->md_fd);
    }
    if (md->md_tempfile) {
        unlink(md->md_tempfile);
    }
    md_free(md);
    return(0);
}

/*
 * maildrop_write_header
 *
 * Function
 *	Write mail header to temporary file.
 *
 * Argument
 *	struct maildrop *md	Pointer to maildrop structure.
 *	char *headerf		Header field name.
 *	char *headerv		Header field value.
 *
 * Return value
 *	0	Normal end.
 *	-1	Abnormal end.
 */
int
maildrop_write_header(struct maildrop *md, char *headerf, char *headerv)
{
    ssize_t ret;
    char *header, *hp;
    int len;

    len = strlen(headerf) + strlen(headerv) + 3;

    header = malloc(len + 1);
    if (!header) {
	log(ERR_MEMORY_ALLOCATE, "maildrop_write_header",
	    "header", strerror(errno));
        return(-1);
    }
    sprintf(header, "%s: %s\n", headerf, headerv);

    hp = header;

    while (1) {
        ret = write(md->md_fd, hp, len);
        if (ret < 0) {
            free(header);
	    log(ERR_IO_WRITE, "maildrop_write_header", strerror(errno));
            return(-1);
        }
        if (ret == len) {
            break;
        }
        hp += ret;
        len -= ret;
    }

    free(header);
    return(0);
}

/*
 * maildrop_write_body
 *
 * Function
 *	Write mail body to temporary file.
 *
 * Argument
 *	struct maildrop *md	Pointer to maildrop structure.
 *	char *body		Pointer to the start of body data.
 *	int size		The amount of data pointed to by body.
 *
 * Return value
 *	0	Normal end.
 *	-1	Abnormal end.
 */
int
maildrop_write_body(struct maildrop *md, u_char *body, int size)
{
    ssize_t ret;
    int i;

    for (i = 0; i < size; i++, body++) {
        if (md->md_cr) {
            if (*body != '\n') {
                ret = write(md->md_fd, "\r", 1);
                if (ret < 0) {
		    log(ERR_IO_WRITE, "maildrop_write_body", strerror(errno));
                    return(-1);
                }
            }
            md->md_cr = 0;
        }
        if (*body == '\r') {
            md->md_cr = 1;
            continue;
        }
        ret = write(md->md_fd, body, 1);
        if (ret < 0) {
	    log(ERR_IO_WRITE, "maildrop_write_body", strerror(errno));
            return(-1);
        }
    }
    return(0);
}


/******** ******** ******** ********
 * Internal function
 ******** ******** ******** ********/
int
md_movefile(struct maildrop *md)
{
    char *sfile;
    int len;

    /* Make folder name */
    sfile = md_makefolder(md);
    if (!sfile) {
        return(-1);
    }
#ifdef DEBUG
    printf("DEBUG: md_movefile: sfile=[%s]\n", sfile);
#endif

    /* Make saved filename */
    len = strlen(sfile);
    if (md_makesavefile(md, sfile + len) < 0) {
        free(sfile);
        return(-1);
    }
#ifdef DEBUG
    printf("DEBUG: md_movefile: sfile=[%s]\n", sfile);
#endif

    /* Move temporary file -> saved file */
    if (rename(md->md_tempfile, sfile) < 0) {
	log(ERR_FILE_RENAME, "md_movefile", sfile, strerror(errno));
        free(sfile);
        return(-1);
    }

    free(sfile);
    return(0);
}

char *
md_makefolder(struct maildrop *md)
{
    struct tm *lt;
    char *folder, *p, *curp, *replacename;
    int percent;
    int len;
    int fcount;
    int ret;
    int buflen;

    /* Count "%f" */
    fcount = count_replaceaddr(md->md_mailfolder);
    if (fcount > 0) {

	/* Set folder name */
	if (md->md_replaceaddr == NULL) { 

	    buflen = (md->md_from_len * 2) + 1;

	    replacename = (char *)malloc(buflen);
            if (!replacename) {
	        log(ERR_MEMORY_ALLOCATE,
	            "md_makefolder", "replacename", strerror(errno));
                return(NULL);
            }

	    ret = check_ascii(md->md_from, replacename);
	    if (ret < 0) {
		md->md_replaceaddr = strdup(UNKNOWN);
        	if (md->md_replaceaddr == NULL) {
            	    log(ERR_MEMORY_ALLOCATE,
                	"maildrop_open", "md->md_replaceaddr", strerror(errno));
	            free(replacename);
            	    return(NULL);
        	}
        	md->md_replaceaddr_len = sizeof(UNKNOWN);

	    } else {
	        md->md_replaceaddr = strdup(replacename);
	        if (md->md_replaceaddr == NULL) {
                    log(ERR_MEMORY_ALLOCATE,
            	        "maildrop_open", "md->md_replaceaddr", strerror(errno));
	            free(replacename);
            	    return(NULL);
	        }
       	        md->md_replaceaddr_len = ret;
	    }
	    free(replacename);
	}

        /* Replace Delimiter */
	replace_delimiter(md->md_replaceaddr, DOT, md->md_dotdelimiter);
	replace_delimiter(md->md_replaceaddr, SLASH, md->md_slashdelimiter);
    }

    /* Get localtime */
    lt = localtime(&md->md_curtime);
    if (!lt) {
	log(ERR_TIME_LOCALTIME, "md_makefolder", strerror(errno));
        return(NULL);
    }

    /* Allocate memory for folder name */
    /* (BABEDIR)/(FOLDER)/new/0000000000.0000000000.(HOSTNAME) */
    folder = (char *) malloc(md->md_maildir_len +
                            md->md_mailfolder_len * 2 +
                            (md->md_replaceaddr_len - 2) * fcount +
                            md->md_hostname_len + SAVEFILE_LEN);
    if (!folder) {
	log(ERR_MEMORY_ALLOCATE, "md_makefolder", "folder", strerror(errno));
        return(NULL);
    }

    /* Make folder name */
    curp = folder;

    for (p = md->md_maildir; *p; p++) {
        *curp++ = *p;
    }
    *curp++ = '/';

    percent = 0;
    for (p = md->md_mailfolder; *p; p++) {
        if (*p == '.' && p != md->md_mailfolder) {
            if (percent) {
                *curp = '%';
                *(curp + 1) = '\0';
            } else {
                *curp = '\0';
            }
#ifdef DEBUG
            printf("DEBUG: Dot: folder=[%s]\n", folder);
#endif
            /* Make directory for folder */
            len = strlen(folder);
            if (md_makemaildir(folder, len) < 0) {
                free(folder);
                return(NULL);
            }
        }

        if (*p == '%') {
            if (percent) {
                *curp++ = '%';
            }
            percent = 1;
            continue;
        }

        if (percent) {
            switch (*p) {
                case 'y':
                    sprintf(curp, "%04d", lt->tm_year + 1900);
                    curp += 4;
                    break;
                case 'm':
                    sprintf(curp, "%02d", lt->tm_mon + 1);
                    curp += 2;
                    break;
                case 'd':
                    sprintf(curp, "%02d", lt->tm_mday);
                    curp += 2;
                    break;
                case 'f':
                    sprintf(curp, "%s", md->md_replaceaddr);
                    curp += md->md_replaceaddr_len;
                    break;
                default:
                    *curp++ = '%';
                    *curp++ = *p;
                    break;
            }
        } else {
            *curp++ = *p;
        }
        percent = 0;
    }

    if (percent) {
        *curp++ = '%';
    }

    /* Delete the last dot */
    if (*(curp - 1) == '.') {
        curp--;
    }

    *curp = '\0';

#ifdef DEBUG
    printf("DEBUG: Last: folder=[%s]\n", folder);
#endif
    /* Make directory for folder */
    len = strlen(folder);
    if (md_makemaildir(folder, len) < 0) {
        free(folder);
        return(NULL);
    }

    return(folder);
}

int
md_makesavefile(struct maildrop *md, char *folder)
{
    struct stat stbuf;

    /* Get inode */
    if (stat(md->md_tempfile, &stbuf) < 0) {
	log(ERR_FILE_GET_INFO, "md_makesavefile",
	    md->md_tempfile, strerror(errno));
        return(-1);
    }

    /* Add save filename to folder */
    sprintf(folder, SAVEFILE_NAME,
            md->md_curtime, stbuf.st_ino, md->md_hostname);

    return(0);
}

void
md_free(struct maildrop *md)
{
    if (md->md_maildir) {
        free(md->md_maildir);
    }
    if (md->md_mailfolder) {
        free(md->md_mailfolder);
    }
    if (md->md_from) {
        free(md->md_from);
    }
    if (md->md_replaceaddr) {
	free(md->md_replaceaddr);
    }
    if (md->md_tempfile) {
        free(md->md_tempfile);
    }
    if (md->md_hostname) {
        free(md->md_hostname);
    }
    if (md->md_mydomain) {
	free_filter(md->md_mydomain);
    }
    free(md);
}

int
md_makemaildir(char *dir, int size)
{
    int ret;
    char *subdir, *lastp;

    /* Make base directory */
    ret = md_makesubdir(dir);
#ifdef DEBUG
    printf("DEBUG: md_makesubdir(%s): ret=[%d]\n", dir, ret);
#endif
    if (ret > 0) {
        return(-1);
    }

    /* Allocat memory for sub directory name */
    subdir = (char *) malloc(size + 5);
    if (!subdir) {
	log(ERR_MEMORY_ALLOCATE, "md_makemaildir", "subdir", strerror(errno));
        return(-1);
    }

    /* Make tmp directory */
    strncpy(subdir, dir, size);
    lastp = subdir + size;

    strcpy(lastp, "/tmp");
    ret = md_makesubdir(subdir);
#ifdef DEBUG
    printf("DEBUG: md_makesubdir(%s): ret=[%d]\n", subdir, ret);
#endif
    if (ret > 0) {
        free(subdir);
        return(-1);
    }

    /* Make cur directory */
    strcpy(lastp, "/cur");
    ret = md_makesubdir(subdir);
#ifdef DEBUG
    printf("DEBUG: md_makesubdir(%s): ret=[%d]\n", subdir, ret);
#endif
    if (ret > 0) {
        free(subdir);
        return(-1);
    }

    /* Make new directory */
    strcpy(lastp, "/new");
    ret = md_makesubdir(subdir);
#ifdef DEBUG
    printf("DEBUG: md_makesubdir(%s): ret=[%d]\n", subdir, ret);
#endif
    if (ret > 0) {
        free(subdir);
        return(-1);
    }

    free(subdir);
    return(0);
}

int
md_makesubdir(char *dirname)
{
    struct stat stbuf;

    if (stat(dirname, &stbuf) < 0) {
        if (errno != ENOENT) {
	    log(ERR_FILE_GET_INFO, "md_makesubdir", dirname, strerror(errno));
            return(1);
        }

        if (mkdir(dirname, 0700) < 0) {
	    log(ERR_DIRECTORY_MAKE, "md_makesubdir", dirname, strerror(errno));
            return(2);
        }
    } else {
        if (!S_ISDIR(stbuf.st_mode)) {
	    log(ERR_DIRECTORY_NOT_DIRECTORY, "md_makesubdir", dirname);
            return(3);
        }
    }
    return(0);
}

/*
 * replace_delimiter
 *
 * args:
 *  char *address	address (replaced)
 *  char delimiter	delimiter
 *  char newdelimiter	new delimiter
 * return:
 *  none
 */
void
replace_delimiter(char *address, char delimiter, char newdelimiter)
{
    char *p = address;
    while ((p = strchr(p, delimiter)) != NULL) {
	*p++ = newdelimiter;
    }
}

/*
 * count_replaceaddr
 *
 * args:
 *  char *str		string
 * return:
 *  count		the number of '%f'
 */
int
count_replaceaddr(char *str)
{
    char *p = str;
    int count = 0;

    while ((p = strchr(p, '%')) != NULL) {
	p++;
	if (*p == 'f') {
	    count++;
	}
    }
    return count;
}

/*
 * push_list
 *
 * args:
 * return:
 *  0		ok
 *  -1		error
 */
int
push_rcptlist(struct rcptaddr  **head, char *str)
{
    int i;

    /* check current size */
    if (*head == NULL) {
	i = 0;
    } else {
	for (i = 0; (*head + i)->rcpt_addr != NULL; i++) {
	    ;
	}
    }

    /* (re)allocate memory */
    *head = realloc(*head, sizeof(struct rcptaddr) * (i + 2));
    if (*head == NULL) {
	log(ERR_MEMORY_ALLOCATE, "push_list", "head", strerror(errno));
	return -1;
    }

    /* copy string */
    (*head + i)->rcpt_addr = strdup(str);
    if ((*head + i)->rcpt_addr == NULL) {
	log(ERR_MEMORY_ALLOCATE, "push_list",
	    "(*head + i)->rcpt_addr", strerror(errno));
	return -1;
    }
    (*head + i)->rcpt_addr_len = strlen(str);

    /* end with NULL */
    (*head + i + 1)->rcpt_addr = NULL;

    return 0;
}

void
free_rcptlist(struct rcptaddr *head)
{
    int i;
    for (i = 0; (head + i)->rcpt_addr != NULL; i++) {
	free((head + i)->rcpt_addr);
    }
    free(head);
}

/*
 * check_ascii
 *
 * args:
 *  char *str		mail address
 *  char *retbuf	return string
 * return:
 *  -1		not ascii
 *  >0		retbuf length
 */
int
check_ascii(char *str, char *retbuf)
{
    int ret;
    int i, j;

    for (i = 0, j = 0; str[i] != '\0'; i++) {

	/* Ascii Check */
        ret = isascii(str[i]);

	/* Not Ascii */
        if (ret == 0) {
	    log(ERR_MAILADDR_UNKNOWN_TYPE, "check_ascii", str);
	    return -1;
	}

	/* '&' -> "&-" */
        if (str[i] == '&') {
            retbuf[j] = '&';
            j++;
            retbuf[j] = '-';
            j++;
            continue;
        }

	/* Change to Lowercase */
        retbuf[j] = tolower(str[i]);
        j++;
    }
    retbuf[j] = '\0';
    return(j);
}
