/*
 * Copyright (c) 2006-2010 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core.layout;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import jp.sf.maskat.core.MaskatElement;
import jp.sf.maskat.core.MaskatElementVisitor;
import jp.sf.maskat.core.event.Event;
import jp.sf.maskat.core.event.EventDef;
import jp.sf.maskat.core.event.Source;
import jp.sf.maskat.core.event.Target;

import org.apache.commons.beanutils.BasicDynaBean;
import org.apache.commons.beanutils.DynaClass;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.PredicateUtils;

/**
 * マスカット部品の動的に生成されるモデルクラスです
 * 
 * DynaComponetクラスはorg.apache.commons.DynaBeanインターフェース
 * を実装しマスカット部品のJavaBeanオブジェクトとして利用されます。
 * 
 * このクラスは拡張ポイント「jp.sf.maskat.core.widgets」の定義を行う
 * ことで自動作成されます。静的にJavaBeanを作成する必要がありません。
 */
public class DynaComponent extends BasicDynaBean implements Component {

	/**
	 * シリアルバージョンUID
	 */
	private static final long serialVersionUID = 925589796525491481L;

	/**
	 * プロパティ変更通知
	 */
	private PropertyChangeSupport support;

	/**
	 * 親要素
	 */
	private MaskatElement parent;

	/**
	 * デフォルトコンストラクタです
	 */
	public DynaComponent() {
		super(null);
		this.support = new PropertyChangeSupport(this);
	}

	/**
	 * コンストラクタです
	 * 
	 * @param dynaClass ClassオブジェクトをシュミレーションしたDynaClass
	 */
	public DynaComponent(DynaClass dynaClass) {
		super(dynaClass);
		this.support = new PropertyChangeSupport(this);
		setDefaultVlaue();
	}

	/**
	 * デフォルト値を定義します
	 * 
	 * 全てのプロパティに対して定義されているデフォルト値を設定します
	 * デフォルト値はClassオブジェクトをシュミレーションしたDynaClass
	 * から取得します
	 */
	private void setDefaultVlaue() {
		DynaComponentClass dynaClass = (DynaComponentClass) getDynaClass();
		if (dynaClass.isWidthResizable()) {
			this.set("width", new Integer(0));
		}
		if (dynaClass.isHeightResizable()) {
			this.set("height", new Integer(0));
		}
		Map defaultMap = this.getDefaultValues();
		for (Iterator ite = defaultMap.keySet().iterator(); ite.hasNext();) {
			String name = (String) ite.next();
			this.set(name, defaultMap.get(name));
		}
		DynaProperty[] properties = (DynaProperty[]) this.dynaClass
				.getDynaProperties();
		for (int i = 0; i < properties.length; i++) {
			if (Boolean.class.isAssignableFrom(properties[i].getType())) {
				if (!(this.get(properties[i].getName()) instanceof Boolean)) {
					this.set(properties[i].getName(), Boolean.FALSE);
				}
			}
		}
	}

	/**
	 * ClassオブジェクトをシュミレーションしたDynaClassを設定します
	 * 
	 * @param dynaClass  ClassオブジェクトをシュミレーションしたDynaClass
	 */
	public void setDynaClass(DynaClass dynaClass) {
		this.dynaClass = dynaClass;
		setDefaultVlaue();
	}

    /**
     * {@inheritDoc}
     */
	public void set(String name, int index, Object value) {
		Object oldValue = get(name, index);
		super.set(name, index, value);
		support.firePropertyChange(name, oldValue, value);
	}

    /**
     * {@inheritDoc}
     */
	public void set(String name, Object value) {
		Object oldValue = get(name);
		super.set(name, value);
		if ("name".equals(name)) {
			updateComponentIdInEvent((String) oldValue, (String) value);
		}
		support.firePropertyChange(name, oldValue, value);
	}

    /**
     * {@inheritDoc}
     */
	public void set(String name, String key, Object value) {
		Object oldValue = get(key, name);
		super.set(name, key, value);
		support.firePropertyChange(name, oldValue, value);
	}

    /**
     * {@inheritDoc}
     */
	public void remove(String name, String key) {
		Object oldValue = get(key, name);
		super.remove(name, key);
		support.firePropertyChange(name, oldValue, null);
	}

	/**
	 * 親要素を取得します
	 * 
	 * @return 親要素
	 */
	public MaskatElement getParent() {
		return parent;
	}

	/**
	 * 親要素を設定します
	 * 
	 * @param parent 親要素
	 */
	public void setParent(MaskatElement parent) {
		this.parent = parent;
	}

	/**
	 * 子要素をリストで取得します
	 * 
	 * このメソッドをオーバライドすることにより子要素の取得
	 * ロジックを変更することができます
	 * 
	 * @return 子要素リスト
	 */
	protected List getRawChildren() {
		List result = (List) get("children");
		if (result == null) {
			result = new ArrayList();
			set("children", result);
		}
		return result;
	}

    /**
     * {@inheritDoc}
     */
	public List getChildren() {
		return getRawChildren();
	}

    /**
     * {@inheritDoc}
     */
	public List getTypedChildren(Class clazz) {
		List result = new ArrayList();
		CollectionUtils.select(getRawChildren(), PredicateUtils
				.instanceofPredicate(clazz), result);
		return result;
	}

    /**
     * {@inheritDoc}
     */
	public List getTypedChildren(String name) {
		List result = new ArrayList();
		List children = getRawChildren();
		for (int i = 0; i < children.size(); i++) {
			Object obj = children.get(i);
			if (obj instanceof DynaComponent) {
				DynaComponent dynaComponent = (DynaComponent) obj;
				if (name.equals(dynaComponent.getDynaClass().getName())) {
					result.add(dynaComponent);
				}
			}
		}
		return result;
	}
		
    /**
     * {@inheritDoc}
     */
	public void addChild(Object child) {
		getRawChildren().add(child);
		if (child instanceof MaskatElement) {
			((MaskatElement) child).setParent(this);
		}
		support.firePropertyChange("addChild", "", null);
	}

    /**
     * {@inheritDoc}
     */
	public void addChildToIdx(Object child, int idx) {
		getRawChildren().add(idx, child);
		if (child instanceof MaskatElement) {
			((MaskatElement) child).setParent(this);
		}
		support.firePropertyChange("addChild", "", null);
	}
	
    /**
     * {@inheritDoc}
     */
	public int getChildIdx(Object child) {
		return getRawChildren().indexOf(child);
	}

    /**
     * {@inheritDoc}
     */
	public Object getChildByTypeIdx(Class clazz, int idx) {
		List result = new ArrayList();
		CollectionUtils.select(getRawChildren(), PredicateUtils
				.instanceofPredicate(clazz), result);
		if (result.size() <= idx) {
			return null;
		}
		return result.get(idx);
	}

    /**
     * {@inheritDoc}
     */
	public void removeAllChildren() {
		getRawChildren().clear();
		support.firePropertyChange("removeChild", "", null);
	}

    /**
     * {@inheritDoc}
     */
	public void removeChild(Object obj) {
		getRawChildren().remove(obj);
		support.firePropertyChange("removeChild", "", null);
	}

    /**
     * {@inheritDoc}
     */
	public void removeAllByType(Class clazz) {
		List elements = getRawChildren();
		for (int i = 0; i < elements.size(); i++) {
			Object element = elements.get(i);
			if (clazz.isInstance(element)) {
				elements.remove(element);
			}
		}
		support.firePropertyChange("removeChild", "", null);
	}

    /**
     * {@inheritDoc}
     */
	public void accept(MaskatElementVisitor visitor) {
		List elements = getChildren();
		for (int i = 0; i < elements.size(); i++) {
			Object element = elements.get(i);
			visitor.visit(element);
			if (element instanceof MaskatElement) {
				((MaskatElement) element).accept(visitor);
			}
		}
	}

    /**
     * {@inheritDoc}
     */
	public Layout getLayout() {
		return getParent() == null ? null : ((LayoutElement) getParent())
				.getLayout();
	}

    /**
     * {@inheritDoc}
     */
	public String getName() {
		return values.containsKey("name") ? (String) get("name") : "";
	}

    /**
     * {@inheritDoc}
     */
	public void setName(String name) {
		set("name", name);
	}

    /**
     * {@inheritDoc}
     */
	public int getTop() {
		return values.containsKey("top") ? ((Integer) get("top")).intValue()
				: 0;
	}
    /**
     * {@inheritDoc}
     */
	public void setTop(int top) {
		set("top", new Integer(top));
	}

    /**
     * {@inheritDoc}
     */
	public int getLeft() {
		return values.containsKey("left") ? ((Integer) get("left")).intValue()
				: 0;
	}

    /**
     * {@inheritDoc}
     */
	public void setLeft(int left) {
		set("left", new Integer(left));
	}

    /**
     * {@inheritDoc}
     */
	public int getHeight() {
		return values.containsKey("height") ? ((Integer) get("height"))
				.intValue() : 0;
	}

    /**
     * {@inheritDoc}
     */
	public void setHeight(int height) {
		set("height", new Integer(height));
	}

    /**
     * {@inheritDoc}
     */
	public int getWidth() {
		return values.containsKey("width") ? ((Integer) get("width"))
				.intValue() : 0;
	}

    /**
     * {@inheritDoc}
     */
	public void setWidth(int width) {
		set("width", new Integer(width));
	}

    /**
     * {@inheritDoc}
     */
	public void setConstraint(int left, int top, int width, int height) {
		if (getDynaClass().getDynaProperty("left") != null) {
			super.set("left", new Integer(left));
		}
		if (getDynaClass().getDynaProperty("top") != null) {
			super.set("top", new Integer(top));
		}
		if (getDynaClass().getDynaProperty("width") != null) {
			super.set("width", new Integer(width));
		}
		if (getDynaClass().getDynaProperty("height") != null) {
			super.set("height", new Integer(height));
		}
		support.firePropertyChange("constraint", null, null);
	}

    /**
     * {@inheritDoc}
     */
	public int getTabIndex() {
		return ((Integer) get("tabIndex")).intValue();
	}

    /**
     * {@inheritDoc}
     */
	public void setTabIndex(int tabIndex) {
		set("tabIndex", new Integer(tabIndex));
	}

    /**
     * {@inheritDoc}
     */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		support.addPropertyChangeListener(listener);
	}

    /**
     * {@inheritDoc}
     */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		support.removePropertyChangeListener(listener);
	}

    /**
     * {@inheritDoc}
     */
	public Component getContainer() {
		return (Component) getParent();
	}

    /**
     * {@inheritDoc}
     */
	public String[] getEventTypes() {
		return ((DynaComponentClass) getDynaClass()).getEventTypes();
	}

    /**
     * {@inheritDoc}
     */	
	public Map getDefaultValues() {
		return ((DynaComponentClass) getDynaClass()).getDefaultValues();
	}

    /**
     * {@inheritDoc}
     */
	public boolean hasEvent(String eventType) {
		String[] eventTypes = getEventTypes();
		if (eventTypes != null) {
			for (int i = 0; i < eventTypes.length; i++) {
				if (eventTypes[i].equals(eventType)) {
					return true;
				}
			}
		}
		return false;
	}

    /**
     * {@inheritDoc}
     */
	public Event getEvent(String eventType) {
		if (hasEvent(eventType)) {
			EventDef eventDef = getLayout().getLayoutDef().getEventDef();
			if (eventDef != null) {
				jp.sf.maskat.core.event.Component component = eventDef
						.findComponent(getName());
				if (component != null) {
					return component.findEvent(eventType);
				}
			}
		}
		return null;
	}

    /**
     * {@inheritDoc}
     */
	public void setEvent(Event event) {
		if (hasEvent(event.getId())) {
			EventDef eventDef = getLayout().getLayoutDef().getEventDef();
			jp.sf.maskat.core.event.Component component = eventDef
					.findComponent(getName());
			if (component == null) {
				component = new jp.sf.maskat.core.event.Component();
				component.setId(getName());
				eventDef.addChild(component);
			}
			component.setEvent(event);
			support.firePropertyChange("#event", null, event);
		}
	}

    /**
     * {@inheritDoc}
     */
	public Object clone() throws CloneNotSupportedException {
		DynaComponent dynaBean = (DynaComponent) super.clone();
		dynaBean.values = (HashMap) dynaBean.values.clone();
		dynaBean.set("children", new ArrayList());
		dynaBean.setParent(null);

		List children = this.getChildren();
		for (int i = 0; i < children.size(); i++) {
			Object child = children.get(i);
			if (child instanceof MaskatElement) {
				dynaBean.addChild(((MaskatElement) child).clone());
			} else if (child instanceof DynaComponent) {
				dynaBean.addChild(((DynaComponent) child).clone());
			} else {
				dynaBean.addChild(child);
			}
		}
		dynaBean.support = new PropertyChangeSupport(dynaBean);
		return dynaBean;
	}

    /**
     * {@inheritDoc}
     */
	public List getAllDescendants(MaskatElement parent, Class descendant) {
		ArrayList result = new ArrayList();
		List children = parent.getChildren();
		for (int i = 0; i < children.size(); i++) {
			Object child = children.get(i);
			if (descendant.equals(child.getClass())) {
				result.add(child);
			}
			if (child instanceof MaskatElement) {
				result.addAll(getAllDescendants((MaskatElement) child,
						descendant));
			}
		}
		return result;
	}

	/**
	 * 有効な属性か判定します
	 * 
	 * DynaComponentで制御用に利用する属性 children, parentを除いた
	 * 属性を有効な属性として判定します。
	 * 
	 * @param propName プロパティ名
	 * @return 有効な属性の場合 trueを返却します
	 */
	public static boolean isAttributeProperty(String propName) {
		return !("children".equals(propName) || "parent".equals(propName));
	}

	/**
	 * コンポーネント名の変更時にイベント定義にあるコンポーネント名も
	 * 合わせて新しいコンポーネント名に修正します
	 * 
	 * @param oldId 修正前のコンポーネント名
	 * @param newId 修正を行うコンポーネント名
	 */
	private void updateComponentIdInEvent(String oldId, String newId) {
		if (oldId == null || oldId.equals(newId)) {
			return;
		}
		if (getParent() == null || getLayout() == null
				|| getLayout().getLayoutDef() == null
				|| getLayout().getLayoutDef().getEventDef() == null) {
			return;
		}

		EventDef eventDef = getLayout().getLayoutDef().getEventDef();
		jp.sf.maskat.core.event.Component component = eventDef
				.findComponent(oldId);
		if (component != null) {
			component.setId(newId);
		}

		// TODO should update all Source.obj , Source.idxRef, Target.in,
		// Target.out who equals to oldId ?
		List sources = getAllDescendants(eventDef, Source.class);
		for (int i = 0; i < sources.size(); i++) {
			Source source = (Source) sources.get(i);
			if (oldId.equals(source.getObj())) {
				source.setObj(newId);
			}
			if (oldId.equals(source.getIdxRef())) {
				source.setIdxRef(newId);
			}
		}
		List targets = getAllDescendants(eventDef, Target.class);
		for (int i = 0; i < targets.size(); i++) {
			Target target = (Target) targets.get(i);
			if (oldId.equals(target.getIn())) {
				target.setIn(newId);
			}
			if (oldId.equals(target.getOut())) {
				target.setOut(newId);
			}
		}
	}

	/**
	 * cdata-section, textのコンテキストを取得します
	 * このメソッドはbetwixtからのみ呼び出されます
	 * 
	 * @return cata-sectionまたはtextのコンテキスト
	 */
	public String getContext() {
		return values.containsKey("context") ? (String) get("context") : null;
	}

	/**
	 *  cdata-section, textのコンテキストを設定します
	 *  このメソッドはbetwixtからのみ呼び出されます
	 * 
	 * @param context cata-sectionまたはtextのコンテキスト
	 */
	public void setContext(String context) {
		if (values.containsKey("context")) {
			set("context", context);
		}
	}
}
