/*
 * Copyright (c) 2006-2010 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core.event;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * イベント定義の対象となるコンポーネントクラスです。
 */
public class Component extends AbstractEventElement {

	/**
	 * コンポーネント名
	 * イベントが発生する部品のname属性
	 */
    private String id;

    /**
     * イベントマップ
     * このコンポーネントが持っているイベントの一覧
     */
    private Map events = new HashMap();

    /**
     * コンポーネントIDを取得します。
     * コンポーネントIDとはイベントが発生する部品のname属性です。
     * 
     * @return コンポーネントID
     */
    public String getId() {
        return id;
    }

    /**
     * コンポーネントIDを格納します。
     * 
     * @param id コンポーネントID
     */
    public void setId(String id) {
        this.id = id;
    }

    /**
     * {@inheritDoc}
     */
    public void addChild(Object child) {
        super.addChild(child);
        if (child instanceof Event) {
            addEvent((Event) child);
        }
    }

    /**
     * イベント一覧に指定イベントを追加します
     * 
     * @param event 追加したいイベント
     */
    private void addEvent(Event event) {
        events.put(event.getId(), event);
    }

    /**
     * イベント一覧から指定されたIDのイベントを削除します
     * 
     * @param id 削除を行いたいイベントのID
     * @return 削除されたイベント
     */
    public Event removeEvent(String id) {
        return (Event) events.remove(id);
    }

    /**
     * イベント一覧から指定されたIDのイベントを取得します
     * 
     * @param id 取得したいイベントのID
     * @return 取得されたイベント、存在しない場合にはnull
     */
    public Event findEvent(String id) {
        return (Event) events.get(id);
    }

    /**
     * このコンポーネントにイベントを追加します
     * 既に同じイベントが登録されていた場合には上書きとなります
     * 
     * @param event 追加したいイベント
     */
    public void setEvent(Event event) {
        int index = getChildIndex(event);
        if (index != -1) {
            List list = getChildren();
            list.set(index, event);
            event.setParent(this);
            addEvent(event);
        } else {
            addChild(event);
        }
    }

    /**
     * このコンポーネントの子要素にある指定されたイベント
     * の格納位置を取得します
     * 
     * @param event 対象となるイベント
     * @return 格納されている位置、存在しない場合は-1
     */
    private int getChildIndex(Event event) {
        List list = getChildren();
        for (int i = 0; i < list.size(); i++) {
            Object child = list.get(i);
            if (child instanceof Event) {
                if (event.getId().equals(((Event) child).getId())) {
                    return i;
                }
            }
        }
        return -1;
    }

    /**
     * このコンポーネントにイベントが存在するか判定します
     * 
     * @return イベントが存在する場合、trueが返却されます
     */
    public boolean hasEvents() {
        return !events.isEmpty();
    }

    /**
     * このコンポーネントに登録されているイベントの中に
     * 無効なイベントが存在するか判定します
     * 
     * @return 無効なイベントが存在する場合、trueが返却されます
     */
    public boolean hasEnabledEvents() {
        if (events.isEmpty()) {
            return false;
        }
        Iterator it = events.values().iterator();
        while (it != null && it.hasNext()) {
            if (((Event) it.next()).isEnable()) {
                return true;
            }
        }
        return false;
    }

    /**
     * このコンポーネントに登録されているイベントを配列として
     * 取得します
     * 
     * @return 登録されているイベント配列
     */
    public Event[] getAllEvents() {
        if (events.isEmpty()) {
            return new Event[0];
        }
        return (Event[]) events.values().toArray(new Event[0]);
    }
}
