/*
 * Copyright (c) 2006-2009 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.ui.editors.layout;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import jp.sf.maskat.core.layout.ComponentRegistry;
import jp.sf.maskat.core.layout.UnknownComponentClass;
import jp.sf.maskat.ui.ISharedImages;
import jp.sf.maskat.ui.MaskatUIPlugin;
import jp.sf.maskat.ui.PreferenceConstants;
import jp.sf.maskat.ui.editors.layout.requests.BeanCreationFactory;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.gef.palette.CombinedTemplateCreationEntry;
import org.eclipse.gef.palette.CreationToolEntry;
import org.eclipse.gef.palette.MarqueeToolEntry;
import org.eclipse.gef.palette.PaletteContainer;
import org.eclipse.gef.palette.PaletteDrawer;
import org.eclipse.gef.palette.PaletteEntry;
import org.eclipse.gef.palette.PaletteGroup;
import org.eclipse.gef.palette.PaletteRoot;
import org.eclipse.gef.palette.SelectionToolEntry;
import org.eclipse.gef.palette.ToolEntry;
import org.eclipse.gef.requests.CreationFactory;
import org.eclipse.gef.ui.palette.PaletteCustomizer;
import org.eclipse.gef.ui.palette.customize.DefaultEntryPage;
import org.eclipse.gef.ui.palette.customize.DrawerEntryPage;
import org.eclipse.gef.ui.palette.customize.EntryPage;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Text;


public class PluggablePaletteCustomizer extends PaletteCustomizer {

	private static ImageDescriptor DEFAULT_IMAGE_DESCRIPTOR = MaskatUIPlugin
			.getImageDescriptor(ISharedImages.IMG_COMPONENTICON);
	
	private PaletteRoot paletteRoot;

	private Map entries;

	public PluggablePaletteCustomizer() {
		super();
		paletteRoot = new PaletteRoot();
		entries = new HashMap();
		initializePalette();
		revertToSaved();
	}

	public PaletteRoot getPaletteRoot() {
		return paletteRoot;
	}

	private void initializePalette() {
		PaletteGroup group = new PaletteGroup("tools"); //$NON-NLS-1$
		ToolEntry entry = new SelectionToolEntry();
		paletteRoot.setDefaultEntry(entry);
		group.add(entry);
		group.add(new MarqueeToolEntry());
		paletteRoot.add(group);

		IExtensionPoint point = Platform.getExtensionRegistry().getExtensionPoint(
				MaskatUIPlugin.PLUGIN_ID, "paletteEntries");
		IExtension[] extensions = point.getExtensions();

		/* First we add palette containers */
		for (int i = 0; i < extensions.length; i++) {
			IConfigurationElement[] elements = extensions[i].getConfigurationElements();
			for (int j = 0; j < elements.length; j++) {
				if ("drawer".equals(elements[j].getName())) {
					addPaletteDrawer(elements[j]);
				}
			}
		}

		/* Then add palette entries */
		for (int i = 0; i < extensions.length; i++) {
			IConfigurationElement[] elements = extensions[i].getConfigurationElements();
			for (int j = 0; j < elements.length; j++) {
				if ("creationTool".equals(elements[j].getName())) {
					addCreationToolEntry(elements[j]);
				}
			}
		}
		IPreferenceStore store = MaskatUIPlugin.getDefault().getPreferenceStore();
		String storeString = store.getString(PreferenceConstants.PALETTE_ENTRIES);
		createPaletteEntry(parseStoreEntry(storeString), null);
		
		/* 既存リストに存在しない新たなDrawerおよびCreattionToolを追加する
		 * このDrawerにリンクしているCreationToolはplugin.xmlの定義順とり、
		 * エントリは非表示となる。
		 */
		for (Iterator ite = entries.keySet().iterator(); ite.hasNext();) {
			PaletteEntry p = (PaletteEntry) entries.get(ite.next());
			PaletteEntry px = p;
			PaletteEntry root = null;
			while (px.getParent() != paletteRoot) {
				px = px.getParent();
				if (px == null) {
					if (!(p instanceof PaletteDrawer)) {
						if (root != null) {
//							root.setVisible(false);
							paletteRoot.add(root);
						} else {
							p.setVisible(false);
//							paletteRoot.add(p);
						}
					}
					break;
				}
				root = px;
			}
		}
		entries.clear();
	}

	private void createPaletteEntry(StoreEntry entry, StoreContainer palent) {
		if (!(entry instanceof RootEntry)) {
			PaletteEntry p = (PaletteEntry) entries.get(palent.getId());
			PaletteEntry c = (PaletteEntry) entries.get(entry.getId());
			if (c != null) {
				if (p instanceof PaletteDrawer) {
					((PaletteDrawer) p).remove(c);
					((PaletteDrawer) p).add(c);
					
				} else if (p == null) {
					paletteRoot.add(c);
				}
				if (!(c instanceof PaletteDrawer)) {
					entries.remove(entry.getId());
				} else {
					if (c instanceof PaletteDrawer && entry instanceof StoreDrawer) {
						StoreDrawer dr = (StoreDrawer) entry;
						int state = dr.isInitiallyOpen() ?
								PaletteDrawer.INITIAL_STATE_OPEN :
								PaletteDrawer.INITIAL_STATE_CLOSED;
						
						if (dr.isInitiallyOpen() && dr.isInitiallyPinned()) {
							state = PaletteDrawer.INITIAL_STATE_PINNED_OPEN;
						}
						((PaletteDrawer) c).setInitialState(state);
					}
				}
				c.setVisible(entry.isVisible());
			}
		}
		if (entry instanceof StoreContainer) {
			StoreContainer container = (StoreContainer) entry;
			List children = container.getChildren();
			for (int i = 0; i < children.size(); i++) {
				createPaletteEntry((StoreEntry) children.get(i), container);
			}
		}
	}
	
	public boolean canDelete(PaletteEntry entry) {
		return false;
	}

	public List getNewEntryFactories() {
		return new ArrayList();
	}

	public EntryPage getPropertiesPage(PaletteEntry entry) {
		if (entry instanceof PaletteDrawer) {
			return new WidgetDrawerEntryPage();
		}
		return new WidgetEntryPage();
	}

	private class WidgetDrawerEntryPage extends DrawerEntryPage {
		protected Text createDescText(Composite panel) {
			Text text = super.createDescText(panel);
			text.setEditable(false);
			return text;
		}
		protected Text createNameText(Composite panel) {
			Text text = super.createNameText(panel);
			text.setEditable(false);
			return text;
		}
	}
	
	private class WidgetEntryPage extends DefaultEntryPage {
		protected Text createDescText(Composite panel) {
			Text text = super.createDescText(panel);
			text.setEditable(false);
			return text;
		}
		protected Text createNameText(Composite panel) {
			Text text = super.createNameText(panel);
			text.setEditable(false);
			return text;
		}		
	}
	
	private void addPaletteDrawer(IConfigurationElement element) {
		PaletteDrawer drawer = new PaletteDrawer(element.getAttribute("label"),
				MaskatUIPlugin.getImageDescriptor(element, "icon"));
		String id = element.getAttribute("id");
		drawer.setId(id);
		entries.put(id, drawer);
		
		String path = element.getAttribute("path");
		if (path != null && entries.containsKey(path)) {
			PaletteEntry entryForPath = (PaletteEntry) entries.get(path);
			if (entryForPath instanceof PaletteContainer) {
				((PaletteContainer) entryForPath).add(drawer);
			} else {
				entryForPath.getParent().add(drawer);
			}
		} else {
//			paletteRoot.add(drawer);
		}
	}

	private void addCreationToolEntry(IConfigurationElement element) {
		String namespaceURI = element.getAttribute("namespaceURI");
		String name = element.getAttribute("name");
		CreationFactory factory = null;

		try {
			if (element.getAttribute("factory") != null) {
				factory = (CreationFactory) element.createExecutableExtension("factory");
			} else {
				IConfigurationElement[] children = element.getChildren("property");
				Map properties = null;
				if (children != null) {
					properties = new HashMap();
					for (int i = 0; i < children.length; i++) {
						String key = children[i].getAttribute("name");
						String value = children[i].getAttribute("value");
						properties.put(key, value);
					}
				}
				Object type = ComponentRegistry.getComponentType(namespaceURI, name);
				type = type == null ? new UnknownComponentClass() : type;
				factory = new BeanCreationFactory(type, properties);
			}
//		} catch (UnknownComponentException e) {
//			IStatus status = new Status(IStatus.ERROR, MaskatUIPlugin.PLUGIN_ID, 0,
//					"Unknown component type: " + namespaceURI + "#" + name, e);
//			MaskatUIPlugin.log(status);
		} catch (CoreException e) {
			MaskatUIPlugin.log(e.getStatus());
		}

		ImageDescriptor iconSmall = MaskatUIPlugin.getImageDescriptor(element, "iconSmall"); 
		ImageDescriptor iconLarge = MaskatUIPlugin.getImageDescriptor(element, "iconLarge");
		CreationToolEntry entry = new CombinedTemplateCreationEntry(
				element.getAttribute("label"),
				element.getAttribute("description"),
				factory,
				(iconSmall != null) ? iconSmall : DEFAULT_IMAGE_DESCRIPTOR,
				(iconLarge != null) ? iconLarge : DEFAULT_IMAGE_DESCRIPTOR);
		
		String id = element.getAttribute("id");
		entry.setId(id);
		entries.put(id, entry);

		String path = element.getAttribute("path");
		if (path != null && entries.containsKey(path)) {
			PaletteEntry entryForPath = (PaletteEntry) entries.get(path);
			if (entryForPath instanceof PaletteContainer) {
				((PaletteContainer) entryForPath).add(entry);
			} else {
				entryForPath.getParent().add(entry);
			}
		} else {
//			paletteRoot.add(entry);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.gef.ui.palette.PaletteCustomizer#revertToSaved()
	 */
	public void revertToSaved() {
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.gef.ui.palette.PaletteCustomizer#save()
	 */
	public void save() {
		IPreferenceStore store = MaskatUIPlugin.getDefault().getPreferenceStore();
		StringBuffer sb = new StringBuffer();
		List children = paletteRoot.getChildren();
		for (int i = 0; i < children.size(); i++) {
			PaletteEntry entry = (PaletteEntry) children.get(i);
			if (!(entry instanceof PaletteGroup)) {
				sb.append(getStoreString(entry));
			}
		}
		store.setValue(PreferenceConstants.PALETTE_ENTRIES, sb.toString());
	}
	
	private String getStoreString(PaletteEntry entry) {
		StringBuffer sb = new StringBuffer();
		sb.append(entry.getId());
		sb.append(":");
		sb.append(entry.getParent() != null ? entry.getParent().getId() : "");
		sb.append(":");
		sb.append(entry.isVisible());
		
		if (entry instanceof PaletteDrawer) {
			PaletteDrawer drawer = (PaletteDrawer) entry;
			sb.append(":");
			sb.append(drawer.isInitiallyOpen());
			sb.append(":");
			sb.append(drawer.isInitiallyPinned());
			sb.append(";");
			
			List children = ((PaletteContainer) entry).getChildren();
			for (int i = 0; i < children.size(); i++) {
				sb.append(getStoreString((PaletteEntry) children.get(i)));
			}
			
		} else {
			sb.append(";");
		}
		return sb.toString();
	}
	
	private StoreEntry parseStoreEntry(String value) {
		RootEntry root = new RootEntry();
		if (value == null) {
			return root;
		}
		String[] entries = value.split(";");
		for (int i = 0; i < entries.length; i++) {
			if ("".equals(entries[i].trim())) {
				continue;
			}
			String[] values = entries[i].split(":");
			if (values.length < 3 ) {
				continue;
			}
			String id = values[0];
			String parentId = values[1];
			boolean visible = new Boolean(values[2]).booleanValue();
			StoreEntry entry;

			if (values.length == 5) {
				entry = new StoreDrawer(
						id, visible,
						new Boolean(values[3]).booleanValue(),
						new Boolean(values[4]).booleanValue());
			} else {
				entry = new StoreEntry(id, visible); 
			}
			root.addChild(entry, parentId);
		}
		return root;
	}
	
	private class RootEntry extends StoreContainer {}
	private class StoreDrawer extends StoreContainer {
		private boolean initiallyOpen = false;
		private boolean initiallyPinned = false;
		
		public StoreDrawer(String id, boolean visible, boolean open, boolean pin) {
			super(id, visible);
			this.initiallyOpen = open;
			this.initiallyPinned = pin;
		}
		public boolean isInitiallyOpen() {
			return initiallyOpen;
		}
		public void setInitiallyOpen(boolean initiallyOpen) {
			this.initiallyOpen = initiallyOpen;
		}
		public boolean isInitiallyPinned() {
			return initiallyPinned;
		}
		public void setInitiallyPinned(boolean initiallyPinned) {
			this.initiallyPinned = initiallyPinned;
		}
		
	}
	private class StoreContainer extends StoreEntry {
		private List children = new ArrayList();
		public StoreContainer() {
			super("", false);
		}
		public StoreContainer(String id, boolean visible) {
			super(id, visible);
		}
		public List getChildren() {
			return children;
		}
		public void addChild(StoreEntry child) {
			children.add(child);
		}
		public boolean addChild(StoreEntry child, String parentId) {
			if ("".equals(parentId) || getId().equals(parentId)) {
				addChild(child);
				return true;
			} else {
				for (int i = 0; i < children.size(); i++) {
					StoreEntry entry = (StoreEntry) children.get(i);
					if (entry instanceof StoreContainer) {
						boolean result = ((StoreContainer) entry).
							addChild(child, parentId);
						if (result) {
							return true;
						}
					}
				}
			}
			return false;
		}
		public void setChildren(List children) {
			this.children = children;
		}
	}
	
	private class StoreEntry {
		private String id = "";
		private boolean visible = false;
		
		public StoreEntry() {}
		public StoreEntry(String id, boolean visible) {
			this.id = id;
			this.visible = visible;
		}
		public String getId() {
			return id;
		}
		public void setId(String id) {
			this.id = id;
		}
		public boolean isVisible() {
			return visible;
		}
		public void setVisible(boolean visible) {
			this.visible = visible;
		}
	}
}
