/*
 * Copyright (c) 2006-2009 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.sf.maskat.core.layout.DynaComponent;


/**
 * MaskatElementインターフェースを実装するクラスのスパークラス
 */
public abstract class AbstractMaskatElement implements MaskatElement {

	private MaskatElement parent;
	
	private List children = new ArrayList();
	
	public MaskatElement getParent() {
		return parent;
	}

	public void setParent(MaskatElement parent) {
		this.parent = parent;
	}
	
	public List getChildren() {
		return children;
	}
	
	public void addChild(Object child) {
		if (child == null || children.contains(child)) {
			return;
		}
		if (child instanceof MaskatElement) {
			((MaskatElement) child).setParent(this);
		}
		children.add(child);
	}
	
	public void removeChild(Object obj) {
		children.remove(obj);
	}

	public Object clone() throws CloneNotSupportedException {
		AbstractMaskatElement object = (AbstractMaskatElement) super.clone();
		object.children = new ArrayList();
		object.parent = null;
		List elements = getChildren();
		for (int i = 0; i < elements.size(); i++) {
			Object element = elements.get(i);
			if (element instanceof MaskatElement) {
				object.addChild(((MaskatElement) element).clone());				
			} else if (element instanceof DynaComponent) {
				object.addChild(((DynaComponent) element).clone());			
			} else {
				object.addChild(element);			
			}
		}
		return object;
	}
	
	public void removeAllChildren() {
		List elements = new ArrayList(getChildren());
		for (int i = 0; i < elements.size(); i++) {
			removeChild(elements.get(i));
		}
	}

	public void removeAllByType(Class clazz) {
		List elements = new ArrayList(getChildren());
		for (int i = 0; i < elements.size(); i++) {
			Object element = elements.get(i);
			if (clazz.isInstance(element)) {
				removeChild(element);
			}
		}		
	}
	
	public void accept(MaskatElementVisitor visitor) {
		List elements = getChildren();
		for (int i = 0; i < elements.size(); i++) {
			Object element = elements.get(i);
			visitor.visit(element);
			if (element instanceof MaskatElement) {
				((MaskatElement) element).accept(visitor);
			}
		}
	}

	public List getAllDescendants(MaskatElement parent, Class descendant) {
		ArrayList result = new ArrayList();
		List children = parent.getChildren();
		for (int i = 0; i < children.size(); i++) {
			Object child = children.get(i);
			if (descendant.equals(child.getClass())) {
				result.add(child);
			}
			if (child instanceof MaskatElement) {
				result.addAll(getAllDescendants((MaskatElement) child, descendant));
			}
		}
		return result;
	}
	
	public Map getDefaultValues() {
		return new HashMap();
	}
	
	public Object getChildByTypeIdx(Class clazz, int idx) {
		if (children.size() <= idx) {
			return null;
		}
		for (int i = 0; i < children.size(); i++) {
			if (clazz.isInstance(children.get(i))) {
				if (idx == 0) {
					return children.get(i);
				}
				idx--;
			}
		}
		return null;
	}
	
	public int getChildIdx(Object child) {
		return (children == null) ? -1 : children.indexOf(child);
	}

	public void addChildToIdx(Object child, int idx) {
		if (child == this) {
			throw new IllegalArgumentException("Could not add an object as his own child");
		}
		if (children.contains(child)) {
			throw new IllegalStateException("object already exists");
		}

		if (child instanceof MaskatElement) {
			((MaskatElement) child).setParent(this);
		}
		children.add(idx, child);
	}
	
	public List getTypedChildren(Class clazz) {
		ArrayList list = new ArrayList();
		for (int i = 0; i < children.size(); i++) {
			if (children.get(i).getClass().equals(clazz)) {
				list.add(children.get(i));
			}
		}
		return list;
	}
}
