/*
 * Copyright (c) 2006-2009 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.ui.editors.layout.actions;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import jp.sf.maskat.core.MaskatElement;
import jp.sf.maskat.core.layout.Component;
import jp.sf.maskat.core.layout.DynaComponent;
import jp.sf.maskat.core.layout.DynaComponentClass;
import jp.sf.maskat.core.layout.Layout;
import jp.sf.maskat.core.layout.LayoutElement;
import jp.sf.maskat.ui.MaskatUIPlugin;
import jp.sf.maskat.ui.Messages;
import jp.sf.maskat.ui.editors.layout.LayoutGraphicalEditor;
import jp.sf.maskat.ui.editors.layout.commands.AddBasicDefCommand;
import jp.sf.maskat.ui.editors.layout.editparts.LayoutElementEditPart;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.draw2d.FigureCanvas;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.GraphicalEditPart;
import org.eclipse.gef.GraphicalViewer;
import org.eclipse.gef.RequestConstants;
import org.eclipse.gef.commands.Command;
import org.eclipse.gef.internal.GEFMessages;
import org.eclipse.gef.requests.ChangeBoundsRequest;
import org.eclipse.gef.ui.actions.Clipboard;
import org.eclipse.gef.ui.actions.SelectionAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.ActionFactory;


public class PasteComponentAction extends SelectionAction {

	private LayoutElement targetDef = null;
	private LayoutGraphicalEditor editor = null;

	//public PasteComponentAction(IWorkbenchPart part) {
	public PasteComponentAction(LayoutGraphicalEditor editor) {
		super(editor);
		this.editor = editor;
		
		setId(ActionFactory.PASTE.getId());
		setText(GEFMessages.PasteAction_Label);
		setToolTipText(GEFMessages.PasteAction_Tooltip);
		ISharedImages sharedImages = PlatformUI.getWorkbench()
				.getSharedImages();
		setImageDescriptor(sharedImages.getImageDescriptor(
				ISharedImages.IMG_TOOL_PASTE));
		setDisabledImageDescriptor(sharedImages.getImageDescriptor(
				ISharedImages.IMG_TOOL_PASTE_DISABLED));
		
	}

	protected boolean calculateEnabled() {
		// check the content in the clipboard can be pasted.
		if (!checkClipboard()) {
			return false;
		}
		List objects = this.getSelectedObjects();
		if (objects.isEmpty()) {
			return false;
		}
		if (objects.size() > 1) {
			return false;
		}
		Object selectedObj = objects.get(0);
		if (!(selectedObj instanceof EditPart)) {
			return false;
		}
		Object model = ((EditPart) selectedObj).getModel();
		if (!(model instanceof LayoutElement)) {
			return false;
		}
		targetDef = (LayoutElement) model;
		
		List basicDefList = (List) Clipboard.getDefault().getContents();
		return canBePasted(basicDefList);
	}

	private boolean checkClipboard() {
		Object clipboardContent = Clipboard.getDefault().getContents();
		if (!(clipboardContent instanceof List)) {
			return false;
		}
		for (int i = 0; i < ((List) clipboardContent).size(); i++) {
			if (!(((List) clipboardContent).get(i) instanceof MaskatElement)) {
				return false;
			}
		}
		return true;
	}

	public void run() {
		if (!checkClipboard()) {
			return;
		}
		Command command = null;

		List basicDefList = (List) Clipboard.getDefault().getContents();
		List source = new ArrayList();
		for (Iterator it = basicDefList.iterator(); it.hasNext();) {
			LayoutElement def = (LayoutElement) it.next();
			try {
				LayoutElement clone = (LayoutElement) def.clone();
				source.add(clone);
			} catch (CloneNotSupportedException e) {
				MessageDialog.openError(this.getWorkbenchPart().getSite().getShell(),
						Messages.getString("layout.cmd.msg.error.title"), //$NON-NLS-1$
						Messages.getString("layout.cmd.paste.msg.error")); //$NON-NLS-1$
				MaskatUIPlugin.log(new Status(IStatus.ERROR, MaskatUIPlugin.PLUGIN_ID,
						IStatus.ERROR, e.getMessage(), e));
				return;
			}
		}
		for (Iterator it = source.iterator(); it.hasNext();) {
			LayoutElement def = (LayoutElement) it.next();
			if (def instanceof Component && targetDef instanceof Component) {
				Component compDef = (Component) def;
				if (compDef instanceof DynaComponent) {
					DynaComponentClass dynaClass = (DynaComponentClass)
						((DynaComponent) compDef).getDynaClass();
					if (dynaClass.isTopMovable()) {
						Component comp = (Component) targetDef;
						if (!(comp instanceof Layout)) {
							int parentHeight = comp.getHeight();
							int childTop = compDef.getTop() + compDef.getHeight() + 10;
							int height = parentHeight < childTop ? parentHeight / 2 : childTop;
							compDef.setTop(height);
						} else {
							compDef.setTop(compDef.getTop() + 10);
						}
					}
					if (dynaClass.isLeftMovable()) {
						Component comp = (Component) targetDef;
						if (!(comp instanceof Layout)) {
							int parentWidth = comp.getWidth();
							int childLeft = compDef.getLeft() + compDef.getWidth() + 10;
							int width = parentWidth < childLeft ? parentWidth / 2 : childLeft;
							compDef.setLeft(width);
						} else {
							compDef.setLeft(compDef.getLeft() + 10);
						}
					}
					
				} else {
					compDef.setLeft(compDef.getLeft() + 10);
					compDef.setTop(compDef.getTop() + 10);					
				}
			}
			if (command == null) {
				command = genCommand(targetDef, def);
			} else {
				command = command.chain(genCommand(targetDef, def));
			}
		}
		execute(command);

		// もう一度コピー
		List toCopy = new ArrayList();
		for (Iterator it = source.iterator(); it.hasNext();) {
			LayoutElement def = (LayoutElement) it.next();
			try {
				toCopy.add(def.clone());
			} catch (CloneNotSupportedException e) {
			}
		}
		if (canBePasted(toCopy)) {
			Clipboard.getDefault().setContents(toCopy);
		} else {
			this.refresh();
		}
	}

	private Command genCommand(LayoutElement parentDef, LayoutElement childDef) {
		return new AddBasicDefCommand(targetDef, childDef);
	}

	private boolean canBePasted(List basicDefList) {
		GraphicalViewer viewer = (GraphicalViewer) this.editor.getAdapter(
					GraphicalViewer.class);
				
		List sourceEPList = new ArrayList();
		for (Iterator it = basicDefList.iterator(); it.hasNext();) {
			LayoutElement def = (LayoutElement) it.next();
			GraphicalEditPart ep = (GraphicalEditPart)
				viewer.getEditPartFactory().createEditPart(null, def);
			
			ep.setModel(def);
			FigureCanvas canvas = (FigureCanvas) viewer.getControl();
			Rectangle rec = canvas.getViewport().getBounds();
			ep.getFigure().setBounds(new Rectangle(rec.width, rec.height, 0, 0));
			sourceEPList.add(ep);
		}
		List objects = this.getSelectedObjects();
		EditPart targetEP = (EditPart) objects.get(0);
		if (targetEP instanceof LayoutElementEditPart) {
			for (int i = 0; i < sourceEPList.size(); i++) {
				if (!((LayoutElementEditPart) targetEP).canAddChild(
						((EditPart) sourceEPList.get(i)).getModel())) {
					return false;
				}
				EditPart sourceEditPart = (EditPart) sourceEPList.get(i);
				if (sourceEditPart instanceof LayoutElementEditPart) {
					if (!((LayoutElementEditPart) sourceEditPart).
							canAddParent(targetEP.getModel())) {
						return false;
					}
				}
			}
		}
		
		ChangeBoundsRequest request = new ChangeBoundsRequest(RequestConstants.REQ_ADD);
		request.setEditParts(sourceEPList);
		Command command = targetEP.getCommand(request);
		return command == null ? false : command.canExecute();
	}
}
