package org.maskat.core.gen;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;


import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import com.sun.org.apache.xerces.internal.parsers.DOMParser;

/**
 * イベントXMLパーサ
 * <pre>
 * イベントXMLを解析して
 * ClassContentを生成する
 * </pre>
 *
 */
public class EventXmlParser {
	
	Document document;
	
	/**
	 * コンストラクタ
	 * @param eventXml 解析対象XMLパス
	 * @throws SAXException
	 * @throws IOException
	 */
	public EventXmlParser(String eventXml) throws SAXException, IOException {
		DOMParser parser = new DOMParser();
		DtdIgnore dtdIgnore = new DtdIgnore(eventXml);
		parser.parse(new InputSource(dtdIgnore.getReader()));
		document = parser.getDocument();
	}

	/**
	 * 解析結果からパッケージ名を取得
	 * <pre>
	 * パッケージ名は
	 * headerタグのnameが"maskat_layoutID"のもののvalue。
	 * <b>&lt;header name="maskat_layoutID" value="org.maskat.demo"/&gt;</b>
	 * この場合、パッケージ名は"org.maskat.demo"となる。
	 * </pre>
	 * @return パッケージ名
	 */
	public String getPacageName() {
		NodeList headerList = document.getElementsByTagName("header");
		for (int i = 0; i < headerList.getLength(); i++) {
			Node node = headerList.item(i);
			if("maskat_layoutID".equals(node.getAttributes().getNamedItem("name").getTextContent())) {
				return node.getAttributes().getNamedItem("value").getTextContent();
			}
		}
		return "";
	}
	
	/**
	 * すべてのクラスコンテンツ取得
	 * <pre>
	 * すべてのcomponentタグを対象にClassContent作成を試みる。
	 * </pre>
	 * @return ClassContent配列
	 */
	public ClassContent[] getClassContents() {
		NodeList componentList = document.getElementsByTagName("component");
		List classContentList = new ArrayList(0);
		for (int i = 0; i < componentList.getLength(); i++) {
			ClassContent content = getClassContent(componentList.item(i));
			if (content.hasMetod()) {
				classContentList.add(content);
			}
		}
		return (ClassContent[]) classContentList.toArray(new ClassContent[classContentList.size()]);
	}
	
	/**
	 * クラスコンテンツ取得
	 * <pre>
	 * componetタグのidがクラス名に、
	 * componentタグ内のeventタグがメソッド候補になる。
	 * 
	 * eventタグのtypeが未設定もしくは"remote"に指定されている場合、
	 * idをメソッド名としてメソッドリストに追加する。
	 * </pre>
	 * @param node 作成対象ノード
	 * @return 作成したClassContentインスタンス
	 */
	private ClassContent getClassContent(Node node) {
		
		ClassContent content = new ClassContent();
		
		content.setClassName(node.getAttributes().getNamedItem("id").getTextContent());
		
		NodeList children = node.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {
			Node child = children.item(i);
			if ("event".equals(child.getNodeName())) {
				if (null == child.getAttributes().getNamedItem("type") ||
						"remote".equals(child.getAttributes().getNamedItem("type").getTextContent())) {
					content.addClassMedhod(child.getAttributes().getNamedItem("id").getTextContent());
				}
			}
		}
		return content;
	}
}
