package org.kaoriha.marimite;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

class Section {
	public static final long ID_EMPTY = -1;
	private static final String SPLITTER = " - ";
	private static final ConcurrentMap<Section, Section> INSTANCE = new ConcurrentHashMap<Section, Section>();

	private final String start;
	private final String end;
	private final int hashCode;
	private long id = ID_EMPTY;

	private Section(String start, String end) {
		this.start = start;
		this.end = end;
		hashCode = start.hashCode() ^ end.hashCode();
	}

	public static Section getInstance(String start, String end) {
		Section temp = new Section(start, end);
		Section r = INSTANCE.get(temp);
		if (r != null) {
			return r;
		}

		if (start.contains(SPLITTER) || end.contains(SPLITTER)) {
			throw new IllegalArgumentException("must not contain '" + SPLITTER + "'");
		}

		INSTANCE.putIfAbsent(temp, temp);
		return INSTANCE.get(temp);
	}

	@Override
	public int hashCode() {
		return hashCode;
	}

	public String getEnd() {
		return end;
	}

	public String getStart() {
		return start;
	}

	@Override
	public boolean equals(Object obj) {
		Section s = (Section) obj;
		return s.start.equals(start) && s.end.equals(end);
	}

	@Override
	public String toString() {
		return new StringBuilder(start).append(SPLITTER).append(end).toString();
	}

	public static Section fromString(String s) {
		String ss[] = s.split(SPLITTER);
		if (ss.length != 2) {
			throw new IllegalArgumentException("bad string");
		}
		return Section.getInstance(ss[0], ss[1]);
	}

	public long getId() {
		return id;
	}

	public void setId(long id) {
		this.id = id;
	}

	public long getIdAsPersistent(Connection conn) throws SQLException {
		return getIdAsPersistent(conn, true);
	}

	public long getIdAsPersistent(Connection conn, boolean create)
			throws SQLException {
		if (getId() != Section.ID_EMPTY) {
			return getId();
		}

		PreparedStatement sidByName = conn
				.prepareStatement("select id from section where start=? and goal=?");
		sidByName.setString(1, getStart());
		sidByName.setString(2, getEnd());
		ResultSet rs = null;
		try {
			rs = sidByName.executeQuery();
			if (rs.next()) {
				long r = rs.getLong(1);
				setId(r);
				return r;
			}
		} finally {
			rs.close();
		}

		if (!create) {
			return Section.ID_EMPTY;
		}

		PreparedStatement insSection = conn
				.prepareStatement("insert into section (id, start, goal) values (NEXT VALUE FOR seq_section_id, ?, ?)");
		insSection.setString(1, getStart());
		insSection.setString(2, getEnd());
		insSection.executeUpdate();
		try {
			conn.commit();
		} catch (SQLException e) {
			conn.rollback();
			throw e;
		}
		insSection.close();

		return getIdAsPersistent(conn);
	}

	public void disposeId() {
		id = ID_EMPTY;
	}
}
