/*
 * Copyright (c) since 2006 NTT DATA CORPORATION.
 * All rights reserved.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <dirent.h>
#include "postgres.h"
#include "pgsenna2.h"
#include "miscadmin.h"
#include "access/genam.h"
#include "access/heapam.h"
#include "catalog/catalog.h"
#include "commands/async.h"
#include "executor/executor.h"
#include "libpq/libpq.h"
#include "nodes/relation.h"
#include "nodes/pg_list.h"
#include "storage/lock.h"
#include "storage/smgr.h"
#include "utils/fmgroids.h"
#include "utils/relcache.h"
#include "utils/resowner.h"
#include "utils/syscache.h"

HTAB *locktags = NULL;

PG_FUNCTION_INFO_V1(pgs2_lock);
PG_FUNCTION_INFO_V1(pgs2_unlock);
PG_FUNCTION_INFO_V1(pgs2_fsync);
PG_FUNCTION_INFO_V1(pgs2_lock_all);
PG_FUNCTION_INFO_V1(pgs2_unlock_all);
PG_FUNCTION_INFO_V1(pgs2_fsync_all);
static void lock_release(LOCKTAG *ltag);
inline static void gen_pathname(const char *path, char *buffer, int fno);
void do_dir_fsync(const char *dirname);
sen_rc pgs2_sen_index_fsync(const char *path);
sen_rc pgs2_sen_io_fsync(const char *path);
sen_rc pgs2_sen_file_fsync(const char *path);
void sen_str_itoh(unsigned int i, char *p, unsigned int len);


void tagshash_init(HTAB **locktags)
{
  if (*locktags == NULL)
    {
      HASHCTL ctl;
      memset(&ctl, 0, sizeof(ctl));
      ctl.keysize = sizeof(LOCKTAG);
      ctl.entrysize = sizeof(LOCKTAG);
      ctl.hash = tag_hash;
      *locktags = hash_create("ludia locktags",
                             1024,
                             &ctl,
                             HASH_ELEM | HASH_FUNCTION);
      if (*locktags == NULL)
        elog(ERROR, "pgsenna2: tagshash_init falied.");
    }
}


Datum
pgs2_unlock_all(PG_FUNCTION_ARGS)
{
  if (locktags)
    {
      HASH_SEQ_STATUS status;
      LOCKTAG *hentry = NULL;
      hash_seq_init(&status, locktags);
      while ((hentry = (LOCKTAG *) hash_seq_search(&status)) != NULL)
        {
          lock_release(hentry);
        }
      hash_destroy(locktags);
      locktags = NULL;
    }
  PG_RETURN_VOID();
}


Datum
pgs2_lock(PG_FUNCTION_ARGS)
{
  Oid indexrelid = PG_GETARG_OID(0);
#if defined(POSTGRES82) || defined(POSTGRES83)
  LockAcquireResult res = LOCKACQUIRE_OK;
#endif
  LOCKTAG tmptag;
  LOCKTAG *locktag = NULL;
  bool found = false;
#if defined(POSTGRES82) || defined(POSTGRES83)
  ResourceOwner currentOwner;
#endif

  Relation indrel = relation_open(indexrelid, AccessShareLock);
  if (indrel->rd_rel->relkind != RELKIND_INDEX)
    {
      elog(WARNING, "pgsenna2: \"%s\" is not an index", RelationGetRelationName(indrel));
      relation_close(indrel, AccessShareLock);
      PG_RETURN_VOID();
    }
  else
    {
      relation_close(indrel, AccessShareLock);
    }

  SET_LOCKTAG_RELATION(tmptag, MyDatabaseId, indexrelid);
  tagshash_init(&locktags);
  locktag = (LOCKTAG *)hash_search(locktags,
                                   (const void*)&tmptag,
                                   HASH_ENTER,
                                   &found);
  SET_LOCKTAG_RELATION(*locktag, MyDatabaseId, indexrelid);
#if defined(POSTGRES82) || defined(POSTGRES83)
  currentOwner = CurrentResourceOwner;
  PG_TRY();
  {
    CurrentResourceOwner = CurTransactionResourceOwner;
    res = LockAcquire(locktag, ShareUpdateExclusiveLock, false, false);
    if (res == LOCKACQUIRE_OK)
      {
      }
    else if (res == LOCKACQUIRE_ALREADY_HELD)
      {
        elog(WARNING, "pgsenna2: index lock already held.");
        LockRelease(locktag, ShareUpdateExclusiveLock, false);
      }
    else if (res == LOCKACQUIRE_NOT_AVAIL)
      {
        elog(ERROR, "pgsenna2: index lock failed (LOCKACQUIRE_NOT_AVAIL).");
      }
  }
  PG_CATCH();
  {
    /* Ensure CurrentResourceOwner is restored on error */
    CurrentResourceOwner = currentOwner;
    PG_RE_THROW();
  }
  PG_END_TRY();
  CurrentResourceOwner = currentOwner;

#endif 
  PG_RETURN_VOID();
}


Datum
pgs2_unlock(PG_FUNCTION_ARGS)
{
  Oid indexrelid = PG_GETARG_OID(0);
  if (locktags)
    {
      LOCKTAG *ltag = NULL;
      LOCKTAG tmptag;
      SET_LOCKTAG_RELATION(tmptag, MyDatabaseId, indexrelid);
      ltag = (LOCKTAG *)hash_search(locktags,
                                    (const void*)&tmptag,
                                    HASH_FIND,
                                    NULL);
      if (ltag)
        {
          lock_release(ltag);
          ltag = (LOCKTAG *)hash_search(locktags,
                                        (const void*)&tmptag,
                                        HASH_REMOVE,
                                        NULL);
        }
      else
        {
          elog(WARNING, "pgsenna2: no hash entry for %d", indexrelid);
        }
    }
  else
    {
      elog(WARNING, "pgsenna2: locktags hash is not initialized.");
    }
  PG_RETURN_VOID();
}


static void lock_release(LOCKTAG *ltag)
{
#if defined(POSTGRES82) || defined(POSTGRES83)
  bool res = false;
  ResourceOwner currentOwner = CurrentResourceOwner;
  PG_TRY();
  {
    CurrentResourceOwner = CurTransactionResourceOwner;
    res = LockRelease(ltag, ShareUpdateExclusiveLock, false);
    /*
      if (res == false)
      elog(WARNING, "pgsenna2: lock release failed.");
    */
  }
  PG_CATCH();
  {
    /* Ensure CurrentResourceOwner is restored on error */
    CurrentResourceOwner = currentOwner;
    PG_RE_THROW();
  }
  PG_END_TRY();
  CurrentResourceOwner = currentOwner;
#endif
}


Datum
pgs2_fsync(PG_FUNCTION_ARGS)
{
    if (locktags)
    {
      HASH_SEQ_STATUS status;
      LOCKTAG *hentry = NULL;
      hash_seq_init(&status, locktags);
      while ((hentry = (LOCKTAG *) hash_seq_search(&status)) != NULL)
        {
          Oid indexrelid = (Oid)(hentry->locktag_field2);
#ifdef POSTGRES81
          Relation indrel = index_open(indexrelid);
#else
          Relation indrel = index_open(indexrelid, AccessShareLock);
#endif
          char idxpath[MAXPGPATH];
          char *rpath;
          RelationOpenSmgr(indrel);
          rpath = relpath(indrel->rd_smgr->smgr_rnode);
          snprintf(idxpath, MAXPGPATH, "%s/%s", DataDir, rpath);
          pfree(rpath);
          RelationCloseSmgr(indrel);
#ifdef POSTGRES81
          index_close(indrel);
#else
          index_close(indrel, AccessShareLock);
#endif
          pgs2_sen_index_fsync(idxpath);
        }
    }
  PG_RETURN_VOID();
}


Datum
pgs2_lock_all(PG_FUNCTION_ARGS)
{
  Oid am_oid[3];
  ScanKeyData skey[3];
  SysScanDesc relscan;
  Relation relrel;
  int i;

  tagshash_init(&locktags);

  am_oid[0] = GetSysCacheOid(AMNAME, PointerGetDatum("fulltext"), 0, 0, 0);
  am_oid[1] = GetSysCacheOid(AMNAME, PointerGetDatum("fulltextb"), 0, 0, 0);
  am_oid[2] = GetSysCacheOid(AMNAME, PointerGetDatum("fulltextu"), 0, 0, 0);
  elog(LOG, "pgsenna2: am_fulltext_oid: %d, am_fulltextb_oid: %d, am_fulltextu_oid: %d",
       am_oid[0], am_oid[1], am_oid[2]);

  for (i = 0; i < 3; i++)
    {
      ScanKeyInit(skey + i,
                  Anum_pg_class_relam,
                  BTEqualStrategyNumber, F_OIDEQ,
                  ObjectIdGetDatum(am_oid[i]));
    }
  
  relrel = heap_open(RelationRelationId, AccessShareLock);
  for (i = 0; i < 3; i++)
    {
      HeapTuple pg_class_tup;
      relscan = systable_beginscan(relrel, InvalidOid, false, SnapshotNow, 1, skey + i);
      while (HeapTupleIsValid(pg_class_tup = systable_getnext(relscan)))
        {
          Oid indexrelid = HeapTupleGetOid(pg_class_tup);
          DirectFunctionCall1(pgs2_lock, indexrelid);
        }
      systable_endscan(relscan);
    }
  heap_close(relrel, AccessShareLock);
  PG_RETURN_VOID();
}


Datum
pgs2_fsync_all(PG_FUNCTION_ARGS)
{
  char *dirname = (char *)palloc(sizeof(char) * MAXPGPATH);
  snprintf(dirname, MAXPGPATH, "%s/%s", DataDir, DatabasePath);
  do_dir_fsync(dirname);
  pfree(dirname);
  PG_RETURN_VOID();
}


void
do_dir_fsync(const char *dirname)
{
  DIR *dir = opendir(dirname);
  if (dir)
    {
      struct dirent *ent;
      while ((ent = readdir(dir)))
        {
          const char *name = ent->d_name;
          char absolute_dirname[MAXPGPATH];
          if (!strcmp(name, ".") || !strcmp(name, ".."))
            {
              continue;
            }
          sprintf(absolute_dirname, "%s/%s", dirname, ent->d_name);
          do_dir_fsync(absolute_dirname);
        }
      closedir(dir);
    }
  else
    {
      int len = strlen(dirname) - 4;
      if (len > 0 &&
          dirname[len] == '.' &&
          dirname[len + 1] == 'S' &&
          dirname[len + 2] == 'E' &&
          dirname[len + 3] == 'N')
        {
          char *idxname = strdup(dirname);
          idxname[len] = '\0';
          pgs2_sen_index_fsync(idxname);
        }
    }
}


sen_rc
pgs2_sen_index_fsync(const char *path)
{
  sen_rc rc;
  char buffer[PATH_MAX];
  if (!path || strlen(path) > PATH_MAX - 8) { return sen_invalid_argument; }
  snprintf(buffer, PATH_MAX, "%s.SEN", path);
  if ((rc = pgs2_sen_io_fsync(buffer))) { goto exit; }
  snprintf(buffer, PATH_MAX, "%s.SEN.i", path);
  if ((rc = pgs2_sen_io_fsync(buffer))) { goto exit; } // sen_inv_remove
  snprintf(buffer, PATH_MAX, "%s.SEN.i.c", path);
  if ((rc = pgs2_sen_io_fsync(buffer))) { goto exit; }
  snprintf(buffer, PATH_MAX, "%s.SEN.l", path);
  if ((rc = pgs2_sen_io_fsync(buffer))) { goto exit; }
  snprintf(buffer, PATH_MAX, "%s.SEN.v", path);
  rc = pgs2_sen_io_fsync(buffer); // sen_vgram_remove
exit :
  return rc;
}


sen_rc
pgs2_sen_io_fsync(const char *path)
{
   if (pgs2_sen_file_fsync(path) == sen_success) {
    int fno;
    char buffer[PATH_MAX];
    for (fno = 1; ; fno++) {
      gen_pathname(path, buffer, fno);
      if (pgs2_sen_file_fsync(buffer) != sen_success)
        break;
    }
    return sen_success;
  }
  return sen_file_operation_error;
}


sen_rc
pgs2_sen_file_fsync(const char *path)
{
  struct stat s;
  
  if (!stat(path, &s))
    {
      int fd = open(path, O_RDWR);
      if (fd)
        {
          int rc;
          rc = fsync(fd);
          if (rc == -1)
            {
              close(fd);
              elog(ERROR, "pgsenna2: fsync failed (%s), due to %d", path, errno);
            }
          elog(LOG, "pgsenna2: fsynced %s", path);
          close(fd);
          return sen_success;
        }
      else
        {
          return sen_invalid_argument;
        }
    }
  else 
    {
      return sen_invalid_argument;
    }
}


inline static void
gen_pathname(const char *path, char *buffer, int fno)
{
  size_t len = strlen(path);
  memcpy(buffer, path, len);
  if (fno) {
    buffer[len] = '.';
    sen_str_itoh(fno, buffer + len + 1, 3);
  } else {
    buffer[len] = '\0';
  }
}
