/*
 * Copyright (c) since 2006 NTT DATA CORPORATION.
 * All rights reserved.
 */

#include "postgres.h"
#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "fmgr.h"
#include <mb/pg_wchar.h>
#include "pgsenna2.h"
#include "funcapi.h"
#include "utils/guc.h"

PG_FUNCTION_INFO_V1(pgs2pdftotext1);
PG_FUNCTION_INFO_V1(pgs2pdftotext2);
PG_FUNCTION_INFO_V1(pgs2snippet1);
PG_FUNCTION_INFO_V1(pgs2norm);
PG_FUNCTION_INFO_V1(pgs2textporter1);
PG_FUNCTION_INFO_V1(pgs2textporter2);
PG_FUNCTION_INFO_V1(pgs2textporteroption);

#define TMPFILE_UNLINK(path) tmpfile_unlink(path, __FILE__, __LINE__)

void
tmpfile_unlink(char *path, const char* file, int line)
{
  if (unlink(path) == -1) {
    elog(WARNING, "pgsenna2: failed to unlink temporary file (%s) %s:%d",
	 strerror(errno), file, line);
  }
}

Datum
pgs2pdftotext1(PG_FUNCTION_ARGS)
{
  text *path_a = (text*)PG_GETARG_TEXT_P(0);
  char *path = NULL;
  char command[512];
  int buf_size = 4096;
  int buf_read = 0;
  int buf_read_total = 0;
  FILE *stdout_pdftotext;
  text *filtered;

  path = text2cstr(path_a);
  snprintf(command, sizeof(command), "pdftotext %s -",path);
  command[511] = '\0';
  stdout_pdftotext = popen(command, "r");
  if (stdout_pdftotext == (FILE *)-1) {
    elog(ERROR, "pgsenna2: failed to popen for pdftotext (%s)", strerror(errno));
  }
  filtered = PGS2_PALLOC(VARHDRSZ + buf_size);
  while ((buf_read = fread(VARDATA(filtered) + buf_read_total,
                           sizeof(char), buf_size - buf_read_total,
                           stdout_pdftotext))) {
    if (buf_read == (buf_size - buf_read_total)) {
      buf_size *= 2;
      filtered = repalloc(filtered, VARHDRSZ + buf_size);
      if (filtered == NULL) {
        elog(ERROR, "pgsenna2: failed to palloc for return value");
      }
    }
    buf_read_total += buf_read;
  }
  if (ferror(stdout_pdftotext)) {
    elog(ERROR, "pgsenna2: failed to fread temporary PDFfile");
  }
  if (pclose(stdout_pdftotext) != 0) {
    elog(ERROR, "pgsenna2: failed pdftotext 1 (%s)", strerror(errno));
  }
  /* varatt_size include the size of itself */
#ifdef POSTGRES83
  SET_VARSIZE_4B(filtered, VARHDRSZ + buf_read_total);
#else
  VARATT_SIZEP(filtered) = VARHDRSZ + buf_read_total;
#endif
  PG_RETURN_TEXT_P(filtered);
}

Datum
pgs2pdftotext2(PG_FUNCTION_ARGS)
{
  bytea *pdfdata = (bytea*)PG_GETARG_BYTEA_P(0);
  char path[64] = "/tmp/pgs2_XXXXXX";
  int byte_wrote = 0;
  int byte_wrote_total = 0;
  FILE *tmpfile;
  int fd = -1;
  char command[512];
  int buf_size = 4096;
  int buf_read = 0;
  int buf_read_total = 0;
  FILE *stdout_pdftotext;
  text *filtered;

#ifndef WIN32
  fd = mkstemp(path);
#endif
  if (fd == -1) {
    elog(ERROR, "pgsenna2: failed mkstemp for temporary PDFfile (%s)", strerror(errno));
  }
  close(fd);
  tmpfile = fopen(path, "wb");
  if (tmpfile == NULL) {
    TMPFILE_UNLINK(path);
    elog(ERROR, "pgsenna2: failed to fopen temporary PDFfile (%s)", strerror(errno));
  }
  while ((byte_wrote = fwrite(VARDATA(pdfdata) + byte_wrote_total, sizeof(char),
                              VARSIZE(pdfdata) - VARHDRSZ - byte_wrote_total,
                              tmpfile))) {
    byte_wrote_total += byte_wrote;
  }
  if (fclose(tmpfile) != 0) {
    TMPFILE_UNLINK(path);
    elog(ERROR, "pgsenna2: failed to fclose temporary PDFfile (%s)", strerror(errno));
  }
  snprintf(command, sizeof(command), "pdftotext %s -", path);
  //  snprintf(command, sizeof(command), "wvWare --charset UTF-8 %s -", path);
  //  snprintf(command, sizeof(command), "ppthtml %s -", path);
  command[511] = '\0';
  stdout_pdftotext = popen(command, "r");
  if (stdout_pdftotext == (FILE *)-1) {
    TMPFILE_UNLINK(path);
    elog(ERROR, "pgsenna2: failed to popen for pdftotext (%s)", strerror(errno));
  }
  filtered = palloc(VARHDRSZ + buf_size);
  if (filtered == NULL) {
    TMPFILE_UNLINK(path);
    elog(ERROR, "pgsenna2: failed to palloc for return value");
  }
  while ((buf_read = fread(VARDATA(filtered) + buf_read_total,
                           sizeof(char), buf_size - buf_read_total,
                           stdout_pdftotext))) {
    if (buf_read == (buf_size - buf_read_total)) {
      buf_size *= 2;
      filtered = repalloc(filtered, VARHDRSZ + buf_size);
      if (filtered == NULL) {
        TMPFILE_UNLINK(path);
        elog(ERROR, "pgsenna2: failed to palloc for return value");
      }
    }
    buf_read_total += buf_read;
  }
  if (ferror(stdout_pdftotext)) {
    TMPFILE_UNLINK(path);
    elog(ERROR, "pgsenna2: failed to fread temporary PDFfile");
  }
  if (pclose(stdout_pdftotext) != 0) {
    TMPFILE_UNLINK(path);
    elog(ERROR, "pgsenna2: failed pdftotext 2 (%s)", strerror(errno));
  }
  /* varatt_size include the size of itself */
#ifdef POSTGRES83
  SET_VARSIZE_4B(filtered, VARHDRSZ + buf_read_total);
#else
  VARATT_SIZEP(filtered) = VARHDRSZ + buf_read_total;
#endif
  TMPFILE_UNLINK(path);
  PG_RETURN_TEXT_P(filtered);
}

Datum pgs2snippet1(PG_FUNCTION_ARGS)
{
  sen_rc rc = sen_success;
  int flags = PG_GETARG_INT32(0);
  size_t width = (size_t)PG_GETARG_INT32(1);
  unsigned int max_results = PG_GETARG_INT32(2);
  text *defaultopentag_ = (text*)PG_GETARG_TEXT_P(3);
  const char *defaultopentag = text2cstr(defaultopentag_);
  text *defaultclosetag_ = (text*)PG_GETARG_TEXT_P(4);
  const char *defaultclosetag = text2cstr(defaultclosetag_);
  sen_snip_mapping *mapping = (sen_snip_mapping *)PG_GETARG_POINTER(5);
  text *keywords_ = (text*)PG_GETARG_TEXT_P(6);
  char *keywords = text2cstr(keywords_);
  text *document_ = (text*)PG_GETARG_TEXT_P(7);
  char *document = text2cstr(document_);
  text *result = NULL;
  unsigned int result_len = 0;
  sen_snip *snip = NULL;
  unsigned int nresults = 0;
  unsigned int max_tagged_len = 0;
  unsigned int strlen_dot = strlen(defaultopentag);
  unsigned int strlen_dct = strlen(defaultclosetag);

  snip = sen_query_snip(get_query(keywords), flags, width, max_results, 1,
                        &defaultopentag, &strlen_dot,
                        &defaultclosetag, &strlen_dct,
                        mapping);
  if (!snip) {
    elog(ERROR, "pgsenna2: sen_query_snip() failed");
  }
  rc = sen_snip_exec(snip, document, strlen(document),
                     &nresults, &max_tagged_len);
  if (rc != sen_success) {
    elog(ERROR, "pgsenna2: sen_snip_exec() failed %d", rc);
  }
  result = PGS2_PALLOC(VARHDRSZ + max_tagged_len);
  memset(VARDATA(result), 0, max_tagged_len);
  rc = sen_snip_get_result(snip, 0, VARDATA(result), &result_len);
  if (rc == sen_invalid_argument) {
    rc = sen_snip_close(snip);
    if (rc != sen_success) {
      elog(ERROR, "pgsenna2: sen_snip_close() failed1 %d", rc);
    }
    pfree((char *)defaultopentag);
    pfree((char *)defaultclosetag);
    pfree(keywords);
    pfree(document);
    PG_RETURN_NULL();
  } else if (rc != sen_success) {
    elog(ERROR, "pgsenna2: sen_snip_get_result() failed %d", rc);
  }
  rc = sen_snip_close(snip);
  if (rc != sen_success) {
    elog(ERROR, "pgsenna2: sen_snip_close() failed2 %d", rc);
  }
  pfree((char *)defaultopentag);
  pfree((char *)defaultclosetag);
  pfree(keywords);
  pfree(document);
  max_tagged_len --; /* cutting null-end */
#ifdef POSTGRES83
  SET_VARSIZE_4B(result, VARHDRSZ + max_tagged_len);
#else
  VARATT_SIZEP(result) = VARHDRSZ + max_tagged_len;
#endif
  PG_RETURN_TEXT_P(result);
}

Datum pgs2norm(PG_FUNCTION_ARGS)
{
  const char *source = text2cstr((text*)PG_GETARG_TEXT_P(0));
  int source_len;
  int flags = PG_GETARG_INT32(1);
  text *filtered;
  int filtered_len;

  if (flags != 0 && flags != 1) {
    elog(ERROR, "pgsenna2: invalid value at pgs2norm");
  }
  source_len = strlen(source);
  filtered = (text *) PGS2_PALLOC(VARHDRSZ + source_len);
  filtered_len = sen_str_normalize(source, source_len, get_sen_encoding(),
                                   flags, VARDATA(filtered), source_len);
  if (filtered_len == -1) {
    elog(ERROR, "pgsenna2: sen_nstr_open failed during pgs2norm");
  }
  pfree((char *)source);
#ifdef POSTGRES83
  SET_VARSIZE_4B(filtered, VARHDRSZ + filtered_len);
#else
  VARATT_SIZEP(filtered) = VARHDRSZ + filtered_len;
#endif
  PG_RETURN_TEXT_P(filtered);
}

#ifdef TEXTPORTER
typedef struct {
  unsigned char groupname[256];
  unsigned char deflangname[256];
  int bbigendian;
  unsigned int option;
  unsigned int option1;
  long long size;
  unsigned short csv_c;
  int log_flag;
} _TEXTINFO;

int _DMC_GetText_V4(unsigned char *path_bin, unsigned char *path_txt,
                    unsigned char *groupname, unsigned char *deflangname,
                    int bbigendian, unsigned int option, unsigned int option1,
                    long long size, unsigned short csv_c, int flag);
int __DMC_GetText_V4(char *path_bin, char *path_txt) ;
_TEXTINFO get_textinfo();

static inline int textporter_error_check(int iRet, char *path_txt, char *path_bin)
{
  if (iRet != 0) {
    const char *option;
    char textporter_error[64];
    int null_strlen;
    int i = 0;
    option = getludiaoption("ludia.textporter_error");
    if (option) {
      strncpy(textporter_error, option, 64);
    } else {
      strncpy(textporter_error, TEXTPORTER_ERROR, 64);
    }
    while (textporter_error[i] != '\0') {
      textporter_error[i] = tolower(textporter_error[i]);
      i++;
    }
    if (textporter_error[0] == 'd' &&
        textporter_error[1] == 'e' &&
        textporter_error[2] == 'b' &&
        textporter_error[3] == 'u' &&
        textporter_error[4] == 'g') {
      elog(DEBUG1, "pgsenna2: textporter errorcode = %d, %s", iRet, path_bin);
    } else if (textporter_error[0] == 'l' &&
               textporter_error[1] == 'o' &&
               textporter_error[2] == 'g') {
      elog(LOG, "pgsenna2: textporter errorcode = %d, %s", iRet, path_bin);
    } else if (textporter_error[0] == 'i' &&
               textporter_error[1] == 'n' &&
               textporter_error[2] == 'f' &&
               textporter_error[3] == 'o') {
      elog(INFO, "pgsenna2: textporter errorcode = %d, %s", iRet, path_bin);
    } else if (textporter_error[0] == 'n' &&
               textporter_error[1] == 'o' &&
               textporter_error[2] == 't' &&
               textporter_error[3] == 'i' &&
               textporter_error[4] == 'c' &&
               textporter_error[5] == 'e') {
      elog(NOTICE, "pgsenna2: textporter errorcode = %d, %s", iRet, path_bin);
    } else if (textporter_error[0] == 'w' &&
               textporter_error[1] == 'a' &&
               textporter_error[2] == 'r' &&
               textporter_error[3] == 'n' &&
               textporter_error[4] == 'i' &&
               textporter_error[5] == 'n' &&
               textporter_error[6] == 'g') {
      elog(WARNING, "pgsenna2: textporter errorcode = %d, %s", iRet, path_bin);
    } else {
      TMPFILE_UNLINK(path_txt);
      elog(ERROR, "pgsenna2: textporter errorcode = %d, %s", iRet, path_bin);
      return 1;
    }
    null_strlen = strlen(textporter_error);
    if (textporter_error[null_strlen - 4] == 'n' &&
        textporter_error[null_strlen - 3] == 'u' &&
        textporter_error[null_strlen - 2] == 'l' &&
        textporter_error[null_strlen - 1] == 'l') {
      return 1;
    }
  }
  return 0;
}

Datum
pgs2textporter1(PG_FUNCTION_ARGS)
{
  text *path_bin_ = (text*)PG_GETARG_TEXT_P(0);
  unsigned char *path_bin = text2cstr(path_bin_);  /* from */
  unsigned char path_txt[64] = "/tmp/pgs2_XXXXXX";  /* to */
  FILE *tmpfile_txt;
  int fd = -1;
  int buf_size = 4096;
  int buf_read_total = 0;
  char tmp_str[buf_size];
  text *filtered;
  int iRet;

#ifndef WIN32
  fd = mkstemp(path_txt);
#endif
  if (fd == -1) {
    elog(ERROR, "pgsenna2: failed mkstemp for temporary txt-file 1(%s)", strerror(errno));
  }
  close(fd);

  iRet = __DMC_GetText_V4(path_bin, path_txt);

  if (textporter_error_check(iRet, path_txt, path_bin) == 1) {
    TMPFILE_UNLINK(path_txt);
    PG_RETURN_NULL();
  }
  filtered = palloc(VARHDRSZ + buf_size);
  if (filtered == NULL) {
    TMPFILE_UNLINK(path_txt);
    elog(ERROR, "pgsenna2: failed to palloc for return value");
  }
  tmpfile_txt = fopen(path_txt, "r");
  if (tmpfile_txt) {
    while (true) {
      if (fgets(tmp_str, buf_size, tmpfile_txt) == NULL) {
	break;
      } else {
	memcpy(VARDATA(filtered) + buf_read_total, tmp_str, strlen(tmp_str));
	buf_read_total += strlen(tmp_str);
	filtered = repalloc(filtered, VARHDRSZ + buf_read_total + buf_size);
      }
    }
    fclose(tmpfile_txt);
  }
  TMPFILE_UNLINK(path_txt);
  /* varatt_size include the size of itself */
  VARATT_SIZEP(filtered) = VARHDRSZ + buf_read_total;
  PG_RETURN_TEXT_P(filtered);
}

Datum
pgs2textporter2(PG_FUNCTION_ARGS)
{
  bytea *pdfdata = (bytea*)PG_GETARG_BYTEA_P(0);
  unsigned char path_bin[64] = "/tmp/pgs2_XXXXXX";
  unsigned char path_txt[64] = "/tmp/pgs2_XXXXXX";
  int byte_wrote = 0;
  int byte_wrote_total = 0;
  FILE *tmpfile_bin;
  FILE *tmpfile_txt;
  int fd = -1;
  int buf_size = 4096;
  int buf_read_total = 0;
  char tmp_str[buf_size];
  text *filtered;
  int iRet;

#ifndef WIN32
  fd = mkstemp(path_bin);
#endif
  if (fd == -1) {
    elog(ERROR, "pgsenna2: failed mkstemp for temporary binary-file 2(%s)", strerror(errno));
  }
  close(fd);
  tmpfile_bin = fopen(path_bin, "wb");
  if (tmpfile_bin == NULL) {
    TMPFILE_UNLINK(path_bin);
    elog(ERROR, "pgsenna2: failed to fopen temporary binary-file 2(%s)", strerror(errno));
  }
  while ((byte_wrote = fwrite(VARDATA(pdfdata) + byte_wrote_total, sizeof(char),
                              VARSIZE(pdfdata) - VARHDRSZ - byte_wrote_total,
                              tmpfile_bin))) {
    byte_wrote_total += byte_wrote;
  }
  if (fclose(tmpfile_bin) != 0) {
    TMPFILE_UNLINK(path_bin);
    elog(ERROR, "pgsenna2: failed to fclose temporary binary-file 2(%s)", strerror(errno));
  }
#ifndef WIN32
  fd = mkstemp(path_txt);
#endif
  if (fd == -1) {
    TMPFILE_UNLINK(path_bin);
    elog(ERROR, "pgsenna2: failed mkstemp for temporary txt-file 2(%s)", strerror(errno));
  }
  close(fd);

  iRet = __DMC_GetText_V4(path_bin, path_txt);

  TMPFILE_UNLINK(path_bin);
  if (textporter_error_check(iRet, path_txt, "") == 1) {
    TMPFILE_UNLINK(path_txt);
    PG_RETURN_NULL();
  }
  filtered = palloc(VARHDRSZ + buf_size);
  if (filtered == NULL) {
    TMPFILE_UNLINK(path_txt);
    elog(ERROR, "pgsenna2: failed to palloc for return value");
  }
  tmpfile_txt = fopen(path_txt, "r");
  if (tmpfile_txt) {
    while (true) {
      if (fgets(tmp_str, buf_size, tmpfile_txt) == NULL) {
	break;
      } else {
	memcpy(VARDATA(filtered) + buf_read_total, tmp_str, strlen(tmp_str));
	buf_read_total += strlen(tmp_str);
	filtered = repalloc(filtered, VARHDRSZ + buf_read_total + buf_size);
      }
    }
    fclose(tmpfile_txt);
  }
  TMPFILE_UNLINK(path_txt);
  /* varatt_size include the size of itself */
  VARATT_SIZEP(filtered) = VARHDRSZ + buf_read_total;
  PG_RETURN_TEXT_P(filtered);
}


_TEXTINFO get_textinfo() {
  _TEXTINFO _textinfo;
  _textinfo.option = TEXTPORTER_OPTION;
  _textinfo.option1 = TEXTPORTER_OPTION1;
  _textinfo.bbigendian = TEXTPORTER_BBIGENDIAN;
  _textinfo.size = TEXTPORTER_SIZE;
  _textinfo.csv_c = TEXTPORTER_CSV_C;
  _textinfo.log_flag = 0;

  {
    const char *option;
    option = getludiaoption("ludia.textporter_groupname");
    if (option && strcmp(option, "auto")) {
        strncpy(_textinfo.groupname, option, 256);
    } else {
      switch (GetDatabaseEncoding()) {
      case PG_UTF8:
        strcpy(_textinfo.groupname, "UTF-8");
        break;
      case PG_EUC_JP:
        strcpy(_textinfo.groupname, "EUC-JP");
        break;
      default:
        strcpy(_textinfo.groupname, "UTF-8");
      }
    }
  }
  {
    const char *option;
    option = getludiaoption("ludia.textporter_deflangname");
    if (option) {
      strncpy(_textinfo.deflangname, option, 256);
    } else {
      strncpy(_textinfo.deflangname, TEXTPORTER_DEFLANGNAME, 256);
    }
  }
  {
    const char *option;
    option = getludiaoption("ludia.textporter_bbigendian");
    if (option) {
      _textinfo.bbigendian = atoi(option);
      if (_textinfo.bbigendian < 0) {
        elog(ERROR, "pgsenna2: value of ludia.textporter_option is invalid: %x",
             _textinfo.bbigendian);
      }
    }
  }
  {
    const char *option;
    option = getludiaoption("ludia.textporter_option");
    if (option) {
      _textinfo.option = atoi(option);
      if (_textinfo.option < 0) {
        elog(ERROR, "pgsenna2: value of ludia.textporter_option is invalid: %x",
             _textinfo.option);
      }
    }
  }
  {
    const char *option;
    option = getludiaoption("ludia.textporter_option1");
    if (option) {
      _textinfo.option1 = atoi(option);
      if (_textinfo.option < 0) {
        elog(ERROR, "pgsenna2: value of ludia.textporter_option1 is invalid: %x",
             _textinfo.option1);
      }
    }
  }
  {
    const char *option;
    char log[16];
    option = getludiaoption("ludia.textporter_log");
    if (option) {
      strncpy(log, option, 16);
    } else {
      strncpy(log, TEXTPORTER_LOG, 16);
    }
    if (!strcmp(log, "on")) {
      _textinfo.log_flag = 1;  /* on  */
    } else {
      _textinfo.log_flag = 0;  /* off */
    }
  }
  {
    const char *option;
    option = getludiaoption("ludia.textporter_size");
    if (option) {
      _textinfo.size = atoi(option);
      if (_textinfo.size < 0) {
        elog(ERROR, "pgsenna2: value of ludia.textporter_size is invalid: %lld",
             _textinfo.size);
      }
    }
  }
  {
    const char *option;
    option = getludiaoption("ludia.textporter_csv_c");
    if (option) {
      _textinfo.csv_c = (unsigned int)atoi(option);
    }
  }
  elog (DEBUG1, "pgsenna2: textporter GroupName=%s", _textinfo.groupname);
  elog (DEBUG1, "pgsenna2: textporter DefLangName=%s", _textinfo.deflangname);
  elog (DEBUG1, "pgsenna2: textporter bBigEndian=%d", _textinfo.bbigendian);
  elog (DEBUG1, "pgsenna2: textporter Option=0x%x, Option1=0x%x",
        _textinfo.option, _textinfo.option1);
  elog (DEBUG1, "pgsenna2: textporter Size=%lld, Csv_c=%d",
        _textinfo.size, _textinfo.csv_c);
  elog (DEBUG1, "pgsenna2: textporter Log=%s(%d)",
        _textinfo.log_flag ? "on" : "off", _textinfo.log_flag);
  return _textinfo;
}


int __DMC_GetText_V4(char *path_bin, char *path_txt) {
  int iRet = 1;
  _TEXTINFO _textinfo;

  _textinfo = get_textinfo();
  elog (DEBUG1, "pgsenna2: textporter Source-file=%s, Dest-file=%s",
        path_bin, path_txt);
  iRet = _DMC_GetText_V4(path_bin, path_txt, _textinfo.groupname,
                         _textinfo.deflangname, _textinfo.bbigendian,
                         _textinfo.option, _textinfo.option1,
                         _textinfo.size, _textinfo.csv_c, _textinfo.log_flag);
  return iRet;
}

Datum
pgs2textporteroption(PG_FUNCTION_ARGS) {
  TupleDesc tupdesc;
  char *values[8];
  Datum result;
  AttInMetadata *attinmeta;
  _TEXTINFO _textinfo;

  _textinfo = get_textinfo();
  if (get_call_result_type(fcinfo, NULL, &tupdesc) != TYPEFUNC_COMPOSITE) {
    ereport(ERROR,
            (errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
             errmsg("pgsenna2: function returning record called in context "
                    "that cannot accept type record")));
  }
  attinmeta = TupleDescGetAttInMetadata(tupdesc);
  values[0] = PGS2_PALLOC(256);
  snprintf(values[0], 256, "%s", _textinfo.groupname);
  values[1] = PGS2_PALLOC(256);
  snprintf(values[1], 256, "%s", _textinfo.deflangname);
  values[2] = PGS2_PALLOC(16);
  snprintf(values[2], 16, "%d", _textinfo.bbigendian);
  values[3] = PGS2_PALLOC(16);
  snprintf(values[3], 16, "%d", _textinfo.option);
  values[4] = PGS2_PALLOC(16);
  snprintf(values[4], 16, "%d", _textinfo.option1);
  values[5] = PGS2_PALLOC(16);
  snprintf(values[5], 16, "%lld", _textinfo.size);
  values[6] = PGS2_PALLOC(16);
  snprintf(values[6], 16, "%d", _textinfo.csv_c);
  values[7] = PGS2_PALLOC(16);
  snprintf(values[7], 16, "%d", _textinfo.log_flag);
  result = HeapTupleGetDatum(BuildTupleFromCStrings(attinmeta, values));
  PG_RETURN_DATUM(result);
}

#endif   /* TEXTPORTER */
