/*
 * Copyright (c) since 2006 NTT DATA CORPORATION.
 * All rights reserved.
 */

#include "postgres.h"
#include "executor/executor.h"
#include "mb/pg_wchar.h"
#include "miscadmin.h"
#include "storage/lock.h"
#include "utils/memutils.h"
#include "utils/guc.h"
#include "pgsenna2.h"

#ifdef WIN32
int client_min_messages;
int log_min_messages;
int log_min_error_statement;
#endif

void *pgs2_palloc(size_t size, const char* file, int line);
void *pgs2_palloc0(size_t size, const char* file, int line);
char *text2cstr(text *t);
char *datum2cstr(Datum v);
const char *getludiaoption(const char *option);
CurrentId* start_currentid(void);
void reset_currentid(CurrentId* currentid);
bool comp_currentid(CurrentId* currentid);
sen_encoding get_sen_encoding(void);
int my_query_term_callback(const char* term, unsigned int len, void* func_arg);
sen_query *get_query(const char *str);
void update_index_with_char(index_info *ii, const void *key,
                            unsigned int section, char *oldc, char *newc);
void update_index_with_datum(index_info *ii, const void *key,
                             unsigned int section, Datum oldv, Datum newv,
                             Oid oid_type);
unsigned int update_index_with_tuple(IndexInfo* indexInfo, index_info *ii,
                                     EState *estate, HeapTuple oldtup,
                                     HeapTuple newtup, Oid oid_type);
Datum get_tuple_Datum(ExprContext *econtext, HeapTuple tup,
                      int attnum, ExprState *estate);


void *
pgs2_palloc(size_t size, const char* file, int line)
{
  void *res;
  if (!AllocSizeIsValid(size)) {
    elog(ERROR, "pgsenna2: palloc size is invalid %ld, %s:%d",
         (long int)size, file, line);
  }
  res = palloc(size);
  if (!res) {
    elog(ERROR, "pgsenna2: palloc failed %ld, %s:%d",
         (long int)size, file, line);
  }
  return res;
}


void *
pgs2_palloc0(size_t size, const char* file, int line)
{
  void *res;
  if (!AllocSizeIsValid(size)) {
    elog(ERROR, "pgsenna2: palloc size is invalid %ld, %s:%d",
         (long int)size, file, line);
  }
  res = palloc0(size);
  if (!res) {
    elog(ERROR, "pgsenna2: palloc failed %ld, %s:%d",
         (long int)size, file, line);
  }
  return res;
}


/*
  You must 'pfree' after using 'text2cstr','datum2cstr' !!!
  Forgetting 'pfree' causes memory leak.
*/
char *
text2cstr(text *t)
{
  int32 size = VARSIZE(t) - VARHDRSZ;
  char *c = PGS2_PALLOC(size + 1);
  memcpy(c, t->vl_dat, size);
  c[size] = '\0';
  return c;
}


char *
datum2cstr(Datum v)
{
  text *t = DatumGetTextP(v);
  char *c = text2cstr(t);
  if (t != (text *) v) {
    pfree(t);
  }
  return c;
}


const char*
getludiaoption(const char *option)
{
  static int result = -1;

  if (result == -1) {
    const char *ccs = GetConfigOption("custom_variable_classes");
    result = 0;
    if (ccs != NULL) {
      int dotPos = strlen("ludia");
      const char *start = ccs;
      for (;; ++ccs) {
        int c = *ccs;
        if (c == 0 || c == ',') {
          if (dotPos == ccs - start && strncmp(start, "ludia", dotPos) == 0) {
            result = 1;
            break;
          }
          if (c == 0) {
            break;
          }
          start = ccs + 1;
        }
      }
    }
  }
  if (result) {
    return GetConfigOption(option);
  }
  return NULL;
}


CurrentId*
start_currentid(void) {
  CurrentId *currentid = (CurrentId*)malloc(sizeof(CurrentId));
  if (!currentid) {
    elog(ERROR, "pgsenna2: malloc failed while start_currentid");
  }
  reset_currentid(currentid);
  currentid->cid --;
  return currentid;
}


void
reset_currentid(CurrentId* currentid) {
  currentid->xid = GetCurrentTransactionId();
#if defined(POSTGRES81) || defined(POSTGRES82)
  currentid->cid = GetCurrentCommandId();
#else
  currentid->cid = GetCurrentCommandId(false);
#endif
}


bool
comp_currentid(CurrentId* currentid) {
  if (currentid->xid == GetCurrentTransactionId() &&
#if defined(POSTGRES81) || defined(POSTGRES82)
      currentid->cid == GetCurrentCommandId()) {
#else
      currentid->cid == GetCurrentCommandId(false)) {
#endif
    return true;
  }
  return false;
}


sen_encoding
get_sen_encoding(void)
{
  sen_encoding encoding;

  switch (GetDatabaseEncoding()) {
  case PG_UTF8:
    encoding = sen_enc_utf8;
    break;
  case PG_EUC_JP:
    encoding = sen_enc_euc_jp;
    break;
  case PG_SJIS:
    encoding = sen_enc_sjis;
    break;
  default:
    encoding = sen_enc_default;
  }
  return encoding;
}


int my_query_term_callback(const char* term, unsigned int len, void* func_arg) {
  int* n = (int*)func_arg;
  (*n)++;
  elog(DEBUG1, "pgsenna2: term %d (%.*s)", *n, len, term);
  return 1;
}


sen_query*
get_query(const char *str)
{
  static char *cache_str = NULL;
  static sen_query *cache_q = NULL;

  if (!cache_str) {
    cache_str = (char*)malloc(sizeof(char));
    if (!cache_str) {
      elog(ERROR, "pgsenna2: malloc failed while get_query");
    }
    cache_str[0] = '\0';
  }
  if (strcmp(cache_str, str)) {
    sen_rc rc;
    if (cache_q) {
      rc = sen_query_close(cache_q);
      if (rc != sen_success) {
        cache_q = NULL;
        elog(ERROR, "pgsenna2: sen_query_close failed (%d)", rc);
      }
    }
    cache_str = (char *)realloc(cache_str, (strlen(str) + 1) * sizeof(char));
    if (!cache_str) {
      elog(ERROR, "pgsenna2: realloc failed while get_query %ld",
           (long int)((strlen(str) + 1) * sizeof(char)));
    }
    strncpy(cache_str, str, strlen(str));
    cache_str[strlen(str)] = '\0';
    if ((cache_q = sen_query_open(cache_str, strlen(cache_str), sen_sel_or,
                                  PGS2_MAX_N_EXPRS, get_sen_encoding()))) {
      if (sen_query_rest(cache_q, NULL)) {
        elog(WARNING, "pgsenna2: too many expressions (%d)",
             sen_query_rest(cache_q, NULL));
      }
      if (client_min_messages <= DEBUG1 ||
          log_min_messages <= DEBUG1 ||
          log_min_error_statement <= DEBUG1) {
        int n = 0;
        sen_query_term(cache_q, &my_query_term_callback, &n);
      }
    } else {
      elog(ERROR, "pgsenna2: sen_query_open failed");
    }
  }
  return cache_q;
}


void
update_index_with_char(index_info *ii, const void *key, unsigned int section,
                       char *oldc, char *newc) {
  sen_rc rc = sen_success;
  sen_values *oldvalues = NULL;
  sen_values *newvalues = NULL;
  LOCKTAG locktag;
  LockAcquireResult res = LOCKACQUIRE_OK;

  if (oldc) {
    oldvalues = sen_values_open();
    rc = sen_values_add(oldvalues, oldc, strlen(oldc), 0);
    if (rc != sen_success) {
      elog(ERROR, "pgsenna2: sen_values_add %d %d %d",
           rc, (int)strlen(oldc), section);
    }
  }
  if (newc) {
    newvalues = sen_values_open();
    rc = sen_values_add(newvalues, newc, strlen(newc), 0);
    if (rc != sen_success) {
      elog(ERROR, "pgsenna2: sen_values_add %d %d %d.",
           rc, (int)strlen(newc), section);
    }
  }
  SET_LOCKTAG_RELATION(locktag, MyDatabaseId, ii->relid);
#if defined(POSTGRES82) || defined(POSTGRES83)
  res = LockAcquire(&locktag, ShareUpdateExclusiveLock, 0, 0);
#endif
  if(res == LOCKACQUIRE_OK) {
    rc = sen_index_update(ii->index, key, section + 1, oldvalues, newvalues);
  } else {
    elog(ERROR, "pgsenna2: cannot LockAcquire while update_index (%d)", res);
  }
#if defined(POSTGRES82) || defined(POSTGRES83)
  LockRelease(&locktag, ShareUpdateExclusiveLock, 0);
#endif
  if (rc != sen_success) {
    elog(ERROR, "pgsenna2: sen_index_update failed %d,%d", rc, section);
  }
  if (oldvalues) {
    rc = sen_values_close(oldvalues);
    if (rc != sen_success) {
      elog(ERROR, "pgsenna2: sen_values_close failed %d %d", rc, section);
    }
  }
  if (newvalues) {
    rc = sen_values_close(newvalues);
    if (rc != sen_success) {
      elog(ERROR, "pgsenna2: sen_values_close failed %d %d.", rc, section);
    }
  }
}


/* while using array, section doesn't correspond with column-number. */
/* So, while using array, you cannot use multi-column-index. */
void
update_index_with_datum(index_info *ii, const void *key, unsigned int section,
                        Datum oldv, Datum newv, Oid oid_type)
{
  if (oid_type == TEXTOID) {
    char *oldc = NULL;
    char *newc = NULL;

    if (oldv) {
      oldc = datum2cstr(oldv);
    }
    if (newv) {
      newc = datum2cstr(newv);
    }
    if (oldc || newc) {
      update_index_with_char(ii, key, section, oldc, newc);
    }
    if (oldc) {
      pfree(oldc);
    }
    if (newc) {
      pfree(newc);
    }
  } else {
    if (oldv) {
      ArrayType* atype = DatumGetArrayTypeP(oldv);
      char* v_ptr = ARR_DATA_PTR(atype);
      int i;
#ifdef POSTGRES81
      bits8 *bitmap = NULL;
#else
      bits8 *bitmap = ARR_NULLBITMAP(atype);
#endif
      int bitmask = 1;
      int null_flag = 0;
      for (i = 0; i < *ARR_DIMS(atype); i++) {
        if (!bitmap || (*bitmap & bitmask)) {
          char *c = datum2cstr(fetch_att(v_ptr, false, -1));
          update_index_with_char(ii, key, i, c, NULL);
          null_flag = 1;
#if defined(POSTGRES81) || defined(POSTGRES82)
          v_ptr = att_addlength(v_ptr, -1, PointerGetDatum(v_ptr));
          v_ptr = (char *) att_align(v_ptr, 'i');
#else
          v_ptr = att_addlength_pointer(v_ptr, -1, PointerGetDatum(v_ptr));
          v_ptr = (char *) att_align_nominal(v_ptr, 'i');
#endif
          /* SELECT typalign FROM pg_type WHERE typalign LIKE '%text%'; --'i' */
          pfree(c);
        }
        if (bitmap) {
          bitmask <<= 1;
          if (bitmask == 0x100) {
            bitmap++;
            bitmask = 1;
          }
        }
      }
      // If no records is inserted into fulltext-index, 
      // records of index differ from records of table.
      // So, "" is inserted into fulltext-index.
      if (null_flag == 0) {
        update_index_with_char(ii, key, i, "", NULL);
      }
    }
    if (newv) {
      ArrayType* atype = DatumGetArrayTypeP(newv);
      char* v_ptr = ARR_DATA_PTR(atype);
      int i;
#ifdef POSTGRES81
      bits8 *bitmap = NULL;
#else
      bits8 *bitmap = ARR_NULLBITMAP(atype);
#endif
      int bitmask = 1;
      int null_flag = 0;
      for (i = 0; i < *ARR_DIMS(atype); i++) {
        if (!bitmap || (*bitmap & bitmask)) {
          char *c = datum2cstr(fetch_att(v_ptr, false, -1));
          update_index_with_char(ii, key, i, NULL, c);
#if defined(POSTGRES81) || defined(POSTGRES82)
          v_ptr = att_addlength(v_ptr, -1, PointerGetDatum(v_ptr));
          v_ptr = (char *) att_align(v_ptr, 'i');
#else
          v_ptr = att_addlength_pointer(v_ptr, -1, PointerGetDatum(v_ptr));
          v_ptr = (char *) att_align_nominal(v_ptr, 'i');
#endif
          /* SELECT typalign FROM pg_type WHERE typalign LIKE '%text%'; --'i' */
          pfree(c);
        }
        if (bitmap) {
          bitmask <<= 1;
          if (bitmask == 0x100) {
            bitmap++;
            bitmask = 1;
          }
        }
      }
      if (null_flag == 0) {
        update_index_with_char(ii, key, i, NULL, "");
      }
    }
  }
}


unsigned int update_index_with_tuple(IndexInfo* indexInfo,
                                     index_info *ii,
                                     EState *estate,
                                     HeapTuple oldtup,
                                     HeapTuple newtup,
                                     Oid oid_type)
{
  ExprContext *econtext = GetPerTupleExprContext(estate);
  int j;
  unsigned int num_records = 0;
  ListCell *indexpr_item = list_head(indexInfo->ii_ExpressionsState);

  for (j = 0; j < indexInfo->ii_NumIndexAttrs; j++) {
    int keycol = indexInfo->ii_KeyAttrNumbers[j];
    Datum oldDatum = 0;
    Datum newDatum = 0;

    if (keycol != 0) { /* plain index */
      if (oldtup) {
        oldDatum = get_tuple_Datum(econtext, oldtup, keycol, NULL);
      }
      if (newtup) {
        newDatum = get_tuple_Datum(econtext, newtup, keycol, NULL);
      }
    } else { /* index on expressions */
      if (indexpr_item == NULL) {
        elog(ERROR, "pgsenna2: wrong number of index expressions");
      }
      if (oldtup) {
        oldDatum = get_tuple_Datum(econtext, oldtup, 0,
                                   (ExprState *)lfirst(indexpr_item));
      }
      if (newtup) {
        newDatum = get_tuple_Datum(econtext, newtup, 0,
                                   (ExprState *)lfirst(indexpr_item));
      }
      indexpr_item = lnext(indexpr_item);
    }
    if (oldDatum || newDatum) {
      if (oldDatum) {
        update_index_with_datum(ii, (&(oldtup->t_self)),
                                j, oldDatum, 0, oid_type);
      }
      if (newDatum) {
        update_index_with_datum(ii, (&(newtup->t_self)),
                                j, 0, newDatum, oid_type);
      }
      num_records += 1;
    }
  }
  return num_records;
}


Datum get_tuple_Datum(ExprContext *econtext,
                     HeapTuple tup,
                     int attnum,
                     ExprState *estate)
{
  Datum iDatum;
  bool isNull;

  ExecStoreTuple(tup, econtext->ecxt_scantuple, InvalidBuffer, false);
  if (!estate) {
    iDatum = slot_getattr(econtext->ecxt_scantuple, attnum, &isNull);
  } else {
    iDatum = ExecEvalExprSwitchContext(estate, econtext, &isNull, NULL);
  }
  if (isNull) {
    iDatum = 0;
  }
  ExecClearTuple(econtext->ecxt_scantuple);
  return iDatum;
}
