/*
 * Copyright (c) 2006-2008 NTT DATA CORPORATION.
 * All rights reserved.
 */

#include "postgres.h"
#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>
#include "pgsenna2.h"
#include "miscadmin.h"
#include "funcapi.h"
#include "utils/guc.h"

PG_FUNCTION_INFO_V1(pgs2destroy);
PG_FUNCTION_INFO_V1(pgs2indexinfo);
PG_FUNCTION_INFO_V1(pgs2indexinfodb);
PG_FUNCTION_INFO_V1(pgs2getlexicon);
PG_FUNCTION_INFO_V1(pgs2version);
PG_FUNCTION_INFO_V1(pgs2seninfo);
PG_FUNCTION_INFO_V1(pgs2getoption);

Datum _pgs2indexinfo(FunctionCallInfo fcinfo, char *path);

#ifdef WIN32
#if defined(POSTGRES81)
extern DLLIMPORT char *DataDir;
extern DLLIMPORT char *DatabasePath;
#endif /* POSTGRES81 */
#if defined(POSTGRES82)
extern DLLIMPORT char *DatabasePath;
#endif /* POSTGRES82 */
#endif /* WIN32 */

typedef struct {
  char *filename;
  int dead_flag;   // 0 == alive, 1 == dead
  int key_size;
  int flags;
  int initial_n_segments;
  sen_encoding encoding;
  unsigned nrecords_keys;
  unsigned file_size_keys;
  unsigned nrecords_lexicon;
  unsigned file_size_lexicon;
  unsigned long long inv_seg_size;
  unsigned long long inv_chunk_size;
  void *next_list;
} sen_index_list;

typedef struct {
  int id;
  char *lexicon_array;
  int flag;
} sym_array;


static void
do_dir(const char *dirname, int *n)
{
  DIR *dir = opendir(dirname);

  if (dir) {
    struct dirent *ent;
    while ((ent = readdir(dir))) {
      const char *name = ent->d_name;
      char absolute_dirname[MAXPGPATH];
      if (!strcmp(name, ".") || !strcmp(name, "..")) {
        continue;
      }
      sprintf(absolute_dirname, "%s/%s", dirname, ent->d_name);
      do_dir(absolute_dirname, n);
    }
    closedir(dir);
  } else {
    const char *name = dirname;
    int len = strlen(name) - 4;
    if (len > 0 &&
        name[len] == '.' &&
        name[len + 1] == 'S' &&
        name[len + 2] == 'E' &&
        name[len + 3] == 'N') {
      struct stat s;
      char *idxname = strdup(name);
      idxname[len] = '\0';
      if (stat(idxname, &s) == -1 && errno == ENOENT) {
        sen_rc rc;

        elog(DEBUG1, "pgsenna2: sen_index_remove %s", idxname);
        rc = sen_index_remove(idxname);
        /*
        if (rc != sen_success) {
          elog(ERROR, "pgsenna2: sen_index_remove failed while pgs2destroy (%d)",
               rc);
        }
        */
        (*n)++;
      }
    }
  }
}

Datum
pgs2destroy(PG_FUNCTION_ARGS)
{
  int n = 0;
  DIR *dir = opendir(".");

  if (dir) {
    do_dir(DataDir, &n);
    closedir(dir);
  } else {
    elog(ERROR, "pgsenna2: pgs2destroy cannot opendir (%s)", strerror(errno));
  }
  PG_RETURN_INT32(n);
}

static void
do_dir_indexinfo(const char *dirname, int *n, sen_index_list *si_list)
{
  DIR *dir = opendir(dirname);

  if (dir) {
    struct dirent *ent;
    while ((ent = readdir(dir))) {
      const char *name = ent->d_name;
      char absolute_dirname[MAXPGPATH];
      if (!strcmp(name, ".") || !strcmp(name, "..")) {
        continue;
      }
      sprintf(absolute_dirname, "%s/%s", dirname, ent->d_name);
      do_dir_indexinfo(absolute_dirname, n, si_list);
    }
    closedir(dir);
  } else {
    const char *name = dirname;
    int len = strlen(name) - 4;
    if (len > 0 &&
        name[len] == '.' &&
        name[len + 1] == 'S' &&
        name[len + 2] == 'E' &&
        name[len + 3] == 'N') {
      char *idxname = strdup(name);
      struct stat s;
      sen_index_list *si_list_next;
      sen_index_list *si_list_tmp;
      sen_index *sindex;
      sen_rc rc = sen_success;

      idxname[len] = '\0';
      sindex = sen_index_open(idxname);
      if (sindex == NULL) {
        elog(ERROR, "pgsenna2: pgs2indexinfo cannot open senna index %s",
             idxname);
      }
      si_list_next = (sen_index_list *) PGS2_PALLOC(sizeof(*si_list_next));
      sen_index_info(sindex,
                     &(si_list_next->key_size),
                     &(si_list_next->flags),
                     &(si_list_next->initial_n_segments),
                     &(si_list_next->encoding),
                     &(si_list_next->nrecords_keys),
                     &(si_list_next->file_size_keys),
                     &(si_list_next->nrecords_lexicon),
                     &(si_list_next->file_size_lexicon),
                     &(si_list_next->inv_seg_size),
                     &(si_list_next->inv_chunk_size));
      rc = sen_index_close(sindex);
      if (rc != sen_success) {
        elog(ERROR, "pgsenna2: sen_index_close failed while pgs2indexinfo (%d)",
             rc);
      }
      si_list_tmp = si_list;
      while(si_list_tmp->next_list != NULL) {
        si_list_tmp = si_list_tmp->next_list;
      }
      si_list_tmp->next_list = si_list_next;
      si_list_next->filename = idxname;
      if (stat(idxname, &s) == -1 && errno == ENOENT) {
        si_list_next->dead_flag = 1;
      } else {
        si_list_next->dead_flag = 0;
      }
      si_list_next->next_list = NULL;
      (*n)++;
    }
  }
}

Datum
_pgs2indexinfo(FunctionCallInfo fcinfo, char *path)
{
  int n = 0;
  FuncCallContext     *funcctx;
  int                  call_cntr;
  int                  max_calls;
  TupleDesc            tupdesc;
  AttInMetadata       *attinmeta;
  sen_index_list *si_list;

  if (SRF_IS_FIRSTCALL()) {
    DIR *dir = opendir(".");
    MemoryContext   oldcontext;
    funcctx = SRF_FIRSTCALL_INIT();
    oldcontext = MemoryContextSwitchTo(funcctx->multi_call_memory_ctx);
    if (get_call_result_type(fcinfo, NULL, &tupdesc) != TYPEFUNC_COMPOSITE)
      ereport(ERROR,
              (errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
               errmsg("pgsenna2: function returning record called in context "
                      "that cannot accept type record")));
    attinmeta = TupleDescGetAttInMetadata(tupdesc);
    funcctx->attinmeta = attinmeta;

    si_list = (sen_index_list*) PGS2_PALLOC(sizeof(*si_list));
    si_list->next_list = NULL;
    funcctx->user_fctx = (void *)si_list;
    if (dir) {
      do_dir_indexinfo(path, &n, si_list);
      closedir(dir);
    } else {
      elog(ERROR, "pgsenna2: pgs2indexinfo cannot opendir (%s)", strerror(errno));
    }
    funcctx->max_calls = n;
    MemoryContextSwitchTo(oldcontext);
  }
  funcctx = SRF_PERCALL_SETUP();
  si_list = (sen_index_list *) funcctx->user_fctx;
  call_cntr = funcctx->call_cntr;
  max_calls = funcctx->max_calls;
  attinmeta = funcctx->attinmeta;
  if (call_cntr < max_calls) {
    int i;
    char       **values;
    HeapTuple    tuple;
    Datum        result;

    si_list = si_list->next_list;
    for (i = 0; i < call_cntr; i++) {
      si_list = si_list->next_list;
    }
    values = (char **) PGS2_PALLOC(12 * sizeof(char *));
    values[0] = (char *) PGS2_PALLOC(MAXPGPATH * sizeof(char));
    for(i = 1; i < 12; i++) {
      values[i] = (char *) PGS2_PALLOC(24 * sizeof(char));
    }
    snprintf(values[0], MAXPGPATH, "%s", si_list->filename);
    snprintf(values[1], 24, "%d", si_list->dead_flag);
    snprintf(values[2], 24, "%d", si_list->key_size);
    snprintf(values[3], 24, "%d", si_list->flags);
    snprintf(values[4], 24, "%d", si_list->initial_n_segments);
    snprintf(values[5], 24, "%d", si_list->encoding);
    snprintf(values[6], 24, "%u", si_list->nrecords_keys);
    snprintf(values[7], 24, "%u", si_list->file_size_keys);
    snprintf(values[8], 24, "%u", si_list->nrecords_lexicon);
    snprintf(values[9], 24, "%u", si_list->file_size_lexicon);
    snprintf(values[10], 24, "%llu", si_list->inv_seg_size);
    snprintf(values[11], 24, "%llu", si_list->inv_chunk_size);
    tuple = BuildTupleFromCStrings(attinmeta, values);
    result = HeapTupleGetDatum(tuple);
    for(i = 0; i < 12; i++) {
      pfree(values[i]);
    }
    pfree(values);
    SRF_RETURN_NEXT(funcctx, result);
  } else {
    SRF_RETURN_DONE(funcctx);
  }
}

Datum
pgs2indexinfo(PG_FUNCTION_ARGS) {
  PG_RETURN_DATUM(_pgs2indexinfo(fcinfo, DataDir));
}

Datum
pgs2indexinfodb(PG_FUNCTION_ARGS) {
#if WIN32 && defined(POSTGRES83)
  elog(ERROR, "pgsenna2: cannot use pgs2indexinfodb at PG83Windows");
#endif
  PG_RETURN_DATUM(_pgs2indexinfo(fcinfo, DatabasePath));
}

#ifndef WIN32
void *sen_inv_cursor_open(sen_inv *inv, unsigned key, int with_pos);

static sym_array **
do_dir_getlexicon(const char *dirname, int *n, sym_array **s_array, char *filename)
{
  DIR *dir = opendir(dirname);

  if (dir) {
    struct dirent *ent;
    while ((ent = readdir(dir))) {
      const char *name = ent->d_name;
      if (!strcmp(name, ".") || !strcmp(name, "..")) {
        continue;
      }
      sym_array **tmp_s_array;
      char absolute_dirname[MAXPGPATH];
      sprintf(absolute_dirname, "%s/%s", dirname, ent->d_name);
      tmp_s_array = do_dir_getlexicon(absolute_dirname, n,
                                      s_array, filename);
      if (tmp_s_array != NULL) {
        closedir(dir);
        return tmp_s_array;
      }
    }
    closedir(dir);
  } else {
    const char *name = dirname;
    int len = strlen(name) - 4;
    char absolute_dirname[MAXPGPATH];
    sprintf(absolute_dirname, "%s", dirname);
    if (len > 0 &&
        name[len] == '.' &&
        name[len + 1] == 'S' &&
        name[len + 2] == 'E' &&
        name[len + 3] == 'N') {
      char *idxname = strdup(name);
      int len2 = strlen(filename);
      int i, flag = 1;
      idxname[len] = '\0';
      sen_rc rc = sen_success;

      for(i = len2; i > 0; i--) {
        if(name[len-i] != filename[len2-i]) {
          flag = 0;
          break;
        }
      }
      if (flag) {
        sen_index *sindex;
        char *keybuf;
        int sid = 1;
        keybuf = (char *) PGS2_PALLOC(sizeof(char *) * SEN_SYM_MAX_KEY_SIZE);
        sindex = sen_index_open(idxname);
        if (sindex == NULL) {
          elog(ERROR, "pgsenna2: pgs2getlexicon cannot open senna index %s",
               idxname);
        }
        (*n) = 0;
        s_array = (sym_array **) PGS2_PALLOC(sen_sym_size(sindex->lexicon) *
                                        sizeof(**s_array));
        while (true) {
          keybuf[0] = '\0';
          sen_sym_key(sindex->lexicon, sid, keybuf, SEN_SYM_MAX_KEY_SIZE);
          if (sid == SEN_SYM_NIL) {
            break;
          }
          s_array[*n] = (sym_array *) PGS2_PALLOC(sizeof(*s_array[*n]));
          s_array[*n]->id = sid;
          s_array[*n]->lexicon_array = (char *) PGS2_PALLOC(strlen(keybuf) *
                                                            sizeof(char));
          if(sen_inv_cursor_open(sindex->inv , sid, 1)) {
            s_array[*n]->flag = 1;
          } else {
            s_array[*n]->flag = 0;
          }
          strcpy(s_array[*n]->lexicon_array, keybuf);
          sid = sen_sym_next(sindex->lexicon, sid);
          (*n)++;
        }
        sen_index_close(sindex);
        if (rc != sen_success) {
          elog(ERROR, "pgsenna2: sen_index_close failed while pgs2getlexicon (%d)",
               rc);
        }
        return s_array;
      }
    }
  }
  return NULL;
}
#endif /* WIN32 */

Datum
pgs2getlexicon(PG_FUNCTION_ARGS)
{
  int n = 0;
  text *filename_a = (text*)PG_GETARG_TEXT_P(0);
  char *filename = NULL;
  FuncCallContext     *funcctx;
  int                  call_cntr;
  int                  max_calls;
  TupleDesc            tupdesc;
  AttInMetadata       *attinmeta;
  sym_array           **s_array = NULL;

  if (SRF_IS_FIRSTCALL()) {
    DIR *dir = opendir(".");
    MemoryContext   oldcontext;
    funcctx = SRF_FIRSTCALL_INIT();
    oldcontext = MemoryContextSwitchTo(funcctx->multi_call_memory_ctx);
    if (get_call_result_type(fcinfo, NULL, &tupdesc) != TYPEFUNC_COMPOSITE)
      ereport(ERROR,
              (errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
               errmsg("pgsenna2: function returning record called in context "
                      "that cannot accept type record")));
    attinmeta = TupleDescGetAttInMetadata(tupdesc);
    funcctx->attinmeta = attinmeta;

    if (dir) {
      filename = text2cstr(filename_a);
#ifdef WIN32
      elog(ERROR, "pgsenna2: cannot use pgs2getlexion at Windows");
#else
      s_array = do_dir_getlexicon(DataDir, &n, s_array, filename);
#endif
      closedir(dir);
    } else {
      elog(ERROR, "pgsenna2: pgs2getlexicon cannot opendir (%s)", strerror(errno));
    }
    funcctx->user_fctx = s_array;
    funcctx->max_calls = n;
    MemoryContextSwitchTo(oldcontext);
  }
  funcctx = SRF_PERCALL_SETUP();
  s_array = (sym_array **) funcctx->user_fctx;
  call_cntr = funcctx->call_cntr;
  max_calls = funcctx->max_calls;
  attinmeta = funcctx->attinmeta;
  if (call_cntr < max_calls) {
    int i;
    char       **values;
    HeapTuple    tuple;
    Datum        result;
    int len;

    len = strlen(s_array[call_cntr]->lexicon_array) + 1;
    values = (char **) PGS2_PALLOC(3 * sizeof(char *));
    values[0] = (char *) PGS2_PALLOC(16 * sizeof(char));
    values[1] = (char *) PGS2_PALLOC(len * sizeof(char));
    values[2] = (char *) PGS2_PALLOC(16 * sizeof(char));
    snprintf(values[0], 16, "%d", s_array[call_cntr]->id);
    snprintf(values[1], len, "%s", s_array[call_cntr]->lexicon_array);
    snprintf(values[2], 16, "%d", s_array[call_cntr]->flag);
    tuple = BuildTupleFromCStrings(attinmeta, values);
    result = HeapTupleGetDatum(tuple);
    for(i = 0; i < 3; i++) {
      pfree(values[i]);
    }
    pfree(values);
    SRF_RETURN_NEXT(funcctx, result);
  } else {
    SRF_RETURN_DONE(funcctx);
  }
}


Datum
pgs2version(PG_FUNCTION_ARGS)
{
  text *buffer;

  buffer = (text *) PGS2_PALLOC(VARHDRSZ + strlen(PGS2_PACKAGE_STRING));
#ifdef POSTGRES83
  SET_VARSIZE_4B(buffer, VARHDRSZ + strlen(PGS2_PACKAGE_STRING));
#else
  VARATT_SIZEP(buffer) = VARHDRSZ + strlen(PGS2_PACKAGE_STRING);
#endif
  memcpy((char *)(VARDATA(buffer)), PGS2_PACKAGE_STRING,
         strlen(PGS2_PACKAGE_STRING));
  PG_RETURN_TEXT_P(buffer);
}

Datum
pgs2seninfo(PG_FUNCTION_ARGS)
{
  TupleDesc tupdesc;
  char *values[6];
  Datum result;
  AttInMetadata *attinmeta;
  char *version;
  char *configure_options;
  char *config_path;
  sen_encoding default_encoding;
  unsigned int initial_n_segments;
  unsigned int partial_match_threshold;

  version = (char *) PGS2_PALLOC(24);
  configure_options = (char *) PGS2_PALLOC(256);
  config_path = (char *) PGS2_PALLOC(MAXPGPATH);
  sen_info(&version, &configure_options, &config_path,
           &default_encoding, &initial_n_segments, &partial_match_threshold);
  if (get_call_result_type(fcinfo, NULL, &tupdesc) != TYPEFUNC_COMPOSITE) {
    ereport(ERROR,
            (errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
             errmsg("pgsenna2: function returning record called in context "
                    "that cannot accept type record")));
  }
  attinmeta = TupleDescGetAttInMetadata(tupdesc);
  values[0] = PGS2_PALLOC(24);
  snprintf(values[0], 24, "%s", version);
  values[1] = PGS2_PALLOC(256);
  snprintf(values[1], 256, "%s", configure_options);
  values[2] = PGS2_PALLOC(MAXPGPATH);
  snprintf(values[2], MAXPGPATH, "%s", config_path);
  values[3] = PGS2_PALLOC(24);
  snprintf(values[3], 24, "%d", default_encoding);
  values[4] = PGS2_PALLOC(24);
  snprintf(values[4], 24, "%d", initial_n_segments);
  values[5] = PGS2_PALLOC(24);
  snprintf(values[5], 24, "%d", partial_match_threshold);
  result = HeapTupleGetDatum(BuildTupleFromCStrings(attinmeta, values));
  PG_RETURN_DATUM(result);
}

Datum
pgs2getoption(PG_FUNCTION_ARGS)
{
  TupleDesc tupdesc;
  char *values[6];
  Datum result;
  AttInMetadata *attinmeta;
  int max_n_sort_result;
  char enable_seqscan[16];
  int sen_index_flags;
  int max_n_index_cache;
  int initial_n_segments;
  char seqscan_flags;

  {
    const char *option = getludiaoption("ludia.max_n_sort_result");
    if (option) {
      max_n_sort_result = atoi(option);
    } else {
      max_n_sort_result = MAX_N_SORT_RESULT;
    }
  }
  {
    const char *option = getludiaoption("ludia.enable_seqscan");
    if (option) {
      strncpy(enable_seqscan, option, 16);
    } else {
      strncpy(enable_seqscan, ENABLE_SEQSCAN, 16);
    }
  }
  {
    const char *option = getludiaoption("ludia.seqscan_flags");
    if (option) {
      seqscan_flags = atoi(option);
    } else {
      seqscan_flags = SEQSCAN_FLAGS;
    }
  }
  {
    const char *option = getludiaoption("ludia.sen_index_flags");
    if (option) {
      sen_index_flags = atoi(option);
    } else {
      sen_index_flags = SEN_INDEX_FLAGS;
    }
  }
  {
    const char *option = getludiaoption("ludia.max_n_index_cache");
    if (option) {
      max_n_index_cache = atoi(option);
    } else {
      max_n_index_cache = MAX_N_INDEX_CACHE;
    }
  }
  {
    const char *option = getludiaoption("ludia.initial_n_segments");
    if (option) {
      initial_n_segments = atoi(option);
    } else {
      initial_n_segments = INITIAL_N_SEGMENTS;
    }
  }
  if (get_call_result_type(fcinfo, NULL, &tupdesc) != TYPEFUNC_COMPOSITE) {
    ereport(ERROR,
            (errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
             errmsg("pgsenna2: function returning record called in context "
                    "that cannot accept type record")));
  }
  attinmeta = TupleDescGetAttInMetadata(tupdesc);
  values[0] = PGS2_PALLOC(16);
  snprintf(values[0], 16, "%d", max_n_sort_result);
  values[1] = PGS2_PALLOC(16);
  snprintf(values[1], 16, "%s", enable_seqscan);
  values[2] = PGS2_PALLOC(16);
  snprintf(values[2], 16, "%d", seqscan_flags);
  values[3] = PGS2_PALLOC(16);
  snprintf(values[3], 16, "%d", sen_index_flags);
  values[4] = PGS2_PALLOC(16);
  snprintf(values[4], 16, "%d", max_n_index_cache);
  values[5] = PGS2_PALLOC(16);
  snprintf(values[5], 16, "%d", initial_n_segments);
  result = HeapTupleGetDatum(BuildTupleFromCStrings(attinmeta, values));
  PG_RETURN_DATUM(result);
}
