/*
 * Copyright (c) 2006-2008 NTT DATA CORPORATION.
 * All rights reserved.
 */

#include "postgres.h"
#include "access/genam.h"
#include "catalog/index.h"
#include "catalog/namespace.h"
#include "utils/syscache.h"
#include "utils/fmgroids.h"
#include "pgsenna2.h"

#ifdef POSTGRES82

#include "xlog.c"

typedef struct LudiaRelInfoNode
{
  RelFileNode node; /* hash key */
  Oid indrelid;
  TupleTableSlot *slot;
  List *iilist;
} LudiaRelInfoNode;

typedef struct LudiaIndexInfoNode
{
  Oid indexrelid;
  IndexInfo *indexInfo;
} LudiaIndexInfoNode;

PG_FUNCTION_INFO_V1(pgs2recovery);
PG_FUNCTION_INFO_V1(_pgs2recovery);
HTAB *init_ludiarelhash(EState *estate);
void destroy_ludiarelhash(HTAB *ludiarelhash);
void update_index_for_recovery(ItemPointer tid, EState *estate, LudiaRelInfoNode *relinfo, Oid oid_type);


Datum
pgs2recovery(PG_FUNCTION_ARGS)
{
  EState *estate = CreateExecutorState();
  HTAB *ludiarelhash = init_ludiarelhash(estate);
  sen_check_init();
#if 1
  CheckPoint checkPoint;
  bool wasShutdown;
  XLogRecPtr RecPtr;
  XLogRecPtr LastRec;
  XLogRecPtr checkPointLoc;
  /*  XLogRecPtr minRecoveryLoc; */
  /*  XLogRecPtr EndOfLog; */
  XLogRecord *record;
  bool needNewTimeLine = false;

  ControlFile = (ControlFileData*)palloc(sizeof(ControlFileData));
  ReadControlFile();
  if (ControlFile->logSeg == 0 ||
      ControlFile->state < DB_SHUTDOWNED ||
      ControlFile->state > DB_IN_PRODUCTION ||
      !XRecOffIsValid(ControlFile->checkPoint.xrecoff))
    ereport(FATAL,
            (errmsg("control file contains invalid data")));

  recoveryTargetTLI = ControlFile->checkPointCopy.ThisTimeLineID;
  readRecoveryCommandFile();
  expectedTLIs = readTimeLineHistory(recoveryTargetTLI);

  if (!list_member_int(expectedTLIs, (int) ControlFile->checkPointCopy.ThisTimeLineID))
    {
      ereport(FATAL,
              (errmsg("requested timeline %u is not a child of database system timeline %u",
                      recoveryTargetTLI,
                      ControlFile->checkPointCopy.ThisTimeLineID)));
    }

  checkPointLoc = ControlFile->checkPoint;
  record = ReadCheckpointRecord(checkPointLoc, 1);
  if (record != NULL)
    {
      ereport(LOG,
              (errmsg("checkpoint record is at %X/%X",
                      checkPointLoc.xlogid, checkPointLoc.xrecoff)));
    }
  else
    {
      checkPointLoc = ControlFile->prevCheckPoint;
      record = ReadCheckpointRecord(checkPointLoc, 2);
      if (record != NULL)
        {
          ereport(LOG,
                  (errmsg("using previous checkpoint record at %X/%X",
                          checkPointLoc.xlogid, checkPointLoc.xrecoff)));
          InRecovery = true;		/* force recovery even if SHUTDOWNED */
        }
      else
        ereport(ERROR,
                (errmsg("could not locate a valid checkpoint record")));
    }

  LastRec = checkPointLoc;
  RecPtr = checkPointLoc;
  memcpy(&checkPoint, XLogRecGetData(record), sizeof(CheckPoint));
  wasShutdown = (record->xl_info == XLOG_CHECKPOINT_SHUTDOWN);
  ereport(LOG,
          (errmsg("redo record is at %X/%X; undo record is at %X/%X; shutdown %s",
                  checkPoint.redo.xlogid, checkPoint.redo.xrecoff,
                  checkPoint.undo.xlogid, checkPoint.undo.xrecoff,
                  wasShutdown ? "TRUE" : "FALSE")));
  ereport(LOG,
          (errmsg("next transaction ID: %u/%u; next OID: %u",
                  checkPoint.nextXidEpoch, checkPoint.nextXid,
                  checkPoint.nextOid)));
  ereport(LOG,
          (errmsg("next MultiXactId: %u; next MultiXactOffset: %u",
                  checkPoint.nextMulti, checkPoint.nextMultiOffset)));

  ThisTimeLineID = checkPoint.ThisTimeLineID;
  RedoRecPtr = checkPoint.redo;
  /* XLogCtl->Insert.RedoRecPtr = checkPoint.redo; */
  if (XLByteLT(RecPtr, checkPoint.redo))
    ereport(PANIC,
            (errmsg("invalid redo in checkpoint record")));
  if (checkPoint.undo.xrecoff == 0)
    checkPoint.undo = RecPtr;

  XLogInitRelationCache();
  /*
   * Find the first record that logically follows the checkpoint --- it
   * might physically precede it, though.
   */
  if (XLByteLT(checkPoint.redo, RecPtr))
    {
      /* back up to find the record */
      record = ReadRecord(&(checkPoint.redo), PANIC);
    }
  else
    {
      /* just have to read next record after CheckPoint */
      record = ReadRecord(NULL, LOG);
    }


  if (record != NULL)
    {
      bool		recoveryContinue = true;
      bool		recoveryApply = true;
      /* ErrorContextCallback errcontext; */

      InRedo = true;
      ereport(LOG,
              (errmsg("redo starts at %X/%X",
                      ReadRecPtr.xlogid, ReadRecPtr.xrecoff)));
      /*
       * main redo apply loop
       */
      do
        {
#ifdef WAL_DEBUG
          if (XLOG_DEBUG)
            {
              StringInfoData buf;

              initStringInfo(&buf);
              appendStringInfo(&buf, "REDO @ %X/%X; LSN %X/%X: ",
                               ReadRecPtr.xlogid, ReadRecPtr.xrecoff,
                               EndRecPtr.xlogid, EndRecPtr.xrecoff);
              xlog_outrec(&buf, record);
              appendStringInfo(&buf, " - ");
              RmgrTable[record->xl_rmid].rm_desc(&buf,
                                                 record->xl_info,
                                                 XLogRecGetData(record));
              elog(LOG, "%s", buf.data);
              pfree(buf.data);
            }
#endif /* WAL_DEBUG */
          if (recoveryStopsHere(record, &recoveryApply))
            {
              needNewTimeLine = true;		/* see below */
              recoveryContinue = false;
              if (!recoveryApply)
                break;
            }
          if (record->xl_info & XLR_BKP_BLOCK_MASK)
            {
            }

          if (record->xl_rmid == RM_HEAP_ID)
            {
              uint8       info = record->xl_info & ~XLR_INFO_MASK;
              info &= XLOG_HEAP_OPMASK;
              if (info == XLOG_HEAP_INSERT)
                {
                  xl_heap_insert *xlrec = (xl_heap_insert *) XLogRecGetData(record);
                  LudiaRelInfoNode* lrinode = (LudiaRelInfoNode*)hash_search(ludiarelhash,
                                                                             &(xlrec->target.node),
                                                                             HASH_FIND,
                                                                             NULL);
                  if (lrinode)
                    update_index_for_recovery(&(xlrec->target.tid), estate, lrinode, TEXTOID);
                }
              if (info == XLOG_HEAP_UPDATE)
                {
                  xl_heap_update *xlrec = (xl_heap_update *) XLogRecGetData(record);
                  LudiaRelInfoNode* lrinode = (LudiaRelInfoNode*)hash_search(ludiarelhash,
                                                                             &(xlrec->target.node),
                                                                             HASH_FIND,
                                                                             NULL);
                  if (lrinode)
                    update_index_for_recovery(&(xlrec->newtid), estate, lrinode, TEXTOID);
                }
            }
          LastRec = ReadRecPtr;
          record = ReadRecord(NULL, LOG);
        } while (record != NULL && recoveryContinue);
    }

  if (InArchiveRecovery)
    {
      unlink(RECOVERY_COMMAND_DONE);
      if (rename(RECOVERY_COMMAND_FILE, RECOVERY_COMMAND_DONE) != 0)
        ereport(FATAL,
                (errcode_for_file_access(),
                 errmsg("could not rename file \"%s\" to \"%s\": %m",
                        RECOVERY_COMMAND_FILE, RECOVERY_COMMAND_DONE)));
      ereport(LOG, (errmsg("archive recovery complete")));
    }
  pfree(ControlFile);
#endif
  destroy_ludiarelhash(ludiarelhash);
  FreeExecutorState(estate);

  PG_RETURN_VOID();
}


HTAB *init_ludiarelhash(EState *estate)
{
  HTAB *ludiarelhash = NULL;
  Oid am_oid[3];
  ScanKeyData skey[3];
  SysScanDesc relscan;
  Relation relrel;
  int i;
  HASHCTL ctl;

  memset(&ctl, 0, sizeof(ctl));
  ctl.keysize = sizeof(RelFileNode);
  ctl.entrysize = sizeof(LudiaRelInfoNode);
  ctl.hash = tag_hash;
  ludiarelhash = hash_create("Ludia relhash", 512,
                              &ctl, HASH_ELEM | HASH_FUNCTION);

  am_oid[0] = GetSysCacheOid(AMNAME, PointerGetDatum("fulltext"), 0, 0, 0);
  am_oid[1] = GetSysCacheOid(AMNAME, PointerGetDatum("fulltextb"), 0, 0, 0);
  am_oid[2] = GetSysCacheOid(AMNAME, PointerGetDatum("fulltextu"), 0, 0, 0);
  elog(LOG, "pgsenna2: am_fulltext_oid: %d, am_fulltextb_oid: %d, am_fulltextu_oid: %d",
       am_oid[0], am_oid[1], am_oid[2]);

  for (i = 0; i < 3; i++)
    {
      ScanKeyInit(skey + i,
                  Anum_pg_class_relam,
                  BTEqualStrategyNumber, F_OIDEQ,
                  ObjectIdGetDatum(am_oid[i]));
    }

  relrel = heap_open(RelationRelationId, AccessShareLock);
  for (i = 0; i < 3; i++)
    {
      HeapTuple pg_class_tup;
      relscan = systable_beginscan(relrel, InvalidOid, false, SnapshotNow, 1, skey + i);
      while (HeapTupleIsValid(pg_class_tup = systable_getnext(relscan)))
        {
          Oid indexrelid = HeapTupleGetOid(pg_class_tup);
          Relation indrel = index_open(indexrelid, AccessShareLock);
          IndexInfo* indexInfo = BuildIndexInfo(indrel);
          if (indexInfo->ii_Expressions != NIL && indexInfo->ii_ExpressionsState == NIL)
            {
              indexInfo->ii_ExpressionsState = (List *)ExecPrepareExpr((Expr *)indexInfo->ii_Expressions,
                                                                       estate);
            }
          Relation hrel = heap_open(indrel->rd_index->indrelid, AccessShareLock);
          LudiaIndexInfoNode* lnode = palloc(sizeof(LudiaIndexInfoNode));
          lnode->indexrelid = indexrelid;
          lnode->indexInfo = indexInfo;
          bool found = false;
          LudiaRelInfoNode* val = (LudiaRelInfoNode*)hash_search(ludiarelhash,
                                                                 &(hrel->rd_node),
                                                                 HASH_ENTER,
                                                                 &found);
          if (found)
            {
              val->iilist = lcons(lnode, val->iilist);
            }
          else
            {
              Relation reln;
              val->node = hrel->rd_node;
              val->indrelid = indrel->rd_index->indrelid;
              reln= heap_open(val->indrelid, AccessShareLock);
              val->slot = MakeSingleTupleTableSlot(RelationGetDescr(reln));
              heap_close(reln, AccessShareLock);
              val->iilist = lcons(lnode, NIL);
            }
          elog(LOG, "pgsenna2: index: %s (relNode: %d), heap: %s (relNode: %d)",
               indrel->rd_rel->relname.data, indrel->rd_node.relNode,
               hrel->rd_rel->relname.data, hrel->rd_node.relNode);
          heap_close(hrel, AccessShareLock);
          index_close(indrel, AccessShareLock);
        }
      systable_endscan(relscan);
    }
  heap_close(relrel, AccessShareLock);
  return ludiarelhash;
}


void destroy_ludiarelhash(HTAB *ludiarelhash)
{
  HASH_SEQ_STATUS status;
  LudiaRelInfoNode *hentry = NULL;
  hash_seq_init(&status, ludiarelhash);
  while ((hentry = (LudiaRelInfoNode *) hash_seq_search(&status)) != NULL)
    {
      ExecDropSingleTupleTableSlot(hentry->slot);
    }
  hash_destroy(ludiarelhash);
}


void update_index_for_recovery(ItemPointer tid,
                               EState *estate,
                               LudiaRelInfoNode *lrinode,
                               Oid oid_type)
{
  Relation reln = heap_open(lrinode->indrelid, AccessShareLock);
  Buffer buffer;
  HeapTupleData htup;
  htup.t_self = *tid;
  /* bool isNull; */
  ExprContext *econtext = GetPerTupleExprContext(estate);
  econtext->ecxt_scantuple = lrinode->slot;
  if (heap_fetch(reln, SnapshotNow, &htup, &buffer, false, NULL) == false)
    {
      heap_close(reln, AccessShareLock);      
#ifdef WAL_DEBUG
      if (XLOG_DEBUG) elog(LOG, "pgsenna2: tuple failed time qual");
#endif /* WAL_DEBUG */
      return;
    }

#if 0
#ifdef WAL_DEBUG
  if (XLOG_DEBUG)
    {
      text *t;
      char *c;
      Datum iDatum = fastgetattr(&htup, 1, RelationGetDescr(reln), &isNull);
      t = DatumGetTextP(iDatum);
      c = text2cstr(t);
      elog(LOG, "pgsenna2: iDatum: %s", c);
      pfree(c);
      if (t != (text *) iDatum)
        {
          pfree(t);
        }
    }
#endif /* WAL_DEBUG */
#endif /* 0 */

  ListCell *iilist = list_head(lrinode->iilist);
  while (iilist != NULL)
    {
      LudiaIndexInfoNode *iinode = (LudiaIndexInfoNode *)lfirst(iilist);
      Relation indrel = index_open(iinode->indexrelid, ShareUpdateExclusiveLock);
      index_info *ii = index_info_open(indrel, 0, 0);
      index_close(indrel, ShareUpdateExclusiveLock);
      update_index_with_tuple(iinode->indexInfo, ii, estate, NULL, &htup, oid_type);
      iilist = lnext(iilist);
    }
  heap_close(reln, AccessShareLock);
  ReleaseBuffer(buffer);
}
#endif /* POSTGRES82 */
