/************************************************************
 *
 * COPYRIGHT (C) HITACHI,LTD. 2003 ALL RIGHTS RESERVED.
 * WRITTEN BY HITACHI SYSTEMS DEVELOPMENT LABORATORY,
 *            HITACHI CENTRAL RESEARCH LABORATORY.
 *
 * Created by M. Hiramatsu <hiramatu@sdl.hitachi.co.jp>
 *
 ************************************************************/

#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/time.h>
#include <linux/sched.h>
#include <asm/uaccess.h>
#include <linux/lkst_private.h>
#include "lkst_mod_watchdog_log.h"

static int	evhandler_id = LKST_EVHANDLER_ID_VOID;
static int	evhandler_id2 = LKST_EVHANDLER_ID_VOID;

static int	lkst_mod_init(void);
static void	lkst_mod_cleanup(void);

module_init(lkst_mod_init);
module_exit(lkst_mod_cleanup);

MODULE_AUTHOR("M.Hiramatsu <hiramatu@sdl.hitachi.co.jp>");
MODULE_DESCRIPTION("LKST module event record with heart beat.");
MODULE_LICENSE("GPL");

MODULE_PARM(evhandler_id, "i");
MODULE_PARM_DESC(evhandler_id, "id of event-hanlder to regist.");

static atomic_t sleepmax = ATOMIC_INIT(0);
static unsigned long jif_old = 0;

#define TASK_STACK_SZ (8192-sizeof(struct task_struct))

struct task_struct_ex {
	struct task_struct task;
	char _stack[TASK_STACK_SZ];
};

#define TASKBACK_MAX 256
static int task_backuped = -1;
static struct task_struct_ex task_backup_table[TASKBACK_MAX];

static void backup_tasks(void) 
{
	int i = 0;
	struct task_struct *p;
	for_each_task(p) {
		memcpy(&task_backup_table[i], p, 
		       sizeof(struct task_struct_ex));
		if ( (++i) == TASKBACK_MAX ) break;
	}
	task_backuped = i;
}

static void lkst_evhandler_watchdog(void *phookrec, int event_type,
				    lkst_arg_t arg1, lkst_arg_t arg2,
				    lkst_arg_t arg3, lkst_arg_t arg4)
{
	unsigned long max = (unsigned long)atomic_read(&sleepmax);
	if (max) {
		unsigned long diff = (jiffies - jif_old) / HZ;
		lkst_evhandlerprim_entry_log(event_type,
					     arg1, arg2,
					     arg3, arg4);
		if ( max <= diff ) {
			atomic_set(&sleepmax, 0);
			backup_tasks();
		}
	}
}

static void lkst_evhandler_watchpuppy(void *phookrec, int event_type,
				      lkst_arg_t arg1, lkst_arg_t arg2,
				      lkst_arg_t arg3, lkst_arg_t arg4)
{
	unsigned long max = (unsigned long)atomic_read(&sleepmax);
	if (max) {
		lkst_evhandlerprim_entry_log(event_type,
					     arg1, arg2,
					     arg3, arg4);
	}
}

static int lkst_evhandler_watchdog_ctrl(void *buf, size_t bufsize)
{
	int i = 0;
	if (buf != NULL) {
		i = simple_strtol((char *) buf, NULL, 0); /* set record process id */
		if ( i == 0 ) { /* clear */
			jif_old = jiffies;
		}else if ( i > 0 ) { /* start / set limit */
			jif_old = jiffies;
			atomic_set(&sleepmax, i);
			printk("lkst watchdog start\t");
			printk("sleepmax = %d\n", i);
		} else { /* stop */
			atomic_set(&sleepmax, 0);
			printk("lkst watchdog stop\t");
			printk("backuped tasks = %d\n", task_backuped);
		}
		return 0;
	} else 
		return -EINVAL;
}

static int lkst_evhandler_watchpuppy_ctrl(void *buf, size_t bufsize)
{
	int size = task_backuped*sizeof(struct task_struct_ex);
	struct taskparam *p = buf;
	if (buf != NULL && p->size >= size ) {
		if (task_backuped > 0)
			return copy_to_user(p->addr, task_backup_table,
					    size);
		else
			return -EBUSY;
	}else 
		return size;
}

static int lkst_mod_init()
{
	int	retval;

	/* Register an event handler for light logging. */
	retval = lkst_evhandler_register(evhandler_id,
					 DOGNAME,
					 &lkst_evhandler_watchdog,
					 &lkst_evhandler_watchdog_ctrl);
	if (retval < 0) {
		printk(KERN_ERR "cannot register event-handler function!\n");
		return retval;
	}
	evhandler_id = (lkst_evhandler_id) retval;

	retval = lkst_evhandler_register(evhandler_id+1,
					 PUPPYNAME,
					 &lkst_evhandler_watchpuppy,
					 &lkst_evhandler_watchpuppy_ctrl);
	if (retval < 0) {
		printk(KERN_ERR "cannot register event-handler function!\n");
		return retval;
	}
	evhandler_id2 = (lkst_evhandler_id) retval;

	printk("task_backup_table = 0x%p size=%d\n",
	       task_backup_table, sizeof(task_backup_table));
	return 0;
}

static void lkst_mod_cleanup()
{
	/* deregister the event handler. */
	if (evhandler_id != LKST_EVHANDLER_ID_VOID)
		lkst_evhandler_unregister((lkst_evhandler_id)evhandler_id);
	if (evhandler_id2 != LKST_EVHANDLER_ID_VOID)
		lkst_evhandler_unregister((lkst_evhandler_id)evhandler_id2);

	return;
}

