/*-*-c++-*-
 * $Id: cmainwin.h,v 1.2 2002/03/05 14:47:14 holzheu Exp $
 *
 * This file is part of qlcrash, a GUI for the dump-analysis tool lcrash.
 *
 * Copyright (C) 2001 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * Authors:
 * Michael Geselbracht (let@users.sourceforge.net)
 * Fritz Elfert (elfert@de.ibm.com)
 * Michael Holzheu (holzheu@de.ibm.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */
#ifndef CMAINWIN_H
#define CMAINWIN_H

#include <qwidget.h>
#include <qmap.h>

#include "crashtypes.h"
#include "cnconsole.h"

class CGraphManager;
class CGraphObject;
class QLcrashDoc;
class CListView;
class StructViewItem;
class CTrace;
class CTraceLowcore;
class CPrintExprDialog;
class CWalkExpressionDialog;

class QApplication;
class QWorkspace;
class QSplitter;
class QListView;
class QListViewItem;
class QPopupMenu;

/**Main window consisting of the console and the workspace.
  *@author Michael Geselbracht
  */
class CMainWin : public QWidget {
	Q_OBJECT

public: 
	CMainWin(QLcrashApp* app, const char *name=0);
	~CMainWin();
	
	// layout all objects in active window.
	void layoutActiveWindow();
	
	QFont consoleFont();
	void setConsoleFont(QFont);
	
	inline QWorkspace* workspace() {
		return oWorkspace;
	}
	
	// setup a frame for a QWorkspace child
	void setFrame(QWidget* w);
	
	virtual void setCursor(const QCursor& c);
	virtual void unsetCursor();
	
	// scroll viewport of active graph manager
	void scrollLeft();
	void scrollRight();
	void scrollUp();
	void scrollDown();
	
	// find a struct member in selected object
	void findItem();
	
	// display a graph print dialog
	void graphPrintDialog(const QString& def=QString::null, bool rewindCursor=false);
	
	// display a graph walk dialog
	void graphWalkDialog(const QString& type=QString::null, const QString& member=QString::null, const QString& addr=QString::null);
	
	// display a new trace window
	void graphTrace(const QString& = QString::null);
	
	inline CPrintExprDialog* printDumpDialog() {
		return oPrintExprDialog;
	}
	
	// initialize command completion
	void initCommandCompletion(CNConsole* con=0);
	
signals:
	void sigNewChildWin(QWidget*);

public slots:
	// display a new struct view
	void slotAddTree(CListView* root);
	
	// display/update a task list
	void slotTaskList(TaskListType*);
	
	// display/update a trace list
	void slotTraceList();
	
	// show a new console
	void slotNewConsole();
	
	// display a list with global symbols
	void slotSymtabList();
	
	// update global symbols
	void slotSymtabListUpdate();
	
	// display a debug feature window
	void
	slotNewS390dbf();
	
	// Restore the children's original sizes
	void restoreChildren();

protected:
	virtual void resizeEvent(QResizeEvent*);
	virtual bool eventFilter(QObject*, QEvent*);
	virtual void focusInEvent(QFocusEvent*);
	
	// return appropriate size for a new list view
	QSize calculateViewSize(QListView*);
	
	// Display a dynamic array
	void displayArray(QListViewItem*);
	
	// Display walk view
	void displayWalk(QListViewItem*);
	
	// Display dump view
	void displayDump(QListViewItem*);
	
	// Add a tree view. Used by svdoubleclick, displayarray and displaywalk
	void displayTree(QListView* tree, CGraphObject* parent, CGraphManager* man, const QString& caption, const QString& wirelabel);
	
	// Display a new graph manager
	void graphPrint(const QString& expr, int count=1);
	
protected slots:
	// symbol list item clicked
	void slotSLClicked(QListViewItem*);
	
	// symbol list item double clicked
	void slotSLDoubleClicked(QListViewItem*);
	
	// context menu for symbol list
	void slotSLRightClicked(QListViewItem*, const QPoint&, int);
	
	// struct view item double clicked
	void slotSVDoubleClicked(QListViewItem*);
	
	// copy value if current list item to the clipboard
	void slotSVClicked(int, QListViewItem*, const QPoint&, int);
	
	// context menu for struct view item
	void slotSVRightClicked(QListViewItem*, const QPoint&, int);
	
	// task list view item clicked
	void slotTLClicked(QListViewItem*);
	
	// context menu for task list item
	void slotTLRightClicked(QListViewItem*, const QPoint&, int);
	
	// task list view item double clicked
	void slotTLDoubleClicked(QListViewItem*);
	
	// a graph manager has been closed
	void slotGMClosed(CGraphManager*);
	
	// a task list view has been closed
	void slotTLClosed(CListView*);
	
	// send a string to a console
	void slotToConsole(CNConsole*, QString);
	
	// send a string to a console
	void slotErrToConsole(CNConsole*, QString);
	
	// a console has been closed
	void slotConsoleClosed(CNConsole*);
	
	// a list view has been closed
	void slotSVClosed(QWidget*);
	
	// set initial geometry
	void slotTimeout();
	
	// keep track of dump views
	void slotWindowActivated(QWidget*);
	
	// Display lowcore info
	void slotLowcore(CTrace* view, const QString& addr);

private:
	void initPopups();

private:
	QWorkspace* oWorkspace;
	QLcrashDoc* oDoc;
	CListView* oTaskList;
	QSplitter* oSplitter;
	
	// the main (first) console
	CNConsole* oConsole;
	
	// which graph manager instance contains a specific list view?
	QMap<QListView*, CGraphManager*> oList2Manager;
	
	// which graph object contains a specific list view?
	QMap<QListView*, CGraphObject*> oList2Object;
	
	QList<CNConsole> oConsoleList;
	
	QPopupMenu* oSVPopup;
	QPopupMenu* oTLPopup;
	QPopupMenu* oSLPopup;
	QWidget* oActiveWindow;
	CTrace* oTrace;
	CTraceLowcore* oTraceLowcore;
	CListView* oSymtabList;
	
	CPrintExprDialog* oPrintExprDialog;
	CWalkExpressionDialog* oWalkExprDialog;
};

#endif
