/*
 * Kernel header file for Linux crash dumps.
 *
 * Created by: Matt Robinson (yakker@sgi.com)
 *
 * Copyright 1999 Silicon Graphics, Inc. All rights reserved.
 *
 * This code is released under version 2 of the GNU GPL.
 */

/* This header file holds the architecture specific crash dump header */
#ifndef _ASM_DUMP_H
#define _ASM_DUMP_H

/* definitions */
#define DUMP_ASM_MAGIC_NUMBER     0xdeaddeadULL  /* magic number            */
#define DUMP_ASM_VERSION_NUMBER   0x3            /* version number          */

#ifdef __KERNEL__
#include <asm/pal.h>
#ifdef CONFIG_SMP
extern int (*dump_ipi_function_ptr)(struct pt_regs *);
extern void dump_send_ipi(void);
#else /* !CONFIG_SMP */
#define dump_send_ipi()
#endif

#else  /* !__KERNEL__ */
/* necessary header files */
#include <asm/ptrace.h>                          /* for pt_regs             */
#include <linux/threads.h>
#endif /* __KERNEL__ */

/* 
 * mkswap.c calls getpagesize() to get the system page size,
 * which is not  necessarily the same as the hardware page size.
 *
 * For ia64 the kernel PAGE_SIZE can be configured from 4KB ... 16KB.
 *
 * The physical memory is layed out out in the hardware/minimal pages.
 * This is the size we need to use for dumping physical pages.
 *
 * Note ths hardware/minimal page size being use in;
 *      arch/ia64/kernel/efi.c`efi_memmap_walk():
 *	    curr.end   = curr.start + (md->num_pages << 12);
 *
 * Since the system page size could change between the kernel we boot
 * on the the kernel that cause the core dume we may want to have something
 * more constant like the maximum system page size (See include/asm-ia64/page.h).
 */
#define DUMP_MIN_PAGE_SHIFT                 	12
#define DUMP_MIN_PAGE_SIZE                  	(1UL << DUMP_MIN_PAGE_SHIFT)
#define DUMP_MIN_PAGE_MASK                  	(~(DUMP_MIN_PAGE_SIZE - 1))
#define DUMP_MIN_PAGE_ALIGN(addr)           	(((addr) + DUMP_MIN_PAGE_SIZE - 1) & DUMP_MIN_PAGE_MASK)

#define DUMP_MAX_PAGE_SHIFT                 	16
#define DUMP_MAX_PAGE_SIZE                  	(1UL << DUMP_MAX_PAGE_SHIFT)
#define DUMP_MAX_PAGE_MASK                  	(~(DUMP_MAX_PAGE_SIZE - 1))
#define DUMP_MAX_PAGE_ALIGN(addr)           	(((addr) + DUMP_MAX_PAGE_SIZE - 1) & DUMP_MAX_PAGE_MASK)


#undef	DUMP_PAGE_SHIFT				/* Redefining Default for ia64  */
#undef	DUMP_PAGE_SIZE				/* "	"	"	"	*/
#undef	DUMP_PAGE_MASK				/* "    "       "       "       */
#undef	DUMP_PAGE_ALIGN				/* "    "       "       "       */
#undef	DUMP_HEADER_OFFSET			/* "    "       "       "       */

#define DUMP_HEADER_OFFSET              	DUMP_MAX_PAGE_SIZE

#define DUMP_EF_PAGE_SHIFT			DUMP_MIN_PAGE_SHIFT

#define DUMP_PAGE_SHIFT				DUMP_MIN_PAGE_SHIFT
#define DUMP_PAGE_SIZE				DUMP_MIN_PAGE_SIZE
#define DUMP_PAGE_MASK				DUMP_MIN_PAGE_MASK
#define DUMP_PAGE_ALIGN(addr)			DUMP_MIN_PAGE_ALIGN(addr)

/*
 * Structure: dump_header_asm_t
 *  Function: This is the header for architecture-specific stuff.  It
 *            follows right after the dump header.
 */
typedef struct _dump_header_asm_s {

        /* the dump magic number -- unique to verify dump is valid */
        uint64_t             dha_magic_number;

        /* the version number of this dump */
        uint32_t             dha_version;

        /* the size of this header (in case we can't read it) */
        uint32_t             dha_header_size;

        /* pointer to pt_regs */
	struct pt_regs      *dha_pt_regs;

	/* the dump registers */
	struct pt_regs       dha_regs;

        /* the rnat register saved after flushrs */
        uint64_t             dha_rnat;

	/* the pfs register saved after flushrs */
	uint64_t             dha_pfs;

	/* the bspstore register saved after flushrs */
	uint64_t             dha_bspstore;

	/* smp specific */
	uint32_t	     dha_smp_num_cpus;
	int		     dha_dumping_cpu;	
	struct pt_regs	     dha_smp_regs[NR_CPUS];
	void *		     dha_smp_current_task[NR_CPUS];
	void *		     dha_stack[NR_CPUS];

} dump_header_asm_t;

#if 0
/*
 * Arch Specific Functions should be declared here,
 * instead of in dump_base.c where they currently 
 * are defined.
 */
#ifdef __KERNEL__
extern void __dump_open(struct file *, uint64_t);
extern void __dump_cleanup(void);
extern void __dump_init(uint64_t);
extern int __dump_configure_header(dump_header_asm_t *, struct pt_regs *);
extern unsigned int  __dump_silence_system(unsigned int);
extern unsigned int  __dump_resume_system(unsigned int);
extern unsigned int  __dump_resume_system(unsigned int);
extern void * __dump_memcpy(void *, const void *, size_t);
#endif /* __KERNEL__ */
#endif

#endif /* _ASM_DUMP_H */
