/*
 * Kernel header file for Linux crash dumps.
 *
 * Created by: Matt Robinson (yakker@sgi.com)
 *
 * Copyright 1999 Silicon Graphics, Inc. All rights reserved.
 *
 * This code is released under version 2 of the GNU GPL.
 */

/* This header file holds the architecture specific crash dump header */
#ifndef _ASM_DUMP_H
#define _ASM_DUMP_H

/* necessary header files */
#include <asm/ptrace.h>                          /* for pt_regs             */
#include <linux/threads.h>

/* definitions */
#define DUMP_ASM_MAGIC_NUMBER     0xdeaddeadULL  /* magic number            */
#define DUMP_ASM_VERSION_NUMBER   0x2            /* version number          */


/*
 * Structure: dump_header_asm_t
 *  Function: This is the header for architecture-specific stuff.  It
 *            follows right after the dump header.
 */
typedef struct _dump_header_asm_s {

        /* the dump magic number -- unique to verify dump is valid */
        uint64_t             dha_magic_number;

        /* the version number of this dump */
        uint32_t             dha_version;

        /* the size of this header (in case we can't read it) */
        uint32_t             dha_header_size;

	/* the esp for i386 systems */
	uint32_t             dha_esp;

	/* the eip for i386 systems */
	uint32_t             dha_eip;

	/* the dump registers */
	struct pt_regs       dha_regs;

	/* smp specific */
	uint32_t	     dha_smp_num_cpus;
	int		     dha_dumping_cpu;	
	struct pt_regs	     dha_smp_regs[NR_CPUS];
	void *		     dha_smp_current_task[NR_CPUS];
	void *		     dha_stack[NR_CPUS];
} dump_header_asm_t;

#ifdef __KERNEL__
static inline void get_current_regs(struct pt_regs *regs)
{
	__asm__ __volatile__("movl %%ebx,%0" : "=m"(regs->ebx));
	__asm__ __volatile__("movl %%ecx,%0" : "=m"(regs->ecx));
	__asm__ __volatile__("movl %%edx,%0" : "=m"(regs->edx));
	__asm__ __volatile__("movl %%esi,%0" : "=m"(regs->esi));
	__asm__ __volatile__("movl %%edi,%0" : "=m"(regs->edi));
	__asm__ __volatile__("movl %%ebp,%0" : "=m"(regs->ebp));
	__asm__ __volatile__("movl %%eax,%0" : "=m"(regs->eax));
	__asm__ __volatile__("movl %%esp,%0" : "=m"(regs->esp));
	__asm__ __volatile__("movw %%ss, %%ax;" :"=a"(regs->xss));
	__asm__ __volatile__("movw %%cs, %%ax;" :"=a"(regs->xcs));
	__asm__ __volatile__("movw %%ds, %%ax;" :"=a"(regs->xds));
	__asm__ __volatile__("movw %%es, %%ax;" :"=a"(regs->xes));
	__asm__ __volatile__("pushfl; popl %0" :"=m"(regs->eflags));
	regs->eip = (unsigned long)current_text_addr();
	
}

extern volatile int dump_in_progress;
extern unsigned long irq_affinity[];
extern dump_header_asm_t dump_header_asm;

#ifdef CONFIG_SMP
extern int (*dump_ipi_function_ptr)(struct pt_regs *);
extern void dump_send_ipi(void);
#else
#define dump_send_ipi()
#endif
#endif /* __KERNEL__ */

#endif /* _ASM_DUMP_H */
