/*
 * include/kl_error.h
 *
 * Copyright 1999 Silicon Graphics, Inc. All rights reserved.
 */

/**
 ** This header file contains basic definitions and declarations
 ** for the KLIB error handling facility.
 **
 **/

#ifndef __KL_ERROR_H
#define __KL_ERROR_H

/* Error Classes
 */
#define KLEC_APP        0
#define KLEC_KLIB       1
#define KLEC_MEM	2
#define KLEC_SYM	3
#define KLEC_KERN	4

#define KLEC_CLASS_MASK 0x00000000ff000000
#define KLEC_CLASS_SHIFT 24
#define KLEC_ECODE_MASK 0x0000000000ffffff
#define KLEC_TYPE_MASK  0xffffffff00000000
#define KLEC_TYPE_SHIFT 32
#define KLEC_CLASS(e) ((e & KLEC_CLASS_MASK) >> KLEC_CLASS_SHIFT)
#define KLEC_ECODE(e) (e & KLEC_ECODE_MASK)
#define KLEC_TYPE(e) ((e & KLEC_TYPE_MASK) >> KLEC_TYPE_SHIFT)

extern uint64_t klib_error;
void kl_reset_error(void);
void kl_print_error(void);

/** 
 ** Some macros for accessing data in klib_error 
 **/
#define KLIB_ERROR		klib_error
#define KL_ERROR 		klib_error
#define KL_ERRORVAL 		klib_errorval
#define KL_ERRORFP 		stderr

/* Error codes
 *
 * There are basically two types of error codes -- with each type
 * residing in a single word in a two word error code value. The lower
 * 32-bits contains an error class and code that represents exactly 
 * WHAT error occurred (e.g., non-numeric text in a numeric value 
 * entered by a user, bad virtual address, etc.). 
 * 
 * The upper 32-bits represents what type of data was being referenced 
 * when the error occurred (e.g., bad proc struct). Having two tiers of 
 * error codes makes it easier to generate useful and specific error 
 * messages. Note that is possible to have situations where one or the 
 * other type of error codes is not set. This is OK as long as at least 
 * one type s set.
 */

/** General klib error codes
 **/
#define KLE_KLIB (KLEC_KLIB << KLEC_CLASS_SHIFT)
#define KLE_NO_MEMORY				(KLE_KLIB|1)
#define KLE_OPEN_ERROR				(KLE_KLIB|2)
#define KLE_ZERO_BLOCK 				(KLE_KLIB|3)  
#define KLE_INVALID_VALUE 			(KLE_KLIB|4)  
#define KLE_NULL_BUFF 				(KLE_KLIB|5)  
#define KLE_ZERO_SIZE 				(KLE_KLIB|6)  
#define KLE_ACTIVE 				(KLE_KLIB|7)  


#define KLE_MISC_ERROR 				(KLE_KLIB|97)
#define KLE_NOT_SUPPORTED 			(KLE_KLIB|98)  
#define KLE_UNKNOWN_ERROR 			(KLE_KLIB|99)  

/** memory error codes
 **/
#define KLE_MEM (KLEC_MEM << KLEC_CLASS_SHIFT)
#define KLE_BAD_MAP_FILE			(KLE_MEM|1)
#define KLE_BAD_DUMP	  			(KLE_MEM|2)
#define KLE_BAD_DUMPTYPE			(KLE_MEM|3)
#define KLE_INVALID_LSEEK 			(KLE_MEM|4) 
#define KLE_INVALID_READ 			(KLE_MEM|5) 
#define KLE_BAD_MEMINFO 			(KLE_MEM|6) 
#define KLE_INVALID_PADDR 			(KLE_MEM|7)  
#define KLE_INVALID_VADDR 			(KLE_MEM|8)  
#define KLE_INVALID_VADDR_ALIGN 		(KLE_MEM|9)  
#define KLE_INVALID_MAPPING 		        (KLE_MEM|10)  
#define KLE_CMP_ERROR 		        	(KLE_MEM|11)  
#define KLE_INVALID_DUMP_MAGIC 		        (KLE_MEM|12)  
#define KLE_KERNEL_MAGIC_MISMATCH               (KLE_MEM|13)
#define KLE_NO_END_SYMBOL                       (KLE_MEM|14)
#define KLE_INVALID_DUMP_HEADER			(KLE_MEM|15)
#define KLE_DUMP_INDEX_CREATION			(KLE_MEM|16)
#define KLE_DUMP_HEADER_ONLY			(KLE_MEM|17)

/** symbol error codes
 **/
#define KLE_SYM (KLEC_SYM << KLEC_CLASS_SHIFT)
#define KLE_NO_SYMTAB                     	(KLE_SYM|1)
#define KLE_NO_SYMBOLS                     	(KLE_SYM|2)
#define KLE_INVALID_TYPE                        (KLE_SYM|3)
#define KLE_NO_MODULE_LIST                      (KLE_SYM|4)

/** kernel data error codes
 **/
#define KLE_KERN (KLEC_KERN << KLEC_CLASS_SHIFT)
#define KLE_INVALID_KERNELSTACK 		(KLE_KERN|1)  
#define KLE_INVALID_STRUCT_SIZE 		(KLE_KERN|2)  
#define KLE_BEFORE_RAM_OFFSET	 		(KLE_KERN|3)  
#define KLE_AFTER_MAXPFN 			(KLE_KERN|4)  
#define KLE_AFTER_PHYSMEM  			(KLE_KERN|5)  
#define KLE_AFTER_MAXMEM 			(KLE_KERN|6)  
#define KLE_PHYSMEM_NOT_INSTALLED 		(KLE_KERN|7)  
#define KLE_NO_DEFTASK	 			(KLE_KERN|8)  
#define KLE_PID_NOT_FOUND 			(KLE_KERN|9)  
#define KLE_DEFTASK_NOT_ON_CPU 			(KLE_KERN|10)  
#define KLE_NO_CURCPU 				(KLE_KERN|11)  
#define KLE_NO_CPU 				(KLE_KERN|12)  
#define KLE_SIG_ERROR 				(KLE_KERN|13)  

/** Error codes that indicate what type of data was bad. These are
 ** placed in the upper 32-bits of klib_error.
 **/
#define KLE_BAD_TASK_STRUCT    	(((uint64_t)1)<<32)
#define KLE_BAD_SYMNAME         (((uint64_t)2)<<32)
#define KLE_BAD_SYMADDR         (((uint64_t)3)<<32)
#define KLE_BAD_FUNCADDR        (((uint64_t)4)<<32)
#define KLE_BAD_STRUCT          (((uint64_t)5)<<32)
#define KLE_BAD_FIELD           (((uint64_t)6)<<32)
#define KLE_BAD_PC              (((uint64_t)7)<<32)
#define KLE_BAD_RA              (((uint64_t)8)<<32)
#define KLE_BAD_SP              (((uint64_t)9)<<32)
#define KLE_BAD_EP              (((uint64_t)10)<<32)
#define KLE_BAD_SADDR           (((uint64_t)11)<<32)
#define KLE_BAD_KERNELSTACK     (((uint64_t)12)<<32)
#define KLE_BAD_LINENO          (((uint64_t)13)<<32)
#define KLE_MAP_FILE          	(((uint64_t)14)<<32)
#define KLE_DUMP          	(((uint64_t)15)<<32)
#define KLE_BAD_STRING          (((uint64_t)16)<<32)

#endif /* __KL_ERROR_H */
