/*
 djprobe -- Direct jmp-insertion probe
 Copyright (c) 2005 Hitachi,Ltd.,
 Created by Masami Hiramatsu<hiramatu@sdl.hitachi.co.jp>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */
#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <asm/system.h>
#include "djprobe.h"

#define _DEBUG_(a) a

/* jmp code manipulators */
struct __arch_jmp_op {
	char op;
	long raddr;
} __attribute__((packed));
/* insert jmp code */
static inline void __set_jmp_op(void *from, void *to)
{
	struct __arch_jmp_op *jop = (struct __arch_jmp_op *)from;

	jop->raddr=(long)(to) - ((long)(from) + 5);
	smp_mb();
	jop->op = RELATIVEJUMP_INSTRUCTION;
}
/* switch back to the kprobes */
static inline void __set_breakpoint_op(void *dest, void *orig)
{
	struct __arch_jmp_op *jop = (struct __arch_jmp_op *)dest, 
		*jop2 = (struct __arch_jmp_op *)orig;

	jop->op = BREAKPOINT_INSTRUCTION;
	smp_mb();
	jop->raddr = jop2->raddr;
}
#include <asm/segment.h>
/*
 * djprobe call back function: called from stub.
 */
void asmlinkage djprobe_callback(struct djprobe_instance * djpi,
				 struct pt_regs * regs)
{
	/*TODO: use list*/
	if (djpi->djp && djpi->djp->handler)
		djpi->djp->handler(djpi->djp, regs);
}

/*
 * Copy post processing instructions
 * Target instructions MUST be relocatable.
 */
static int arch_prepare_djprobe_instance(struct djprobe_instance *djpi,
					 unsigned long size)
{
	kprobe_opcode_t *stub;
	stub = djpi->stub.insn;

	/* copy arch-dep-instance from template */
	_DEBUG_(printk("djpi %p :copy from %p, to %p : size %ld\n", 
		       djpi, (void*)&arch_tmpl_stub_entry,(void*)stub, ARCH_STUB_SIZE));
	memcpy((void*)stub, (void*)&arch_tmpl_stub_entry, ARCH_STUB_SIZE);

	/* set probe information */
	*((long*)(stub + ARCH_STUB_VAL_IDX)) = (long)djpi;
	/* set probe function */
	*((long*)(stub + ARCH_STUB_CALL_IDX)) = (long)djprobe_callback;

	/* copy instructions into the middle of axporbe instance */
	memcpy((void*)(stub + ARCH_STUB_INST_IDX),
	       (void*)djpi->kp.addr, size);
	djpi->stub.size = size;

	/* set returning jmp instruction at the tail of axporbe instance*/
	__set_jmp_op(stub + ARCH_STUB_END_IDX, 
		     (void*)((long)djpi->kp.addr + size));

	return 0;
}

/*
 * Insert "jmp" instruction into the probing point.
 */
static int arch_install_djprobe_instance(struct djprobe_instance *djpi)
{
	kprobe_opcode_t *stub;
	stub = djpi->stub.insn;
	__set_jmp_op((void*)djpi->kp.addr, (void*)stub);
	return 0;
}
/*
 * Write back original instructions & kprobes
 */
static int arch_uninstall_djprobe_instance(struct djprobe_instance *djpi)
{
	kprobe_opcode_t *stub;
	stub = &djpi->stub.insn[ARCH_STUB_INST_IDX];
	__set_breakpoint_op((void*)djpi->kp.addr, (void*)stub);
	return 0;
}

/*
 * DJProbe has 3 phases. 
 * The 1st phase is the inserting phase. In this phase, a probe is driven by 
 * the kprobe. The kprobe's pre_handler checks safety of the cpu on which it
 * works. If it finished checking all cpus, it goes to the 2nd phase. If not, 
 * it changes execution point to the head of stub code that the probe has.
 * The 2nd phase is the djprobe working phase. In this phase, the probe is 
 * driven by the djprobe. The kernel execution path is changed to jump into 
 * the stub code directly.
 * The 3rd phase is the removing phase. In this phase, the probe is driven by
 * the kprobe again. The kprobe's pre_handler checks safety of the cpu on 
 * which it works. If it finished checking all cpus, it removes the kprobe 
 * from the code. If not, it resumes execution.
 */

/*
 * The djprobe do not refer it's list when probe function called.
 * This list is operated on registering and unregistering djprobe.
 * Thus, It is not required processing speed. I decided using a 'list'.
 */
static DEFINE_SPINLOCK(djprobe_lock);
static LIST_HEAD(djprobe_list);
static int nr_instances = 0;

static void work_free_djprobe_instances(void *data)
{
	struct list_head *pos;
	struct djprobe_instance *djpi;
	unsigned long flags;

	spin_lock_irqsave(&djprobe_lock, flags);
	list_for_each(pos, &djprobe_list) {
		djpi = container_of(pos, struct djprobe_instance, list);
		if (DJPI_EMPTY(djpi) && djpi->state == DJST_RELEASING) {
			pos = pos->prev; /* because *pos will be removed from list*/
			list_del(&djpi->list);
			nr_instances -- ;
			/* fixup dummy instruction */
			djpi->kp.ainsn.insn[0] = djpi->stub.insn[ARCH_STUB_INST_IDX];
			unregister_kprobe(&(djpi->kp));
			kfree(djpi);
		}
	}
	spin_unlock_irqrestore(&djprobe_lock, flags);
}

/* defered free worker */
static DECLARE_WORK(djprobe_release_work, 
		    work_free_djprobe_instances, NULL);

/*
 * safety check handler
 */
static int bypass_pre_handler(struct kprobe * kp,
			      struct pt_regs * regs)
{
	struct djprobe_instance *djpi = 
		container_of(kp,struct djprobe_instance, kp);
	kprobe_opcode_t *stub = djpi->stub.insn;
	int cpu = smp_processor_id();

	cpu_set(cpu, djpi->checked_cpus); /* check this cpu */

	if (cpus_weight(djpi->checked_cpus) == num_online_cpus()) {
		_DEBUG_(printk("all cpus are checked %d\n",DJPI_EMPTY(djpi)));
		cpus_clear(djpi->checked_cpus);
		/* if all cpus were checked, (un)install djprobe.*/
		if (DJPI_EMPTY(djpi)) {
			djpi->state = DJST_RELEASING;
			/* kick the defered releasing process */
			schedule_work(&djprobe_release_work);
		} else {
			/* write djprobe over kprobe */
			arch_install_djprobe_instance(djpi);
			djpi->state = DJST_ENABLE;
		}
	}

	if (DJPI_EMPTY(djpi)) {
		kp->ainsn.insn[0] = djpi->stub.insn[ARCH_STUB_INST_IDX];
		return 0;
	} else {
		regs->eip = (unsigned long)stub;
		regs->eflags |= TF_MASK;
		regs->eflags &= ~IF_MASK;
		/*
		 * dummy return code :
		 * This code is to avoid to fix eip value by 
		 * resume_execute() of kprobes
		 */
		kp->ainsn.insn[0] = RETURN_INSTRUCTION;
		return 1; /* already prepared */
	}
}

static int attach_djprobe(struct djprobe * djp)
{
	struct djprobe_instance *djpi;
	int ret = 0;
	spin_lock(&djprobe_lock);

	list_for_each_entry(djpi, &djprobe_list, list) {
		if (djpi->kp.addr == djp->addr) {
			if (!DJPI_EMPTY(djpi)) {
				ret = -EBUSY; /* already used ... */
				goto out;
			}
			if (djpi->state == DJST_RELEASING)
				djpi->state = DJST_DISABLE;/* recycle */
			djp->inst = djpi;
			djpi->djp = djp; /*TODO: use list*/
			goto out;
		}
	}
	
	/* could not find */
	djpi = kmalloc(sizeof(struct djprobe_instance),GFP_KERNEL);
	if (djpi == NULL) {
		ret = -ENOMEM; /* memory allocation error */
		goto out;
	}

	/* initialize */
	memset(djpi, 0, sizeof(struct djprobe_instance));
	INIT_LIST_HEAD(&djpi->list);
	djpi->state = DJST_DISABLE;
	djpi->kp.addr = djp->addr;
	cpus_clear(djpi->checked_cpus);
	arch_prepare_djprobe_instance(djpi, djp->size); /*TODO : remove size*/
	nr_instances ++ ;
	list_add(&djpi->list, &djprobe_list);

	djp->inst = djpi;
	djpi->djp = djp; /*TODO: use list*/
	/* first arming -- must register */
	djpi->kp.pre_handler = bypass_pre_handler;
	ret = register_kprobe(&(djpi->kp));
	if (ret < 0) { /* failed to attach */
		djp->inst = NULL;
		list_del(&djpi->list);
		nr_instances --;
		kfree(djpi);
	}
out:
	spin_unlock(&djprobe_lock);
	return ret;
}

static void detach_djprobe(struct djprobe * djp)
{
	struct djprobe_instance *djpi = djp->inst;	

	spin_lock(&djprobe_lock);
	djp->inst = NULL;
	djpi->djp = NULL; /*TODO: use list*/
	if (DJPI_EMPTY(djpi)) {
		/* defered releasing */
		arch_uninstall_djprobe_instance(djpi);
		/* bottom half will be processed in kprobes */
	}
	spin_unlock(&djprobe_lock);
}

int register_djprobe(struct djprobe *djp)
{
	if (djp == NULL || djp->addr == NULL || 
	    djp->size > ARCH_STUB_INSN_MAX || 
	    djp->size < ARCH_STUB_INSN_MIN || 
	    djp->inst != NULL)
		return -EINVAL;
	
	return attach_djprobe(djp);
}

void unregister_djprobe(struct djprobe *djp)
{
	if (djp == NULL || djp->inst == NULL) {
		return ;
	}
	detach_djprobe(djp);
}

EXPORT_SYMBOL_GPL(register_djprobe);
EXPORT_SYMBOL_GPL(unregister_djprobe);

/*
 * Initiation/Termination Functions.
 */ 

static int init_djprobe(void) 
{
	/* initialize the list for instances */
	nr_instances = 0;
	return 0;
}

static void exit_djprobe(void)
{
	if ( nr_instances )
		BUG();/*TODO*/
}

module_init(init_djprobe);
module_exit(exit_djprobe);
MODULE_AUTHOR("M.Hiramatsu <hiramatu@sdl.hitachi.co.jp>");
MODULE_LICENSE("GPL");


