/*
 LKST log analyzer

 Copyright (C) HITACHI,LTD. 2004-2005
 WRITTEN BY HITACHI SYSTEMS DEVELOPMENT LABORATORY,
 Created by M.Hiramatsu <hiramatu@sdl.hitachi.co.jp>
  
 The development of this program is partly supported by IPA
 (Information-Technology Promotion Agency, Japan).
  
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <logfile.h>
#include <lkstla.h>

#define VERSION "1.0.1"

#define ANALYZER(anlz) \
	extern struct gate_analyzer anlz##_analyzer;
#include "analyzers.list"
#undef  ANALYZER

extern struct analysis_formatter log_formatter;
extern struct analysis_formatter stat_formatter;
extern struct analysis_formatter dist_formatter;

struct gate_analyzer *analyzer_list[] = {
#define ANALYZER(anlz) \
	&anlz##_analyzer,
#include "analyzers.list"
#undef  ANALYZER
	NULL,
};
struct analysis_formatter *formatter_list[] = {
	&log_formatter,
	&stat_formatter,
	&dist_formatter,
	NULL,
};
#define FMT_DEFAULT 0

#define for_each_analyzer(ga, i) \
	for (i=0, ga = analyzer_list[i]; ga!=NULL; ga = analyzer_list[++i])

#define for_each_formatter(fmt, i) \
	for (i=0, fmt = formatter_list[i]; fmt!=NULL; fmt = formatter_list[++i])

#define IE_ASSERT(v) if (!(v)) { \
	fprintf(stderr,"%s:%d:Internal assertion error\n",__FILE__,__LINE__); \
	exit(-1);}

static void option_print(struct command_option *opt, const char* suffix)
{
	printf("       %s : %s %s\n",opt->format,opt->description,suffix);
}

static char * option_optcat(struct command_option *opt, char * dest, int len)
{
	return strncat(dest, opt->opt, (len-strlen(dest))-1);
}

static int opt_handler_help(int c, char *opt);

static struct command_option help_option = {
	.opt = "h",
	.format = "-h",
	.description = "show this message.",
	.handler = opt_handler_help,
};
extern struct command_option key_option;
extern struct command_option pid_option;
extern struct command_option time_option;

struct command_option *global_options[] = {
	&help_option,
	&key_option,
	&pid_option,
	&time_option,
	NULL,
};

void usage(char * message);

static int opt_handler_help(int c, char *opt)
{
	usage("");
	return 0;
}

#define MAXOPTS GA_MAXOPTS+4+3
#define OPTSLEN (MAXOPTS*2)

struct optlist {
	int nr_opts;
	char stropts[OPTSLEN];
	struct command_option *opts[MAXOPTS];
};

int optlist_addopt(struct optlist *optl, struct command_option *opt)
{
	if (optl->nr_opts < MAXOPTS &&
	    option_optcat(opt, optl->stropts, OPTSLEN)!=NULL) {
		optl->opts[optl->nr_opts++] = opt;
		return optl->nr_opts;
	}
	return -1;
}

int optlist_handle(struct optlist *optl, int c, char *opt) 
{
	int i;
	for (i=0;i<optl->nr_opts;i++) {
		if (optl->opts[i]->opt[0] == c) {
			return optl->opts[i]->handler(c,optarg);
		}
	}
	return -1;
}

#ifdef DEBUG
void verify_analyzer(struct gate_analyzer *ga)
{
	IE_ASSERT(ga);
	IE_ASSERT((ga->name)&&(ga->description)&&(ga->title)&&(ga->format_data)
		  &&(ga->delta_title));
	fprintf(stderr,"verifying %s ...",ga->name);
	IE_ASSERT(ga->init);
	IE_ASSERT(ga->max > ga->min && ga->ranks > 0);
	IE_ASSERT((ga->get_type)&&(ga->get_alias)&&(ga->info_key));
	IE_ASSERT((ga->get_pid)&&(ga->get_time)&&(ga->get_delta));
	if (ga->analyze_logs == analyze_sessions)
		IE_ASSERT(ga->session_key);
	if (ga->nr_options)
		IE_ASSERT(ga->options);
	fprintf(stderr,"OK\n");
}
#endif

int main (int argc, char * argv[])
{
	int c,ret=0;
	int i;
	struct optlist optl = {0,{0},{NULL,}};
	struct lkstloglist loglist={0,{0,}};
	struct gate_analyzer *ga;
	struct analysis_formatter *fmt, *chosen_fmt;

#ifdef DEBUG
	for_each_analyzer(ga,i) {
		verify_analyzer(ga);
	}
#endif
	if (argc<2) usage("");
	
	/*decide analyzer*/
	for_each_analyzer(ga,i) {
		if (strcmp(argv[1], ga->name) == 0) break;
	}
	if (ga == NULL) usage("Error: choose analyzer");
	argc--;	argv++;

	/*make getopt format string*/
	for_each_formatter(fmt,i) {
		fmt->active = 0;
		IE_ASSERT(optlist_addopt(&optl, &fmt->opt)>0);
	}
	for (i=0; i<ga->nr_options; i++) {
		IE_ASSERT(optlist_addopt(&optl, ga->options[i])>0);
	}
	for(i=0;global_options[i]!=NULL;i++) {
		IE_ASSERT(optlist_addopt(&optl, global_options[i])>0);
	}

	/*parse options*/
	while ((c=getopt(argc, argv, optl.stropts)) != -1) { 
		ret = optlist_handle(&optl,c,optarg);
		if (ret<0) {
			fprintf(stderr,"Error: faild to handle an option (-%c)",c);
			usage("");
		}
	}
	
	/*select a formatter*/
	chosen_fmt = NULL;
	for_each_formatter(fmt,i) {
		if (fmt->active) {
			if (chosen_fmt == NULL)
				chosen_fmt = fmt;
			else
				usage("Error: choose one formatter");
		}
	}
	if (chosen_fmt == NULL) chosen_fmt = formatter_list[FMT_DEFAULT];
	
	/*open logfiles*/
	c = optind;
	while(c < argc) {
		if (add_loglist(&loglist, argv[c]) < 0) {
			cleanup_loglist(&loglist);
			usage("Error: failed to open logfile");
		}
		c++;
	}
	if (loglist.nr_logfile == 0) usage("Error: please specify logfile at lease one");

	/*init analyzer*/
	if (ga->init() < 0) usage("Error: failed to initialize analyzer");
	
	/*analyzing*/
	ret = chosen_fmt->pre_analyze(ga);
	if (ret < 0) goto end_of_main;
	ret = ga->analyze_logs(ga, &loglist, chosen_fmt);
	if (ret < 0) goto end_of_main;
	chosen_fmt->post_analyze(ga);
	ret = 0;
 end_of_main:
	cleanup_loglist(&loglist);
	return ret;
}

void usage(char * message)
{
	int i,j;
	struct gate_analyzer *ga;
	struct analysis_formatter *fmt;
	
	printf("%s\n",message);
	printf("lkstla -- LKST log analizer Version %s\n", VERSION);
	printf("Copyright (C) 2004-2005, HITACHI, LTD.,\n");
	printf("Usage: lkstla <analyzer> [format option] [option] logfile [logfile] ... \n");
	printf("Or   : lkstla -h\n");
	printf("<analyzer>\n");
	for_each_analyzer(ga,i) {
		printf("        %-9s : %s\n",ga->name,ga->description);
	}
	printf("<format options>\n");
	for_each_formatter(fmt,i) {
		option_print(&fmt->opt,(i==FMT_DEFAULT)?"(Default Format)":"");
	}
	printf("<options>\n");
	for(i=0;global_options[i]!=NULL;i++) {
		option_print(global_options[i],"");
	}
	printf("<extra options>\n");
	for_each_analyzer(ga,i) {
		if (ga->nr_options == 0) continue;
		printf("  <options for %s>\n",ga->name);
		for (j=0; j < ga->nr_options; j++) {
			option_print(ga->options[j],"");
		}
	}
	exit(1);
}
