/*
 LKST log analyzer base functions

 Copyright (C) HITACHI,LTD. 2004-2005
 WRITTEN BY HITACHI SYSTEMS DEVELOPMENT LABORATORY,
 Created by M.Hiramatsu <hiramatu@sdl.hitachi.co.jp>
  
 The development of this program is partly supported by IPA
 (Information-Technology Promotion Agency, Japan).
                                                                                                                             
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <logfile.h>
#include <lkstla.h>
#include <timespec.h>

#define LOG_HASH_BITS 6
static generic_slots_t *gsl_session = NULL;

#define MAXKEYS 255

static struct {
	int nr_keys;
	unsigned int key[MAXKEYS];
} keyparam = {0, {0,}};

static int opt_handler_key(int c, char *opt)
{
	unsigned int key,i;
	char *ptr = opt-1;
	i = keyparam.nr_keys;
	do {
		opt = ptr+1;
		key = strtoul(opt,&ptr,0);
		if ( ptr == opt ) break;
		keyparam.key[i++] = key;
	} while (*ptr==',');
	keyparam.nr_keys = i;
	return 0;
}

static int key_filter(int key) 
{
	if (keyparam.nr_keys) {
		int i;
		for (i=0;i<keyparam.nr_keys;i++) {
			if (key == keyparam.key[i]) break;
		}
		if (i == keyparam.nr_keys) return 0;
	}
	return 1;
}

#define MAXPIDS 255

static struct {
	int nr_pids;
	int pid[MAXPIDS];
} pidparam = {0, {0,}};

static int opt_handler_pid(int c, char *opt)
{
	int pid,i;
	char *ptr = opt-1;
	i = pidparam.nr_pids;
	do {
		opt = ptr+1;
		pid = strtol(opt,&ptr,10);
		if ( ptr == opt ) break;
		pidparam.pid[i++] = pid;
	} while (*ptr==',');
	pidparam.nr_pids = i;
	return 0;
}

static int pid_filter(int pid) 
{
	if (pidparam.nr_pids) {
		int i;
		for (i=0;i<pidparam.nr_pids;i++) {
			if (pid == pidparam.pid[i]) break;
		}
		if (i == pidparam.nr_pids) return 0;
	}
	return 1;
}

#define MAX_TIMES 255

static int nr_times = 0;
static struct {
	double start, end;
	int rev;
} timeparams[MAX_TIMES] ={{0,0}};

#include <ctype.h>

static int opt_handler_time(int c, char *opt)
{
	char *ptr = opt-1;
	do {
		opt = ptr + 1;
		if (*opt == '!') {
			timeparams[nr_times].rev = 1;
			opt++;
		}else
			timeparams[nr_times].rev = 0;
		timeparams[nr_times].start = strtod(opt,&ptr);
		if (ptr == opt) return -1;
		opt = ptr;
		if (*opt!='\0') {
			if (*opt =='+' || *opt ==':') {
				if (*opt == '+') {
					timeparams[nr_times].end =
						timeparams[nr_times].start;
				}else {
					timeparams[nr_times].end = 0;
				}
				opt++;
				timeparams[nr_times].end += strtod(opt,&ptr);
				if (ptr == opt) return -1;
			}
		}
		nr_times++;
	} while(*ptr == ',');
	return 0;
}

static int time_filter(double t)
{
	int i;
	if (nr_times == 0)
		return 1;
	for(i=0;i<nr_times;i++) {
		if (timeparams[i].rev) {
			if ((t < timeparams[i].start) ||
			    (timeparams[i].end != 0 && t > timeparams[i].end))
				return 1;
		} else {
			if ((t >= timeparams[i].start) &&
			    (timeparams[i].end == 0 || t <= timeparams[i].end))
				return 1;
		}
	}
	return 0;
}

struct command_option key_option = {
	.opt = "k:",
	.format = "-k<key[,key[,...]]>",
	.description = "specify infokey filter.",
	.handler = opt_handler_key,
};

struct command_option pid_option = {
	.opt = "p:",
	.format = "-p<pid[,pid[,...]]>",
	.description = "specify PID filter.",
	.handler = opt_handler_pid,
};

struct command_option time_option = {
	.opt = "t:",
	.format = "-t<[!]start[+span|:end][,...]]>",
	.description = "specify time filter.",
	.handler = opt_handler_time,
};

/*default special functions*/
static struct timespec zero_ts={0,0};

int get_pid_logpid(struct lkst_log_record *start,
		   struct lkst_log_record *end)
{
	if(start) {
		return start->posix.log_pid;
	} else
		return -1;
}
double get_time_logtime(struct timespec *ts,
			struct lkst_log_record *start,
			struct lkst_log_record *end)
{
	if (start) {
		*ts = start->posix.log_time;
	} else {
		*ts = zero_ts;
	}
	return ts2d(*ts);
}
double get_delta_logtime(struct lkst_log_record *start,
			 struct lkst_log_record *end) 
{
	struct timespec ts;
	if (start && end) {
		ts = end->posix.log_time;
		timespec_dec(ts,start->posix.log_time);
		return ts2d(ts);
	} else {
		return (double)0;
	}
}

slot_t *slotfilter_pass(slot_t *slot)
{
	return slot;
}

slot_t *slotfilter_stop(slot_t *slot)
{
	return NULL;
}

int init_none(void) 
{
	return 0;
}

slot_hkey session_key_none(struct lkst_log_record *rec)
{
	return HKEY_UNUSED;
}

/*
 * function: analyze_logs
 * desc: analyze logfile and match same context entries.
 * args: llist -- list of logfiles
 *       ga -- gate analyzer
 *       fn -- matched entry handler
 * retval: 0 -- success
 *        -1 -- failed to read header.
 */

int analyze_sessions(struct gate_analyzer *ga, struct lkstloglist *llist,
		     struct analysis_formatter *fmt) 
{
	struct lkst_log_record *rec, *stopper, *starter, *ender;
	struct timespec ts;
	slot_t *slot;
	slot_hkey key, ikey;
	int pid;
	
	/*init session slots*/
	gsl_session = new_generic_slots(LOG_HASH_BITS, sizeof(struct lkst_log_record));
	if (gsl_session == NULL) {
		printf("error: fail to memory allocation\n");
		return -ENOMEM;
	}
	rec = stopper = NULL;
	while ((rec = loglist_earliest_entry(llist, rec)) != NULL) {
		if (rec->posix.log_event_type == LKST_ETYPE_LKST_OVWRTN_REC) {
			lkstlogfile_t *lf;
			release_all_slots(gsl_session);//dispose all sessions
			lf = find_logfile_from_entry(llist,rec);
			if (!lf) goto end_analyze; //debug!
			do{//read next valid entry
				if(logfile_read_one(lf)<1) {
					goto end_analyze;
				}
			}while (lf->entry.posix.log_event_type == 
				LKST_ETYPE_LKST_OVWRTN_REC);
			stopper = &(lf->entry);
		}
		if (stopper) {
			if( stopper != rec)
				continue;
			else
				stopper = NULL;
		}
		
		switch(ga->get_type(rec)) {
		case GAT_START:
			key = ga->session_key(rec);
			slot = get_free_slot(gsl_session, key); /*reserve slot*/
			if (slot!=NULL) 
				*(struct lkst_log_record*)slot_data(slot) = *rec;
			break;
		case GAT_END:
			key = ga->session_key(rec);
			for (slot = find_slot(gsl_session, key);
			     slot != NULL; 
			     slot = find_slot(gsl_session, key)) {
				starter = slot_data(slot);
				ender = rec;
				/*filtering*/
				if (!key_filter(ikey = ga->info_key(starter, ender)) ||
				    !pid_filter(pid = ga->get_pid(starter, ender)) ||
				    !time_filter(ga->get_time(&ts, starter,
							      ender))) break;
				fmt->callback(ga, ikey, &ts,
					      ga->get_delta(starter, ender),
					      pid);
				release_slot(gsl_session, key);
				if(!ga->multi_session) break;
			}
			break;
		default:
			break;
		}
	}
end_analyze:
	free_generic_slots(gsl_session);
	gsl_session = NULL;
	return 0;
}

int analyze_events(struct gate_analyzer *ga, struct lkstloglist *llist,
		   struct analysis_formatter *fmt) 
{
	struct lkst_log_record *rec;
	struct timespec ts;
	int pid;
	slot_hkey ikey;
	
	rec = NULL;
	while ((rec = loglist_earliest_entry(llist, rec)) != NULL) {
		if (rec->posix.log_event_type == LKST_ETYPE_LKST_OVWRTN_REC)
			continue;
		if (ga->get_type(rec) != GAT_EVENT) continue;
		
		;
		/*filtering*/
		if (!key_filter(ikey = ga->info_key(rec, rec)) ||
		    !pid_filter(pid = ga->get_pid(rec, rec)) ||
		    !time_filter(ga->get_time(&ts, rec, rec)) )
			continue;
		fmt->callback(ga, ikey, &ts,
			      ga->get_delta(rec, rec),
			      pid);
	}
	return 0;
}

