/*
 LKST log gate analyzer

 Copyright (C) HITACHI,LTD. 2004-2005
 WRITTEN BY HITACHI SYSTEMS DEVELOPMENT LABORATORY,
 Created by M.Hiramatsu <hiramatu@sdl.hitachi.co.jp>
  
 The development of this program is partly supported by IPA
 (Information-Technology Promotion Agency, Japan).
  
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */
#ifndef __LKSTLA_H__
#define __LKSTLA_H__

#include <linux/lkst.h>
#include <linux/lkst_buffer.h>
#include <timespec.h>
#include <logfile.h>
#include <slot.h>

struct command_option {
	char * opt;
	char * format;
	char * description;
	int (* handler)(int , char *);
};

struct analysis_formatter;

#define GAF_TIME_EV 1
#define GAF_LONG_EV 0

typedef enum {
	GAT_IGNORE=0,
	GAT_EVENT=1,
	GAT_START=2,
	GAT_END=3,
		/*for future use*/
//	GAT_PAUSE=4,
//	GAT_PEND=5,
} ga_type_t;

#define GA_MAXOPTS 64

/*
 * struct : gate_analyzer
 * description : event gate analyzer 
 */
struct gate_analyzer {
	/* common interfaces & proparties */
	char * name;		/*analyzer name*/
	char * description;	/*analyzer description*/	

	char * title;		/*info key and info alias title*/
	char * format_data;	/*info key and info alias format(for printf)*/
	char * delta_title;

	/* initialization function */
	int (*init)(void);
	
	/* optionals */
	int nr_options;
	struct command_option * options[GA_MAXOPTS]; /*extra options*/

	/*main analyzer loop*/
	int (*analyze_logs)(struct gate_analyzer *ga, 
			    struct lkstloglist *llist,
			    struct analysis_formatter *fmt);

	/*for distribution formatter -- default values*/
	int flag;
	int ranks;
	double max;
	double min;

	/* get event type */
	ga_type_t (*get_type)(struct lkst_log_record *);

	/* make session key (use session analyzers only)*/
	slot_hkey (*session_key)(struct lkst_log_record *);

	/* get alias name of information key */
	char *    (*get_alias)(slot_hkey key);

	/* in event analyzer, pass same record to following functions*/
	/* make information key (for classification) */
	slot_hkey (*info_key)(struct lkst_log_record *,
			      struct lkst_log_record *);

	/* for pid filter */
	int (*get_pid)(struct lkst_log_record *,
		       struct lkst_log_record *);
	/* get event time & time filter */
	double (*get_time)(struct timespec *ts, 
			   struct lkst_log_record *,
			   struct lkst_log_record *);
	/* get delta value */
	double (*get_delta)(struct lkst_log_record *,
			    struct lkst_log_record *);

	/*session correspondence*/
	int multi_session;
};

/*default special functions*/
extern int init_none(void);
extern slot_hkey session_key_none(struct lkst_log_record *rec);
extern int get_pid_logpid(struct lkst_log_record *,
			  struct lkst_log_record *);
extern double get_time_logtime(struct timespec *,
			       struct lkst_log_record *,
			       struct lkst_log_record *);
extern double get_delta_logtime(struct lkst_log_record *,
				struct lkst_log_record *);

extern int analyze_sessions(struct gate_analyzer *ga,
			    struct lkstloglist *llist,
			    struct analysis_formatter *fmt);
extern int analyze_events(struct gate_analyzer *ga,
			  struct lkstloglist *llist,
			  struct analysis_formatter *fmt);

#define GA_INIT_DEF_TIME \
	.flag = GAF_TIME_EV, \
	.ranks = 8, \
	.min = 0.0000001,\
	.max = 100, 

#define GA_INIT_DEF_LONG \
	.flag = GAF_LONG_EV, \
	.ranks = 9, \
	.min = 0,\
	.max = 100, 

#define GA_INIT_RET_EVENT_TIME \
	.get_time = get_time_logtime,

#define GA_INIT_1TO1_SESSION \
	.multi_session = 0,

#define GA_INIT_1TON_SESSION \
	.multi_session = 1,

#define GA_INIT_RET_SESSION_TIME \
	.get_time = get_time_logtime, \
	.get_delta = get_delta_logtime, \
	GA_INIT_DEF_TIME

#define GA_INIT_RET_EVENT_PID \
	.get_pid = get_pid_logpid,

#define GA_INIT_NO_EXTRAOPTS \
	.nr_options = 0, \
	.options[0] = NULL,

#define GA_INIT_SESSION_ANALYZER \
	.analyze_logs = analyze_sessions,

#define GA_INIT_EVENT_ANALYZER \
	.analyze_logs = analyze_events, \
	.session_key = session_key_none,

struct analysis_formatter {
	struct command_option opt;
	int active;
	
	/* analyze handler */
	int (*pre_analyze)(struct gate_analyzer *ga);
	void (*callback)(struct gate_analyzer *ga, slot_hkey key, 
			 struct timespec *ts, double delta, int pid);
	void (*post_analyze)(struct gate_analyzer *ga);
};

#endif
