/***************************************************************************
 *   Copyright (C) 2005 by SUZUKI Tasuku                                   *
 *   tasuku@linux-life.net                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "kkdwidget.h"
#include "kkdcanvas.h"
#include "kkdview.h"
#include "kkddoc.h"
#include "kkdpaper.h"
#include "kkdtextitem.h"
#include "kkdimageitem.h"
#include "kkdsubscribetemplate/kkdsubscribetemplatedialog.h"
#include "../kreetingkard.h"
#include "../kkaddressbook/kkacontactdata.h"

#include <qdom.h>
#include <kurl.h>
#include <kdebug.h>
#include <kaction.h>
#include <kconfig.h>
#include <klocale.h>
#include <kapplication.h>
#include <kcursor.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <kiconloader.h>
#include <kfontdialog.h>
#include <tkcoloractions.h>
#include <kprinter.h>

using namespace KKDesigner;

KKDWidget::KKDWidget( KCommandHistory* cmdHistory, QWidget* parent, const char* name )
	: KKWidgetBase( cmdHistory, parent, name )
	, m_filename( QString::null )
{
	m_doc = new KKDDoc( cmdHistory, this );

	m_view = new KKDView( KKDItemBase::DesignMode, m_doc, this );

	m_strlActions << "file_open"
				  << "file_open_recent"
				  << "file_save"
				  << "file_save_as"
				  << "file_print"
				  << "file_close"
				  << "edit_cut"
				  << "edit_copy"
				  << "edit_paste"
				  << "edit_select_all"
				  ;

	//BEGIN File Menu
	KStdAction::open( this, SLOT( fileOpen() ), actionCollection() );
	KStdAction::openRecent( this, SLOT( openRecent( const KURL& ) ), actionCollection() );
	KStdAction::save( this, SLOT( fileSave() ), actionCollection() );
	KStdAction::saveAs( this, SLOT( fileSaveAs() ), actionCollection() );
	KStdAction::print( this, SLOT( print() ), actionCollection() );
	KStdAction::close( this, SLOT( fileClose() ), actionCollection() );
	//END File Menu

	//BEGIN Edit Menu
	new KAction( i18n( "Insert &Text" ), "frame_text", KShortcut(Qt::Key_Insert), m_view, SLOT( insertText() ), actionCollection(), "inserttext" );
	new KAction( i18n( "Insert &Image" ), "frame_image", KShortcut(), m_view, SLOT( insertImage() ), actionCollection(), "insertimage" );
	new KAction( i18n( "Insert &Addressee" ), "insert_addressee", KShortcut(), m_view, SLOT( insertAddressee() ), actionCollection(), "insertaddressee" );
	new KAction( i18n( "Insert A&ddress" ), "insert_address", KShortcut(), m_view, SLOT( insertAddress() ), actionCollection(), "insertaddress" );
	new KAction( i18n( "Insert &Zipcode" ), "insert_zipcode", KShortcut(), m_view, SLOT( insertZipcode() ), actionCollection(), "insertzipcode" );

	new KAction( i18n( "&Delete Item" ), "eraser", KShortcut(Qt::Key_Delete), m_doc, SLOT( remove() ), actionCollection(), "remove" );

	new KAction( i18n( "&Raise" ), "top", 0, m_doc, SLOT( raise() ), actionCollection(), "raise" );

	new KAction( i18n( "&Lower" ), "bottom", 0, m_doc, SLOT( lower() ), actionCollection(), "lower" );

	KRadioAction* actR = new KRadioAction( i18n( "Horizontal" ), "text_left", 0, this, SLOT( setOrientation() ), actionCollection(), "horizontal" );
	actR->setExclusiveGroup( "Orientation" );

	actR = new KRadioAction( i18n( "Vertical" ), "text_top", 0, this, SLOT( setOrientation() ), actionCollection(), "vertical" );
	actR->setExclusiveGroup( "Orientation" );

	//BEGIN Align related
	actR = new KRadioAction( i18n( "Align Left" ), "text_left", 0, this, SLOT( setAlign() ), actionCollection(), "alignleft" );
	actR->setExclusiveGroup( "Align" );

	actR = new KRadioAction( i18n( "Align Center" ), "text_center", 0, this, SLOT( setAlign() ), actionCollection(), "aligncenter" );
	actR->setExclusiveGroup( "Align" );

	actR = new KRadioAction( i18n( "Align Right" ), "text_right", 0, this, SLOT( setAlign() ), actionCollection(), "alignright" );
	actR->setExclusiveGroup( "Align" );

	actR = new KRadioAction( i18n( "Align Top" ), "text_top", 0, this, SLOT( setAlign() ), actionCollection(), "aligntop" );
	actR->setExclusiveGroup( "Align" );

	actR = new KRadioAction( i18n( "Align Middle" ), "text_middle", 0, this, SLOT( setAlign() ), actionCollection(), "alignmiddle" );
	actR->setExclusiveGroup( "Align" );

	actR = new KRadioAction( i18n( "Align Bottom" ), "text_bottom", 0, this, SLOT( setAlign() ), actionCollection(), "alignbottom" );
	actR->setExclusiveGroup( "Align" );
	//END Align related

	//BEGIN Font related
	// see http://webcvs.kde.org/koffice/kword/kwview.cc?rev=1.1093&view=auto
#ifdef KFONTACTION_HAS_CRITERIA_ARG
	KFontAction* actFF = new KFontAction( KFontChooser::SmoothScalableFonts,
										i18n( "Font Family" ), 0,
										actionCollection(), "fontfamily" );
#else
	KFontAction* actFF = new KFontAction( i18n( "Font Family" ), 0,
										actionCollection(), "fontfamily" );
#endif
	connect( actFF, SIGNAL( activated( const QString& ) ), this, SLOT( setFontFamily( const QString& ) ) );

	KFontSizeAction* actFS = new KFontSizeAction( i18n( "Font Size" ), 0, actionCollection(), "fontsize" );
	connect( actFS, SIGNAL( fontSizeChanged( int ) ), this, SLOT( setFontSize( int ) ) );

	new KToggleAction( i18n( "Bold" ), "text_bold", KShortcut(Qt::CTRL + Qt::Key_B), this, SLOT( setFontBold() ), actionCollection(), "fontbold" );

	new KToggleAction( i18n( "Italic" ), "text_italic", KShortcut(Qt::CTRL + Qt::Key_I), this, SLOT( setFontItalic() ), actionCollection(), "fontitalic" );

	new KToggleAction( i18n( "Underline" ), "text_under", KShortcut(Qt::CTRL + Qt::Key_Underscore), this, SLOT( setFontUnderline() ), actionCollection(), "fontunderline" );

	new KToggleAction( i18n( "StrikeOut" ), "text_strike", KShortcut(Qt::CTRL + Qt::Key_Minus), this, SLOT( setFontStrikeOut() ), actionCollection(), "fontstrikeout" );

	TKSelectColorAction* actC = new TKSelectColorAction( i18n( "Color..." ), TKSelectColorAction::TextColor, this, SLOT( setColor() ), actionCollection(), "color", true );
	actC->setDefaultColor( QColor() );
	//END Font related

	new KToggleAction( i18n( "Lock" ), "encrypted", KShortcut(Qt::CTRL + Qt::Key_L), this, SLOT( setLocked() ), actionCollection(), "locked" );
	//END Edit Menu

// 	new KAction( i18n( "&Subscribe template" ), "sendtemplate", 0, this, SLOT( subscribeTemplate() ), actionCollection(), "subscribetemplate" );
// 	setActionEnabled( "subscribetemplate", true );

	new KAction( i18n( "&Font Installer" ), "kcmfontinst", 0, this, SLOT( fontinst() ), actionCollection(), "fontinst" );

	KStdAction::selectAll( m_doc, SLOT( selectAll() ), actionCollection() );
	KStdAction::cut( m_doc, SLOT( cut() ), actionCollection() );
	KStdAction::copy( m_doc, SLOT( copy() ), actionCollection() );
	KStdAction::paste( m_doc, SLOT( paste() ), actionCollection() );


	KStdAction::preferences( this, SLOT( preferences() ), actionCollection() );
	setEditActionEnabled();


	connect( m_doc, SIGNAL( selectedChanged() ), this, SLOT( setActionLocked() ) );
	connect( m_doc, SIGNAL( selectedChanged() ), this, SLOT( setActionFontFamily() ) );
	connect( m_doc, SIGNAL( selectedChanged() ), this, SLOT( setActionFontSize() ) );
	connect( m_doc, SIGNAL( selectedChanged() ), this, SLOT( setActionFontBold() ) );
	connect( m_doc, SIGNAL( selectedChanged() ), this, SLOT( setActionFontItalic() ) );
	connect( m_doc, SIGNAL( selectedChanged() ), this, SLOT( setActionFontUnderline() ) );
	connect( m_doc, SIGNAL( selectedChanged() ), this, SLOT( setActionFontStrikeOut() ) );
	connect( m_doc, SIGNAL( selectedChanged() ), this, SLOT( setActionOrientation() ) );
	connect( m_doc, SIGNAL( selectedChanged() ), this, SLOT( setActionAlign() ) );
	connect( m_doc, SIGNAL( selectedChanged() ), this, SLOT( setActionColor() ) );

	connect( m_doc, SIGNAL( lockedChanged() ), this, SLOT( setActionLocked() ) );
	connect( m_doc, SIGNAL( fontFamilyChanged() ), this, SLOT( setActionFontFamily() ) );
	connect( m_doc, SIGNAL( fontSizeChanged() ), this, SLOT( setActionFontSize() ) );
	connect( m_doc, SIGNAL( fontBoldChanged() ), this, SLOT( setActionFontBold() ) );
	connect( m_doc, SIGNAL( fontItalicChanged() ), this, SLOT( setActionFontItalic() ) );
	connect( m_doc, SIGNAL( fontUnderlineChanged() ), this, SLOT( setActionFontUnderline() ) );
	connect( m_doc, SIGNAL( fontStrikeOutChanged() ), this, SLOT( setActionFontStrikeOut() ) );
	connect( m_doc, SIGNAL( orientationChanged() ), this, SLOT( setActionOrientation() ) );
	connect( m_doc, SIGNAL( alignChanged() ), this, SLOT( setActionAlign() ) );
	connect( m_doc, SIGNAL( colorChanged() ), this, SLOT( setActionColor() ) );

	setActionLocked();
	setActionFontFamily();
	setActionFontSize();
	setActionFontBold();
	setActionFontItalic();
	setActionFontUnderline();
	setActionFontStrikeOut();
	setActionOrientation();
	setActionAlign();
	setActionColor();
}

KKDWidget::~KKDWidget()
{
}

void KKDWidget::resetCaption()
{
	QFileInfo info( m_filename );
	emit captionChanged( info.baseName() );
}

//BEGIN File Menu
void KKDWidget::fileNew( const QString& filename )
{
	if( !checkChangedAndSaved() ) return;

	QApplication::setOverrideCursor( KCursor::waitCursor() );
	m_doc->open( filename );
	QApplication::restoreOverrideCursor();

	m_filename = "";
	setEditActionEnabled();

	resetCaption();
}

void KKDWidget::fileOpen()
{
	if( !checkChangedAndSaved() ) return;
	QString filename = KFileDialog::getOpenFileName( ":<KreetingKard>", i18n( "*.kk|KreetingKard File (*.kk)" ) );
// 	KURL url( "/usr/kde/3.3/share/apps/kreetingkard/templates/Nenga_Simple_Landscape.kk" );	//for test
	if( filename.isEmpty() ) return;

	((KRecentFilesAction*)action( "file_open_recent" ))->addURL( KURL( filename ) );
	fileOpen( filename );
}

void KKDWidget::fileOpen( const QString& filename, bool blCheckChanged )
{
	if( blCheckChanged )
	{
		if( !checkChangedAndSaved() ) return;
	}

	QApplication::setOverrideCursor( KCursor::waitCursor() );
	m_doc->open( filename );
	QApplication::restoreOverrideCursor();

	m_filename = filename;
	setEditActionEnabled();

	resetCaption();
}

void KKDWidget::openRecent( const KURL& url )
{
	if( !checkChangedAndSaved() ) return;
	fileOpen( url.path() );
}

bool KKDWidget::fileSave()
{
	if( m_filename.isEmpty() )
	{
		return fileSaveAs();
	}
	QApplication::setOverrideCursor( KCursor::waitCursor() );
	if( !m_doc->save( m_filename ) )
	{
		QApplication::restoreOverrideCursor();
		return false;
	}
	QApplication::restoreOverrideCursor();

	commandHistory()->clear();

	resetCaption();

	actionCollection()->action( "file_save" )->setEnabled( true );
	return true;
}

bool KKDWidget::fileSaveAs()
{
	KURL url = KFileDialog::getSaveFileName( ":<KreetingKard>", i18n( "*.kk|KreetingKard File (*.kk)" ), this );
	if( url.isEmpty() ) return false;
	if( !url.isLocalFile() )
	{
		KMessageBox::sorry( this, i18n( "KreetingKard cannot handle remote files yet") );
		return false;
	}
	if( QFile::exists( url.path() ) )
	{
		if( KMessageBox::questionYesNo( this, i18n( "A file named \"%1\" already exists. Are you sure you want to overwrite it?" ).arg( url.fileName() ), i18n( "Overwrite File?" ), KGuiItem( i18n( "Overwrite" ), SmallIcon( "filesave" ) ) ) == KMessageBox::No )
		{
			return false;
		}
	}
	m_filename = url.path();
	if( !fileSave() ) return false;
	((KRecentFilesAction*)actionCollection()->action( "file_open_recent" ))->addURL( url );
	return true;

}

void KKDWidget::print()
{
	KKDPaper* paper = m_doc->current();
	KPrinter printer;

	printer.setDocName( "KreetingKard" );
// 	printer.setPageSize( KPrinter::A6 );
	printer.setFullPage( true );
	printer.setColorMode( KPrinter::Color );

	printer.setOrientation( ( paper->size().width() > paper->size().height() ) ?  KPrinter::Landscape : KPrinter::Portrait );

	switch( paper->printType() )
	{
		case KKDPaper::Insert:
		{
			QPtrList< KMainWindow >* list = KMainWindow::memberList;
			QPtrList<KKAddressBook::KKAContactData> data =  ((KreetingKard*)list->at(0))->printData();

			if( data.count() == 0 ) return;
			if( !printer.setup( this ) ) return;

			bool newPage = false;
			QPainter p;
			p.begin( &printer );
			for( KKAddressBook::KKAContactData* d = data.first(); d; d = data.next() )
			{
				KKAddressBook::KKAAddressData::LocationType l = d->current()->locationType();

				d->setCurrent( KKAddressBook::KKAAddressData::Home );
				if( d->current()->printStatus() == KKAddressBook::KKAAddressData::Print )
				{
					if( newPage )
					{
						printer.newPage();
					}
					else
					{
						newPage = true;
					}
					m_doc->print( p, d );
				}

				d->setCurrent( KKAddressBook::KKAAddressData::Office );
				if( d->current()->printStatus() == KKAddressBook::KKAAddressData::Print )
				{
					if( newPage )
					{
						printer.newPage();
					}
					else
					{
						newPage = true;
					}
					m_doc->print( p, d );
				}
				d->setCurrent( l );
			}
			p.end();
			int ans = KMessageBox::questionYesNo( this, i18n( "Do you change status Print to Printed?" ), i18n( "Print" ) );
			if( ans == KMessageBox::Ok )
			{
				for( KKAddressBook::KKAContactData* d = data.first(); d; d = data.next() )
				{
					KKAddressBook::KKAAddressData::LocationType l = d->current()->locationType();

					d->setCurrent( KKAddressBook::KKAAddressData::Home );
					if( d->current()->printStatus() == KKAddressBook::KKAAddressData::Print )
					{
						d->current()->setPrintStatus( KKAddressBook::KKAAddressData::Printed );
					}

					d->setCurrent( KKAddressBook::KKAAddressData::Office );
					if( d->current()->printStatus() == KKAddressBook::KKAAddressData::Print )
					{
						d->current()->setPrintStatus( KKAddressBook::KKAAddressData::Printed );
					}
					d->setCurrent( l );
				}
			}
			break;
		}
		case KKDPaper::Plain:
		{

			if( !printer.setup( this ) ) return;

			bool newPage = false;
			QPainter p;
			p.begin( &printer );

			for( int i = 0; i != printer.numCopies(); i++ )
			{
				if( newPage )
				{
						printer.newPage();
				}
				else
				{
						newPage = true;
				}
				m_doc->print( p );
			}
			p.end();
			break;
		}
	}
}

void KKDWidget::fileClose()
{
	if( !checkChangedAndSaved() ) return;
	m_doc->close();

	m_filename = QString::null;

	resetCaption();
	setEditActionEnabled();
}
//END File Menu

//BEGIN Edit Menu
void KKDWidget::setLocked()
{
	m_doc->setLocked( ((KToggleAction*)action( "locked" ))->isChecked() );
	setActionLocked();
}

void KKDWidget::setFontFamily( const QString& f )
{
	m_doc->setFontFamily( f );
	setActionFontFamily();
}

void KKDWidget::setFontSize( int s )
{
	m_doc->setFontSize( s );
	setActionFontSize();
}

void KKDWidget::setFontBold()
{
	m_doc->setFontBold( ((KToggleAction*)action( "fontbold" ))->isChecked() );
	setActionFontBold();
}

void KKDWidget::setFontItalic()
{
	m_doc->setFontItalic( ((KToggleAction*)action( "fontitalic" ))->isChecked() );
	setActionFontItalic();
}

void KKDWidget::setFontUnderline()
{
	m_doc->setFontUnderline( ((KToggleAction*)action( "fontunderline" ))->isChecked() );
	setActionFontUnderline();
}

void KKDWidget::setFontStrikeOut()
{
	m_doc->setFontStrikeOut( ((KToggleAction*)action( "fontstrikeout" ))->isChecked() );
	setActionFontStrikeOut();
}

void KKDWidget::setOrientation()
{
	if( ((KRadioAction*)action( "horizontal" ))->isChecked() )
	{
		m_doc->setOrientation( Horizontal );
	}
	else if( ((KRadioAction*)action( "vertical" ))->isChecked() )
	{
		m_doc->setOrientation( Vertical );
	}
	setActionOrientation();
}

void KKDWidget::setAlign()
{
	AlignmentFlags align = AlignAuto;
	if( ((KRadioAction*)action( "alignleft" ))->isChecked() )
	{
		align = AlignLeft;
	}
	else if( ((KRadioAction*)action("aligncenter" ))->isChecked() )
	{
		align = AlignHCenter;
	}
	else if( ((KRadioAction*)action("alignright" ))->isChecked() )
	{
		align = AlignRight;
	}
	else if( ((KRadioAction*)action("aligntop" ))->isChecked() )
	{
		align = AlignTop;
	}
	else if( ((KRadioAction*)action("alignmiddle" ))->isChecked() )
	{
		align = AlignVCenter;
	}
	else if( ((KRadioAction*)action("alignbottom" ))->isChecked() )
	{
		align = AlignBottom;
	}
	m_doc->setAlign( align );
	setActionAlign();
}

void KKDWidget::setColor()
{
	m_doc->setColor( ((TKSelectColorAction*)action( "color" ))->color() );
	setActionColor();
}
//END Edit Menu

//BEGIN Set action's value
void KKDWidget::setActionLocked()
{
	((KToggleAction*)action( "locked" ))->setChecked( m_doc->locked() );
}

void KKDWidget::setActionFontFamily()
{
	((KFontAction*)action( "fontfamily" ))->setFont( m_doc->fontFamily() );
}

void KKDWidget::setActionFontSize()
{
	int s = m_doc->fontSize();
	if( s == 0 ) return;
	((KFontSizeAction*)action( "fontsize" ))->setFontSize( s );
}

void KKDWidget::setActionFontBold()
{
	((KToggleAction*)action( "fontbold" ))->setChecked( m_doc->fontBold() );
}

void KKDWidget::setActionFontItalic()
{
	((KToggleAction*)action( "fontitalic" ))->setChecked( m_doc->fontItalic() );
}

void KKDWidget::setActionFontUnderline()
{
	((KToggleAction*)action( "fontunderline" ))->setChecked( m_doc->fontUnderline() );
}

void KKDWidget::setActionFontStrikeOut()
{
	((KToggleAction*)action( "fontstrikeout" ))->setChecked( m_doc->fontStrikeOut() );
}

void KKDWidget::setActionOrientation()
{
	Orientation o = m_doc->orientation();
	switch( o )
	{
		case Horizontal:
			((KRadioAction*)action( "horizontal" ))->setChecked( true );
			break;
		case Vertical:
			((KRadioAction*)action( "vertical" ))->setChecked( true );
			break;
		case (Orientation)-1:
			((KRadioAction*)action( "horizontal" ))->setChecked( false );
			((KRadioAction*)action( "vertical" ))->setChecked( false );
			break;
	}
	((KRadioAction*)action( "horizontal" ))->setEnabled( o > -2 );
	((KRadioAction*)action( "vertical" ))->setEnabled( o > -2 );
}

void KKDWidget::setActionAlign()
{
	AlignmentFlags a = m_doc->align();
	switch( a )
	{
		case AlignLeft:
			((KRadioAction*)action( "alignleft" ))->setChecked( true );
			break;
		case AlignHCenter:
			((KRadioAction*)action( "aligncenter" ))->setChecked( true );
			break;
		case AlignRight:
			((KRadioAction*)action( "alignright" ))->setChecked( true );
			break;
		case AlignTop:
			((KRadioAction*)action( "aligntop" ))->setChecked( true );
			break;
		case AlignVCenter:
			((KRadioAction*)action( "alignmiddle" ))->setChecked( true );
			break;
		case AlignBottom:
			((KRadioAction*)action( "alignbottom" ))->setChecked( true );
			break;
		case -1:
			((KRadioAction*)action( "alignleft" ))->setChecked( false );
			((KRadioAction*)action( "aligncenter" ))->setChecked( false );
			((KRadioAction*)action( "alignright" ))->setChecked( false );
			((KRadioAction*)action( "aligntop" ))->setChecked( false );
			((KRadioAction*)action( "alignmiddle" ))->setChecked( false );
			((KRadioAction*)action( "alignbottom" ))->setChecked( false );
			break;
	}
	bool enabled = ( a > -2 );
	((KRadioAction*)action( "alignleft" ))->setEnabled( enabled );
	((KRadioAction*)action( "aligncenter" ))->setEnabled( enabled );
	((KRadioAction*)action( "alignright" ))->setEnabled( enabled );
	((KRadioAction*)action( "aligntop" ))->setEnabled( enabled );
	((KRadioAction*)action( "alignmiddle" ))->setEnabled( enabled );
	((KRadioAction*)action( "alignbottom" ))->setEnabled( enabled );
}

void KKDWidget::setActionColor()
{
	((TKSelectColorAction*)action( "color" ))->setCurrentColor( m_doc->color() );
}
//END Set action's value

void KKDWidget::subscribeTemplate()
{
	if( !checkChangedAndSaved() ) return;
	fileSave();
	KKDSubscribeTemplate::KKDSubscribeTemplateDialog dlg( m_filename, this );
	for( KKDCanvas* canvas = m_view->canvas().first(); canvas; canvas = m_view->canvas().next() )
	{
		QPixmap pix = QPixmap::grabWidget( canvas );
		dlg.addPixmap( pix );
	}
	dlg.exec();
}

void KKDWidget::fontinst()
{
	kapp->startServiceByDesktopName( "kcmfontinst" );
}

void KKDWidget::readConfig( KConfig* conf )
{
	conf->setGroup("Designer");

	if( conf->hasKey( "HSplit" ) )
	{
		setSizes( conf->readIntListEntry( "HSplit" ) );
	}

	QString filename = conf->readEntry( "file", "" );
	if( !filename.isEmpty() )
		fileOpen( filename );

	((KRecentFilesAction*)action( "file_open_recent" ))->loadEntries( conf );
}

void KKDWidget::saveConfig( KConfig* conf )
{
	conf->setGroup("Designer");
	conf->writeEntry( "HSplit", sizes() );
	conf->writeEntry( "file", m_filename );
	((KRecentFilesAction*)action( "file_open_recent" ))->saveEntries( conf );
}

bool KKDWidget::closable()
{
	return checkChangedAndSaved();
}

void KKDWidget::tabChanged( QWidget* wgt )
{
	bool blEnabled = ( wgt == this );
	for( QStringList::ConstIterator it = m_strlActions.begin(); it != m_strlActions.end(); ++it )
	{
		setActionEnabled( *it, blEnabled );
	}
	if( blEnabled ) setEditActionEnabled();
	resetCaption();
}

bool KKDWidget::checkChangedAndSaved()
{
	bool blRtn = true;
	if( action( "edit_undo" )->isEnabled() )
	{
		int ans = KMessageBox::questionYesNoCancel( this, i18n("This document is modified.\nDo you save change?"), i18n("Confirm") );
		switch( ans )
		{
			case KMessageBox::Yes:
				blRtn = fileSave();
				if( blRtn )
				{
// 					m_doc->commandHistory()->clear();
				}
				break;
			case KMessageBox::No:
				break;
			case KMessageBox::Cancel:
				blRtn = false;
				break;
		}
	}
	return blRtn;
}

void KKDWidget::setEditActionEnabled()
{
	bool blEnabled = !m_filename.isNull();
	setActionEnabled( "file_save", blEnabled );
	setActionEnabled( "file_save_as", blEnabled );
	setActionEnabled( "file_print", blEnabled );
	setActionEnabled( "file_close", blEnabled );
	setActionEnabled( "inserttext", blEnabled );
	setActionEnabled( "insertimage", blEnabled );
	setActionEnabled( "insertaddressee", blEnabled );
	setActionEnabled( "insertaddress", blEnabled );
	setActionEnabled( "insertzipcode", blEnabled );
	setActionEnabled( "remove", blEnabled );
	setActionEnabled( "edit_select_all", blEnabled );

	setActionEnabled( "alignleft", blEnabled );
	setActionEnabled( "aligncenter", blEnabled );
	setActionEnabled( "alignright", blEnabled );
	setActionEnabled( "aligntop", blEnabled );
	setActionEnabled( "alignmiddle", blEnabled );
	setActionEnabled( "alignbottom", blEnabled );

	setActionEnabled( "fontfamily", blEnabled );
	setActionEnabled( "fontsize", blEnabled );
	setActionEnabled( "fontbold", blEnabled );
	setActionEnabled( "fontitalic", blEnabled );
	setActionEnabled( "fontunderline", blEnabled );
	setActionEnabled( "fontstrikeout", blEnabled );

	setActionEnabled( "color", blEnabled );
	setActionEnabled( "locked", blEnabled );

	setActionEnabled( "raise", blEnabled );
	setActionEnabled( "lower", blEnabled );
	setActionEnabled( "edit_cut", false );
	setActionEnabled( "edit_copy", false );
	setActionEnabled( "edit_paste", false );
	setActionEnabled( "view_zoom_in", false );
	setActionEnabled( "view_zoom_out", false );
}
