/***************************************************************************
 *   Copyright (C) 2005 by SUZUKI Tasuku                                   *
 *   tasuku@linux-life.net                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "kkdtextitem.h"

#include <qregexp.h>

#include <kiconloader.h>
#include <klocale.h>
#include <kdebug.h>

using namespace KKDesigner;

KKDTextItem::KKDTextItem( const QPoint& l
						, const QSize& s
						, const QFont& f
						, const QColor& c
						, Orientation o
						, const QString& t
						, AlignmentFlags a
						, QObject* parent
						)
	: KKDTextBase( l, s, f, c, o, parent )
	, m_text( t )
	, m_align( a )
{
}

KKDTextItem::KKDTextItem( QDomElement e, QObject* parent )
	: KKDTextBase( e, parent )
	, m_text( e.attribute( "text" ) )
	, m_align( (AlignmentFlags)e.attribute( "align" ).toInt() )
{
}

KKDTextItem::KKDTextItem( const KKDTextItem* item, QObject* parent )
	: KKDTextBase( (KKDTextBase*)item, parent )
	, m_text( item->m_text )
	, m_align( item->m_align )
{
}

KKDTextItem::~KKDTextItem()
{
}

//BEGIN Properties
void KKDTextItem::setText( const QString& t )
{
	if( locked() ) return;
	if( m_text == t ) return;
	QRect r = rect();
	m_text = t;
	resize();
	emit repaint( r | rect() );
	emit textChanged();
}

void KKDTextItem::setAlign( AlignmentFlags a )
{
	if( locked() ) return;
	if( m_align == a ) return;
	Orientation o = orientation();
	QRect r( rect() );
	setOrientation( a & AlignHorizontal_Mask ? Horizontal : Vertical );
	m_align = a;
	QSize s( size() );
	emit repaint( r|rect() );
	emit alignChanged();
}
//END   Properties

QDomElement KKDTextItem::toElement( QDomDocument& doc, const QString& t )
{
	QString tag( t == QString::null ? tagName() : t );
	QDomElement elem = KKDTextBase::toElement( doc, tag );
	elem.setAttribute( "text", m_text );
	elem.setAttribute( "align", (int)m_align );
	return elem;
}

/**
 * @english
 * @short Draw string
 * according to follow elements<br>
 * print, country, data, align<br>
 * For example, japanese zipcode is a number of seven digits.<br>
 * If print, string is KKAAddress::zipcode.<br>
 * Draw horizontal.<br>
 * Location must 8:8:8[:1]:8:8:8:8.
 *
 * @param r Draw rectangle
 * @param p QPainter
 * @param zoom Zoom
 * @param d ContactData to insert when printing
 * @endenglish
 */
void KKDTextItem::drawItemBefore( QRect& r, QPainter& p, DrawMode m, const KKAddressBook::KKAContactData* d )
{
	KKDTextBase::drawItemBefore( r, p, m, d );

	if( m_align & AlignVertical_Mask )
	{
		QStringList org = QStringList::split( "\n", m_text, true );
		QStringList disp;
		for( QStringList::Iterator it = org.begin(); it != org.end(); ++it )
		{
			QStringList line = QStringList::split( QString(""), *it, true );
			line.pop_front();
			line.pop_back();
			QString tmp = line.join( "\n" );

			QRegExp rx( "([0-9])\n([0-9])" );
			while( rx.search( tmp ) != -1 )
			{
				tmp.replace( rx, "\\1\\2" );
			}
			disp.append( tmp );
		}

		int* width = new int[disp.count()];
		QFontMetrics met( font() );
		for( unsigned int i = 0; i != disp.count(); i++ )
		{
			width[i] = 0;
			QStringList line = QStringList::split( "\n", disp[i], true );
			for( QStringList::Iterator it = line.begin(); it != line.end(); ++it )
			{
				if( width[i] < met.width( *it ) )
				{
					width[i] = met.width( *it );
				}
			}
		}
		int right = r.width();
		for( unsigned int i = 0; i != disp.count(); i++ )
		{
			right -= width[i];
		}
		switch( AlignRight )
		{
			case AlignLeft:
				right = r.right() - right;
				break;
			case AlignHCenter:
				right = r.right() - right / 2;
				break;
			case AlignRight:
				right = r.right();
				break;
		}
		for( unsigned int i = 0; i != disp.count(); i++ )
		{
			QRect rct( right - width[i], r.top(), width[i], r.height() );
			if( r.contains( rct ) )
			{
				p.drawText( rct, m_align | AlignHCenter, disp[i] );
			}
			right -= width[i];
		}
		delete[] width;
	}
	else if( m_align & AlignHorizontal_Mask )
	{
		p.drawText( r, m_align | AlignVCenter, m_text );
	}
}

void KKDTextItem::drawItemAfter( QRect& r, QPainter& p, DrawMode m, const KKAddressBook::KKAContactData* d )
{
	KKDTextBase::drawItemAfter( r, p, m, d );
}

void KKDTextItem::resize()
{
	setSize( calcSize( text() ) );
}

void KKDTextItem::refontsize()
{
	QSize s = calcSize( text() );
	if( s.width() > rect().width() || s.height() > rect().height() )
	{
		while( 1 )
		{
			setFontSize( fontSize() - 1 );
			s = calcSize( text() );
			if( s.width() <= rect().width() && s.height() <= rect().height() )
			{
				break;
			}
		}
	}
	else if( s.width() < rect().width() || s.height() < rect().height() )
	{
		while( 1 )
		{
			setFontSize( fontSize() + 1 );
			s = calcSize( text() );
			if( s.width() > rect().width() || s.height() > rect().height() )
			{
				break;
			}
		}
		setFontSize( fontSize() - 1 );
	}
}
