/***************************************************************************
 *   Copyright (C) 2005 by SUZUKI Tasuku                                   *
 *   tasuku@linux-life.net                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef _KKDITEMBASE_H_
#define _KKDITEMBASE_H_

#include <qobject.h>
#include <qdom.h>
#include <qpainter.h>

namespace KKAddressBook
{
	class KKAContactData;
}

namespace KKDesigner
{
	/**
	 * @english
	 * @short Base class of each items.
	 * It has common properties such as "size" and methods.
	 * @author SUZUKI Tasuku <tasuku@linux-life.net>
	 * @endenglish
	 *
	 * @japanese
	 * @short アイテムクラスの基底クラス
	 * サイズなどの共通のプロパティとメソッドを含みます。
	 * @author 鈴木 佑 <tasuku@linux-life.net>
	 * @endjapanese
	 *
	 * @see KKDImageItem
	 * @see KKDPaper
	 * @see KKDTextItem
	 */
	class KKDItemBase : public QObject
	{
		Q_OBJECT
		Q_PROPERTY( QPoint location READ location WRITE setLocation )
		Q_PROPERTY( QSize size READ size WRITE setSize )
		Q_PROPERTY( bool locked READ locked WRITE setLocked )
		Q_PROPERTY( bool selected READ selected WRITE setSelected )
		Q_PROPERTY( QRect rect READ rect )
		Q_PROPERTY( ItemType type READ type )
		Q_ENUMS( ItemType )
	public:
		enum DrawMode
		{
			/**
			 * used by template view
			 */
			PreviewMode,
			/**
			 * used by designer view
			 */
			DesignMode,
			/**
			 * used by address view
			 */
			AddressMode,
			/**
			 * used for printing
			 */
			PrintMode,
		};
		enum ItemType
		{
			Unknown,
			/**
			 * @see KKDTextItem
			 */
			TextItem,
			/**
			 * @see KKDImageItem
			 */
			ImageItem,
			/**
			 * @see KKDPaper
			 */
			Paper,
			/**
			 * @see KKDAddresseeItem
			 */
			Addressee,
			/**
			 * @see KKDAddressItem
			 */
			Address,
			/**
			 * @see KKDZipcodeItem
			 */
			Zipcode,
		};
		KKDItemBase( int x, int y, int width, int height, QObject* parent = 0 );
		KKDItemBase( QDomElement e, QObject* parent = 0 );
		KKDItemBase( const KKDItemBase* item, QObject* parent = 0 );

		virtual ~KKDItemBase();

		/**
		 * @returns tag name for xml.
		 */
		static const QString tagName(){ return QString( "Base" ); }

		virtual QDomElement toElement( QDomDocument& doc, const QString& t = QString::null );

		void drawItem( QPainter& p, DrawMode m, const KKAddressBook::KKAContactData* d = 0 );

		//BEGIN Properties
		/**
		 * @english
		 * @returns location
		 * @endenglish
		 * @japanese
		 * @returns 位置
		 * @endjapanese
		 */
		const QPoint& location() const { return m_location; }

		/**
		 * @english
		 * @returns size
		 * @endenglish
		 * @japanese
		 * @returns サイズ
		 * @endjapanese
		 */
		const QSize& size() const { return m_size; }

		/**
		 * @english
		 * @returns locked
		 * @endenglish
		 * @japanese
		 * @returns 保護状態
		 * @endjapanese
		 */
		bool locked() const { return m_locked; }

		/**
		 * @english
		 * @returns selected
		 * @endenglish
		 * @japanese
		 * @returns 選択状態
		 * @endjapanese
		 */
		bool selected() const { return m_selected; }

		/**
		 * @english
		 * @returns rectangle
		 * @endenglish
		 * @japanese
		 * @returns 矩形
		 * @endjapanese
		 */
		QRect rect() const { return QRect( location(), size() ); }

		/**
		 * @english
		 * @returns item type
		 * @endenglish
		 * @japanese
		 * @returns アイテムの種類
		 * @endjapanese
		 */
		virtual ItemType type() const = 0;
		//END   Properties

	public slots:
		//BEGIN Properties
		void setLocation( const QPoint& l ){ setLocation( l.x(), l.y() ); }
		void setLocation( int x, int y );
		void setSize( const QSize& s ){ setSize( s.width(), s.height() ); }
		void setSize( int w, int h );
		void setLocked( bool l );
		void setSelected( bool b );
		//END   Properties

		void slotRepaint(){ emit repaint( rect() ); }

	signals:
		//BEGIN Properties
		void locationChanged();
		void sizeChanged();
		void lockedChanged();
		void selectedChanged();
		//END   Properties
		void repaint( const QRect& );

	protected:
		virtual void drawItemBefore( QRect& r, QPainter& p, DrawMode m, const KKAddressBook::KKAContactData* d );
		virtual void drawItemAfter( QRect& r, QPainter& p, DrawMode m, const KKAddressBook::KKAContactData* d );

	private:
		QPoint m_location;
		QSize m_size;
		bool m_locked;
		bool m_selected;
	};
}

#endif /* _KKDITEMBASE_H_ */
