/***************************************************************************
 *   Copyright (C) 2005 by SUZUKI Tasuku                                   *
 *   tasuku@linux-life.net                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "kkdcmds.h"

#include "kkdtextbase.h"
#include "kkdtextitem.h"

using namespace KKDesigner;

//BEGIN KKDCmdBase
KKDCmdBase::KKDCmdBase( KKDDoc* doc )
{
	m_doc = doc;
	m_paper = m_doc->at( doc->currentIndex() );
	if( m_paper->selected() )
	{
		m_items.append( m_paper );
	}
	else
	{
		for( KKDItemBase* item = m_paper->first(); item; item = m_paper->next() )
		{
			if( item->selected() ) m_items.append( item );
		}
	}
}

void KKDCmdBase::execute()
{
	m_doc->setCurrent( m_paper );
	for( KKDItemBase* item = items().first(); item; item = items().next() )
	{
		execute( item );
	}
	m_doc->setSelected( items(), true, true );
	emit executed();
}

void KKDCmdBase::unexecute()
{
	m_doc->setCurrent( m_paper );
	for( KKDItemBase* item = items().first(); item; item = items().next() )
	{
		unexecute( item );
	}
	m_doc->setSelected( items(), true, true );
	emit unexecuted();
}

void KKDCmdBase::begin()
{
	QPtrList<KKDItemBase> remove;
	for( KKDItemBase* item = items().first(); item; item = items().next() )
	{
		if( !begin( item ) )
		{
			remove.append( item );
		}
	}
	for( KKDItemBase* item = remove.first(); item; item = remove.next() )
	{
		items().removeRef( item );
	}
}

void KKDCmdBase::end()
{
	QPtrList<KKDItemBase> remove;
	for( KKDItemBase* item = items().first(); item; item = items().next() )
	{
		if( !end( item ) )
		{
			remove.append( item );
		}
	}
	for( KKDItemBase* item = remove.first(); item; item = remove.next() )
	{
		items().removeRef( item );
	}
	setName( i18n( "Change %1 of %2 item(s)" ).arg( propertyName() ).arg( items().count() ) );
}

QString KKDCmdBase::name() const
{
	return m_name;
}

void KKDCmdBase::setName( const QString &name )
{
	m_name = name;
}
//END KKDCmdBase

//BEGIN KKDCmdOrientation
KKDCmdOrientation::KKDCmdOrientation( KKDDoc* doc )
	: KKDCmdBase( doc )
{
	connect( this, SIGNAL( executed() ), doc, SIGNAL( datafieldChanged() ) );
	connect( this, SIGNAL( unexecuted() ), doc, SIGNAL( datafieldChanged() ) );
}

void KKDCmdOrientation::execute( KKDItemBase* item )
{
	((KKDTextBase*)item)->setOrientation( m_after[item] );
}

void KKDCmdOrientation::unexecute( KKDItemBase* item )
{
	((KKDTextBase*)item)->setOrientation( m_before[item] );
}

bool KKDCmdOrientation::begin( KKDItemBase* item )
{
	if( !item->inherits( "KKDesigner::KKDTextBase" ) ) return false;
	m_before[item] = ((KKDTextBase*)item)->orientation();
	return true;
}

bool KKDCmdOrientation::end( KKDItemBase* item )
{
	bool rtn = ( m_before[item] != ((KKDTextItem*)item)->orientation() );
	if( rtn )
	{
		m_after[item] = ((KKDTextBase*)item)->orientation();
	}
	else
	{
		m_before.remove( item );
	}
	return rtn;
}

QString KKDCmdOrientation::propertyName()
{
	return i18n( "orientation" );
}
//END KKDCmdData

//BEGIN KKDCmdImage
KKDCmdImage::KKDCmdImage( KKDDoc* doc )
	: KKDCmdBase( doc )
{
	connect( this, SIGNAL( executed() ), doc, SIGNAL( imageChanged() ) );
	connect( this, SIGNAL( unexecuted() ), doc, SIGNAL( imageChanged() ) );
}

void KKDCmdImage::execute( KKDItemBase* item )
{
	((KKDImageItem*)item)->setImage( m_after[item] );
}

void KKDCmdImage::unexecute( KKDItemBase* item )
{
	((KKDImageItem*)item)->setImage( m_before[item] );
}

bool KKDCmdImage::begin( KKDItemBase* item )
{
	if( !item->inherits( "KKDesigner::KKDImageItem" ) ) return false;
	m_before[item] = ((KKDImageItem*)item)->image();
	return true;
}

bool KKDCmdImage::end( KKDItemBase* item )
{
	bool rtn = ( m_before[item] != ((KKDImageItem*)item)->image() );
	if( rtn )
	{
		m_after[item] = ((KKDImageItem*)item)->image();
	}
	else
	{
		m_before.remove( item );
	}
	return rtn;
}

QString KKDCmdImage::propertyName()
{
	return i18n( "image" );
}
//END KKDCmdImage

//BEGIN KKDCmdImagePrint
KKDCmdImagePrint::KKDCmdImagePrint( KKDDoc* doc )
	: KKDCmdBase( doc )
{
	connect( this, SIGNAL( executed() ), doc, SIGNAL( imagePrintChanged() ) );
	connect( this, SIGNAL( unexecuted() ), doc, SIGNAL( imagePrintChanged() ) );
}

void KKDCmdImagePrint::execute( KKDItemBase* item )
{
	((KKDPaper*)item)->setImagePrint( m_after[item] );
}

void KKDCmdImagePrint::unexecute( KKDItemBase* item )
{
	((KKDPaper*)item)->setImagePrint( m_before[item] );
}

bool KKDCmdImagePrint::begin( KKDItemBase* item )
{
	if( !item->inherits( "KKDesigner::KKDPaper" ) ) return false;
	m_before[item] = ((KKDPaper*)item)->imagePrint();
	return true;
}

bool KKDCmdImagePrint::end( KKDItemBase* item )
{
	bool rtn = ( m_before[item] != ((KKDPaper*)item)->imagePrint() );
	if( rtn )
	{
		m_after[item] = ((KKDPaper*)item)->imagePrint();
	}
	else
	{
		m_before.remove( item );
	}
	return rtn;
}

QString KKDCmdImagePrint::propertyName()
{
	return i18n( "print image" );
}
//END KKDCmdImagePrint

//BEGIN KKDCmdInsert
KKDCmdInsert::KKDCmdInsert( KKDDoc* doc )
	: KKDCmdBase( doc )
{
	connect( this, SIGNAL( executed() ), doc, SIGNAL( inserted() ) );
	connect( this, SIGNAL( unexecuted() ), doc, SIGNAL( removed() ) );
}

void KKDCmdInsert::execute( KKDItemBase* item )
{
	m_parent[item]->prepend( item );
	document()->insertByCommand( m_parent[item], item );
}

void KKDCmdInsert::unexecute( KKDItemBase* item )
{
	m_parent[item]->remove( item );
	document()->removeByCommand( m_parent[item], item );
}

bool KKDCmdInsert::begin( KKDItemBase* item )
{
	if( item->inherits( "KKDesigner::KKDPaper" ) ) return false;
	m_parent[item] = (KKDPaper*)item->parent();
	return true;
}

bool KKDCmdInsert::end( KKDItemBase* )
{
	return true;
}

QString KKDCmdInsert::propertyName()
{
	return i18n( "insert" );
}
//END KKDCmdInsert

//BEGIN KKDCmdRemove
KKDCmdRemove::KKDCmdRemove( KKDDoc* doc )
	: KKDCmdBase( doc )
{
	connect( this, SIGNAL( executed() ), doc, SIGNAL( removed() ) );
	connect( this, SIGNAL( unexecuted() ), doc, SIGNAL( inserted() ) );
}

void KKDCmdRemove::execute( KKDItemBase* item )
{
	m_parent[item]->remove( item );
	document()->removeByCommand( m_parent[item], item );
}

void KKDCmdRemove::unexecute( KKDItemBase* item )
{
	m_parent[item]->insert( m_index[item], item );
	document()->insertByCommand( m_parent[item], item );
}

bool KKDCmdRemove::begin( KKDItemBase* item )
{
	if( item->inherits( "KKDesigner::KKDPaper" ) ) return false;
	if( item->locked() ) return false;
	m_parent[item] = (KKDPaper*)item->parent();
	m_index[item] = m_parent[item]->containsRef( item );
	return true;
}

bool KKDCmdRemove::end( KKDItemBase* item )
{
	if( item->locked() ) return false;
	return item->parent();
}

QString KKDCmdRemove::propertyName()
{
	return i18n( "remove" );
}
//END KKDCmdRemove

//BEGIN KKDCmdLocation
KKDCmdLocation::KKDCmdLocation( KKDDoc* doc )
	: KKDCmdBase( doc )
{
	connect( this, SIGNAL( executed() ), doc, SIGNAL( locationChanged() ) );
	connect( this, SIGNAL( unexecuted() ), doc, SIGNAL( locationChanged() ) );
}

void KKDCmdLocation::execute( KKDItemBase* item )
{
	item->setLocation( m_after[item] );
}

void KKDCmdLocation::unexecute( KKDItemBase* item )
{
	item->setLocation( m_before[item] );
}

bool KKDCmdLocation::begin( KKDItemBase* item )
{
	if( item->inherits( "KKDesigner::KKDPaper" ) ) return false;
	m_before[item] = item->location();
	return true;
}

bool KKDCmdLocation::end( KKDItemBase* item )
{
	bool rtn = ( m_before[item] != item->location() );
	if( rtn )
	{
		m_after[item] = item->location();
	}
	else
	{
		m_before.remove( item );
	}
	return rtn;
}

QString KKDCmdLocation::propertyName()
{
	return i18n( "location" );
}
//END KKDCmdLocation

//BEGIN KKDCmdLocked
KKDCmdLocked::KKDCmdLocked( KKDDoc* doc )
	: KKDCmdBase( doc )
{
	connect( this, SIGNAL( executed() ), doc, SIGNAL( lockedChanged() ) );
	connect( this, SIGNAL( unexecuted() ), doc, SIGNAL( lockedChanged() ) );
}

void KKDCmdLocked::execute( KKDItemBase* item )
{
	item->setLocked( m_after[item] );
}

void KKDCmdLocked::unexecute( KKDItemBase* item )
{
	item->setLocked( m_before[item] );
}

bool KKDCmdLocked::begin( KKDItemBase* item )
{
// 	if( item->inherits( "KKDesigner::KKDPaper" ) ) return false;
	m_before[item] = item->locked();
	return true;
}

bool KKDCmdLocked::end( KKDItemBase* item )
{
	bool rtn = ( m_before[item] != item->locked() );
	if( rtn )
	{
		m_after[item] = item->locked();
	}
	else
	{
		m_before.remove( item );
	}
	return rtn;
}

QString KKDCmdLocked::propertyName()
{
	return i18n( "locked" );
}
//END KKDCmdLocked

//BEGIN KKDCmdSize
KKDCmdSize::KKDCmdSize( KKDDoc* doc )
	: KKDCmdBase( doc )
{
	connect( this, SIGNAL( executed() ), doc, SIGNAL( sizeChanged() ) );
	connect( this, SIGNAL( unexecuted() ), doc, SIGNAL( sizeChanged() ) );
}

void KKDCmdSize::execute( KKDItemBase* item )
{
	item->setSize( m_after[item] );
}

void KKDCmdSize::unexecute( KKDItemBase* item )
{
	item->setSize( m_before[item] );
}

bool KKDCmdSize::begin( KKDItemBase* item )
{
	if( item->inherits( "KKDesigner::KKDPaper" ) ) return false;
	m_before[item] = item->size();
	return true;
}

bool KKDCmdSize::end( KKDItemBase* item )
{
	bool rtn = ( m_before[item] != item->size() );
	if( rtn )
	{
		m_after[item] = item->size();
	}
	else
	{
		m_before.remove( item );
	}
	return rtn;
}

QString KKDCmdSize::propertyName()
{
	return i18n( "size" );
}
//END KKDCmdSize

//BEGIN KKDCmdText
KKDCmdText::KKDCmdText( KKDDoc* doc )
	: KKDCmdBase( doc )
{
	connect( this, SIGNAL( executed() ), doc, SIGNAL( textChanged() ) );
	connect( this, SIGNAL( unexecuted() ), doc, SIGNAL( textChanged() ) );
}

void KKDCmdText::execute( KKDItemBase* item )
{
	((KKDTextItem*)item)->setText( m_after[item] );
}

void KKDCmdText::unexecute( KKDItemBase* item )
{
	((KKDTextItem*)item)->setText( m_before[item] );
}

bool KKDCmdText::begin( KKDItemBase* item )
{
	if( !item->inherits( "KKDesigner::KKDTextItem" ) ) return false;
	m_before[item] = ((KKDTextItem*)item)->text();
	return true;
}

bool KKDCmdText::end( KKDItemBase* item )
{
	bool rtn = ( m_before[item] != ((KKDTextItem*)item)->text() );
	if( rtn )
	{
		m_after[item] = ((KKDTextItem*)item)->text();
	}
	else
	{
		m_before.remove( item );
	}
	return rtn;
}

QString KKDCmdText::propertyName()
{
	return i18n( "text" );
}
//END KKDCmdText

//BEGIN KKDCmdAlign
KKDCmdAlign::KKDCmdAlign( KKDDoc* doc )
	: KKDCmdBase( doc )
{
	connect( this, SIGNAL( executed() ), doc, SIGNAL( alignChanged() ) );
	connect( this, SIGNAL( unexecuted() ), doc, SIGNAL( alignChanged() ) );
}

void KKDCmdAlign::execute( KKDItemBase* item )
{
	((KKDTextItem*)item)->setAlign( m_after[item] );
}

void KKDCmdAlign::unexecute( KKDItemBase* item )
{
	((KKDTextItem*)item)->setAlign( m_before[item] );
}

bool KKDCmdAlign::begin( KKDItemBase* item )
{
	if( !item->inherits( "KKDesigner::KKDTextItem" ) ) return false;
	m_before[item] = ((KKDTextItem*)item)->align();
	return true;
}

bool KKDCmdAlign::end( KKDItemBase* item )
{
	bool rtn = ( m_before[item] != ((KKDTextItem*)item)->align() );
	if( rtn )
	{
		m_after[item] = ((KKDTextItem*)item)->align();
	}
	else
	{
		m_before.remove( item );
	}
	return rtn;
}

QString KKDCmdAlign::propertyName()
{
	return i18n( "text align" );
}
//END KKDCmdAlign

//BEGIN KKDCmdColor
KKDCmdColor::KKDCmdColor( KKDDoc* doc )
	: KKDCmdBase( doc )
{
	connect( this, SIGNAL( executed() ), doc, SIGNAL( textColorChanged() ) );
	connect( this, SIGNAL( unexecuted() ), doc, SIGNAL( textColorChanged() ) );
}

void KKDCmdColor::execute( KKDItemBase* item )
{
	((KKDTextBase*)item)->setColor( m_after[item] );
}

void KKDCmdColor::unexecute( KKDItemBase* item )
{
	((KKDTextBase*)item)->setColor( m_before[item] );
}

bool KKDCmdColor::begin( KKDItemBase* item )
{
	if( !item->inherits( "KKDesigner::KKDTextBase" ) ) return false;
	m_before[item] = ((KKDTextBase*)item)->color();
	return true;
}

bool KKDCmdColor::end( KKDItemBase* item )
{
	bool rtn = ( m_before[item] != ((KKDTextBase*)item)->color() );
	if( rtn )
	{
		m_after[item] = ((KKDTextBase*)item)->color();
	}
	else
	{
		m_before.remove( item );
	}
	return rtn;
}

QString KKDCmdColor::propertyName()
{
	return i18n( "text color" );
}
//END KKDCmdColor

//BEGIN KKDCmdFontFamily
KKDCmdFontFamily::KKDCmdFontFamily( KKDDoc* doc )
	: KKDCmdBase( doc )
{
	connect( this, SIGNAL( executed() ), doc, SIGNAL( fontFamilyChanged() ) );
	connect( this, SIGNAL( unexecuted() ), doc, SIGNAL( fontFamilyChanged() ) );
}

void KKDCmdFontFamily::execute( KKDItemBase* item )
{
	((KKDTextBase*)item)->setFontFamily( m_after[item] );
}

void KKDCmdFontFamily::unexecute( KKDItemBase* item )
{
	((KKDTextBase*)item)->setFontFamily( m_before[item] );
}

bool KKDCmdFontFamily::begin( KKDItemBase* item )
{
	if( !item->inherits( "KKDesigner::KKDTextBase" ) ) return false;
	m_before[item] = ((KKDTextBase*)item)->fontFamily();
	return true;
}

bool KKDCmdFontFamily::end( KKDItemBase* item )
{
	bool rtn = ( m_before[item] != ((KKDTextBase*)item)->fontFamily() );
	if( rtn )
	{
		m_after[item] = ((KKDTextBase*)item)->fontFamily();
	}
	else
	{
		m_before.remove( item );
	}
	return rtn;
}

QString KKDCmdFontFamily::propertyName()
{
	return i18n( "font family" );
}
//END KKDCmdFontFamily

//BEGIN KKDCmdFontSize
KKDCmdFontSize::KKDCmdFontSize( KKDDoc* doc )
	: KKDCmdBase( doc )
{
	connect( this, SIGNAL( executed() ), doc, SIGNAL( fontSizeChanged() ) );
	connect( this, SIGNAL( unexecuted() ), doc, SIGNAL( fontSizeChanged() ) );
}

void KKDCmdFontSize::execute( KKDItemBase* item )
{
	((KKDTextBase*)item)->setFontSize( m_after[item] );
}

void KKDCmdFontSize::unexecute( KKDItemBase* item )
{
	((KKDTextBase*)item)->setFontSize( m_before[item] );
}

bool KKDCmdFontSize::begin( KKDItemBase* item )
{
	if( !item->inherits( "KKDesigner::KKDTextBase" ) ) return false;
	m_before[item] = ((KKDTextBase*)item)->fontSize();
	return true;
}

bool KKDCmdFontSize::end( KKDItemBase* item )
{
	bool rtn = ( m_before[item] != ((KKDTextBase*)item)->fontSize() );
	if( rtn )
	{
		m_after[item] = ((KKDTextBase*)item)->fontSize();
	}
	else
	{
		m_before.remove( item );
	}
	return rtn;
}

QString KKDCmdFontSize::propertyName()
{
	return i18n( "font size" );
}
//END KKDCmdFontSize

//BEGIN KKDCmdFontBold
KKDCmdFontBold::KKDCmdFontBold( KKDDoc* doc )
	: KKDCmdBase( doc )
{
	connect( this, SIGNAL( executed() ), doc, SIGNAL( fontBoldChanged() ) );
	connect( this, SIGNAL( unexecuted() ), doc, SIGNAL( fontBoldChanged() ) );
}

void KKDCmdFontBold::execute( KKDItemBase* item )
{
	((KKDTextBase*)item)->setFontBold( m_after[item] );
}

void KKDCmdFontBold::unexecute( KKDItemBase* item )
{
	((KKDTextBase*)item)->setFontBold( m_before[item] );
}

bool KKDCmdFontBold::begin( KKDItemBase* item )
{
	if( !item->inherits( "KKDesigner::KKDTextBase" ) ) return false;
	m_before[item] = ((KKDTextBase*)item)->fontBold();
	return true;
}

bool KKDCmdFontBold::end( KKDItemBase* item )
{
	bool rtn = ( m_before[item] != ((KKDTextBase*)item)->fontBold() );
	if( rtn )
	{
		m_after[item] = ((KKDTextBase*)item)->fontBold();
	}
	else
	{
		m_before.remove( item );
	}
	return rtn;
}

QString KKDCmdFontBold::propertyName()
{
	return i18n( "bold" );
}
//END KKDCmdFontBold

//BEGIN KKDCmdFontItalic
KKDCmdFontItalic::KKDCmdFontItalic( KKDDoc* doc )
	: KKDCmdBase( doc )
{
	connect( this, SIGNAL( executed() ), doc, SIGNAL( fontItalicChanged() ) );
	connect( this, SIGNAL( unexecuted() ), doc, SIGNAL( fontItalicChanged() ) );
}

void KKDCmdFontItalic::execute( KKDItemBase* item )
{
	((KKDTextBase*)item)->setFontItalic( m_after[item] );
}

void KKDCmdFontItalic::unexecute( KKDItemBase* item )
{
	((KKDTextBase*)item)->setFontItalic( m_before[item] );
}

bool KKDCmdFontItalic::begin( KKDItemBase* item )
{
	if( !item->inherits( "KKDesigner::KKDTextBase" ) ) return false;
	m_before[item] = ((KKDTextBase*)item)->fontItalic();
	return true;
}

bool KKDCmdFontItalic::end( KKDItemBase* item )
{
	bool rtn = ( m_before[item] != ((KKDTextBase*)item)->fontItalic() );
	if( rtn )
	{
		m_after[item] = ((KKDTextBase*)item)->fontItalic();
	}
	else
	{
		m_before.remove( item );
	}
	return rtn;
}

QString KKDCmdFontItalic::propertyName()
{
	return i18n( "italic" );
}
//END KKDCmdFontItalic

//BEGIN KKDCmdFontUnderline
KKDCmdFontUnderline::KKDCmdFontUnderline( KKDDoc* doc )
	: KKDCmdBase( doc )
{
	connect( this, SIGNAL( executed() ), doc, SIGNAL( fontUnderlineChanged() ) );
	connect( this, SIGNAL( unexecuted() ), doc, SIGNAL( fontUnderlineChanged() ) );
}

void KKDCmdFontUnderline::execute( KKDItemBase* item )
{
	((KKDTextBase*)item)->setFontUnderline( m_after[item] );
}

void KKDCmdFontUnderline::unexecute( KKDItemBase* item )
{
	((KKDTextBase*)item)->setFontUnderline( m_before[item] );
}

bool KKDCmdFontUnderline::begin( KKDItemBase* item )
{
	if( !item->inherits( "KKDesigner::KKDTextBase" ) ) return false;
	m_before[item] = ((KKDTextBase*)item)->fontUnderline();
	return true;
}

bool KKDCmdFontUnderline::end( KKDItemBase* item )
{
	bool rtn = ( m_before[item] != ((KKDTextBase*)item)->fontUnderline() );
	if( rtn )
	{
		m_after[item] = ((KKDTextBase*)item)->fontUnderline();
	}
	else
	{
		m_before.remove( item );
	}
	return rtn;
}

QString KKDCmdFontUnderline::propertyName()
{
	return i18n( "underline" );
}
//END KKDCmdFontUnderline

//BEGIN KKDCmdFontStrikeOut
KKDCmdFontStrikeOut::KKDCmdFontStrikeOut( KKDDoc* doc )
	: KKDCmdBase( doc )
{
	connect( this, SIGNAL( executed() ), doc, SIGNAL( fontStrikeOutChanged() ) );
	connect( this, SIGNAL( unexecuted() ), doc, SIGNAL( fontStrikeOutChanged() ) );
}

void KKDCmdFontStrikeOut::execute( KKDItemBase* item )
{
	((KKDTextBase*)item)->setFontStrikeOut( m_after[item] );
}

void KKDCmdFontStrikeOut::unexecute( KKDItemBase* item )
{
	((KKDTextBase*)item)->setFontStrikeOut( m_before[item] );
}

bool KKDCmdFontStrikeOut::begin( KKDItemBase* item )
{
	if( !item->inherits( "KKDesigner::KKDTextBase" ) ) return false;
	m_before[item] = ((KKDTextBase*)item)->fontStrikeOut();
	return true;
}

bool KKDCmdFontStrikeOut::end( KKDItemBase* item )
{
	bool rtn = ( m_before[item] != ((KKDTextBase*)item)->fontStrikeOut() );
	if( rtn )
	{
		m_after[item] = ((KKDTextBase*)item)->fontStrikeOut();
	}
	else
	{
		m_before.remove( item );
	}
	return rtn;
}

QString KKDCmdFontStrikeOut::propertyName()
{
	return i18n( "strikeout" );
}
//END KKDCmdFontStrikeOut

//BEGIN KKDCmdRaise
KKDCmdRaise::KKDCmdRaise( KKDDoc* doc )
	: KKDCmdBase( doc )
{
	connect( this, SIGNAL( executed() ), doc, SIGNAL( positionChanged() ) );
	connect( this, SIGNAL( unexecuted() ), doc, SIGNAL( positionChanged() ) );
}

void KKDCmdRaise::execute( KKDItemBase* item )
{
	m_parent[item]->remove( item );
	m_parent[item]->prepend( item );
	item->slotRepaint();
}

void KKDCmdRaise::unexecute( KKDItemBase* item )
{
	m_parent[item]->remove( item );
	m_parent[item]->insert( m_index[item], item );
	item->slotRepaint();
}

bool KKDCmdRaise::begin( KKDItemBase* item )
{
	if( item->locked() ) return false;
	m_parent[item] = (KKDPaper*)item->parent();
	m_index[item] = m_parent[item]->containsRef( item );
	return true;
}

bool KKDCmdRaise::end( KKDItemBase* item )
{
	if( item->locked() ) return false;
	return item->parent();
}

QString KKDCmdRaise::propertyName()
{
	return i18n( "raise" );
}
//END KKDCmdRaise

//BEGIN KKDCmdLower
KKDCmdLower::KKDCmdLower( KKDDoc* doc )
	: KKDCmdBase( doc )
{
	connect( this, SIGNAL( executed() ), doc, SIGNAL( positionChanged() ) );
	connect( this, SIGNAL( unexecuted() ), doc, SIGNAL( positionChanged() ) );
}

void KKDCmdLower::execute( KKDItemBase* item )
{
	m_parent[item]->remove( item );
	m_parent[item]->append( item );
	item->slotRepaint();
}

void KKDCmdLower::unexecute( KKDItemBase* item )
{
	m_parent[item]->remove( item );
	m_parent[item]->insert( m_index[item], item );
	item->slotRepaint();
}

bool KKDCmdLower::begin( KKDItemBase* item )
{
	if( item->locked() ) return false;
	m_parent[item] = (KKDPaper*)item->parent();
	m_index[item] = m_parent[item]->containsRef( item );
	return true;
}

bool KKDCmdLower::end( KKDItemBase* item )
{
	if( item->locked() ) return false;
	return item->parent();
}

QString KKDCmdLower::propertyName()
{
	return i18n( "lower" );
}
//END KKDCmdLower
