/***************************************************************************
 *   Copyright (C) 2004 by Tasuku Suzuki                                   *
 *   tasuku@linux-life.net                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#define MMPERPIXEL 2.6
#include <math.h>

#include <qfile.h>
#include <qpixmap.h>
#include <qlayout.h>
#include <qdom.h>

#include <klocale.h>
#include <kdebug.h>
#include <kmessagebox.h>

#include "kkview.h"
#include "kkwidget.h"
#include "kkdoc.h"
#include "kkitem.h"

KKView::KKView( QWidget *parent, bool readonly, const char *name )
	: QScrollView( parent, name )
{
	viewport()->setPaletteBackgroundColor( gray );
	QGridLayout* grid = new QGridLayout( viewport(), 3, 3 );

	wgtMain = new QWidget( viewport() );
	grid->addWidget( wgtMain, 1, 1, Qt::AlignCenter );
	addChild( wgtMain );

	m_canvas.setAutoDelete( true );
	m_readonly = readonly;
	m_zoom = 0;
	wgtMain->setBackgroundMode( X11ParentRelative );
	wgtMain->hide();
}

void KKView::setDocument( KKDoc* doc )
{
	m_doc = doc;
	connect( m_doc, SIGNAL( documentChanged() ), this, SLOT( documentChanged() ) );
}

void KKView::preview( const QString &filename )
{
	// To Preview
	// KKView include KKDoc which is not visible
	m_doc = new KKDoc( this );
	setDocument( m_doc );

	QFile file( filename );
	if( !file.open( IO_ReadOnly ) )
	{
		KMessageBox::error( this, i18n( "File open error" ) );
		return;
	}

	QDomDocument doc( "kreetingkard" );
	if( !doc.setContent( &file ) )
	{
		KMessageBox::error( this, i18n( "File is not kreetingkard format" ) );
		return;
	}
	file.close();
	m_doc->setDocument( doc );
	m_doc->clearSelection();
}

void KKView::documentChanged()
{
	while( m_canvas.first() )
	{
		m_canvas.removeFirst();
	}
	wgtMain->hide();
	QListViewItem* item = m_doc->firstChild();
	while( item )
	{
		KKWidget* w = new KKWidget( (KKItem*)item, wgtMain, m_readonly, QString( "wgt%1" ).arg( item->text(0) ) );
		m_canvas.append( w );
		connect( w, SIGNAL( resized() ), this, SLOT( setLayout() ) );
		connect( w, SIGNAL( popup( const QPoint&, bool ) ), this, SIGNAL( popup( const QPoint&, bool ) ) );
		w->show();
		item = item->nextSibling();
	}
	if( m_canvas.count() )
	{
		QListViewItem* item = m_doc->selectedItems().first();
		if( item )
		{
			if( item->parent() ) item = item->parent();
		}
		else
		{
			item = m_doc->firstChild();
		}
		m_canvas.at( item == m_doc->firstChild() ? 0 : 1 )->raise();
		setZoom( 1 );
		wgtMain->show();
	}
}

KKView::~KKView()
{
}

void KKView::setLayout()
{
	QSize size[2] = { m_canvas.at(0)->size(), m_canvas.at(1)->size() };
	double z = zoom();
	if( size[0].width() + size[1].height() != size[0].height() + size[1].width() )
	{
		// different orientation
		wgtMain->setFixedSize( (int)(1480 * z / MMPERPIXEL), (int)(1480 * z / MMPERPIXEL) );
		if( size[0].width() < size[0].height() )
		{
			// 0 - LandScape
			// 1 - Portrait
			m_canvas.at(0)->move( (int)((1480-1000) / 2 / MMPERPIXEL), 0 );
			m_canvas.at(1)->move( 0, (int)((1480-1000) / 2 / MMPERPIXEL) );
		}
		else
		{
			// 0 - LandScape
			// 1 - Portrait
			m_canvas.at(0)->move( 0, (int)((1480-1000) / 2 / MMPERPIXEL) );
			m_canvas.at(1)->move( (int)((1480-1000) / 2 / MMPERPIXEL), 0 );
		}
	}
	else
	{
		if( size[0].width() < size[0].height() )
		{
			// both - LandScape
			wgtMain->setFixedSize( (int)((1000+100) * z / MMPERPIXEL), (int)((1480+100) * z / MMPERPIXEL) );
		}
		else
		{
			// both - Portrait
			wgtMain->setFixedSize( (int)((1480+100) * z / MMPERPIXEL), (int)((1000+100) * z / MMPERPIXEL) );
		}
		m_canvas.at(0)->move( 0, 0 );
		m_canvas.at(1)->move( (int)(100 / MMPERPIXEL), (int)(100 / MMPERPIXEL) );
	}
}

double KKView::zoom()
{
	if( m_zoom < 0 )
	{
		return -1.0 / (double)m_zoom;
	}
	else
	{
		return m_zoom;
	}
}

void KKView::setZoom( int z )
{
	m_zoom = z;
	for( KKWidget* w = m_canvas.first(); w; w = m_canvas.next() )
	{
		w->setZoom( m_zoom );
	}
	emit zoomupEnabledChanged( m_zoom != 7 );
	emit zoomdownEnabledChanged( m_zoom != -4 );
	setLayout();
}

void KKView::zoomIn()
{
	int z = m_zoom + 1;
	if( z == 0 ) z+=2;
	setZoom( z );
}

void KKView::zoomOut()
{
	int z = m_zoom - 1;
	if( z == 0 ) z-=2;
	setZoom( z );
}

void KKView::wheelEvent( QWheelEvent* e )
{
	if( e->delta() > 0 && m_zoom != -4 )
	{
//		zoomOut();
	}
	else if( e->delta() < 0 && m_zoom != 7 )
	{
//		zoomIn();
	}
}

void KKView::insert()
{
	QListViewItem* item = m_doc->selectedItems().first();
	if( item )
	{
		if( item->parent() )  item = item->parent();
	}
	else
	{
		item = m_doc->firstChild();
	}

	m_canvas.at( ( item == m_doc->firstChild() ) ? 0 : 1 )->insert();
}
