/***************************************************************************
 *   Copyright (C) 2004 by Tasuku Suzuki                                   *
 *   tasuku@linux-life.net                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <qregexp.h>
#include <qlayout.h>

#include "kkpropertyitemtextalign.h"

KKPropertyItemTextAlign::KKPropertyItemTextAlign( KKPropertyItemBase* parent, const QString &text )
	: KKPropertyItemBase( parent, text )
{
	setViewPortHeight( 4 );
	editor = new TextAlignEditor( viewport() );
	connect( editor, SIGNAL( accept() ), this, SLOT( accept() ) );
	connect( editor, SIGNAL( abort() ), this, SLOT( abort() ) );
	setVisible( false );
}

KKPropertyItemTextAlign::~KKPropertyItemTextAlign()
{
}

void KKPropertyItemTextAlign::start()
{
	KKPropertyItemBase::start();
	int align = AlignAuto;
	if( text( 1 ).find( QRegExp( "^Top" ) ) != -1 )
	{
		align += AlignTop;
	}
	else if( text( 1 ).find( QRegExp( "^Middle" ) ) != -1 )
	{
		align += AlignVCenter;
	}
	else if( text( 1 ).find( QRegExp( "^Bottom" ) ) != -1 )
	{
		align += AlignBottom;
	}
	if( text( 1 ).find( QRegExp( "Left$" ) ) != -1 )
	{
		align += AlignLeft;
	}
	else if( text( 1 ).find( QRegExp( "Center$" ) ) != -1 )
	{
		align += AlignHCenter;
	}
	else if( text( 1 ).find( QRegExp( "Right$" ) ) != -1 )
	{
		align += AlignRight;
	}

	editor->setAlign( align );
	editor->setFocus();
}

void KKPropertyItemTextAlign::accept()
{
	if( !editor->isVisible() ) return;
	if( editor->align() != 0 )
	{
		document()->setTextAlign( editor->align() );
	}
	KKPropertyItemBase::accept();
}

void KKPropertyItemTextAlign::reset()
{
	KKPropertyItemBase::reset();
	QPtrList<QListViewItem> selectedItems = document()->selectedItems();
	if( selectedItems.count() == 0 )
	{
		setVisible( false );
		return;
	}
	else
	{
		setVisible( ((KKItem*)selectedItems.first())->parent() != NULL );
	}
	unsigned int align = ((KKItem*)selectedItems.first())->textAlign();
	for( KKItem* item = (KKItem*)selectedItems.next(); item; item = (KKItem*)selectedItems.next() )
	{
		if( align != 0 && align != item->textAlign() )
		{
			align = 0;
			break;
		}
	}
	QString strAlign = "";
	switch( align & AlignVertical_Mask )
	{
		case AlignTop:
			strAlign += "Top";
			break;
		case AlignVCenter:
			strAlign += "Middle";
			break;
		case AlignBottom:
			strAlign += "Bottom";
			break;
	}
	switch( align & AlignHorizontal_Mask )
	{
		case AlignLeft:
			strAlign += " Left";
			break;
		case AlignHCenter:
			strAlign += " Center";
			break;
		case AlignRight:
			strAlign += " Right";
			break;
	}
	setText( 1, strAlign );
}

TextAlignEditor::TextAlignEditor( QWidget* parent )
	: QWidget( parent )
{
	(new QGridLayout( this, 3, 3, 2, 2 ))->setAutoAdd( true );
	for( int y = 0; y != 3; y++ )
	{
		for( int x = 0; x != 3; x++ )
		{
			btnAlign[y][x] = new KPushButton( this );
			btnAlign[y][x]->setToggleButton( true );
			if( y != 1 )
			{
				btnAlign[y][x]->setFixedHeight( 24 );
			}
			if( x != 1 )
			{
				btnAlign[y][x]->setFixedWidth( 24 );
			}
			connect( btnAlign[y][x], SIGNAL( toggled( bool ) ), this, SLOT( alignChanged( bool ) ) );
		}
	}
	m_x = -1;
	m_y = -1;
}

TextAlignEditor::~ TextAlignEditor()
{
}

int TextAlignEditor::align()
{
	int a = 0;
	switch( m_x )
	{
		case 0:
			a+= AlignLeft;
			break;
		case 1:
			a+= AlignHCenter;
			break;
		case 2:
			a+= AlignRight;
			break;
	}
	switch( m_y )
	{
		case 0:
			a+= AlignTop;
			break;
		case 1:
			a+= AlignVCenter;
			break;
		case 2:
			a+= AlignBottom;
			break;
	}
	return a;
}

void TextAlignEditor::setAlign( int align )
{
	int x = -1;
	int y = -1;
	switch( align & AlignVertical_Mask )
	{
		case AlignTop:
			y = 0;
			break;
		case AlignVCenter:
			y = 1;
			break;
		case AlignBottom:
			y = 2;
			break;
	}
	switch( align & AlignHorizontal_Mask )
	{
		case AlignLeft:
			x = 0;
			break;
		case AlignHCenter:
			x = 1;
			break;
		case AlignRight:
			x = 2;
			break;
	}
	if( x == -1 || y == -1  )
	{
		if( m_x != -1 && m_y != -1 )
		{
			btnAlign[m_y][m_x]->setOn( false );
		}
	}
	else
	{
		btnAlign[y][x]->setOn( true );
	}
}

void TextAlignEditor::alignChanged( bool bl )
{
	if( !bl )
	{
		m_x = m_y = -1;
		emit abort();
		return;
	}
	int a_x = -1;
	int a_y = -1;
	for( int y = 0; y != 3; y++ )
	{
		for( int x = 0; x != 3; x++ )
		{
			if( btnAlign[y][x]->isOn() )
			{
				if( x == m_x && y == m_y )
				{
					disconnect( btnAlign[y][x], SIGNAL( toggled( bool ) ), this, SLOT( alignChanged( bool ) ) );
					btnAlign[y][x]->setOn( false );
					connect( btnAlign[y][x], SIGNAL( toggled( bool ) ), this, SLOT( alignChanged( bool ) ) );
				}
				else
				{
					a_x = x;
					a_y = y;
				}
			}
		}
	}
	m_x = a_x;
	m_y = a_y;
	emit accept();
}
