/***************************************************************************
 *   Copyright (C) 2004 by Tasuku Suzuki                                   *
 *   tasuku@linux-life.net                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <qlayout.h>
#include <qfile.h>
#include <qlabel.h>
#include <qhbox.h>
#include <qvbox.h>

#include <kmessagebox.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kdebug.h>
#include <kconfig.h>
#include <kstandarddirs.h>
#include <kglobal.h>

#include "dlgaddressbook.h"
#include "zipcode.h"

dlgAddressBook::dlgAddressBook( QWidget* parent, const char* name )
	: KDialogBase( KDialogBase::Plain, i18n("AddressBook"), Ok|Apply|Cancel, Apply, parent, name )
{
	m_currentItem = NULL;
	setupWidgets();
	KConfig *conf = KGlobal::config();
	conf->setGroup("dlgAddressBook");
	if( conf->hasKey( "Split" ) )
	{
		splBase->setSizes( conf->readIntListEntry( "Split" ) );
	}
	if( conf->hasKey( "ColumnWidth" ) )
	{
		QValueList<int> width = conf->readIntListEntry( "ColumnWidth" );
		for( unsigned int i = 0; i != width.count(); i++ )
		{
			tblList->setColumnWidth( i, width[i] );
		}
	}
	resize( configDialogSize( "dlgAddressBook" ) );
	readData();
}

dlgAddressBook::~dlgAddressBook()
{
	KConfig *conf = KGlobal::config();
	conf->setGroup("dlgAddressBook");
	QValueList<int> width;
	for( int i = 0; i != tblList->numCols(); i++ )
	{
		width << tblList->columnWidth( i );
	}
	conf->writeEntry( "ColumnWidth", width );
	conf->writeEntry( "Split", splBase->sizes() );
	saveDialogSize( "dlgAddressBook" );
}

void dlgAddressBook::setupWidgets()
{
	QFrame* page = plainPage();
	splBase = new QSplitter( Horizontal, page );
	(new QHBoxLayout( page ))->setAutoAdd( true );
	splBase->setOpaqueResize( true );

	QVBox* vbox = new QVBox( splBase );
	QHBox* hbox = new QHBox( vbox );
	btnGroupAdd = new KPushButton( SmallIcon( "folder_new" ), "", hbox );
	connect( btnGroupAdd, SIGNAL( clicked() ), this, SLOT( addGroup() ) );
	btnGroupDel = new KPushButton( SmallIcon( "fileclose" ), "", hbox );
	connect( btnGroupDel, SIGNAL( clicked() ), this, SLOT( delGroup() ) );
	new QWidget( hbox );
	btnPersonAdd = new KPushButton( SmallIcon( "filenew" ), "", hbox );
	connect( btnPersonAdd, SIGNAL( clicked() ), this, SLOT( addPerson() ) );
	btnPersonDel = new KPushButton( SmallIcon( "editdelete" ), "", hbox );
	connect( btnPersonDel, SIGNAL( clicked() ), this, SLOT( delPerson() ) );
	new QWidget( hbox );
	lvwLeft = new KListView( vbox );
	lvwLeft->addColumn( "" );
	lvwLeft->header()->hide();
	lvwLeft->setRootIsDecorated( true );
	lvwLeft->setFullWidth( true );
	lvwLeft->setSortColumn( -1 );
	lvwLeft->setAllColumnsShowFocus( true );
	connect( lvwLeft, SIGNAL( selectionChanged( QListViewItem* ) ), this, SLOT( selectionChanged( QListViewItem* ) ) );

	stkRight = new QWidgetStack( splBase );

	tblList = new QTable( stkRight );
	tblList->setNumCols( 4 );
	QStringList list;
	list << i18n( "Direction" ) << i18n( "ZipCode" ) << i18n( "Address" ) << i18n( "Comment" );
	tblList->setColumnLabels( list );
	tblList->setSorting( false );

	vbxDetail = new QVBox( stkRight );
	vbxDetail->setMargin( 10 );
	vbxDetail->setSpacing( 10 );

	QLabel* lbl = new QLabel( i18n( "&Direction" ), vbxDetail );
	txtDirection = new KTextEdit( vbxDetail );
	txtDirection->setTabChangesFocus( true );
	lbl->setBuddy( txtDirection );
	connect( txtDirection, SIGNAL( textChanged() ), this, SLOT( directionChanged() ) );

	lbl = new QLabel( i18n( "&ZipCode" ), vbxDetail );
	 hbox = new QHBox( vbxDetail );
	txtZipCode = new KLineEdit( hbox );
	txtZipCode->setInputMask( "999-9999" );
	connect( txtZipCode, SIGNAL( returnPressed() ), this, SLOT( zipcode2address() ) );
	btnZipCode2Address = new KPushButton( hbox );
	btnZipCode2Address->setPixmap( SmallIcon( "2downarrow" ) );
	btnZipCode2Address->setEnabled( ZipCode::instance().readData() );
	connect( btnZipCode2Address, SIGNAL( clicked() ), this, SLOT( zipcode2address() ) );

	KPushButton* btn = new KPushButton( hbox );
	btn->setPixmap( SmallIcon( "reload" ) );
	connect( btn, SIGNAL( clicked() ), this, SLOT( updateZipCode() ) );

	lbl->setBuddy( txtZipCode );

	lbl = new QLabel( i18n( "&Address" ), vbxDetail );
	txtAddress = new KTextEdit( vbxDetail );
	txtAddress->setTabChangesFocus( true );
	lbl->setBuddy( txtAddress );

	lbl = new QLabel( i18n( "&Comment" ), vbxDetail );
	txtComment = new KTextEdit( vbxDetail );
	txtComment->setTabChangesFocus( true );
	lbl->setBuddy( txtComment );

	setMainWidget( splBase );
}

void dlgAddressBook::slotOk()
{
	if( saveData() ) KDialogBase::slotOk();
}

void dlgAddressBook::slotApply()
{
	if( saveData() ) KDialogBase::slotApply();
}

void dlgAddressBook::readData()
{
	QListViewItem* selectedItem;
	QString datafile = locateLocal( "data", KGlobal::instance()->instanceName() + "/address.xml" );
	QDomDocument doc( "kkaddress" );
	QFile file( datafile );
	if( file.open( IO_ReadOnly ) )
	{
		if( !doc.setContent( &file ) )
		{
			createData( doc );
		}
		file.close();
	}
	else
	{
		createData( doc );
	}

	for( QDomNode np = doc.documentElement().lastChild(); !np.isNull(); np = np.previousSibling() )
	{
		QDomElement group = np.toElement();

		QListViewItem* parent = new QListViewItem( lvwLeft, group.attribute( "title" ) );
		parent->setOpen( group.hasAttribute( "isopen" ) );
		parent->setPixmap( 0, SmallIcon( "folder" ) );
		parent->setRenameEnabled( 0, true );
		if( group.hasAttribute( "isselected" ) ) selectedItem = parent;

		for( QDomNode nc = np.lastChild(); !nc.isNull(); nc = nc.previousSibling() )
		{
			QDomElement person = nc.toElement();
			Address doc;
			doc.direction = person.attribute( "direction" );
			doc.zipcode = person.attribute( "zipcode" );
			doc.address = person.attribute( "address" );
			doc.comment = person.attribute( "comment" );

			QListViewItem* child = new QListViewItem( parent, person.attribute( "direction" ).replace( "\n", " / " ) );
			child->setPixmap( 0, SmallIcon( "kuser" ) );
			if( person.hasAttribute( "isselected" ) ) selectedItem = child;
			m_document[child] = doc;
		}
	}
	if( selectedItem )
	{
		lvwLeft->setSelected( selectedItem, true );
	}
	else
	{
		lvwLeft->setSelected( lvwLeft->firstChild(), true );
	}
}

void dlgAddressBook::createData( QDomDocument& doc )
{
	doc.appendChild( doc.createProcessingInstruction( "xml", "version=\"1.0\" encoding=\"UTF-8\"" ) );
		QDomElement root = doc.createElement( "kkaddress" );
		root.setAttribute( "version", "0.1" );

		QDomElement group = doc.createElement( "group" );
		group.setAttribute( "title", i18n("Friends") );
		group.setAttribute( "isopen", "1" );
		group.setAttribute( "isselected", "1" );
			QDomElement person = doc.createElement( "person" );
			person.setAttribute( "direction", i18n("Ichiro Suzuki\nYumiko\nIkkyu") );
			person.setAttribute( "zipcode", i18n("0000000") );
			person.setAttribute( "address", i18n("Seattle") );
			person.setAttribute( "comment", i18n("Good Job!") );
			group.appendChild( person );

		root.appendChild( group );
	doc.appendChild( root );
}

bool dlgAddressBook::saveData()
{
	cache();
	QString datafile = locateLocal( "data", KGlobal::instance()->instanceName() + "/address.xml" );
	QDomDocument doc( "kkaddress" );
	doc.appendChild( doc.createProcessingInstruction( "xml", "version=\"1.0\" encoding=\"UTF-8\"" ) );
	QDomElement root = doc.createElement( "kkaddress" );
	root.setAttribute( "version", "0.1" );

	QListViewItem* parent = lvwLeft->firstChild();
	while( parent )
	{
		QDomElement group = doc.createElement( "group" );
		group.setAttribute( "title", parent->text( 0 ) );
		if( parent->isOpen() ) group.setAttribute( "isopen", "1" );
		if( parent->isSelected() ) group.setAttribute( "isselected", "1" );
		QListViewItem* child = parent->firstChild();
		while( child )
		{
			QDomElement person = doc.createElement( "person" );
			person.setAttribute( "direction", m_document[child].direction );
			person.setAttribute( "zipcode", m_document[child].zipcode );
			person.setAttribute( "address", m_document[child].address );
			person.setAttribute( "comment", m_document[child].comment );
			if( child->isSelected() ) person.setAttribute( "isselected", "1" );
			group.appendChild( person );
			child = child->nextSibling();
		}
		root.appendChild( group );
		parent = parent->nextSibling();
	}
	doc.appendChild( root );
	QFile file( datafile );
	if ( !file.open( IO_WriteOnly ) ){
		KMessageBox::sorry( this, i18n("Can not open file") );
		return false;
	}
	QTextStream str( &file );
	str << doc;
	file.close();
	return true;
}

void dlgAddressBook::selectionChanged( QListViewItem* item )
{
	if( !item ) return;
	if( item == m_currentItem ) return;
	cache();
	m_currentItem = item;
	if( !item->parent() )
	{
		tblList->setNumRows( item->childCount() );
		int i = 0;
		QListViewItem* child = item->firstChild();
		while( child )
		{
			tblList->setText( i, 0, m_document[child].direction.replace( '\n', " / " ) );
			tblList->setText( i, 1, m_document[child].zipcode.insert( 3, '-' ) );
			tblList->setText( i, 2, m_document[child].address.replace( '\n', " / " ) );
			tblList->setText( i, 3, m_document[child].comment.replace( '\n', " / " ) );
			child = child->nextSibling();
			i++;
		}

		stkRight->raiseWidget( tblList );
	}
	else
	{
		txtDirection->setText( m_document[item].direction );
		txtZipCode->setText( m_document[item].zipcode.insert( 3, '-' ) );
		txtAddress->setText( m_document[item].address );
		txtComment->setText( m_document[item].comment );
		stkRight->raiseWidget( vbxDetail );
	}
}

void dlgAddressBook::cache()
{
	if( !m_currentItem ) return;
	if( !m_currentItem->parent() )
	{
		int i = 0;
		QListViewItem* child = m_currentItem->firstChild();
		while( child )
		{
			m_document[child].direction = tblList->text( i, 0 ).replace( " / ", "\n" );
			m_document[child].zipcode = tblList->text( i, 1 ).remove( '-' );
			m_document[child].address = tblList->text( i, 2 ).replace( " / ", "\n" );
			m_document[child].comment = tblList->text( i, 3 ).replace( " / ", "\n" );
			child = child->nextSibling();
			i++;
		}
	}
	else
	{
		m_document[m_currentItem].direction = txtDirection->text();
		m_document[m_currentItem].zipcode = txtZipCode->text().remove( '-' );
		m_document[m_currentItem].address = txtAddress->text();
		m_document[m_currentItem].comment = txtComment->text();
	}
}

void dlgAddressBook::zipcode2address()
{
	txtAddress->setText( ZipCode::instance()[txtZipCode->text().remove( '-' )] );
	txtAddress->setCursorPosition( 0, txtAddress->text( 0 ).length() - 1 );
	txtAddress->setFocus();
}

void dlgAddressBook::updateZipCode()
{
	btnZipCode2Address->setEnabled( ZipCode::instance().update( this ) );
}

void dlgAddressBook::addGroup()
{
	QListViewItem* item = new QListViewItem( lvwLeft, i18n( "New group" ) );
	item->setPixmap( 0, SmallIcon( "folder" ) );
	item->setRenameEnabled( 0, true );
	item->setOpen( true );
	lvwLeft->setSelected( item, true );
}

void dlgAddressBook::delGroup()
{
	QListViewItem* item = lvwLeft->selectedItem();
	if( item->parent() ) item = item->parent();
	if( KMessageBox::questionYesNo( this, i18n( "delete %1 and including person(s)?" ).arg( item->text( 0 ) ) ) == KMessageBox::Yes )
	{
		lvwLeft->takeItem( item );
	}
	if( lvwLeft->firstChild() )
	{
		lvwLeft->setSelected( lvwLeft->firstChild(), true );
	}
	else
	{
		selectionChanged( 0 );
	}
}

void dlgAddressBook::addPerson()
{
	QListViewItem* group = lvwLeft->selectedItem();
	if( group->parent() ) group = group->parent();
	QListViewItem* item = new QListViewItem( group );
	item->setPixmap( 0, SmallIcon( "kuser" ) );
	lvwLeft->setSelected( item, true );
	txtDirection->setFocus();
}

void dlgAddressBook::delPerson()
{
	QListViewItem* item = lvwLeft->selectedItem();
	QListViewItem* group = item->parent();
	if( !group ) return;
	if( KMessageBox::questionYesNo( this, i18n( "delete %1 ?" ).arg( item->text( 0 ) ) ) == KMessageBox::Yes )
	{
		group->takeItem( item );
	}
	lvwLeft->setSelected( group, true );
}

void dlgAddressBook::directionChanged()
{
	lvwLeft->selectedItem()->setText( 0, txtDirection->text().replace( '\n', " / " ) );
}
