/****************************************************************************
 * KONOHA COPYRIGHT, LICENSE NOTICE, AND DISCRIMER
 *
 * Copyright (c) 2006-2010, Kimio Kuramitsu <kimio at ynu.ac.jp>
 *           (c) 2008-      Konoha Team konohaken@googlegroups.com
 * All rights reserved.
 * You may choose one of the following two licenses when you use konoha.
 * See www.konohaware.org/license.html for further information.
 *
 * (1) GNU Lesser General Public License 3.0 (with K_UNDER_LGPL)
 * (2) Konoha Software Foundation License 1.0
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

#ifdef K_USING_SSA

#define USE_cwb_open    1
#define USE_cwb_size    1
#define USE_STEXT       1

#include"commons.h"

#define VERTEX(n)       ((knh_BasicBlock_t*)knh_Array_n(vertex, n))
#define ANCESTOR(bb)    iArray_n(ancestor, DFN(bb))
#define SEMI(bb)        iArray_n(semi, DFN(bb))
#define BEST(bb)        iArray_n(best, DFN(bb))
#define IDOM(bb)        iArray_n(idom, DFN(bb))
#define SAMEDOM(bb)     iArray_n(samedom, DFN(bb))
#define BUCKET(bb)      ((knh_Array_t*)Array_BB(bucket, bb))
#define PRED(bb)        ((knh_Array_t*)Array_BB(pred, bb))
#define DF(bb)          ((knh_Array_t*)Array_BB(df, bb))
#define DEFSITES(n)     ((knh_Array_t*)knh_Array_n(defsites, n))
#define APHI(n)         ((knh_Array_t*)knh_Array_n(Aphi, n))
#define STACK(n)        ((knh_Array_t*)knh_Array_n(stack, iArray_n(origidx, n)))
#define AORIG(n)        ((knh_Array_t*)knh_Array_n(Aorig, n))
#define PCC(n)          ((knh_Array_t*)knh_Array_n(pcc, n))
#define UNM(n)          ((knh_Array_t*)knh_Array_n(unm, n))
#define LIVEIN(n)       ((knh_Array_t*)knh_Array_n(liveIn, n))
#define LIVEOUT(n)      ((knh_Array_t*)knh_Array_n(liveOut, n))
#define DFN(bb)         (DP(bb)->id)
#define Array_BB(a, bb) knh_Array_n(a, DFN(bb))
#define TOP(a)          iArray_n(a, knh_Array_size(a) - 1)
#define PARENT(bb)      getParent(vertex, bb)
#define PUSH(a, n)      iArray_add(ctx, a, n)
#define iArray_n(a, i)  ((a)->ilist[i])
#define isOBJ(n)        ((n) % 2 == 0 ? 1 : 0)
#define tochar(opcode)  ((opcode == OPCODE_PHI) ? "PHI" : knh_opcode_tochar(opcode))
#define LINK(bb1, bb2)\
	ANCESTOR(bb2) = DFN(bb1);\
	BEST(bb2) = DFN(bb2)
#define SWAP(a, b, size)\
	do {\
		register size_t __size = (size);\
		register char *__a = (char*)(a);\
		register char *__b = (char*)(b);\
		do {\
			char __tmp = *__a;\
			*__a++ = *__b;\
			*__b++ = __tmp;\
		} while (--__size > 0);\
	} while (0)
#define isDEF(opcode, n)  knh_opcode_usedef(opcode, 0)
#define isUSE(opcode, n)  knh_opcode_usedef(opcode, n)
#define isVDEF(opcode, n) ((n == 0 && opcode >= OPCODE_XMOV && opcode <= OPCODE_XBMOV) || \
		((n == 1) && (opcode == OPCODE_OSETIDX || opcode == OPCODE_OSETIDXn || \
		opcode == OPCODE_NSETIDX || opcode == OPCODE_NSETIDXn || opcode == OPCODE_BSETIDX \
		|| opcode == OPCODE_BSETIDXn)))
#define isVUSE(opcode, n) ((n == 1) && ((opcode >= OPCODE_OMOVx && opcode <= OPCODE_bMOVx) || \
		opcode == OPCODE_XMOVx || opcode == OPCODE_OGETIDX || opcode == OPCODE_OGETIDXn \
		|| opcode == OPCODE_NGETIDX || opcode == OPCODE_NGETIDXn || opcode == OPCODE_BGETIDX \
		|| opcode == OPCODE_BGETIDXn))
#define isCALL(opcode)    (opcode == OPCODE_SCALL || opcode == OPCODE_VCALL || \
		opcode == OPCODE_VCALL_ || opcode == OPCODE_CALL)

#if defined(K_USING_THREADEDCODE)
#define TADDR   NULL, 0/*counter*/
#else
#define TADDR   0/*counter*/
#endif/*K_USING_THREADEDCODE*/
#define ASMLINE  0

#define OPCODE_PHI        OPCODE_NOP
typedef struct klr_PHI_t {
	KCODE_HEAD;
	knh_sfpidx_t a;
	knh_sfpidx_t b;
	knh_sfpidx_t c;
} klr_PHI_t;

struct phidata {
	knh_Array_t *pred;
	knh_Array_t *liveIn;
	knh_Array_t *liveOut;
	int max;
};

void knh_BasicBlock_add_(Ctx *ctx, knh_BasicBlock_t *bb, int line, knh_opline_t *op);
knh_BasicBlock_t* new_BasicBlockLABEL(Ctx *ctx);

#ifdef __cplusplus 
extern "C" {
#endif

/* ------------------------------------------------------------------------ */
/* [API] */

/**
 * Add an int value to int array.
 * @param ctx Context
 * @param a   Target Array<int>
 * @param v   Int value
 */
static void iArray_add(Ctx *ctx, knh_Array_t *a, knh_int_t v)
{
	BEGIN_LOCAL(ctx, lsfp, 1);
	lsfp[0].ivalue = v;
	a->api->add(ctx, a, lsfp);
	END_LOCAL(ctx, lsfp);
}

/**
 * Pop the stack.
 * @param ctx   Context
 * @param a     Targe Array<int>
 * @return      A top number of the stack.
 */
static int iArray_pop(Ctx *ctx, knh_Array_t *a)
{
	int v = TOP(a);
	knh_Array_clear(ctx, a, knh_Array_size(a) - 1);
	return v;
}

/**
 * Check an array contains an int object.
 * @param ctx Cotnext
 * @param a   A list of int object
 * @param n   Target int object
 * @return    1 if a contains n
 *            or 0 if a does not contain n.
 */
static knh_bool_t iArray_isContain(knh_Array_t *a, knh_int_t n)
{
	size_t i;
	for (i = 0; i < knh_Array_size(a); i++) {
		if(iArray_n(a, i) == n)
			return 1;
	}
	return 0;
}

static void iArray_insert(Ctx *ctx, knh_Array_t *a, knh_int_t n)
{
	size_t i;
	for (i = 0; i < knh_Array_size(a); i++) {
		if (iArray_n(a, i) == n)
			return;
	}
	iArray_add(ctx, a, n);
}

static void iArray_copy(Ctx *ctx, knh_Array_t *to, knh_Array_t *from)
{
	size_t i;
	for (i = 0; i < knh_Array_size(from); i++) {
		iArray_add(ctx, to, iArray_n(from, i));
	}
}

/* ------------------------------------------------------------------------ */
/* [debug] */

#ifdef K_USING_DEBUG

/**
 * Print BasicBlock.
 * @param ctx Context
 * @param bb  BasicBlock
 */
static void printBB(Ctx *ctx, knh_BasicBlock_t *bb)
{
	size_t i;
	fprintf(stderr, " [%02d] incoming=%02d, visited=%d", DP(bb)->id, DP(bb)->incoming, knh_BasicBlock_isVisited(bb));
	fprintf(stderr, ", next=%02d", (bb->nextNC) ? (int)DFN(bb->nextNC) : (-1));
	fprintf(stderr, ", jump=%02d\n", (bb->jumpNC) ? (int)DFN(bb->jumpNC) : (-1));
	//fprintf(stderr, "| size=%02d, capa=%02d |\n",
	//DP(bb)->size, DP(bb)->capacity);
	for (i = 0; i < DP(bb)->size; i++) {
		knh_opline_t *op = DP(bb)->opbuf + i;
		if (op->opcode == OPCODE_PHI) {
			fprintf(stderr, "L0(?): PHI(%lu) r%d r%d r%d\n", OPCODE_PHI, (int)op->data[0], (int)op->data[1], (int)op->data[2]);
		} else {
			knh_opcode_dump(ctx, op, KNH_STDERR, op);
		}
	}
}

/**
 * Print immediately dominator(idom) and others.
 * @param ctx      Context
 * @param vertex   Linear list of BasicBlocks(~= listNC)
 * @param samedom  It is used if a block has same dominator
 *                 as other block.
 * @param ancestor Ancestor block in Control Flow Graph
 * @param semi     Semidominator of the block
 * @param idom     Immediately dominator of the block
 */
static void printIdom(knh_Array_t *vertex, knh_Array_t *samedom, knh_Array_t *ancestor, knh_Array_t *semi, knh_Array_t *idom)
{
	size_t i;
	for (i = 0; i < knh_Array_size(vertex); i++) {
		fprintf(stderr, "[%02lu] samedom=%02d", i, (int)iArray_n(samedom, i));
		fprintf(stderr, ", ancestor=%02d", (int)iArray_n(ancestor, i));
		fprintf(stderr, ", semi=%02d", (int)iArray_n(semi, i));
		fprintf(stderr, ", idom=%02d\n", (int)iArray_n(idom, i));
	}
}

/**
 * Print Dominance Frontier(DF).
 * @param ctx    Context
 * @param vertex Linear list of BasicBlocks(~= listNC)
 * @param df     Dominance Frontier
 */

static void printLiveness(knh_Array_t *liveIn, knh_Array_t *liveOut)
{
	size_t i, j;
	for (i = 0; i < knh_Array_size(liveIn); i++) {
		fprintf(stderr, "[%02lu] liveIn={", i);
		for (j = 0; j < knh_Array_size(LIVEIN(i)); j++) {
			if (j == 0) {
				fprintf(stderr, "%02d", (int)iArray_n(LIVEIN(i), 0));
			} else {
				fprintf(stderr, ", %02d", (int)iArray_n(LIVEIN(i), j));
			}
		}
		fprintf(stderr, "} liveOut={");
		for (j = 0; j < knh_Array_size(LIVEOUT(i)); j++) {
			if (j == 0) {
				fprintf(stderr, "%02d", (int)iArray_n(LIVEOUT(i), 0));
			} else {
				fprintf(stderr, ", %02d", (int)iArray_n(LIVEOUT(i), j));
			}
		}
		fprintf(stderr, "}\n");
	}
}

static void printTree(Ctx *ctx, knh_Array_t *vertex)
{
	size_t i;
	for (i = 0; i < knh_Array_size(vertex); i++) {
		printBB(ctx, VERTEX(i));
	}
}

static void printArray_Array(knh_Array_t *aa)
{
	size_t i, j;
	for (i = 0; i < knh_Array_size(aa); i++) {
		fprintf(stderr, "[%02lu] {", i);
		for (j = 0; j < knh_Array_size((knh_Array_t*)knh_Array_n(aa, i)); j++) {
			if (j == 0) {
				fprintf(stderr, "%02d", (int)iArray_n((knh_Array_t*)knh_Array_n(aa, i), j));
			} else {
				fprintf(stderr, ", %02d", (int)iArray_n((knh_Array_t*)knh_Array_n(aa, i), j));
			}
		}
		fprintf(stderr, "}\n");
	}
}

static void printArray_Int(knh_Array_t *ai)
{
	size_t i;
	fprintf(stderr, "{");
	for (i = 0; i < knh_Array_size(ai); i++) {
		if (i == 0) {
			fprintf(stderr, "%02d", (int)iArray_n(ai, i));
		} else if (i % 10 == 0) {
			fprintf(stderr, " %02d", (int)iArray_n(ai, i));
		} else if (i % 10 == 9) {
			fprintf(stderr, ", %02d\n", (int)iArray_n(ai, i));
		} else {
			fprintf(stderr, ", %02d", (int)iArray_n(ai, i));
		}
	}
	fprintf(stderr, "}\n");
}

#endif /* K_USING_DEBUG */

/* ------------------------------------------------------------------------ */
/* [convert to SSA form] */

static void addPostBody(Ctx *ctx, knh_BasicBlock_t *bb, knh_BasicBlock_t *bbN)
{
	knh_BasicBlock_t bbtmp;
	knh_BasicBlock_t *bbNEW = new_BasicBlockLABEL(ctx);
	// swap Block
	bbtmp = *bbN;
	*bbN = *bbNEW;
	*bbNEW = bbtmp;
	// add pred
	//knh_Array_add(ctx, pred, new_Array(ctx, CLASS_BasicBlock, 0));
//	bbN->jumpNC = bbNEW;
	if (bb->nextNC == bbN) {
		bbN->jumpNC = bbNEW;
		bb->nextNC = bbNEW;
	} else {
		bbN->nextNC = bbNEW;
		bb->jumpNC = bbNEW;
	}
	DP(bbN)->incoming = 2;
	knh_BasicBlock_setVisited(bbN, 1);
}

/**
 * Do depth-first search(DFS) and number each block with
 * depth-first number(dfnum).
 * @param ctx    Context
 * @param p      Parent block
 * @param n      Target block
 * @param vertex Linear list of BasicBlocks(~= listNC)
 * @param pred   Predecessor of the block
 */
static void depthFirstSearch(Ctx *ctx, knh_BasicBlock_t *p, knh_BasicBlock_t *n, knh_Array_t *vertex, knh_Array_t *pred)
{
	knh_BasicBlock_t *bbN, *bbJ;
	knh_BasicBlock_setVisited(n, 0);
	DP(n)->id = knh_Array_size(vertex);
	knh_Array_add(ctx, vertex, n);
	bbN = n->nextNC;
	bbJ = n->jumpNC;
	if (p != NULL)
		iArray_add(ctx, PRED(n), DFN(p));
	if (bbN != NULL) {
		if (knh_BasicBlock_isVisited(bbN))
			depthFirstSearch(ctx, n, bbN, vertex, pred);
		else
			iArray_add(ctx, PRED(bbN), DFN(n));
	}
	if (bbJ != NULL) {
		if (knh_BasicBlock_isVisited(bbJ))
			depthFirstSearch(ctx, n, bbJ, vertex, pred);
		else
			iArray_add(ctx, PRED(bbJ), DFN(n));
	}
}

/**
 * Get ancestor block with lowest Semidominator.
 * @param ctx      Context
 * @param vertex   Linear list of BasicBlocks(~= listNC)
 * @param v        Target block
 * @param ancestor Ancestor block in Control Flow Graph
 * @param best     A node with lowest dfnum from ancestor to v
 * @param semi     Semidominator of the block
 * @return         The ancestor block with lowest semidominator
 */
static knh_BasicBlock_t* getAncestorWLS(Ctx *ctx, knh_Array_t *vertex, knh_BasicBlock_t *v, knh_Array_t* ancestor, knh_Array_t* best, knh_Array_t* semi)
{
	knh_BasicBlock_t *a = VERTEX(ANCESTOR(v));
	knh_BasicBlock_t *b = NULL;
	if (ANCESTOR(a) != -1) {
		b = getAncestorWLS(ctx, vertex, a, ancestor, best, semi);
		ANCESTOR(v) = ANCESTOR(a);
		if (SEMI(b) < iArray_n(semi, BEST(v)))
			BEST(v) = DFN(b);
	}
	return VERTEX(BEST(v));
}

/**
 * Get parent BasicBlock.
 * @param vertex Linear list of BasicBlocks(~= listNC)
 * @param n      Target block
 * @return       Parent block if it exists
 *               or NULL if it does not exists.
 */
static knh_BasicBlock_t* getParent(knh_Array_t *vertex, knh_BasicBlock_t *n)
{
	size_t i;
	for (i = 0; i < knh_Array_size(vertex); i++) {
		if (VERTEX(i)->nextNC == n || VERTEX(i)->jumpNC == n)
			return VERTEX(i);
	}
	return NULL;
}

/**
 * Set immediately dominator(IDOM) to each BasicBlock.
 * @param ctx    Context
 * @param vertex Linear list of BasicBlocks(~= listNC)
 * @param idom   Immediately dominator of the block
 * @param pred   Predecessor of the block
 */
static void setIdom(Ctx *ctx, knh_Array_t *vertex, knh_Array_t* idom, knh_Array_t* pred)
{
	size_t i, j;
	size_t size = knh_Array_size(vertex);

	BEGIN_LOCAL(ctx, lsfp, 5);
	LOCAL_NEW(ctx, lsfp, 0, knh_Array_t*, bucket, new_Array(ctx, CLASS_Array, size));
	LOCAL_NEW(ctx, lsfp, 1, knh_Array_t*, semi, new_Array(ctx, CLASS_Int, size));
	LOCAL_NEW(ctx, lsfp, 2, knh_Array_t*, ancestor, new_Array(ctx, CLASS_Int, size));
	LOCAL_NEW(ctx, lsfp, 3, knh_Array_t*, samedom, new_Array(ctx, CLASS_Int, size));
	LOCAL_NEW(ctx, lsfp, 4, knh_Array_t*, best, new_Array(ctx, CLASS_Int, size));
	knh_BasicBlock_t *n, *p, *s, *_s, *v, *y;

	for (i = 0; i < size; i++) {
		knh_Array_add(ctx, bucket, new_Array(ctx, CLASS_Int, 0));
		iArray_add(ctx, semi, -1);
		iArray_add(ctx, ancestor, -1);
		iArray_add(ctx, samedom, -1);
		iArray_add(ctx, best, -1);
	}

	for (i = size - 1; i > 0; i--) {
		n = VERTEX(i);
		p = PARENT(n);
		s = p;
		for (j = 0; j < knh_Array_size(PRED(n)); j++) {
			v = VERTEX(iArray_n(PRED(n), j));
			//DBG_P("pred[%d]=%d", j, DFN(PRED(n, j)));
			if (DFN(v) <= DFN(n)) {
				_s = v;
			} else {
				_s = VERTEX(SEMI(getAncestorWLS(ctx, vertex, v, ancestor, best, semi)));
			}
			if (DFN(_s) < DFN(s))
				s = _s;
		}
		SEMI(n) = DFN(s);
		iArray_insert(ctx, BUCKET(s), DFN(n));
		//DBG_P("bucketsize[%02d] = %02d", DFN(s), knh_Array_size(BUCKET(s)));
		LINK(p, n);
		for (j = 0; j < knh_Array_size(BUCKET(p)); j++) {
			v = VERTEX(iArray_n(BUCKET(p), j));
			//DBG_P("bucket[%d]=%d", j, DFN(BUCKET(p, j)));
			y = getAncestorWLS(ctx, vertex, v, ancestor, best, semi);
			if (SEMI(y) == SEMI(v)) {
				IDOM(v) = DFN(p);
			} else {
				SAMEDOM(v) = DFN(y);
			}
		}
		knh_Array_clear(ctx, BUCKET(p), 0);
		//DBG_P("bucketsize[%02d] = %02d", DFN(p), knh_Array_size(BUCKET(p)));
	}
	for (i = 1; i < size; i++) {
		n = VERTEX(i);
		if (SAMEDOM(n) != -1)
			IDOM(n) = IDOM(VERTEX(SAMEDOM(n)));
	}
#ifdef K_USING_DEBUG
	//printIdom(vertex, samedom, ancestor, semi, idom);
#endif
	END_LOCAL(ctx, lsfp);
}

/**
 * Check a block is dominated by another block.
 * @param ctx  Context
 * @param n    Target block
 * @param w    A block that might be dominate the target block
 * @param idom Immediately dominator of the block
 * @return     1 if n is dominated by w
 *             or 0 if n is not dominated by w
 */
static knh_bool_t isDominated(knh_Array_t *vertex, knh_BasicBlock_t *n, knh_BasicBlock_t *w, knh_Array_t* idom)
{
	knh_int_t tmp;
	for (tmp = IDOM(w); tmp != -1; tmp = IDOM(VERTEX(tmp))) {
		if (tmp == DFN(n))
			return 1;
	}
	return 0;
}

/**
 * Compute Dominance Frontier(DF).
 * @param ctx    Context
 * @param vertex Linear list of BasicBlocks(~= listNC)
 * @param n      Target block
 * @param df     Dominance Frontier of the block
 * @param idom   Immediately dominator of the block
 */
static void setDF(Ctx *ctx, knh_Array_t *vertex, knh_BasicBlock_t *n, knh_Array_t *df, knh_Array_t *idom)
{
	size_t i, j;
	size_t size = knh_Array_size(vertex);
	knh_BasicBlock_t *w, *c;
	BEGIN_LOCAL(ctx, lsfp, 1);
	LOCAL_NEW(ctx, lsfp, 0, knh_Array_t*, S, new_Array(ctx, CLASS_Int, 0));
	//DBG_P("S = %p", S);
	knh_BasicBlock_t *y;
	y = n->nextNC;
	if (y != NULL) {
		if (IDOM(y) != DFN(n))
			iArray_insert(ctx, S, DFN(y));
	}
	y = n->jumpNC;
	if (y != NULL) {
		if (IDOM(y) != DFN(n))
			iArray_insert(ctx, S, DFN(y));
	}
	for (i = 0; i < size; i++) {
		if (IDOM(VERTEX(i)) == DFN(n)) {
			c = VERTEX(i);
			setDF(ctx, vertex, c, df, idom);
			for (j = 0; j < knh_Array_size(DF(c)); j++) {
				w = VERTEX(iArray_n(DF(c), j));
				if (!isDominated(vertex, n, w, idom))
					iArray_insert(ctx, S, DFN(w));
			}
		}
	}
	iArray_copy(ctx, DF(n), S);
	END_LOCAL(ctx, lsfp);
}

/**
 * Insert phi function.
 * @param ctx  Context
 * @param n    Target block
 * @param idx  An index of register
 * @param argc A number of Predecessor
 */
static void insertOpline(Ctx *ctx, knh_BasicBlock_t *n, knh_opline_t *op, size_t line)
{
	size_t size = DP(n)->size - line;
	if (size > 0) {
		knh_opline_t *buf;
		size_t bsize = size * sizeof(knh_opline_t);
		knh_cwb_t cwbbuf, *cwb = knh_cwb_open(ctx, &cwbbuf);
		knh_BasicBlock_add_(ctx, n, 0, op);
		buf = DP(n)->opbuf + line;
		knh_Bytes_write(ctx, cwb->ba, new_bytes2((char *)op, sizeof(knh_opline_t)));
		knh_Bytes_write(ctx, cwb->ba, new_bytes2((char *)buf, bsize));
		knh_memcpy(buf, knh_cwb_tochar(ctx, cwb), knh_cwb_size(cwb));
		knh_cwb_close(cwb);
	} else {
		knh_BasicBlock_add_(ctx, n, 0, op);
	}
}

/**
 * Set a list of variable that defined each block n.
 * @param ctx   Context
 * @param Aorig A list of variable that defined each block
 * @param n     Target block
 */
static void setAorig(Ctx *ctx, knh_Array_t *Aorig, knh_BasicBlock_t *n/*, int *max*/)
{
	int i, size;
	knh_opline_t *op;
	size = DP(n)->size;
	for (i = 0; i < size; i++) {
		op = DP(n)->opbuf + i;
		if (isDEF(op->opcode, 0)) {
			if (op->opcode == OPCODE_iINC) {
				op->opcode = OPCODE_iADDn;
				op->data[1] = op->data[0];
				op->data[2] = 1;
			} else if (op->opcode == OPCODE_iDEC) {
				op->opcode = OPCODE_iSUBn;
				op->data[1] = op->data[0];
				op->data[2] = 1;
			} else if (op->opcode == OPCODE_ONMOV) {
				klr_OMOV_t omov = {TADDR, OPCODE_OMOV, ASMLINE, op->data[0], op->data[1]};
				insertOpline(ctx, n, (knh_opline_t*)&omov, i);
				DBG_P("Aorig[%02d] add r%d %s", DFN(n), op->data[0], tochar(op->opcode));
				iArray_add(ctx, Aorig, op->data[0]);
				i += 1;
				op = DP(n)->opbuf + i;
				op->data[0] = op->data[2];
				op->data[1] = op->data[3];
				op->opcode = OPCODE_NMOV;
			} else if (op->opcode == OPCODE_OOMOV) {
				klr_OMOV_t omov = {TADDR, OPCODE_OMOV, ASMLINE, op->data[0], op->data[1]};
				insertOpline(ctx, n, (knh_opline_t*)&omov, i);
				DBG_P("Aorig[%02d] add r%d %s", DFN(n), op->data[0], tochar(op->opcode));
				iArray_add(ctx, Aorig, op->data[0]);
				i += 1;
				op = DP(n)->opbuf + i;
				op->data[0] = op->data[2];
				op->data[1] = op->data[3];
				op->opcode = OPCODE_OMOV;
			} else if (op->opcode == OPCODE_NNMOV) {
				klr_NMOV_t nmov = {TADDR, OPCODE_NMOV, ASMLINE, op->data[0], op->data[1]};
				insertOpline(ctx, n, (knh_opline_t*)&nmov, i);
				DBG_P("Aorig[%02d] add r%d %s", DFN(n), op->data[0], tochar(op->opcode));
				iArray_add(ctx, Aorig, op->data[0]);
				i += 1;
				op = DP(n)->opbuf + i;
				op->data[0] = op->data[2];
				op->data[1] = op->data[3];
				op->opcode = OPCODE_NMOV;
//			} else if (op->opcode == OPCODE_TR) {
//				if (*max < op->data[1] + 1)
//					*max = op->data[1] + 1;
			}
			if ((int)op->data[0] >= 0) {
				DBG_P("Aorig[%02d] add r%d %s", DFN(n), op->data[0], tochar(op->opcode));
				iArray_add(ctx, Aorig, op->data[0]);
//				if (op->opcode == OPCODE_ONMOV || op->opcode == OPCODE_OOMOV || op->opcode == OPCODE_NNMOV) {
//					DBG_P("Aorig[%02d] add r%d %s", DFN(n), op->data[2], tochar(op->opcode));
//					iArray_add(ctx, Aorig, op->data[2]);
//				}
			}
		}
	}
}

/**
 * Put phi function to BasicBlock.
 * @param ctx    Context
 * @param vertex Linear list of BasicBlocks(~= listNC)
 * @param df     Dominance Frontier of the block
 * @param max    Max index of register
 */
static void putPhiFunc(Ctx *ctx, knh_Array_t *vertex, knh_Array_t* df, int *max)
{
	int a;
	size_t i, j;
	size_t size = knh_Array_size(vertex);
	knh_BasicBlock_t *n, *y;
	BEGIN_LOCAL(ctx, lsfp, 4);
	LOCAL_NEW(ctx, lsfp, 0, knh_Array_t*, Aorig, new_Array(ctx, CLASS_Array, size));
	LOCAL_NEW(ctx, lsfp, 1, knh_Array_t*, Aphi, new_Array(ctx, CLASS_Array, 0));
	LOCAL_NEW(ctx, lsfp, 2, knh_Array_t*, defsites, new_Array(ctx, CLASS_Array, 0));
	LOCAL_NEW(ctx, lsfp, 3, knh_Array_t*, W, new_Array(ctx, CLASS_Int, 0));
	for (i = 0; i < size; i++) {
		n = VERTEX(i);
		knh_Array_add(ctx, Aorig, new_Array(ctx, CLASS_Int, 0));
		if (DP(n)->size > 0) {
			setAorig(ctx, AORIG(i), n/*, max*/);
			for (j = 0; j < knh_Array_size(AORIG(i)); j++) {
				a = iArray_n(AORIG(i), j);
				while ((int)knh_Array_size(defsites) <= a) {
					//DBG_P("add%d", a);
					knh_Array_add(ctx, defsites, new_Array(ctx, CLASS_Int, 1));
				}
				iArray_insert(ctx, DEFSITES(a), DFN(n));
			}
		}
	}
	if ((int)knh_Array_size(defsites) > *max)
		*max = (int)knh_Array_size(defsites);
	DBG_P("max=%d", *max);
	for (a = 0; a < (int)knh_Array_size(defsites); a++) {
		knh_Array_add(ctx, Aphi, new_Array(ctx, CLASS_Int, 0));
		if (knh_Array_size(DEFSITES(a)) == 0)
			continue;
		iArray_copy(ctx, W, DEFSITES(a));
		while (knh_Array_size(W) > 0) {
			n = VERTEX(iArray_pop(ctx, W));
			for (i = 0; i < knh_Array_size(DF(n)); i++) {
				y = VERTEX(iArray_n(DF(n), i));
				if (!iArray_isContain(APHI(a), DFN(y))) {
					DBG_ASSERT(DP(y)->incoming == 2);
					DBG_P("[%02d] insert phifunc of r%d", DFN(y), a);
					klr_PHI_t phi = {TADDR, OPCODE_PHI, ASMLINE, a, a, a};
					insertOpline(ctx, y, (knh_opline_t*)&phi, 0);
					iArray_insert(ctx, APHI(a), DFN(y));
					setAorig(ctx, AORIG(DFN(y)), y/*, max*/);
					if (!iArray_isContain(AORIG(DFN(y)), a))
						iArray_insert(ctx, W, DFN(y));
				}
			}
		}
	}
	END_LOCAL(ctx, lsfp);
}

/**
 * Get Predecessor number
 * @param pred Predecessor of the block
 * @param p    Parent block
 * @param n    Target block
 * @return     An index of p in pred.
 */
static int getPredNum(knh_Array_t *pred, knh_BasicBlock_t *p, knh_BasicBlock_t *n)
{
	size_t i;
	for (i = 0; i < knh_Array_size(PRED(n)); i++) {
		if (iArray_n(PRED(n), i) == DFN(p))
			return i;
	}
	DBG_ASSERT(0);
	return -1;
}

static void extendLiveness(Ctx *ctx, knh_Array_t *liveIn, knh_Array_t *liveOut, int idx, knh_BasicBlock_t *n)
{
	int i;
	for (i = DFN(n); i > 0; i--) {
		if (iArray_isContain(LIVEIN(i), idx)) {
			//DBG_P("extend r%d from [%02d] to in[%02d]", idx, DFN(n), i);
			return;
		} else {
			iArray_add(ctx, LIVEIN(i), idx);
		}
		if (iArray_isContain(LIVEOUT(i - 1), idx)) {
			//DBG_P("extend r%d from [%02d] to out[%02d]", idx, DFN(n), i - 1);
			return;
		} else {
			iArray_add(ctx, LIVEOUT(i - 1), idx);
		}
	}
	iArray_add(ctx, LIVEIN(0), idx);
	//DBG_P("extend r%d from [%02d] to begin[%02d]", idx, DFN(n), 0);
}

static void copyArgs(Ctx *ctx, knh_Array_t *stack, knh_Array_t *origidx, knh_BasicBlock_t *n, knh_opline_t *s, int *idx, size_t *pos)
{
	size_t i, shift, min, max;
	int newName;
	knh_opline_t *op;
	if (s->opcode == OPCODE_SCAST/*s->opcode == OPCODE_TR*/) {
		min = s->data[1];
		max = s->data[1] + 2;
		shift = *idx - (s->data[1] - 2);
		*idx += 4;
	} else {
		min = s->data[1];
		max = s->data[2];
		shift = *idx - (s->data[1] - 6);
	}
	DBG_P("shift=%d", shift);
	for (i = min; i < max; i++) {
		if (iArray_n(origidx, i) != -1 && knh_Array_size(STACK(i)) > 1 && knh_Array_size(stack) > (size_t)i) {
//	if (isOBJ(s->data[0] + *idx)) {
//		*idx += 2;
//		iArray_add(ctx, origidx, -1);
//	} else {
//		*idx += 1;
//	}
//	//DBG_P("s->data[0]=%d, esp=%d", s->data[0], ctx->esp->data);
//	iArray_add(ctx, origidx, s->data[0]);
//	DBG_P("replace DEF %s %d to %d", tochar(s->opcode), s->data[0], *idx);
//	PUSH(STACK(s->data[0]), *idx);
//	s->data[0] = *idx;
			if (isOBJ(i)) {
				newName = i + shift;
				while ((int)knh_Array_size(origidx) < newName) {
					iArray_add(ctx, origidx, -1);
				}
				iArray_add(ctx, origidx, i);
				iArray_add(ctx, origidx, -1);
//				while (knh_Array_size(stack) <= i) {
//					knh_Array_add(ctx, stack, new_Array(ctx, CLASS_Int, 1));
//					iArray_add(ctx, (knh_Array_t*)knh_Array_n(stack, knh_Array_size(stack) - 1), knh_Array_size(stack) - 1);
//				}
//				iArray_n(origidx, i + 1) = i;
//				DBG_P("i=%d", i);
				DBG_P("insert OMOV %d to %d", TOP(STACK(i)), newName);
				klr_OMOV_t omov = {TADDR, OPCODE_OMOV, ASMLINE, newName, TOP(STACK(i))};
				insertOpline(ctx, n, (knh_opline_t*)&omov, *pos);
				PUSH(STACK(i), newName);
			} else {
				DBG_P("insert NMOV %d to %d", TOP(STACK(i)), i + shift);
				klr_NMOV_t nmov = {TADDR, OPCODE_NMOV, ASMLINE, i + shift, TOP(STACK(i))};
				insertOpline(ctx, n, (knh_opline_t*)&nmov, *pos);
			}
			*pos += 1;
		}
	}
	*idx = knh_Array_size(origidx);
	op = DP(n)->opbuf + *pos;
	if (op->opcode == OPCODE_SCAST/*op->opcode == OPCODE_TR*/) {
		DBG_P("shift USE %s %d to %d", tochar(op->opcode), op->data[1], op->data[1] + shift);
		op->data[1] += shift;
	} else {
		for (i = 1; i <= 2; i++) {
			DBG_P("shift USE %s %d to %d", tochar(op->opcode), op->data[i], op->data[i] + shift);
			op->data[i] += shift;
		}
	}
	if (op->data[0] >= 0) {
		if (op->opcode == OPCODE_SCAST/*op->opcode == OPCODE_TR*/)
			shift = op->data[1] - 2 - op->data[0];
		newName = op->data[0] + shift;
		if ((int)knh_Array_size(origidx) <= newName) {
			if (isOBJ(op->data[0])) {
				iArray_add(ctx, origidx, op->data[0]);
				iArray_add(ctx, origidx, -1);
			} else {
				iArray_add(ctx, origidx, -1);
				iArray_add(ctx, origidx, op->data[0]);
			}
			*idx += 2;
			DBG_ASSERT(*idx == (int)knh_Array_size(origidx));
		}
		DBG_P("replace DEF %s %d to %d", tochar(op->opcode), op->data[0], newName);
		PUSH(STACK(op->data[0]), newName);
		op->data[0] = newName;
	}
}

static void replaceUse(knh_Array_t *stack, knh_Array_t *origidx, knh_opline_t *s, int pos)
{
	size_t i, shift;
	knh_opline_t *op;
	if (s->opcode == OPCODE_P) {
		if (s->data[4] != 0) {
			op = s - 1;
			i = isOBJ(op->data[0]) ? op->data[0] / 2 : (op->data[0] - 1) / 2;
			DBG_P("replace USE %s %d to %d", tochar(s->opcode), s->data[4], i);
			s->data[4] = i;
			//extendLiveness(ctx, liveIn, liveOut, s->data[4] * 2 + 1, n);
		} else {
			DBG_P("Don't replace USE %s %d", tochar(s->opcode), s->data[4]);
		}
	} else if (pos > 0 && (s->opcode == OPCODE_FASTCALL0/* || s->opcode == OPCODE_SCAST*/ || s->opcode == OPCODE_TR)) {
		op = s - 1;
		if (isDEF(op->opcode, 0) && op->data[0] >= 0) {
			if (iArray_n(origidx, op->data[0]) >= s->data[1] && iArray_n(origidx, op->data[0]) <= s->data[1] + 4) {
				shift = op->data[0] - iArray_n(origidx, op->data[0]);
				DBG_P("shift USE %s %d to %d", tochar(s->opcode), s->data[1], s->data[1] + shift);
				s->data[1] += shift;
			} else {
				DBG_P("replace USE %s %d to %d", tochar(s->opcode), s->data[1], TOP(STACK(s->data[1])));
				s->data[1] = TOP(STACK(s->data[1]));
			}
		} else if (s->data[1] >= 0) {
			DBG_P("replace USE %s %d to %d", tochar(s->opcode), s->data[1], TOP(STACK(s->data[1])));
			s->data[1] = TOP(STACK(s->data[1]));
		}
//	} else if (s->opcode == OPCODE_TR) {
//		if (s->data[1] >= 0) {
//			DBG_P("replace USE %s %d to %d", tochar(s->opcode), s->data[1], TOP(STACK(s->data[1])));
//			s->data[1] = TOP(STACK(s->data[1]));
//		}
	} else {
		size_t osize = knh_opcode_size(s->opcode);
		for (i = 1; i < osize; i++) {
			if (isUSE(s->opcode, i)) {
				DBG_P("replace USE %s %d to %d", tochar(s->opcode), s->data[i], TOP(STACK(s->data[i])));
				s->data[i] = TOP(STACK(s->data[i]));
				//extendLiveness(ctx, liveIn, liveOut, s->data[j], n);
			} else if (isVUSE(s->opcode, i)) {
				DBG_P("replace VUSE %s %d to %d", tochar(s->opcode), s->data[i], TOP(STACK(s->data[i])));
				s->data[i] = TOP(STACK(s->data[i]));
			}
		}
		if (isVDEF(s->opcode, 0)) {
			DBG_P("replace VDEF %s %d to %d", tochar(s->opcode), s->data[0], TOP(STACK(s->data[0])));
			s->data[0] = TOP(STACK(s->data[0]));
		}
	}
}

static void replaceDef(Ctx *ctx, knh_Array_t *stack, knh_Array_t *origidx, knh_opline_t *s, int *idx)
{
	int newName;
	if (isOBJ(s->data[0])) {
		iArray_add(ctx, origidx, s->data[0]);
		iArray_add(ctx, origidx, -1);
		newName = *idx;
	} else {
		iArray_add(ctx, origidx, -1);
		iArray_add(ctx, origidx, s->data[0]);
		newName = *idx + 1;
	}
	//DBG_P("s->data[0]=%d, esp=%d", s->data[0], ctx->esp->data);
	DBG_P("replace DEF %s %d to %d", tochar(s->opcode), s->data[0], newName);
	PUSH(STACK(s->data[0]), newName);
	s->data[0] = newName;
	*idx += 2;
}

static void replaceRix(knh_opline_t *s)
{
//	//if (s->opcode == opcode_phi) {
//		//dbg_p("define r%d in [%02d]", (int)s->data[0], dfn(n));
//		//iArray_add(ctx, LIVEIN(DFN(n)), s->data[0]);
//	//} else if (s->opcode == OPCODE_FASTCALL0 || s->opcode == OPCODE_SCAST || s->opcode == OPCODE_TR) {
	knh_intptr_t newrix = isOBJ(s->data[0] - s->data[1]) ? (s->data[0] - s->data[1]) / 2 : (s->data[0] - s->data[1] - 1) / 2;
	DBG_P("replace RIX %s %d to %d", tochar(s->opcode), s->data[2], newrix);
	s->data[2] = newrix;
	//}
}
/**
 * Rename variable to make SSA form.
 * @param ctx    Cotnext
 * @param vertex Linear list of BasicBlocks(~= listNC)
 * @param n      Target block
 * @param stack  A stack that records rename in each variable
 * @param idx    An index of register
 * @param pred   Predecessor of the block
 * @param idom   Immediately dominator of the block
 */
static void renameVars(Ctx *ctx, knh_Array_t *vertex, knh_BasicBlock_t *n, knh_Array_t *stack, knh_Array_t *origidx, knh_Array_t *pred, knh_Array_t *idom, knh_Array_t *liveIn, knh_Array_t *liveOut, int *idx)
{
	size_t i, j;
	knh_opline_t *s;
	knh_BasicBlock_t *x ,*y;
	for (i = 0; i < DP(n)->size; i++) {
		s = DP(n)->opbuf + i;
		if (isCALL(s->opcode) || s->opcode == OPCODE_SCAST/* || (s->opcode == OPCODE_TR && s->data[1] > 0)*/) {
			copyArgs(ctx, stack, origidx, n, s, idx, &i);
			s = DP(n)->opbuf + i;
			if (s->opcode == OPCODE_SCAST/*s->opcode == OPCODE_TR*/) {
				replaceRix(s);
			}
		} else {
			if (s->opcode != OPCODE_PHI) {
				replaceUse(stack, origidx, s, i);
			}
			if (isDEF(s->opcode, 0) || s->opcode == OPCODE_PHI) {
				if (s->data[0] >= 0) {
					replaceDef(ctx, stack, origidx, s, idx);
				} else {
					DBG_P("Don't replace DEF %s %d", tochar(s->opcode), s->data[0]);
				}
				if (s->opcode == OPCODE_FASTCALL0/* || s->opcode == OPCODE_SCAST*/ || s->opcode == OPCODE_TR) {
					replaceRix(s);
				}
			}
		}
	}
	DBG_P("start %02d", DFN(n));
#ifdef K_USING_DEBUG
	printArray_Int(origidx);
	printArray_Array(stack);
#endif
	y = n->nextNC;
	if (y != NULL) {
		j = getPredNum(pred, n, y) + 1;
		for (i = 0; i < DP(y)->size; i++) {
			s = DP(y)->opbuf + i;
			if (s->opcode == OPCODE_PHI) {
				DBG_P("replace USE %s %d to %d", tochar(s->opcode), s->data[j], TOP(STACK(s->data[j])));
				s->data[j] = TOP(STACK(s->data[j]));
//				extendLiveness(ctx, vertex, liveIn, liveOut, s->data[j], y);
				//DBG_P("phi source r%d is live at end of [%02d]", s->data[j], DFN(n));
				//iArray_add(ctx, LIVEOUT(DFN(n)), (int)s->data[j]);
			} else {
				break;
			}
		}
	}
	y = n->jumpNC;
	if (y != NULL) {
		j = getPredNum(pred, n, y) + 1;
		for (i = 0; i < DP(y)->size; i++) {
			s = DP(y)->opbuf + i;
			if (s->opcode == OPCODE_PHI) {
				DBG_P("replace USE %s %d to %d", tochar(s->opcode), s->data[j], TOP(STACK(s->data[j])));
				s->data[j] = TOP(STACK(s->data[j]));
			} else {
				break;
			}
		}
	}
	for (i = 0; i < knh_Array_size(vertex); i++) {
		x = (knh_BasicBlock_t*)VERTEX(i);
		if (IDOM(x) == DFN(n)) {
			renameVars(ctx, vertex, x, stack, origidx, pred, idom, liveIn, liveOut, idx);
		}
	}
	for (i = DP(n)->size - 1; (int)i >= 0; i--) {
		s = DP(n)->opbuf + i;
		if (isDEF(s->opcode, 0) || s->opcode == OPCODE_PHI) {
			if ((int)s->data[0] > 0 && (int)origidx->ilist[s->data[0]] != -1) {
				DBG_P("pop %02d", s->data[0]);
				DBG_ASSERT(knh_Array_size(STACK(s->data[0])) > 1);
				iArray_pop(ctx, STACK(s->data[0]));
				origidx->ilist[s->data[0]] = -1;
			} else {
				DBG_P("don't pop %d", s->data[0]);
			}
			if (isCALL(s->opcode)) {
				knh_intptr_t min = s->data[1];
				knh_intptr_t max = s->data[2];
				s = DP(n)->opbuf + i - 1;
				while ((int)i > 0 && isDEF(s->opcode, 0) && s->data[0] >= min && s->data[0] < max) {
					if (isOBJ(s->data[0])) {
						DBG_P("pop %02d", s->data[0]);
						DBG_ASSERT(knh_Array_size(STACK(s->data[0])) > 1);
						iArray_pop(ctx, STACK(s->data[0]));
						origidx->ilist[s->data[0]] = -1;
					} else {
						DBG_P("skip pop args %d", s->data[0]);
					}
					i--;
					s = DP(n)->opbuf + i - 1;
				}
			} else if (s->opcode == OPCODE_SCAST/*s->opcode == OPCODE_TR*/) {
				knh_intptr_t min = s->data[1];
				knh_intptr_t max = s->data[1] + 2;
				s = DP(n)->opbuf + i - 1;
				while ((int)i > 0 && isDEF(s->opcode, 0) && s->data[0] >= min && s->data[0] < max) {
					if (isOBJ(s->data[0])) {
						DBG_P("pop %02d", s->data[0]);
						DBG_ASSERT(knh_Array_size(STACK(s->data[0])) > 1);
						iArray_pop(ctx, STACK(s->data[0]));
						origidx->ilist[s->data[0]] = -1;
					} else {
						DBG_P("skip pop args %d", s->data[0]);
					}
					i--;
					s = DP(n)->opbuf + i - 1;
				}
			}
			
			// else if (s->opcode == OPCODE_FASTCALL0 || s->opcode == OPCODE_SCAST) {
//				knh_intptr_t min = s->data[1];
//				knh_intptr_t max = s->data[1] + 3;
//				s = DP(n)->opbuf + i - 1;
//				while (isDEF(s->opcode, 0) && s->data[0] >= min && s->data[0] < max) {
//					DBG_P("skip pop args %d", s->data[0]);
//					i--;
//					s = DP(n)->opbuf + i - 1;
//				}
//			}
		}
	}
	DBG_P("end %02d", DFN(n));
	DBG_P("idx=%02d", *idx);
#ifdef K_USING_DEBUG
	printArray_Int(origidx);
	printArray_Array(stack);
#endif
}

static void setUnvisited(knh_Array_t *vertex)
{
	size_t i;
	for (i = 0; i < knh_Array_size(vertex); i++) {
#ifdef K_USING_DEBUG
		//knh_BasicBlock_print(ctx, VERTEX(i));
#endif
		knh_BasicBlock_setVisited(VERTEX(i), 0);
	}
}

static void splitEdge(Ctx *ctx, knh_BasicBlock_t *bb, knh_BasicBlock_t *bbN)
{
	knh_BasicBlock_t *bbNEW = new_BasicBlockLABEL(ctx);
	if (bb->nextNC == bbN) {
		bbNEW->nextNC = bb->nextNC;
		bb->nextNC = bbNEW;
	} else {
		bbNEW->jumpNC = bb->jumpNC;
		bb->jumpNC = bbNEW;
	}
	DP(bbNEW)->incoming = 1;
	knh_BasicBlock_setVisited(bbNEW, 1);
}

static void adjustBlocks(Ctx *ctx, knh_BasicBlock_t *p, knh_BasicBlock_t *n, size_t *size)
{
	knh_BasicBlock_t *bbN, *bbJ;
	knh_BasicBlock_setVisited(n, 1);
	bbN = n->nextNC;
	bbJ = n->jumpNC;
	if (p != NULL)
		DP(n)->incoming = 1;
	if (bbN != NULL) {
		if (!knh_BasicBlock_isVisited(bbN)) {
			adjustBlocks(ctx, n, bbN, size);
		} else if (DP(bbN)->incoming < 2) {
			DP(bbN)->incoming += 1;
			if (bbJ != NULL) {
				splitEdge(ctx, n, bbN);
				*size += 1;
			}
		} else {
			//DBG_P("add empty block between [%02d] and NEXT[%02d]", DFN(n), DFN(bbN));
			addPostBody(ctx, n, bbN);
#ifdef K_USING_DEBUG
			printBB(ctx, bbN);
#endif
			*size += 1;
		}
	}
	if (bbJ != NULL) {
		if (!knh_BasicBlock_isVisited(bbJ)) {
			adjustBlocks(ctx, n, bbJ, size);
		} else if (DP(bbJ)->incoming < 2) {
			DP(bbJ)->incoming += 1;
			if (bbN != NULL) {
				splitEdge(ctx, n, bbJ);
				*size += 1;
			}
		} else {
			//DBG_P("add empty block between [%02d] and JUMP[%02d]", DFN(n), DFN(bbJ));
			addPostBody(ctx, n, bbJ);
#ifdef K_USING_DEBUG
			printBB(ctx, bbJ);
#endif
			*size += 1;
		}
	}
}

/**
 * Convert to SSA form.
 * @param ctx    Context
 * @param bb     BasicBlock
 * @param df     Dominance Frontier of the block
 * @param pred   Predecessor of the block
 * @param idom   Immediately dominator of the block
 * @param vertex Linear list of BasicBlocks(~= listNC)
 * @param stack  A stack that records rename in each variable
 * @param max    Max index of register
 */
static void convertSSA(Ctx *ctx, knh_BasicBlock_t *bb, knh_Array_t *df, knh_Array_t *pred, knh_Array_t *idom, knh_Array_t *vertex, knh_Array_t *liveIn, knh_Array_t *liveOut, int *max)
{
	int i;
	int idx = 0;
	size_t size = knh_Array_size(bb->listNC);
	DBG_P("size=%d", size);
	adjustBlocks(ctx, NULL, bb, &size);
	for (i = 0; i < (int)size; i++) {
		knh_Array_add(ctx, df, new_Array(ctx, CLASS_Int, 0));
		knh_Array_add(ctx, pred, new_Array(ctx, CLASS_Int, 0));
		iArray_add(ctx, idom, -1);
		knh_Array_add(ctx, liveIn, new_Array(ctx, CLASS_Int, 0));
		knh_Array_add(ctx, liveOut, new_Array(ctx, CLASS_Int, 0));
	}
	depthFirstSearch(ctx, NULL, bb, vertex, pred);
#ifdef K_USING_DEBUG
	printTree(ctx, vertex);
#endif
	setIdom(ctx, vertex, idom, pred);
	setDF(ctx, vertex, bb, df, idom);
#ifdef K_USING_DEBUG
	//printArray_Array(df);
#endif
	putPhiFunc(ctx, vertex, df, &idx);
	*max = idx;
	BEGIN_LOCAL(ctx, lsfp, 2);
	LOCAL_NEW(ctx, lsfp, 0, knh_Array_t*, stack, new_Array(ctx, CLASS_Array, *max));
	LOCAL_NEW(ctx, lsfp, 1, knh_Array_t*, origidx, new_Array(ctx, CLASS_Int, *max));
	for (i = 0; i < *max; i++) {
		knh_Array_add(ctx, stack, new_Array(ctx, CLASS_Int, 1));
		iArray_add(ctx, (knh_Array_t*)knh_Array_n(stack, i), i);
		iArray_add(ctx, origidx, i);
	}
	if ((int)knh_Array_size(stack) > 0) {
		if (isOBJ(idx)) {
		} else {
			iArray_add(ctx, origidx, -1);
			idx += 1;
		}
		DBG_P("idx=%d", idx);
		renameVars(ctx, vertex, bb, stack, origidx, pred, idom, liveIn, liveOut, &idx);
	} else {
		DBG_P("Don't rename!");
	}
	*max = knh_Array_size(origidx);

#ifdef K_USING_DEBUG
	//knh_BasicBlock_printLiveness(liveIn, liveOut);
	//setUnvisited(vertex);
#endif
	END_LOCAL(ctx, lsfp);
}

/* ------------------------------------------------------------------------ */
/* [convert out of SSA form] */

//static int isInterfering(/*int isSource, int a, int b, knh_Array_t *liveIn, knh_Array_t *liveOut*/)
//{
////	if (isSource) {
////	}
//	return 0;
//}

//static int isSameArray(knh_Array_t *a, knh_Array_t *b)
//{
//	size_t i = 0;
//	size_t j = 0;
//	if (knh_Array_size(a) == knh_Array_size(b)) {
//		while (i < knh_Array_size(a)) {
//			while (j < knh_Array_size(b)) {
//				if (knh_Array_n(a, i) == knh_Array_n(b, j)) {
//					i++;
//					b = 0;
//				} else {
//					j++;
//				}
//			}
//			return 0;
//		}
//		return 1;
//	}
//	return 0;
//}


/**
 * Eliminate phi resource interferences based on
 * data-flow and interference graph updates.
 */
//static void eliminatePRI(/*Ctx *ctx, knh_BasicBlock_t *bb, knh_Array_t *liveIn, knh_Array_t *liveOut, knh_Array_t *pcc, knh_Array_t *unm, int max*/)
//{
//	int c, i0, j0, i1, j1, xi, xj, yi, yj;
//	knh_opline_t *op;
//	BEGIN_LOCAL(ctx, lsfp, 1);
//	// candidateResourceSet
//	LOCAL_NEW(ctx, lsfp, 0, knh_Array_t *, crs, new_Array(ctx, CLASS_Array, 0));
//#ifdef K_USING_DEBUG
//	knh_BasicBlock_print(ctx, bb);
//#endif
//	for (c = 0; c < DP(bb)->size; c++) {
//		op = DP(bb)->opbuf + c;
//		if (op->opcode == OPCODE_PHI) {
//			for (i0 = 0; i0 < 2; i0++) {
//				for (j0 = i0 + 1; j0 < 3; j0++) {
//					DBG_P("bb[%02d]", DFN(bb));
//					xi = op->data[i0];
//					DBG_P("x%d=%d", i0, xi);
//					xj = op->data[j0];
//					DBG_P("y%d=%d", j0, yi);
//					for (i1 = 0; i1 < knh_Array_size(PCC(xi)); i1++) {
//						for (j1 = 0; j1 < knh_Array_size(PCC(xj)); j1++) {
//							if (!isSameArray(PCC(xi), PCC(xj))) {
//								yi = iArray_n(PCC(xi), i1);
//								yj = iArray_n(PCC(xj), j1);
//								if (isInterfering(i0, yi, yj, liveIn, liveOut)) {
//
//								}
//							}
//						}
//					}
//				}
//			}
//		}
//	}
//	if (bb->nextNC != NULL && knh_BasicBlock_isVisited(bb->nextNC))
//		eliminatePRI(ctx, bb->nextNC, liveIn, liveOut, pcc, unm, max);
//	if (bb->jumpNC != NULL && knh_BasicBlock_isVisited(bb->jumpNC))
//		eliminatePRI(ctx, bb->jumpNC, liveIn, liveOut, pcc, unm, max);
//	END_LOCAL(ctx, lsfp);
//}

/**
 * Convert out of SSA form.
 * @param ctx Context
 * @param bb  BasicBlock
 */
static void simplePhiElim(Ctx *ctx, knh_Array_t *vertex, void *data)
{
	struct phidata *pdata = (struct phidata*)data;
	int i, j, k;
	knh_opline_t *op;
	knh_BasicBlock_t *target, *bb;
	knh_Array_t *pred    = pdata->pred;
//	knh_Array_t *liveIn  = pdata->liveIn;
//	knh_Array_t *liveOut = pdata->liveOut;
	int max = pdata->max;
//#ifdef K_USING_DEBUG
//	knh_BasicBlock_print(ctx, bb);
//#endif
	BEGIN_LOCAL(ctx, lsfp, 2);
	// phiCongruenceClass
	LOCAL_NEW(ctx, lsfp, 0, knh_Array_t *, pcc, new_Array(ctx, CLASS_Array, 0));
	// unresolvedNaighborMap
	LOCAL_NEW(ctx, lsfp, 1, knh_Array_t *, unm, new_Array(ctx, CLASS_Array, 0));
	for (i = 0; i < max; i++) {
		knh_Array_add(ctx, pcc, new_Array(ctx, CLASS_Int, 0));
		iArray_add(ctx, PCC(i), i);
		knh_Array_add(ctx, unm, new_Array(ctx, CLASS_Int, 0));
	}
//	eliminatePRI(ctx, bb, liveIn, liveOut, pcc, unm, max);
	for (i = 0; i < (int)knh_Array_size(vertex); i++) {
		bb = VERTEX(i);
		size_t size = DP(bb)->size;
		for (j = 0; j < (int)DP(bb)->size; j++) {
			op = DP(bb)->opbuf + j;
			if (op->opcode == OPCODE_PHI) {
				for (k = 0; k <= 1; k++) {
					target = VERTEX(iArray_n(PRED(bb), k));
					if (isOBJ(op->data[0])) {
						klr_OMOV_t omov = {TADDR, OPCODE_OMOV, ASMLINE, op->data[0], op->data[k + 1]};
						//DBG_P("[%02d] add OMOV r%d r%d", DFN(target), op->data[0], op->data[k + 1]);
						knh_BasicBlock_add_(ctx, target, 0, (knh_opline_t*)(&omov));
					} else {
						klr_NMOV_t nmov = {TADDR, OPCODE_NMOV, ASMLINE, op->data[0], op->data[k + 1]};
						//DBG_P("[%02d] add NMOV r%d r%d", DFN(target), op->data[0], op->data[k + 1]);
						knh_BasicBlock_add_(ctx, target, 0, (knh_opline_t*)(&nmov));
					}
					if (target->nextNC != NULL && target->jumpNC != NULL) {
						knh_opline_t *opLAST = DP(target)->opbuf + DP(target)->size - 1;
						SWAP(opLAST, opLAST - 1, sizeof(knh_opline_t));
					}
				}
				op->opcode = OPCODE_NOP;
				size -= 1;
			}
		}
		if (size < DP(bb)->size) {
			knh_opline_t *opD = DP(bb)->opbuf;
			for (j = 0; j < (int)DP(bb)->size; j++) {
				knh_opline_t *opS = DP(bb)->opbuf + j;
				if (opS->opcode == OPCODE_NOP)
					continue;
				if (opD != opS) {
					*opD = *opS;
				}
				opD += 1;
			}
			DP(bb)->size = size;
		}
	}
	END_LOCAL(ctx, lsfp);
}

//#define K_USING_SSA_OPT
#include "opt/nop.c"
#include "opt/cfold.c"
#include "opt/peephole.c"

typedef void (*fbbopt)(Ctx *ctx, knh_BasicBlock_t *bb, knh_Array_t *data, int mask);
/* ------------------------------------------------------------------------ */
/**
 * Optimize the BasicBlocks.
 * @param ctx Context
 * @param bbs BasicBlocks
 */
static void ssa_optimize(/*Ctx *ctx, knh_BasicBlock_t *bbs*/)
{
#ifdef K_USING_SSA_OPT
	struct opt {
		const char *name;
		fbbopt opt;
	} opts[] = {
#ifdef USE_OPT_SIMPLE_CONSTANT_FOLDING
		DEF_SIMPLE_CONSTANT_FOLDING,
#endif
#ifdef USE_OPT_NULL
		DEF_NULL,
#endif
	};
	int flag = knh_BasicBlock_isVisited(bbs);
	size_t i;
	size_t size = (int)(sizeof(x) / sizeof((x)[0]));
	BEGIN_LOCAL(ctx, lsfp, 1);
	LOCAL_NEW(ctx, lsfp, 0, knh_Array_t *, data, new_Array(ctx, CLASS_Int, 0));
	for (i = 0; i < size; i++) {
		DBG_P("opt:%s", opts[i].name);
		flag = !flag;
		opts[i].opt(ctx, bbs, data, flag);
	}
	if (size % 2 == 1) 
		opt_nop(ctx, bbs, data, !flag);
	END_LOCAL(ctx, lsfp);
#endif
}

/* ------------------------------------------------------------------------ */
/* [SSA optimize] */

/**
 * Optimize the KLRcode.
 * @param ctx Context
 * @param bb  BasicBlock
 */
void knh_BasicBlock_optimize(Ctx *ctx, knh_BasicBlock_t *bb)
{
#if 1
	int max;
	size_t size = knh_Array_size(bb->listNC);
	BEGIN_LOCAL(ctx, lsfp, 6);
	LOCAL_NEW(ctx, lsfp, 0, knh_Array_t*, df, new_Array(ctx, CLASS_Array, size));
	LOCAL_NEW(ctx, lsfp, 1, knh_Array_t*, pred, new_Array(ctx, CLASS_Array, size));
	LOCAL_NEW(ctx, lsfp, 2, knh_Array_t*, idom, new_Array(ctx, CLASS_Int, size));
	LOCAL_NEW(ctx, lsfp, 3, knh_Array_t*, vertex, new_Array(ctx, CLASS_BasicBlock, size));
	LOCAL_NEW(ctx, lsfp, 4, knh_Array_t*, liveIn, new_Array(ctx, CLASS_Array, 0));
	LOCAL_NEW(ctx, lsfp, 5, knh_Array_t*, liveOut, new_Array(ctx, CLASS_Array, 0));

	//DBG_P("DF = %p", df);
	convertSSA(ctx, bb, df, pred, idom, vertex, liveIn, liveOut, &max);
	//ssa_optimize(ctx, bb);
	{
		struct phidata pdata = {pred, liveIn, liveOut, max};
		DBG_P("max=%d", max);
		simplePhiElim(ctx, vertex, &pdata);
#ifdef USE_OPT_PEEPHOLE
		opt_peephole(ctx, vertex, 0);
#endif
#ifdef USE_OPT_REMOVE_NOPCODE
		opt_remove_nopcode(ctx, vertex);
#endif
	}
	//setUnvisited(ctx, vertex);
	printTree(ctx, vertex);
	END_LOCAL(ctx, lsfp);
#endif
}

#ifdef __cplusplus
}
#endif

#endif /* K_USING_SSA */
