/****************************************************************************
 * KONOHA COPYRIGHT, LICENSE NOTICE, AND DISCRIMER
 *
 * Copyright (c) 2005-2008, Kimio Kuramitsu <kimio at ynu.ac.jp>
 *           (c) 2008-      Konoha Software Foundation
 * All rights reserved.
 *
 * You may choose one of the following two licenses when you use konoha.
 * See www.konohaware.org/license.html for further information.
 *
 * (1) GNU General Public License 2.0      (with    KONOHA_UNDER_GPL2)
 * (2) Konoha Software Foundation License 1.0
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

/* ************************************************************************ */

#include"commons.h"

/* ************************************************************************ */

#ifdef __cplusplus
extern "C" {
#endif

/* ======================================================================== */
/* Data */
/* ======================================================================== */

/* ------------------------------------------------------------------------ */
/* @data*/
typedef struct {
	char                  *name;
	int                    index;
} knh_StringData_t ;

typedef struct {
	char                  *name;
	knh_struct_t           sid;
	knh_flag_t             flag;
	size_t                 size;
	knh_fstruct_init       finit;
	knh_fstruct_copy       fcopy;
	knh_fstruct_traverse   ftraverse;
	knh_fstruct_compareTo  fcompareTo;
	knh_fstruct_hashCode   fhashCode;
	knh_fstruct_newClass   fnewClass;
} knh_StructData_t ;

typedef struct {
	char *name;
	knh_class_t cid;
	knh_flag_t flag;
	knh_struct_t bcid;
	knh_class_t supcid;
	int field_size;
	int method_size; /* copy supcid if 0 */
	int mapper_size; /* copy Any if 0 */
	knh_fdefault fdefault;
} knh_ClassData_t ;

typedef struct {
	char *name;
	knh_class_t cid;
	knh_class_t bcid;
	knh_class_t p1;
	knh_class_t p2;
} knh_GenericsData_t ;

typedef struct {
	char *name;
	knh_class_t cid;
	knh_type_t r0;
	knh_type_t p1;
	knh_type_t p2;
	knh_type_t p3;
} knh_ClosureData_t ;

typedef struct {
	char *name;
	knh_flag_t flag;
	knh_expt_t eid;
	char *parent;
} knh_ExptData_t ;

typedef struct {
	char *name;
	void *func;
} knh_FuncData_t ;

typedef struct {
	char *name;
	knh_fieldn_t fn;
} knh_FieldNameData_t ;

typedef struct {
	int mfindex;
	knh_ushort_t psize;
	knh_type_t rtype;
	knh_type_t t1;
	knh_fieldn_t f1;
	knh_type_t t2;
	knh_fieldn_t f2;
	knh_type_t t3;
	knh_fieldn_t f3;
	knh_type_t t4;
	knh_fieldn_t f4;
} knh_MethodFieldData_t ;

typedef struct {
	knh_fmethod func;
	knh_flag_t flag;
	knh_class_t cid;
	knh_methodn_t mn;
	knh_ushort_t delta;
	int mfindex;
	void *optfunc;
} knh_MethodData_t ;

typedef struct {
	knh_fmapper func;
	knh_flag_t flag;
	knh_class_t scid;
	knh_class_t tcid;
} knh_MapperData_t ;

typedef struct {
	char *name;
	knh_integer_t ivalue;
} knh_IntConstData_t;

typedef struct {
	char *name;
	knh_float_t fvalue;
} knh_FloatConstData_t;

typedef struct {
	char *name;
	char *value;
} knh_StringConstData_t;

/* ======================================================================== */
/* Object */

#define knh_ObjectField_init_ NULL
#define knh_ObjectField_copy NULL
#define knh_ObjectField_traverse_ NULL
#define knh_ObjectField_compareTo NULL
#define knh_ObjectField_hashCode NULL
#define knh_ObjectField_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_ClassStruct_initField(Ctx *ctx, ClassStruct *cs, knh_class_t self_cid, Object **v)
{
	size_t i;
	knh_cfield_t *cf = cs->fields;
	for(i = 0; i < cs->fsize; i++) {
		if(IS_NULL(cf[i].value) && IS_NNTYPE(cf[i].type)) {
			knh_class_t cid = knh_pmztype_toclass(ctx, cf[i].type, self_cid);
			KNH_INITv(v[i], new_Object_init(ctx, 0, cid, 0));
		}
		else {
			KNH_INITv(v[i], cf[i].value);
		}
	}
}

/* ------------------------------------------------------------------------ */

static
void knh_ObjectField_init(Ctx *ctx, knh_ObjectField_t *of, int init)
{
	knh_class_t cid = of->h.cid;
	if(ctx->tClass[cid].size > 0) {
		//Object **v = (Object**)KNH_MALLOC(ctx, ctx->tClass[cid].size);
		Object **v = (Object**)of->fields;
		size_t offset;
		while((offset = ctx->tClass[cid].offset) != 0) {
			knh_ClassStruct_initField(ctx, ctx->tClass[cid].cstruct, of->h.cid, v + offset);
			cid = ctx->tClass[cid].supcid;
			KNH_ASSERT_cid(cid);
		}
		knh_ClassStruct_initField(ctx, ctx->tClass[cid].cstruct, of->h.cid, v + offset);
		of->fields = v;
		of->bsize = ctx->tClass[of->h.cid].bsize;
	}
	else {
		of->fields = NULL;
		of->bsize = 0;
	}
}

/* ------------------------------------------------------------------------ */

static
void knh_ObjectField_traverse(Ctx *ctx, knh_ObjectField_t *of, knh_ftraverse ftr)
{
	size_t i;
	for(i = 0; i < of->bsize; i++) {
		ftr(ctx, of->fields[i]);
	}
//	if(IS_SWEEP(ftr)) {
//		KNH_FREE(ctx, of->fields, of->bsize * sizeof(Object*));
//		of->fields = NULL;
//		of->bsize = 0;
//	}
}

/* ======================================================================== */
/* Any */

#define knh_Any_init_ NULL
#define knh_Any_copy NULL
#define knh_Any_traverse_ NULL
#define knh_Any_compareTo NULL
#define knh_Any_hashCode NULL
#define knh_Any_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_fgluecopy_NOP(Ctx *ctx, void *ptr, knh_ftraverse ftr)
{
}

/* ------------------------------------------------------------------------ */

static
void knh_Any_init(Ctx *ctx, knh_Glue_t *g, int init)
{
	g->ptr = NULL;
	g->gluefree = knh_fgluecopy_NOP;
}

/* ------------------------------------------------------------------------ */

static
void knh_Any_traverse(Ctx *ctx, knh_Glue_t *g, knh_ftraverse ftr)
{
	g->gluefree(ctx, g->ptr, ftr);
	g->ptr = NULL;
	g->gluefree = knh_fgluecopy_NOP;
}

/* ======================================================================== */
/* Nue */

#define knh_Nue_init_ NULL
#define knh_Nue_copy NULL
#define knh_Nue_traverse_ NULL
#define knh_Nue_compareTo_ NULL
#define knh_Nue_hashCode NULL
#define knh_Nue_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_Nue_init(Ctx *ctx, knh_Nue_t *n, int init)
{
	n->orign = NULL;
}

/* ------------------------------------------------------------------------ */

static
void knh_Nue_traverse(Ctx *ctx, knh_Nue_t *n, knh_ftraverse ftr)
{
	if(n->orign != NULL) {
		KNH_ASSERT(IS_bString(n->orign));
		ftr(ctx, UP(n->orign));
		n->orign = NULL;
	}
}

/* ------------------------------------------------------------------------ */

static
knh_hcode_t knh_Nue_compareTo(Ctx *ctx, Nue *o)
{
	return CLASS_Nue; /* always same value */
}

/* ======================================================================== */
/* Boolean */

#define knh_Boolean_init_ NULL
#define knh_Boolean_copy NULL
#define knh_Boolean_traverse NULL
#define knh_Boolean_compareTo NULL
#define knh_Boolean_hashCode NULL
#define knh_Boolean_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_Boolean_init(Ctx *ctx, Boolean *o, int init)
{
	o->n.bvalue = init;
}

/* ======================================================================== */
/* Number */

#define knh_Number_init NULL
#define knh_Number_copy NULL
#define knh_Number_traverse NULL
#define knh_Number_compareTo NULL
#define knh_Number_hashCode NULL
#define knh_Number_newClass NULL


/* ======================================================================== */
/* Int */

#define knh_Int_init_ NULL
#define knh_Int_copy NULL
#define knh_Int_traverse NULL
#define knh_Int_compareTo_ NULL
#define knh_Int_hashCode_ NULL
#define knh_Int_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_Int_init(Ctx *ctx, Int *n, int init)
{
	n->n.ivalue = (knh_integer_t)init;
}

/* ------------------------------------------------------------------------ */

static
knh_hcode_t knh_Int_hashCode(Ctx *ctx, Int *o)
{
	return (knh_hcode_t)o->n.ivalue;
}

/* ------------------------------------------------------------------------ */

static
int knh_Int_compareTo(Ctx *ctx, Int *o, Int *o2)
{
	if(o->h.cid == CLASS_Int || o2->h.cid == CLASS_Int) {
		return (int)(o->n.ivalue - o2->n.ivalue);
	}
	else {
		IntUnit *u = (IntUnit*)ctx->tClass[o->h.cid].cspec;
		KNH_ASSERT(IS_IntUnit(u));
		return DP(u)->fcmp(u, o->n.ivalue, o2->n.ivalue);
	}
}
/* ======================================================================== */
/* Float */

#define knh_Float_init_ NULL
#define knh_Float_copy NULL
#define knh_Float_traverse NULL
#define knh_Float_compareTo_ NULL
#define knh_Float_hashCode_ NULL
#define knh_Float_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_Float_init(Ctx *ctx, Float *f, int init)
{
	f->n.fvalue = (knh_float_t)init;
}

/* ------------------------------------------------------------------------ */

static
knh_hcode_t knh_Float_hashCode(Ctx *ctx, Int *f)
{
	return (knh_hcode_t)f->n.ivalue;
}

/* ------------------------------------------------------------------------ */

static
int knh_Float_compareTo(Ctx *ctx, Float *o, Float *o2)
{
	if(o->h.cid == CLASS_Float || o2->h.cid == CLASS_Float) {
		return (int)(o->n.fvalue - o2->n.fvalue);
	}
	else {
		FloatUnit *fu = (FloatUnit*)ctx->tClass[o->h.cid].cspec;
		KNH_ASSERT(IS_FloatUnit(fu));
		return DP(fu)->fcmp(fu, o->n.fvalue, o2->n.fvalue);
	}
}

/* ======================================================================== */
/* String */

#define knh_String_init_ NULL
#define knh_String_copy NULL
#define knh_String_traverse_ NULL
#define knh_String_compareTo_ NULL
#define knh_String_hashCode_ NULL
#define knh_String_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_String_init(Ctx *ctx, knh_String_t *s, int init)
{
	s->str = (knh_uchar_t*)""; // for safety
	s->size = 0;
	s->orign = NULL;
}

/* ------------------------------------------------------------------------ */

static
void knh_String_traverse(Ctx *ctx, knh_String_t *s, knh_ftraverse ftr)
{
	if(s->orign != NULL) {
		ftr(ctx, UP(s->orign));
	}
	else if(!knh_String_isTextSgm(s)) {
		if(IS_SWEEP(ftr)) {
			KNH_FREE(ctx, s->str, KNH_SIZE(s->size + 1));
			s->str = NULL;
		}
	}
}

/* ------------------------------------------------------------------------ */

static
knh_hcode_t knh_String_hashCode(Ctx *ctx, String *o)
{
	KNH_ASSERT(IS_bString(o));
	knh_hcode_t h = o->size;
	size_t i;
	for(i = 0; i < o->size; i++) {
		h = o->str[i] + 31 * h;
	}
	return h;
}

/* ------------------------------------------------------------------------ */

static
int knh_String_compareTo(Ctx *ctx, String *o, String *o2)
{
	if(o->h.cid == CLASS_String || o2->h.cid == CLASS_String) {
		size_t i, max = (o->size< o2->size) ? o->size : o2->size;
		for(i = 0; i < max; i++) {
			int res = o->str[i] - o2->str[i];
			if(res != 0) return res;
		}
		return o->size - o2->size;
	}
	else {
		if(o->h.cid == o2->h.cid) {
			StringUnit *u = (StringUnit*)ctx->tClass[o->h.cid].cspec;
			KNH_ASSERT(IS_StringUnit(u));
			return DP(u)->fcmp(u, knh_String_tobytes(o), knh_String_tobytes(o2));
		}
		return (int)(o - o2);
	}
}

/* ======================================================================== */
/* Bytes */

#define knh_Bytes_init_ NULL
#define knh_Bytes_copy NULL
#define knh_Bytes_traverse_ NULL
#define knh_Bytes_compareTo NULL
#define knh_Bytes_hashCode NULL
#define knh_Bytes_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_Bytes_init(Ctx *ctx, knh_Bytes_t *b, int init)
{
	b->size = 0;
	if(init <= 0) {
		b->buf = (knh_uchar_t*)"";
		b->capacity = 0;
	}
	else {
		b->capacity = knh_bytes_newsize(init);
		b->buf = (knh_uchar_t*)KNH_MALLOC(ctx, b->capacity);
	}
}

/* ------------------------------------------------------------------------ */

static
void knh_Bytes_traverse(Ctx *ctx, knh_Bytes_t *b, knh_ftraverse ftr)
{
	if(IS_SWEEP(ftr)) {
		KNH_FREE(ctx, b->buf, b->capacity);
		b->buf = NULL;
		b->capacity = 0;
		b->size = 0;
	}
}

/* ======================================================================== */
/* Tuple2 */

#define knh_Tuple2_init_ NULL
#define knh_Tuple2_copy NULL
#define knh_Tuple2_traverse_ NULL
#define knh_Tuple2_compareTo NULL
#define knh_Tuple2_hashCode NULL
#define knh_Tuple2_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_Tuple2_init(Ctx *ctx, knh_Tuple2_t *t, int init)
{
	KNH_INITv(t->first, KNH_NULL);
	KNH_INITv(t->second, KNH_NULL);
}

/* ------------------------------------------------------------------------ */

static
void knh_Tuple2_traverse(Ctx *ctx, knh_Tuple2_t *t, knh_ftraverse ftr)
{
	ftr(ctx, t->first);
	ftr(ctx, t->second);
}

/* ======================================================================== */
/* Range */

#define knh_Range_init knh_Tuple2_init
#define knh_Range_copy NULL
#define knh_Range_traverse knh_Tuple2_traverse
#define knh_Range_compareTo NULL
#define knh_Range_hashCode NULL
#define knh_Range_newClass NULL

/* ======================================================================== */
/* Array */

#define knh_Array_init_ NULL
#define knh_Array_copy NULL
#define knh_Array_traverse_ NULL
#define knh_Array_compareTo NULL
#define knh_Array_hashCode NULL
#define knh_Array_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_Array_init(Ctx *ctx, knh_Array_t *a, int init)
{
	a->list = NULL;
	a->capacity = 0;
	a->size = 0;
	if(init > 0) {
		knh_Array_grow(ctx, a, init, KNH_NULL);
	}
}

/* ------------------------------------------------------------------------ */

static
void knh_Array_traverse(Ctx *ctx, knh_Array_t *a, knh_ftraverse ftr)
{
	size_t i;
	for(i = 0; i < a->capacity; i++) {
		ftr(ctx, a->list[i]);
	}
	if(a->capacity > 0 && IS_SWEEP(ftr)) {
		if(knh_Array_isDim(a)) {
			knh_array_dfree(ctx, a->list, a->capacity, sizeof(Object*));
		}
		else {
			KNH_FREE(ctx, a->list, a->capacity * sizeof(Object*));
		}
		a->size = 0;
		a->capacity = 0;
	}
}

/* ======================================================================== */
/* IArray */

#define knh_IArray_init_ NULL
#define knh_IArray_copy NULL
#define knh_IArray_traverse_ NULL
#define knh_IArray_compareTo NULL
#define knh_IArray_hashCode NULL
#define knh_IArray_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_IArray_init(Ctx *ctx, knh_IArray_t *a, int init)
{
	a->ilist = NULL;
	a->capacity = 0;
	a->size = 0;
	KNH_ASSERT(init == 0);
//	if(init > 0) {
//		knh_IArray_grow(ctx, a, init, 0);
//	}
}

/* ------------------------------------------------------------------------ */

static
void knh_IArray_traverse(Ctx *ctx, knh_IArray_t *a, knh_ftraverse ftr)
{
	if(a->capacity > 0 && IS_SWEEP(ftr)) {
		if(knh_IArray_isDim(a)) {
			knh_array_dfree(ctx, a->ilist, a->capacity, sizeof(knh_integer_t));
		}
		else {
			KNH_FREE(ctx, a->ilist, a->capacity * sizeof(knh_integer_t));
		}
		a->size = 0;
		a->capacity = 0;
	}
}

/* ======================================================================== */
/* FArray */

#define knh_FArray_init_ NULL
#define knh_FArray_copy NULL
#define knh_FArray_traverse_ NULL
#define knh_FArray_compareTo NULL
#define knh_FArray_hashCode NULL
#define knh_FArray_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_FArray_init(Ctx *ctx, knh_FArray_t *a, int init)
{
	a->flist = NULL;
	a->capacity = 0;
	a->size = 0;
	KNH_ASSERT(init == 0);
//	if(init > 0) {
//		knh_FArray_grow(ctx, a, init, 0.0);
//	}
}

/* ------------------------------------------------------------------------ */

static
void knh_FArray_traverse(Ctx *ctx, knh_FArray_t *a, knh_ftraverse ftr)
{
	if(a->capacity > 0 && IS_SWEEP(ftr)) {
		if(knh_FArray_isDim(a)) {
			knh_array_dfree(ctx, a->flist, a->capacity, sizeof(knh_float_t));
		}
		else {
			KNH_FREE(ctx, a->flist, a->capacity * sizeof(knh_float_t));
		}
		a->size = 0;
		a->capacity = 0;
	}
}

/* ======================================================================== */
/* Iterator */

#define knh_Iterator_init_ NULL
#define knh_Iterator_copy NULL
#define knh_Iterator_traverse_ NULL
#define knh_Iterator_compareTo NULL
#define knh_Iterator_hashCode NULL
#define knh_Iterator_newClass NULL

/* ------------------------------------------------------------------------ */

static
ITRNEXT knh_fitrnext_single(Ctx *ctx, knh_sfp_t *sfp, int n)
{
	Iterator *it = sfp[0].it;
	if(DP(it)->pos == 0) {
		DP(it)->pos = 1;
		KNH_ITRNEXT(ctx, sfp, n, DP(it)->source);
	}
	KNH_ITREND(ctx, sfp, n);
}
/* ------------------------------------------------------------------------ */

static
void knh_Iterator_init(Ctx *ctx, Iterator *it, int init)
{
	knh_Iterator_struct *b = DP(it);
	b->fnext  =  knh_fitrnext_single;
	KNH_INITv(b->source, KNH_VOID);
	KNH_INITv(b->prefetch, KNH_NULL);
	b->pos    =  0;
	b->ref    =  NULL;
	b->count = 0;
	b->start = 0;
	b->end   = (size_t)-1;
}

/* ------------------------------------------------------------------------ */

static
void knh_Iterator_traverse(Ctx *ctx, Iterator *it, knh_ftraverse ftr)
{
	knh_Iterator_struct *b = DP(it);
	ftr(ctx, b->source);
	ftr(ctx, b->prefetch);
}

/* ======================================================================== */
/* DictMap */

#define knh_DictMap_init_ NULL
#define knh_DictMap_copy NULL
#define knh_DictMap_traverse_ NULL
#define knh_DictMap_compareTo NULL
#define knh_DictMap_hashCode NULL
#define knh_DictMap_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_DictMap_init(Ctx *ctx, knh_DictMap_t *d, int init)
{
	d->_list = knh_dictmap_malloc(ctx, (size_t)init);
	d->size = 0;
	d->fcmp = knh_bytes_strcmp;
}

/* ------------------------------------------------------------------------ */

static
void knh_DictMap_traverse(Ctx *ctx, knh_DictMap_t *d, knh_ftraverse ftr)
{
	knh_dict_traverse(ctx, d->_list, ftr);
}

/* ======================================================================== */
/* DictSet */

#define knh_DictSet_init_ NULL
#define knh_DictSet_copy NULL
#define knh_DictSet_traverse knh_DictMap_traverse
#define knh_DictSet_compareTo NULL
#define knh_DictSet_hashCode NULL
#define knh_DictSet_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_DictSet_init(Ctx *ctx, knh_DictSet_t *d, int init)
{
	d->_list = knh_dictset_malloc(ctx, (size_t)init);
	d->size = 0;
	d->fcmp = knh_bytes_strcmp;
}

/* ======================================================================== */
/* HashMap */

#define knh_HashMap_init_ NULL
#define knh_HashMap_copy NULL
#define knh_HashMap_traverse knh_Hash_traverse
#define knh_HashMap_compareTo NULL
#define knh_HashMap_hashCode NULL
#define knh_HashMap_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_Hash_init(Ctx *ctx, knh_Hash_t *hash, int init)
{
	knh_Hash_struct *b = DP(hash);
	b->DBG_name = "HashMap";
	if(init > 0) {
		b->hmax = init;
		b->capacity = knh_bytes_newsize(init);
		b->array = (knh_hashentry_t**)KNH_MALLOC(ctx, sizeof(knh_hashentry_t*) * b->capacity);
		knh_bzero(b->array, sizeof(knh_hashentry_t*) * b->capacity);
	}
	else {
		b->hmax = 0;
		b->capacity = 0;
		b->array = NULL;
	}
	b->size = 0;
	b->tables = NULL;
	b->tables_size = 0;
	b->tables_capacity = 0;
	b->tables_first = NULL;
	b->unused_first = NULL;
	//
	hash->fcmp = knh_Object_compareTo;
}

/* ------------------------------------------------------------------------ */

static
void knh_hashmapentry_traverse(Ctx *ctx, knh_hashentry_t *e, knh_ftraverse ftr)
{
	DBG2_ASSERT(e->key != NULL);
	ftr(ctx, e->key);
	ftr(ctx, e->value);
}

/* ------------------------------------------------------------------------ */
/* @data */

static knh_hash_op knh_hashmap_op = {
	sizeof(knh_hashentry_t),
	NULL,
	knh_hashmapentry_traverse,
};

/* ------------------------------------------------------------------------ */

static
void knh_HashMap_init(Ctx *ctx, knh_HashMap_t *o, int init)
{
	knh_Hash_t *hash = (knh_Hash_t*)o;
	knh_Hash_init(ctx, hash, init);
	DP(hash)->DBG_name = "HashMap";
	hash->fcmp = knh_Object_compareTo;
	hash->hashop = &knh_hashmap_op;
}

/* ------------------------------------------------------------------------ */

static
void knh_Hash_traverse(Ctx *ctx, knh_Hash_t *hash, knh_ftraverse ftr)
{
	knh_Hash_struct *b = DP(hash);
	size_t i, j, esize = hash->hashop->size, tsize = KNH_HASH_TABLESIZE / esize;
	for(j = 0; j < b->tables_size; j++) {
		for(i = 0; i < tsize; i++) {
			knh_hashentry_t *e = KNH_HASHENTRY(b->tables[j], esize, i);
			if(e->key != NULL) {
				hash->hashop->ftraverse(ctx, e, ftr);
			}
		}
	}
	if(IS_SWEEP(ftr)) {
		if(b->array != NULL) {
			KNH_FREE(ctx, b->array, sizeof(knh_hashentry_t*) * b->capacity);
			b->array = NULL;
			b->size = 0;
			b->capacity = 0;
		}
		if(b->tables_size > 0) {
			for(i = 0; i < b->tables_size; i++) {
				KNH_FREE(ctx, b->tables[i], KNH_HASH_TABLESIZE);
			}
			if(b->tables_capacity > 1) {
				KNH_FREE(ctx, b->tables, sizeof(knh_hashentry_t*) * b->capacity);
			}
			b->tables = NULL;
			b->tables_capacity = 0;
			b->tables_size = 0;
		}
	}
}

/* ======================================================================== */
/* HashSet */

#define knh_HashSet_init_ NULL
#define knh_HashSet_copy NULL
#define knh_HashSet_traverse knh_Hash_traverse
#define knh_HashSet_compareTo NULL
#define knh_HashSet_hashCode NULL
#define knh_HashSet_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_hashsetentry_traverse(Ctx *ctx, knh_hashentry_t *e, knh_ftraverse ftr)
{
	DBG2_ASSERT(e->key != NULL);
	ftr(ctx, e->key);
}

/* ------------------------------------------------------------------------ */
/* @data */

static knh_hash_op knh_hashset_op = {
	sizeof(knh_hashentry_t),
	NULL,
	knh_hashsetentry_traverse,
};

/* ------------------------------------------------------------------------ */

static
void knh_HashSet_init(Ctx *ctx, knh_HashSet_t *o, int init)
{
	knh_Hash_t *hash = (knh_Hash_t*)o;
	knh_Hash_init(ctx, hash, init);
	DP(hash)->DBG_name = "HashSet";
	hash->fcmp = knh_Object_compareTo;
	hash->hashop = &knh_hashset_op;
}


/* ======================================================================== */
/* DictIdx */

#define knh_DictIdx_init_ NULL
#define knh_DictIdx_copy NULL
#define knh_DictIdx_traverse_ NULL
#define knh_DictIdx_compareTo NULL
#define knh_DictIdx_hashCode NULL
#define knh_DictIdx_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_DictIdx_init(Ctx *ctx, knh_DictIdx_t *d, int init)
{
	KNH_INITv(d->terms, new_Array(ctx, CLASS_String, init));
	KNH_INITv(d->termsDictSet, new_DictSet(ctx, init));
	d->offset = 0;
}

/* ------------------------------------------------------------------------ */

static
void knh_DictIdx_traverse(Ctx *ctx, knh_DictIdx_t *d, knh_ftraverse ftr)
{
	ftr(ctx, UP(d->terms));
	ftr(ctx, UP(d->termsDictSet));
}

/* ======================================================================== */
/* Class */

#define knh_Class_init_ NULL
#define knh_Class_copy NULL
#define knh_Class_traverse NULL
#define knh_Class_compareTo_ NULL
#define knh_Class_hashCode_ NULL
#define knh_Class_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_Class_init(Ctx *ctx, knh_Class_t *c, int init)
{
	c->cid = (knh_class_t)init;
	c->type = (knh_type_t)init;
}

/* ------------------------------------------------------------------------ */

static
knh_hcode_t knh_Class_hashCode(Ctx *ctx, Class *o)
{
	return (knh_hcode_t)o->cid;
}

/* ------------------------------------------------------------------------ */

static
int knh_Class_compareTo(Ctx *ctx, Class *o, Class *o2)
{
	return knh_strcmp(CLASSN(o->cid), CLASSN(o2->cid));
}

/* ======================================================================== */
/* ClassStruct */

#define knh_ClassStruct_init_ NULL
#define knh_ClassStruct_copy NULL
#define knh_ClassStruct_traverse_ NULL
#define knh_ClassStruct_compareTo NULL
#define knh_ClassStruct_hashCode NULL
#define knh_ClassStruct_newClass NULL

static
void knh_ClassStruct_init(Ctx *ctx, ClassStruct *b, int init)
{
	b->fsize = init;
	if(b->fsize == 0) {
		b->fields = NULL;
	}else {
		b->fields = (knh_cfield_t*)KNH_MALLOC(ctx, b->fsize * sizeof(knh_cfield_t));
		knh_int_t i;
		for(i = 0; i < b->fsize; i++) {
			b->fields[i].flag = 0;
			b->fields[i].type = TYPE_any;
			b->fields[i].fn   = FIELDN_NONAME;
			KNH_INITv(b->fields[i].value, KNH_NULL);
		}
	}
	KNH_INITv(b->methods, KNH_NULL);
}

/* ------------------------------------------------------------------------ */

static
void knh_ClassStruct_traverse(Ctx *ctx, ClassStruct *b, knh_ftraverse gc)
{
	if(b->fields != NULL) {
		knh_int_t i;
		for(i = 0; i < b->fsize; i++) {
			gc(ctx, b->fields[i].value);
		}
		if(IS_SWEEP(gc)) {
			KNH_FREE(ctx, b->fields, b->fsize * sizeof(knh_cfield_t));
		}
	}
	gc(ctx, UP(b->methods));
}

/* ======================================================================== */
/* MethodField */

#define knh_MethodField_init_ NULL
#define knh_MethodField_copy NULL
#define knh_MethodField_traverse_ NULL
#define knh_MethodField_compareTo NULL
#define knh_MethodField_hashCode NULL
#define knh_MethodField_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_MethodField_init(Ctx *ctx, knh_MethodField_t *o, int init)
{
	o->size = (size_t)init;
	if(init > 0) {
		o->params = (knh_mfield_t*)KNH_MALLOC(ctx, init * sizeof(knh_mfield_t));
		size_t i;
		for(i = 0; i < o->size; i++) {
			o->params[i].type = TYPE_Any;
			o->params[i].fn   = FIELDN_NONAME;
		}
	}
	o->rsize = 1;
}

/* ------------------------------------------------------------------------ */

static
void knh_MethodField_traverse(Ctx *ctx, knh_MethodField_t *o, knh_ftraverse ftr)
{
	if(IS_SWEEP(ftr)) {
		KNH_FREE(ctx, o->params, o->size * sizeof(knh_mfield_t));
	}
}

/* ======================================================================== */
/* Method */

#define knh_Method_init_ NULL
#define knh_Method_copy NULL
#define knh_Method_traverse_ NULL
#define knh_Method_compareTo NULL
#define knh_Method_hashCode NULL
#define knh_Method_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_Method_init(Ctx *ctx, Method *mtd, int init)
{
	knh_Method_struct *b = DP(mtd);
	b->flag   = 0;
	b->delta  = 0;
	b->cid    = CLASS_Object;
	b->mn     = METHODN_NONAME;
	b->fproceed  = knh_fmethod_abstract;
	KNH_INITv(b->mf, MF_Any_Any_);
	b->code  = NULL;
}

/* ------------------------------------------------------------------------ */

static
void knh_Method_traverse(Ctx *ctx, Method *mtd, knh_ftraverse gc)
{
	knh_Method_struct *b = DP(mtd);
	gc(ctx, UP(b->mf));
	if((b->flag & KNH_FLAG_MF_OBJECTCODE) == KNH_FLAG_MF_OBJECTCODE) {
		gc(ctx, (Object*)b->code);
		if(IS_SWEEP(gc)) {
			b->code = NULL;
		}
	}
}


/* ======================================================================== */
/* Mapper */

#define knh_Mapper_init_ NULL
#define knh_Mapper_copy NULL
#define knh_Mapper_traverse_ NULL
#define knh_Mapper_compareTo NULL
#define knh_Mapper_hashCode NULL
#define knh_Mapper_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_Mapper_init(Ctx *ctx, Mapper *mpr, int init)
{
	knh_Mapper_struct *b = DP(mpr);
	b->size = 0;
	b->flag = 0;
	b->scid = CLASS_Object;
	b->tcid = CLASS_Object;
	//b->fmap = NULL;
	KNH_INITv(b->mapdata, KNH_NULL);
	KNH_INITv(b->m2, KNH_NULL);
}

/* ------------------------------------------------------------------------ */

static
void knh_Mapper_traverse(Ctx *ctx, Mapper *mpr, knh_ftraverse ftr)
{
	knh_Mapper_struct *b = DP(mpr);
	ftr(ctx, b->mapdata);
	ftr(ctx, UP(b->m2));
}

/* ======================================================================== */
/* ClassMap */

#define knh_ClassMap_init_ NULL
#define knh_ClassMap_copy NULL
#define knh_ClassMap_traverse_ NULL
#define knh_ClassMap_compareTo NULL
#define knh_ClassMap_hashCode NULL
#define knh_ClassMap_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_ClassMap_init(Ctx *ctx, ClassMap *cm, int init)
{
	knh_ClassMap_struct *b = DP(cm);
	b->size   = 0;
	b->capacity = (init <= 3) ? 4 : init;
	b->maplist   = (Mapper**)KNH_MALLOC(ctx, b->capacity * sizeof(Mapper*));
	knh_bzero(b->maplist, b->capacity * sizeof(Mapper*));
}

/* ------------------------------------------------------------------------ */

static
void knh_ClassMap_traverse(Ctx *ctx, ClassMap *cm, knh_ftraverse gc)
{
	knh_ClassMap_struct *b = DP(cm);
	int i;
	for(i = 0; i < b->size; i++) {
		gc(ctx, UP(b->maplist[i]));
	}
	if(IS_SWEEP(gc)) {
		KNH_FREE(ctx, b->maplist, b->capacity * sizeof(Mapper*));
	}
}

/* ======================================================================== */
/* Closure */

#define knh_Closure_init_ NULL
#define knh_Closure_copy NULL
#define knh_Closure_traverse_ NULL
#define knh_Closure_compareTo NULL
#define knh_Closure_hashCode NULL
#define knh_Closure_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_Closure_init(Ctx *ctx, Closure *c, int init)
{
	knh_Closure_struct *b = DP(c);
	KNH_INITv(b->base, KNH_NULL);
	KNH_INITv(b->mtd, KNH_NULL);
	b->stack = NULL;
	b->stacksize = 0;
}

/* ------------------------------------------------------------------------ */

static
void knh_Closure_traverse(Ctx *ctx, Closure *c, knh_ftraverse ftr)
{
	knh_Closure_struct *b = DP(c);
	ftr(ctx, UP(b->base));
	ftr(ctx, UP(b->mtd));
	if(b->stacksize > 0 && knh_Closure_isCopiedStack(c)) {
		size_t i;
		for(i = 0; i < b->stacksize; i++) {
			ftr(ctx, b->stack[i].o);
		}
		if(IS_SWEEP(ftr)) {
			KNH_FREE(ctx, b->stack, sizeof(knh_sfp_t) * b->stacksize);
			b->stack = NULL;
			b->stacksize = 0;
		}
	}
}

/* ======================================================================== */
/* AffineConv */

#define knh_AffineConv_init_ NULL
#define knh_AffineConv_copy NULL
#define knh_AffineConv_traverse NULL
#define knh_AffineConv_compareTo NULL
#define knh_AffineConv_hashCode NULL
#define knh_AffineConv_newClass NULL

static
void knh_AffineConv_init(Ctx *ctx, AffineConv *o, int init)
{
	o->scale = 1.0;
	o->shift = 0.0;
}

/* ======================================================================== */
/* IntUnit */

#define knh_IntUnit_init_ NULL
#define knh_IntUnit_copy NULL
#define knh_IntUnit_traverse_ NULL
#define knh_IntUnit_compareTo NULL
#define knh_IntUnit_hashCode NULL
#define knh_IntUnit_newClass NULL

/* ------------------------------------------------------------------------ */

static
knh_bool_t knh_fenumchk__nop(IntUnit *b, knh_integer_t v)
{
	return 1;
}

/* ------------------------------------------------------------------------ */

static
int knh_fenumcmp__signed(IntUnit *b, knh_integer_t v1, knh_integer_t v2)
{
	return (int)(v1 - v2);
}

/* ------------------------------------------------------------------------ */

static
void knh_fenumfmt__signed(IntUnit *b, char *buf, size_t bufsiz, knh_integer_t v)
{
	knh_snprintf(buf, bufsiz, KNH_INTEGER_FMT, v);
}

/* ------------------------------------------------------------------------ */

static
void knh_IntUnit_init(Ctx *ctx, IntUnit *u, int init)
{
	knh_IntUnit_struct *b = DP(u);
	b->spec.flag = 0;
	b->spec.cid  = CLASS_Int;
	KNH_INITv(b->spec.urn,  TS_EMPTY);
	KNH_INITv(b->spec.tag,  TS_EMPTY);
	KNH_INITv(b->spec.defvalue, ctx->tInt[0-(KNH_TINT_MIN)]);
	//
	b->step = 1;
	b->max  = KNH_INTEGER_MAX;
	b->min  = KNH_INTEGER_MIN;
	b->fchk = knh_fenumchk__nop;
	b->fcmp = knh_fenumcmp__signed;
	b->ffmt = knh_fenumfmt__signed;
}

/* ------------------------------------------------------------------------ */

static
void knh_IntUnit_traverse(Ctx *ctx, IntUnit *u, knh_ftraverse gc)
{
	knh_IntUnit_struct *b = DP(u);
	gc(ctx, UP(b->spec.urn));
	gc(ctx, UP(b->spec.tag));
	gc(ctx, b->spec.defvalue);
}

/* ======================================================================== */
/* FloatUnit */

#define knh_FloatUnit_init_ NULL
#define knh_FloatUnit_copy NULL
#define knh_FloatUnit_traverse_ NULL
#define knh_FloatUnit_compareTo NULL
#define knh_FloatUnit_hashCode NULL
#define knh_FloatUnit_newClass NULL

/* ------------------------------------------------------------------------ */

static
knh_bool_t knh_funitchk__default(FloatUnit *b, knh_float_t v)
{
	return 1;
}

/* ------------------------------------------------------------------------ */

static
int knh_funitcmp__default(FloatUnit *b, knh_float_t v1, knh_float_t v2)
{
	knh_float_t delta = v1 - v2;
	if(delta == 0.0) return 0;
	return delta < 0 ? -1 : 1;
}

/* ------------------------------------------------------------------------ */

static
void knh_funitfmt__default(FloatUnit *b, char *buf, size_t bufsiz, knh_float_t v)
{
	knh_snprintf(buf, bufsiz, KNH_FLOAT_FMT, v);
}

/* ------------------------------------------------------------------------ */

static
void knh_FloatUnit_init(Ctx *ctx, FloatUnit *u, int init)
{
	knh_FloatUnit_struct *b = DP(u);
	b->spec.flag = 0;
	b->spec.cid  = CLASS_Float;
	KNH_INITv(b->spec.urn,  TS_EMPTY);
	KNH_INITv(b->spec.tag,  TS_EMPTY);
	KNH_INITv(b->spec.defvalue, KNH_FLOAT0);
	//
	b->step = 0.0;
	b->max  = KNH_FLOAT_MAX;
	b->min  = KNH_FLOAT_MIN;
	b->fchk = knh_funitchk__default;
	b->fcmp = knh_funitcmp__default;
	b->ffmt = knh_funitfmt__default;
	b->FMT  = KNH_FLOAT_FMT;
}

/* ------------------------------------------------------------------------ */

static
void knh_FloatUnit_traverse(Ctx *ctx, FloatUnit *u, knh_ftraverse gc)
{
	knh_FloatUnit_struct *b = DP(u);
	gc(ctx, UP(b->spec.urn));
	gc(ctx, UP(b->spec.tag));
	gc(ctx, b->spec.defvalue);
}

/* ======================================================================== */
/* Regex */

#define knh_Regex_init_ NULL
#define knh_Regex_copy NULL
#define knh_Regex_traverse_ NULL
#define knh_Regex_compareTo NULL
#define knh_Regex_hashCode NULL
#define knh_Regex_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_Regex_init(Ctx *ctx, Regex *o, int init)
{
	KNH_INITv(o->pattern, TS_EMPTY);
	o->df = knh_System_getRegexDriver(ctx, STEXT("NOP"));
	o->reg = NULL;
}

/* ------------------------------------------------------------------------ */

static
void knh_Regex_traverse(Ctx *ctx, Regex *o, knh_ftraverse ftr)
{

	ftr(ctx, UP(o->pattern));
	if(IS_SWEEP(ftr)) {
		if(o->reg != NULL) {
			o->df->regfree(ctx, o->reg);
			o->df = NULL;
			o->reg = NULL;
		}
	}
}

/* ======================================================================== */
/* BytesConv */

#define knh_BytesConv_init_ NULL
#define knh_BytesConv_copy NULL
#define knh_BytesConv_traverse_ NULL
#define knh_BytesConv_compareTo NULL
#define knh_BytesConv_hashCode NULL
#define knh_BytesConv_newClass NULL

size_t f_bconv__NOP(Ctx *ctx, BytesConv *bc, knh_bytes_t t, knh_Bytes_t *ba)
{
	return 0;
}

/* ------------------------------------------------------------------------ */

static
void knh_BytesConv_init(Ctx *ctx, BytesConv *bc, int init)
{
	KNH_INITv(DP(bc)->name, TS_EMPTY);
	DP(bc)->fbconv = f_bconv__NOP;
#ifdef KNH_USING_ICONV
	DP(bc)->iconv_d = ((iconv_t)-1);
#endif
}

/* ------------------------------------------------------------------------ */

static
void knh_BytesConv_traverse(Ctx *ctx, BytesConv *bc, knh_ftraverse ftr)
{
	ftr(ctx, UP(DP(bc)->name));
	if(IS_SWEEP(ftr)) {
#ifdef KNH_USING_ICONV
		if(DP(bc)->iconv_d != ((iconv_t)-1)) {
			iconv_close(DP(bc)->iconv_d);
			DP(bc)->fbconv = f_bconv__NOP;
		}
#endif
	}
}

/* ======================================================================== */
/* StringUnit */

#define knh_StringUnit_init_ NULL
#define knh_StringUnit_copy NULL
#define knh_StringUnit_traverse_ NULL
#define knh_StringUnit_compareTo NULL
#define knh_StringUnit_hashCode NULL
#define knh_StringUnit_newClass NULL

/* ------------------------------------------------------------------------ */

static
int knh_fvcabcmp__default(StringUnit *o, knh_bytes_t v1, knh_bytes_t v2)
{
	return knh_bytes_strcmp(v1, v2);
}

/* ------------------------------------------------------------------------ */

static
void knh_StringUnit_init(Ctx *ctx, StringUnit *u, int init)
{
	knh_StringUnit_struct *b = DP(u);

	b->spec.flag = 0;
	b->spec.cid  = CLASS_newid;
	KNH_INITv(b->spec.urn, TS_EMPTY);
	KNH_INITv(b->spec.tag,  TS_EMPTY);
	KNH_INITv(b->spec.defvalue, TS_EMPTY);
	//
	b->fnew = new_StringX__fast;
	b->fcmp = knh_fvcabcmp__default;
	b->fbconv = NULL;
	KNH_INITv(b->bconv, KNH_NULL);
	b->charlen = 0;
	b->bytelen = 0;
	KNH_INITv(b->pattern, KNH_NULL);
	KNH_INITv(b->vocabDictIdx, KNH_NULL);
}

/* ------------------------------------------------------------------------ */

static
void knh_StringUnit_traverse(Ctx *ctx, StringUnit *u, knh_ftraverse gc)
{
	knh_StringUnit_struct *b = DP(u);
	gc(ctx, UP(b->spec.urn));
	gc(ctx, UP(b->spec.tag));
	gc(ctx, b->spec.defvalue);

	gc(ctx, UP(b->bconv));
	gc(ctx, UP(b->pattern));
	gc(ctx, UP(b->vocabDictIdx));
}


/* ======================================================================== */
/* InputStream */

#define knh_InputStream_init_ NULL
#define knh_InputStream_copy NULL
#define knh_InputStream_traverse_ NULL
#define knh_InputStream_compareTo NULL
#define knh_InputStream_hashCode NULL
#define knh_InputStream_newClass NULL

static
void knh_InputStream_init(Ctx *ctx, InputStream *in, int init)
{
	knh_InputStream_struct *b = DP(in);
	b->fd = -1;
	KNH_INITv(b->ba, KNH_NULL);
	b->buf = "";
	b->bufpos = 0;
	b->bufend = 0;
	b->bufsiz = 0;
	b->driver = knh_System_getDefaultIODriver();
	KNH_INITv(b->bconv, KNH_NULL);
	KNH_INITv(b->enc, TS_ENCODING);
	KNH_INITv(b->urn, TS_DEVNULL);
	b->size    = 0;
	b->line    = 1;
	b->prev    = '\n';
	b->fileid   = 0;
}

/* ------------------------------------------------------------------------ */

static
void knh_InputStream_traverse(Ctx *ctx, InputStream *in, knh_ftraverse ftr)
{
	knh_InputStream_struct *b = DP(in);
	if(IS_SWEEP(ftr) && b->fd != -1) {
		b->driver.fclose(ctx, b->fd);
		b->fd = -1;
	}
	ftr(ctx, UP(b->ba));
	ftr(ctx, UP(b->enc));
	ftr(ctx, UP(b->bconv));
	ftr(ctx, UP(b->urn));
}

/* ======================================================================== */
/* OutputStream */

#define knh_OutputStream_init_ NULL
#define knh_OutputStream_copy NULL
#define knh_OutputStream_traverse_ NULL
#define knh_OutputStream_compareTo NULL
#define knh_OutputStream_hashCode NULL
#define knh_OutputStream_newClass NULL

void
knh_OutputStream_init(Ctx *ctx, OutputStream *out, int init)
{
	knh_OutputStream_struct *b = DP(out);
	b->fd = -1;
	b->driver = knh_System_getDefaultIODriver();
	if(init == -1) {
		KNH_INITv(b->ba, KNH_NULL);
	}
	else {
		KNH_INITv(b->ba, new_Bytes(ctx, 4096));
	}
	KNH_INITv(b->bconv, KNH_NULL);
	KNH_INITv(b->enc, TS_ENCODING);
	KNH_INITv(b->urn, TS_DEVNULL);
	b->size = 0;
	b->line = 0;
	KNH_INITv(b->NEWLINE, TS_LF);
	KNH_INITv(b->TAB, TS_TAB);
	b->indent = 0;
	b->flag = 0;
}

/* ------------------------------------------------------------------------ */

static
void knh_OutputStream_traverse(Ctx *ctx, OutputStream *out, knh_ftraverse gc)
{
	knh_OutputStream_struct *b = DP(out);
	if(IS_SWEEP(gc) && b->fd != -1) {
		b->driver.fclose(ctx, b->fd);
		b->fd = -1;
	}
	gc(ctx, UP(b->ba));
	gc(ctx, UP(b->enc));
	gc(ctx, UP(b->bconv));
	gc(ctx, UP(b->urn));
	gc(ctx, UP(b->NEWLINE));
	gc(ctx, UP(b->TAB));
}

/* ======================================================================== */
/* Connection */

#define knh_Connection_init_ NULL
#define knh_Connection_copy NULL
#define knh_Connection_traverse_ NULL
#define knh_Connection_compareTo NULL
#define knh_Connection_hashCode NULL
#define knh_Connection_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_Connection_init(Ctx *ctx, Connection *c, int init)
{
	knh_Connection_struct *b = DP(c);
	b->conn = NULL;
	KNH_INITv(b->urn, TS_EMPTY);
	b->df = knh_System_getDefaultDBDriver();
}

/* ------------------------------------------------------------------------ */

static
void knh_Connection_traverse(Ctx *ctx, Connection *c, knh_ftraverse ftr)
{
	knh_Connection_struct *b = DP(c);
	ftr(ctx, UP(b->urn));
	if(IS_SWEEP(ftr) && b->conn != NULL) {
		b->df->dbclose(ctx, b->conn);
		b->conn = NULL;
		b->df = NULL;
	}
}


/* ======================================================================== */
/* ResultSet */

#define knh_ResultSet_init_ NULL
#define knh_ResultSet_copy NULL
#define knh_ResultSet_traverse_ NULL
#define knh_ResultSet_compareTo NULL
#define knh_ResultSet_hashCode NULL
#define knh_ResultSet_newClass NULL

static
void knh_ResultSet_init(Ctx *ctx, ResultSet *rs, int init)
{
	knh_ResultSet_struct *b = DP(rs);
	b->dbcur = NULL;
	b->tcid = CLASS_ResultSet;
	b->column_size = 0;
	b->column = NULL;
	KNH_INITv(b->databuf, new_Bytes(ctx, 256));
	KNH_INITv(b->conn, KNH_NULL);
	b->dbcur_free = knh_dbcurfree__NOP;
	b->count = 0;
}

/* ------------------------------------------------------------------------ */

static
void knh_ResultSet_traverse(Ctx *ctx, ResultSet *rs, knh_ftraverse ftr)
{
	knh_ResultSet_struct *b = DP(rs);
	ftr(ctx, UP(b->databuf));
	int i = 0;
	for(i = 0; i < b->column_size; i++) {
		ftr(ctx, UP(b->column[i].name));
	}
	if(IS_SWEEP(ftr)) {
		if(b->column != NULL) {
			KNH_FREE(ctx, b->column, sizeof(knh_dbschema_t) * b->column_size);
			b->column = NULL;
		}
		if(b->dbcur != NULL) {
			b->dbcur_free(b->dbcur);
			b->dbcur = NULL;
		}
	}
	ftr(ctx, UP(b->conn));
}

/* ======================================================================== */
/* Exception */

#define knh_Exception_init_ NULL
#define knh_Exception_copy NULL
#define knh_Exception_traverse_ NULL
#define knh_Exception_compareTo NULL
#define knh_Exception_hashCode NULL
#define knh_Exception_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_Exception_init(Ctx *ctx, Exception *e, int init)
{
	knh_Exception_struct *b = DP(e);
	b->eid  = 1;
	b->flag = 0;
	KNH_INITv(b->message, TS_EMPTY);
	KNH_INITv(b->bag, KNH_NULL);
	KNH_INITv(b->traces, KNH_NULL);
	b->file = "";
	b->line = 0;
}

/* ------------------------------------------------------------------------ */

static
void knh_Exception_traverse(Ctx *ctx, Exception *e, knh_ftraverse ftr)
{
	knh_Exception_struct *b = DP(e);
	ftr(ctx, UP(b->message));
	ftr(ctx, b->bag);
	ftr(ctx, UP(b->traces));
}


/* ======================================================================== */
/* ExceptionHandler */

#define knh_ExceptionHandler_init_ NULL
#define knh_ExceptionHandler_copy NULL
#define knh_ExceptionHandler_traverse_ NULL
#define knh_ExceptionHandler_compareTo NULL
#define knh_ExceptionHandler_hashCode NULL
#define knh_ExceptionHandler_newClass NULL

static
void knh_ExceptionHandler_init(Ctx *ctx, ExceptionHandler *hdr, int init)
{
	knh_ExceptionHandler_struct *b = DP(hdr);
	b->esp = ctx->esp;
	KNH_INITv(b->caught, KNH_NULL);
}

/* ------------------------------------------------------------------------ */

void
knh_ExceptionHandler_traverse(Ctx *ctx, ExceptionHandler *hdr, knh_ftraverse gc)
{
	knh_ExceptionHandler_struct *b = DP(hdr);
	gc(ctx, UP(b->caught));
}

/* ======================================================================== */
/* Script */

#define knh_Script_init_ NULL
#define knh_Script_copy NULL
#define knh_Script_traverse_ NULL
#define knh_Script_compareTo NULL
#define knh_Script_hashCode NULL
#define knh_Script_newClass NULL

static
void knh_Script_init(Ctx *ctx, Script *s, int init)
{
	int i;
	s->fields = (Object**)KNH_MALLOC(ctx, sizeof(Object*) * KNH_SCRIPT_FIELDSIZE);
	for(i = 0; i < KNH_SCRIPT_FIELDSIZE; i++) {
		KNH_INITv(s->fields[i], KNH_NULL);
	}
}

/* ------------------------------------------------------------------------ */

static
void knh_Script_traverse(Ctx *ctx, Script *s, knh_ftraverse ftr)
{
	int i;
	for(i = 0; i < KNH_SCRIPT_FIELDSIZE; i++) {
		ftr(ctx, s->fields[i]);
	}
//	automatically free
//	if(IS_SWEEP(ftr)) {
//		KNH_FREE(ctx, s->fields, sizeof(Object*) * KNH_SCRIPT_FIELDSIZE);
//	}
}

/* ======================================================================== */
/* NameSpace */

#define knh_NameSpace_init_ NULL
#define knh_NameSpace_copy NULL
#define knh_NameSpace_traverse_ NULL
#define knh_NameSpace_compareTo NULL
#define knh_NameSpace_hashCode NULL
#define knh_NameSpace_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_NameSpace_init(Ctx *ctx, NameSpace *ns, int init)
{
	knh_NameSpace_struct *b = DP(ns);
	KNH_INITv(b->parent, KNH_NULL);
	KNH_INITv(b->nsname, KNH_NULL);
	KNH_INITv(b->script, KNH_NULL);
	KNH_INITv(b->imports, KNH_NULL);
	KNH_INITv(b->lconstDictMap, KNH_NULL);
	KNH_INITv(b->name2cidDictSet, KNH_NULL);
	KNH_INITv(b->tag2urnDictMap, KNH_NULL);
	KNH_INITv(b->func2cidDictSet, KNH_NULL);
	b->dlhdr = NULL;
	b->pkgmeta = NULL;
}

/* ------------------------------------------------------------------------ */

static
void knh_NameSpace_traverse(Ctx *ctx, NameSpace *ns, knh_ftraverse ftr)
{
	knh_NameSpace_struct *b = DP(ns);
	ftr(ctx, UP(b->parent));
	ftr(ctx, UP(b->nsname));
	ftr(ctx, UP(b->script));
	ftr(ctx, UP(b->imports));
	ftr(ctx, UP(b->lconstDictMap));
	ftr(ctx, UP(b->name2cidDictSet));
	ftr(ctx, UP(b->tag2urnDictMap));
	ftr(ctx, UP(b->func2cidDictSet));
	if(b->pkgmeta != NULL && b->pkgmeta->ftraverse != NULL) {
		b->pkgmeta->ftraverse(ctx, ftr);
	}
	if(IS_SWEEP(ftr)) {
		if(b->dlhdr != NULL) {
			knh_dlclose(ctx, b->dlhdr);
			b->dlhdr = NULL;
		}
	}
}

/* ======================================================================== */
/* System */

#define knh_System_init_ NULL
#define knh_System_copy NULL
#define knh_System_traverse_ NULL
#define knh_System_compareTo NULL
#define knh_System_hashCode NULL
#define knh_System_newClass NULL

static knh_uint_t knh_autoSystemId = 0;

/* ------------------------------------------------------------------------ */


static
void knh_System_init(Ctx *ctx, System *o, int init)
{
	knh_System_struct *sys = DP(o);
	char buf[FILENAME_BUFSIZ];
	sys->sysid = knh_autoSystemId++;
	sys->ctxcount = 0;

	KNH_INITv(sys->ns, new_NameSpace(ctx, (NameSpace*)KNH_NULL, TS_EMPTY));
	KNH_INITv(sys->texptNameDictSet, new_DictSet(ctx, KNH_TEXPT_SIZE));

	KNH_INITv(sys->enc,   new_String__T(ctx, konoha_encoding()));
	KNH_INITv(sys->in,    new_InputStream__stdio(ctx, stdin, sys->enc));
	KNH_INITv(sys->out,   new_OutputStream__stdio(ctx, stdout, sys->enc));
	KNH_INITv(sys->err,   new_OutputStream__stdio(ctx, stderr, sys->enc));

	KNH_INITv(sys->props, new_DictMap0(ctx, 64));

	KNH_INITv(sys->tConstDictMap, new_DictMap0(ctx, 256));

	KNH_INITv(sys->tfieldnDictIdx, new_DictIdx0__ignoreCase(ctx, KNH_TFIELDN_SIZE * 2, 0));
	KNH_INITv(sys->tfileidDictIdx, new_DictIdx0(ctx, 32, 0));
	knh_DictIdx_add__fast(ctx, sys->tfileidDictIdx, new_String__T(ctx, "(unknown)"));
	KNH_INITv(sys->tMethodFields, new_Array0(ctx, KNH_TMETHODFIELD_SIZE * 2));

	KNH_INITv(sys->homeDir, new_String(ctx, B(knh_format_homepath(buf,sizeof(buf))), NULL));
	KNH_INITv(sys->sysnsDictMap, new_DictMap0(ctx, 16));
	KNH_INITv(sys->funcDictSet, new_DictSet(ctx, 16));
	KNH_INITv(sys->drvapiDictSet, new_DictSet(ctx, 32));
	KNH_INITv(sys->makespecDictSet, new_DictSet(ctx, 32));

	KNH_INITv(sys->tmetaHashMap, KNH_NULL /*new_HashMap(ctx, 0)*/);
	KNH_INITv(sys->tnameSpaceDictMap, new_DictMap0(ctx, 8));
	KNH_INITv(sys->turnAliasDictMap, new_DictMap0(ctx, 8));

	knh_DictMap_set(ctx, sys->props, new_String__T(ctx, "konoha.encoding"), UP(sys->enc));
	knh_DictMap_set(ctx, sys->props, new_String__T(ctx, "konoha.path.home"), UP(sys->homeDir));
	knh_DictMap_set(ctx, sys->props, new_String__T(ctx, "konoha.path.package"),
			UP(new_String(ctx, B(knh_format_packagedir(buf, sizeof(buf),knh_String_tochar(sys->homeDir))), NULL)));
	knh_DictMap_set(ctx, sys->props, new_String__T(ctx, "konoha.lang"),
			UP(new_String(ctx, B(knh_format_lang(buf, sizeof(buf))), NULL)));
	knh_Asm_setLang(buf);

}

/* ------------------------------------------------------------------------ */

static void knh_System_traverse(Ctx *ctx, System *o, knh_ftraverse ftr)
{
	knh_System_struct *sys = DP(o);

	ftr(ctx, UP(sys->ns));
	ftr(ctx, UP(sys->texptNameDictSet));
	ftr(ctx, UP(sys->enc));
	ftr(ctx, UP(sys->in));
	ftr(ctx, UP(sys->out));
	ftr(ctx, UP(sys->err));

	ftr(ctx, UP(sys->props));
	ftr(ctx, UP(sys->tConstDictMap));

	ftr(ctx, UP(sys->tfieldnDictIdx));
	ftr(ctx, UP(sys->tfileidDictIdx));
	ftr(ctx, UP(sys->tMethodFields));

	ftr(ctx, UP(sys->tmetaHashMap));
	ftr(ctx, UP(sys->tnameSpaceDictMap));
	ftr(ctx, UP(sys->turnAliasDictMap));

	ftr(ctx, UP(sys->homeDir));
	ftr(ctx, UP(sys->sysnsDictMap));
	ftr(ctx, UP(sys->funcDictSet));
	ftr(ctx, UP(sys->drvapiDictSet));
	ftr(ctx, UP(sys->makespecDictSet));
}

/* ======================================================================== */
/* Context */

#define knh_Context_init_ NULL
#define knh_Context_copy NULL
#define knh_Context_traverse_ NULL
#define knh_Context_compareTo NULL
#define knh_Context_hashCode NULL
#define knh_Context_newClass NULL

/* ------------------------------------------------------------------------ */

void knh_Context_init(Ctx *ctx, knh_Context_t *o, int init)
{
	/* Context can be only instatiated in new_Context() */
	KNH_THROWs(ctx, "Abort!!");
}


/* ======================================================================== */
/* Token */

#define knh_Token_init_ NULL
#define knh_Token_copy NULL
#define knh_Token_traverse_ NULL
#define knh_Token_compareTo NULL
#define knh_Token_hashCode NULL
#define knh_Token_newClass NULL

/* ======================================================================== */
/* [struct] */

static
void knh_Token_init(Ctx *ctx, Token *tk, int init)
{
	knh_Token_struct *b = DP(tk);
	b->tt_next = TT_EOT;
	b->type    =  CLASS_Any;
	KNH_INITv(b->data, KNH_NULL);
}

/* ------------------------------------------------------------------------ */

static
void knh_Token_traverse(Ctx *ctx, Token *tk, knh_ftraverse gc)
{
	knh_Token_struct *b = DP(tk);
	gc(ctx, b->data);
}

/* ======================================================================== */
/* Stmt */

#define knh_Stmt_init_ NULL
#define knh_Stmt_copy NULL
#define knh_Stmt_traverse_ NULL
#define knh_Stmt_compareTo NULL
#define knh_Stmt_hashCode NULL
#define knh_Stmt_newClass NULL

static
void knh_Stmt_init(Ctx *ctx, Stmt *stmt, int init)
{
	knh_Stmt_struct *b = DP(stmt);
	b->size = 0;
	b->capacity = init;
	b->type = CLASS_Any;
	b->line_end = 0;
	if(b->capacity > 0) {
		b->terms = (Object**)KNH_MALLOC(ctx, sizeof(Term*) * b->capacity);
		size_t i;
		for(i = 0; i < b->capacity; i++) {
			KNH_INITv(b->terms[i], KNH_NULL);
		}
	}else {
		b->terms = NULL;
	}
	KNH_INITv(b->metaDictMap,  KNH_NULL);
	KNH_INITv(b->next,  KNH_NULL);
}

/* ------------------------------------------------------------------------ */

static
void knh_Stmt_terms_traverse(Ctx *ctx, knh_Stmt_struct *b, knh_ftraverse ftr)
{
	size_t i;
	for(i = 0; i < b->capacity; i++) {
		ftr(ctx, b->terms[i]);
	}
	if(IS_SWEEP(ftr)) {
		KNH_FREE(ctx, b->terms, sizeof(Term*) * b->capacity);
		b->terms = NULL;
		b->capacity = 0;
		b->size = 0;
	}
}

/* ------------------------------------------------------------------------ */

void knh_Stmt_done(Ctx *ctx, Stmt *o)
{
	SP(o)->stt = STT_DONE;
	if(DP(o)->terms != NULL) {
		knh_Stmt_terms_traverse(ctx, DP(o), knh_Object_sweep);
	}
	KNH_SETv(ctx, DP(o)->metaDictMap, KNH_NULL);
}

/* ------------------------------------------------------------------------ */

static
void knh_Stmt_traverse(Ctx *ctx, Stmt *stmt, knh_ftraverse ftr)
{
	knh_Stmt_struct *b = DP(stmt);
	if(b->terms != NULL) {
		knh_Stmt_terms_traverse(ctx, b, ftr);
	}
	ftr(ctx, UP(b->metaDictMap));
	ftr(ctx, UP(b->next));
}

/* ======================================================================== */
/* Asm */

#define knh_Asm_init_ NULL
#define knh_Asm_copy NULL
#define knh_Asm_traverse_ NULL
#define knh_Asm_compareTo NULL
#define knh_Asm_hashCode NULL
#define knh_Asm_newClass NULL

static
void knh_Asm_init(Ctx *ctx, Asm *abr, int init)
{
	knh_Asm_struct *b = DP(abr);
	b->flag = 0;
	b->this_cid = CLASS_Object;

	KNH_INITv(b->ns, knh_Runtime_getNameSpace(ctx, STEXT("main")));
	KNH_INITv(b->mtd,   KNH_NULL);
	b->level = 0;

	knh_int_t i;
	b->vars = (knh_cfield_t*)KNH_MALLOC(ctx, KONOHA_LOCALSIZE * sizeof(knh_cfield_t));
	b->vars_size = 0;
	for(i = 0; i < KONOHA_LOCALSIZE; i++) {
		b->vars[i].flag  = 0;
		b->vars[i].type  = TYPE_Any;
		b->vars[i].fn    = FIELDN_NONAME;
//		b->vars[i].count = 0;
		KNH_INITv(b->vars[i].value, KNH_NULL);
	}

	KNH_INITv(b->registeredStmts, new_Array0(ctx, 8));
	b->stack = 0;
	b->globalidx = -1;

	b->labels = NULL;
	b->labelmax = 0;
	b->labelcapacity = 0;
	KNH_INITv(b->name2labelIdDictSet, new_DictSet(ctx, 256));

	KNH_INITv(b->lstacks, new_Array(ctx, CLASS_String, 8));
	KNH_INITv(b->finallyStmt, KNH_NULL);

	b->fileid = 0;
	b->line = 0;
	KNH_INITv(b->elf, new_Bytes(ctx, 4096));
	KNH_INITv(b->dwarf, new_Bytes(ctx, 1024));

	KNH_INITv(b->constPools, KNH_NULL);
	KNH_INITv(b->exportsMethods, KNH_NULL);
}

/* ------------------------------------------------------------------------ */

static
void knh_Asm_traverse(Ctx *ctx, Asm *abr, knh_ftraverse ftr)
{
	knh_Asm_struct *b = DP(abr);
	size_t i;
	for(i = 0; i < KONOHA_LOCALSIZE; i++) {
		ftr(ctx, b->vars[i].value);
	}
	for(i = 0; i < b->labelcapacity; i++) {
		ftr(ctx, UP(b->labels[i].tklabel));
	}
	if(IS_SWEEP(ftr)) {
		KNH_FREE(ctx, b->vars, KONOHA_LOCALSIZE * sizeof(knh_cfield_t));
		if(b->labels != NULL) {
			KNH_FREE(ctx, b->labels, b->labelcapacity * sizeof(knh_labeltbl_t));
		}
	}

	ftr(ctx, UP(b->ns));
	ftr(ctx, UP(b->mtd));
	ftr(ctx, UP(b->elf));
	ftr(ctx, UP(b->dwarf));
	ftr(ctx, UP(b->registeredStmts));
	ftr(ctx, UP(b->name2labelIdDictSet));
	ftr(ctx, UP(b->lstacks));
	ftr(ctx, UP(b->finallyStmt));

	ftr(ctx, UP(b->exportsMethods));
	ftr(ctx, UP(b->constPools));
}

/* ======================================================================== */
/* KLRCode */

#define knh_KLRCode_init_ NULL
#define knh_KLRCode_copy NULL
#define knh_KLRCode_traverse_ NULL
#define knh_KLRCode_compareTo NULL
#define knh_KLRCode_hashCode NULL
#define knh_KLRCode_newClass NULL

/* ------------------------------------------------------------------------ */

static
void knh_KLRCode_init(Ctx *ctx, KLRCode *o, int init)
{
	knh_KLRCode_struct *b = DP(o);
	b->size = 0;
	b->code = (knh_code_t*)"";
	b->fileid = 0;
	b->nsid = 0;
	b->dwarf = NULL;
	b->dsize = 0;
}

/* ------------------------------------------------------------------------ */

void knh_code_traverse(Ctx *ctx, knh_code_t *pc, knh_ftraverse ftr);

/* ------------------------------------------------------------------------ */

static
void knh_KLRCode_traverse(Ctx *ctx, KLRCode *o, knh_ftraverse ftr)
{
	knh_KLRCode_struct *b = DP(o);
	knh_code_traverse(ctx, b->code, ftr);
	if(IS_SWEEP(ftr)) {
		KNH_FREE(ctx, b->code, KNH_SIZE(b->size));
		if(b->dwarf != NULL) {
			KNH_FREE(ctx, b->dwarf, b->dsize * sizeof(knh_dwarf_t));
		}
	}
}

/* ======================================================================== */
/* [default] */

#define knh_Object_fdefault NULL
#define knh_Any_fdefault NULL
#define knh_This_fdefault NULL
#define knh_Any0_fdefault NULL
#define knh_Any1_fdefault NULL
#define knh_Any2_fdefault NULL
#define knh_Any3_fdefault NULL
#define knh_Nue_fdefault NULL
#define knh_Boolean_fdefault_ NULL
#define knh_Number_fdefault NULL
#define knh_Int_fdefault_ NULL
#define knh_IntX_fdefault_ NULL
#define knh_Float_fdefault_ NULL
#define knh_FloatX_fdefault_ NULL
#define knh_String_fdefault_ NULL
#define knh_StringX_fdefault_ NULL
#define knh_Bytes_fdefault NULL
#define knh_Tuple2_fdefault NULL
#define knh_Range_fdefault NULL
#define knh_Array_fdefault NULL
#define knh_IArray_fdefault NULL
#define knh_FArray_fdefault NULL
#define knh_Iterator_fdefault NULL
#define knh_DictMap_fdefault NULL
#define knh_DictSet_fdefault NULL
#define knh_HashMap_fdefault NULL
#define knh_HashSet_fdefault NULL
#define knh_LRUHashMap_fdefault NULL
#define knh_DictIdx_fdefault NULL
#define knh_Class_fdefault_ NULL
#define knh_ClassStruct_fdefault NULL
#define knh_MethodField_fdefault NULL
#define knh_Method_fdefault NULL
#define knh_Mapper_fdefault NULL
#define knh_ClassMap_fdefault NULL
#define knh_Closure_fdefault NULL
#define knh_AffineConv_fdefault NULL
#define knh_IntUnit_fdefault NULL
#define knh_FloatUnit_fdefault NULL
#define knh_Regex_fdefault NULL
#define knh_BytesConv_fdefault NULL
#define knh_StringUnit_fdefault NULL
#define knh_InputStream_fdefault NULL
#define knh_OutputStream_fdefault NULL
#define knh_Connection_fdefault NULL
#define knh_ResultSet_fdefault NULL
#define knh_Exception_fdefault NULL
#define knh_ExceptionHandler_fdefault_ NULL
#define knh_Script_fdefault NULL
#define knh_NameSpace_fdefault_ NULL
#define knh_System_fdefault_ NULL
#define knh_Context_fdefault_ NULL
#define knh_Token_fdefault NULL
#define knh_Stmt_fdefault NULL
#define knh_Asm_fdefault NULL
#define knh_KLRCode_fdefault NULL

/* ------------------------------------------------------------------------ */

static
Object *knh_Boolean_fdefault(Ctx *ctx, knh_class_t cid)
{
	return KNH_FALSE;
}

/* ------------------------------------------------------------------------ */

static
Object *knh_Class_fdefault(Ctx *ctx, knh_class_t cid)
{
	return (Object*)ctx->tClass[CLASS_Class].class_cid;
}

/* ------------------------------------------------------------------------ */

static
Object *knh_Context_fdefault(Ctx *ctx, knh_class_t cid)
{
	return (Object*)ctx;
}

/* ------------------------------------------------------------------------ */

static
Object *knh_ExceptionHandler_fdefault(Ctx *ctx, knh_class_t cid)
{
	return (Object*)new_Object_bcid(ctx, CLASS_ExceptionHandler, 0);
}


/* ------------------------------------------------------------------------ */

static
Object *knh_Float_fdefault(Ctx *ctx, knh_class_t cid)
{
	return (Object*)KNH_FLOAT0;
}

/* ------------------------------------------------------------------------ */

static
Object *knh_FloatX_fdefault(Ctx *ctx, knh_class_t cid)
{
	FloatUnit *o = (FloatUnit*)ctx->tClass[cid].cspec;
	KNH_ASSERT(IS_FloatUnit(o));
	return DP(o)->spec.defvalue;
}

/* ------------------------------------------------------------------------ */

static
Object *knh_Int_fdefault(Ctx *ctx, knh_class_t cid)
{
	return (Object*)KNH_INT0;
}

/* ------------------------------------------------------------------------ */

static
Object *knh_IntX_fdefault(Ctx *ctx, knh_class_t cid)
{
	IntUnit *o = (IntUnit*)ctx->tClass[cid].cspec;
	KNH_ASSERT(IS_IntUnit(o));
	return DP(o)->spec.defvalue;
}

/* ------------------------------------------------------------------------ */

static
Object *knh_String_fdefault(Ctx *ctx, knh_class_t cid)
{
	return (Object*)TS_EMPTY;
}

/* ------------------------------------------------------------------------ */

static
Object *knh_StringX_fdefault(Ctx *ctx, knh_class_t cid)
{
	StringUnit *u = (StringUnit*)ctx->tClass[cid].cspec;
	KNH_ASSERT(IS_StringUnit(u));
	return DP(u)->spec.defvalue;
}

/* ------------------------------------------------------------------------ */

static
Object *knh_NameSpace_fdefault(Ctx *ctx, knh_class_t cid)
{
	return (Object*)DP(ctx->sys)->ns;
}

/* ------------------------------------------------------------------------ */

static
Object *knh_System_fdefault(Ctx *ctx, knh_class_t cid)
{
	return (Object*)ctx->sys;
}

/* ======================================================================== */
/* [commons] */

#ifdef __cplusplus
}
#endif

#define KNH_CC_METHODAPI
#include"../gen/method_.c"
#include"../api/objectapi.c"
#include"../api/numberapi.c"
#include"../api/stringapi.c"

#include"../api/bytesapi.c"
#include"../api/arrayapi.c"
#include"../api/dictapi.c"
#include"../api/hashapi.c"

#include"../api/systemapi.c"
#include"../api/streamapi.c"
#include"../api/dbapi.c"

#include<konoha/gen/struct_.h>

#ifdef __cplusplus
extern "C" {
#endif

/* ======================================================================== */
/* [load] */

char *konoha_getStructName(Ctx *ctx, knh_struct_t sid)
{
	if(sid < (sizeof(StructName) / sizeof(char*))) {
		return StructName[sid];
	}
	DBG2_P("unknown sid=%d", sid);
	return "STRUCT_unknown";
}

/* ------------------------------------------------------------------------ */

static
void konoha_loadStringData(Ctx *ctx, knh_StringData_t *data)
{
	while(data->name != NULL) {
		KNH_ASSERT(ctx->tString[data->index] == NULL);
		KNH_ASSERT(data->index < (SIZEOF_TSTRING / sizeof(String*)));
		KNH_INITv(ctx->tString[data->index], new_String__T(ctx, data->name));
		data++;
	}
}

/* ------------------------------------------------------------------------ */

void konoha_loadSystemString(Ctx *ctx)
{
	konoha_loadStringData(ctx, StringData);
}

/* ------------------------------------------------------------------------ */
/* StructData */

/* ------------------------------------------------------------------------ */

static void knh_finit__default(Ctx *ctx, Object *o, int init)
{
}

/* ------------------------------------------------------------------------ */

static void knh_ftraverse__default(Ctx *ctx, Object *o, knh_ftraverse f)
{
}

/* ------------------------------------------------------------------------ */

static int knh_fcompareTo__default(Ctx *ctx, Object *o1, Object *o2)
{
	return ((int)o1 - (int)o2);
}

/* ------------------------------------------------------------------------ */

static Object *knh_fcopy__default(Ctx *ctx, Object *o)
{
	return o;
}

/* ------------------------------------------------------------------------ */

static knh_hcode_t knh_fhashCode__default(Ctx *ctx, Object* o1)
{
	return ((knh_hcode_t)o1) / sizeof(Object*);
}

/* ------------------------------------------------------------------------ */

static void knh_fnewClass__default(Ctx *ctx, knh_class_t cid)
{
}

/* ------------------------------------------------------------------------ */

KNHAPI(void) konoha_loadStructData(Ctx *ctx, knh_StructData_t *data)
{
	KNH_LOCK(ctx, ctx->tableLock);
	knh_StructData_t *d = data;
	while(d->name != NULL) {
		if(!(ctx->share->tStructSize < KNH_TSTRUCT_SIZE)) {
			KNH_UNLOCK(ctx, ctx->tableLock);
			KNH_EXIT("enlarge KNH_TSTRUCT_SIZE %d", KNH_TSTRUCT_SIZE);
			return;
		}
		KNH_ASSERT(d->size % sizeof(Object*) == 0);
		int sid = d->sid;
		if(sid == STRUCT_newid) sid = ctx->share->tStructSize;
		KNH_ASSERT(sid == ctx->share->tStructSize);
		ctx->share->tStructSize++;
		ctx->tStruct[sid].name = d->name;
		ctx->tStruct[sid].size = d->size;
		ctx->tStruct[sid].flag = KNH_FLAG_CF2OF(d->flag);
		ctx->tStruct[sid].sid  = (knh_struct_t)sid;
		ctx->tStruct[sid].finit = (d->finit != NULL) ? d->finit : knh_finit__default;
		ctx->tStruct[sid].ftraverse = (d->ftraverse != NULL) ? d->ftraverse : knh_ftraverse__default;
		ctx->tStruct[sid].fcopy = (d->fcopy != NULL) ? d->fcopy : knh_fcopy__default;
		ctx->tStruct[sid].fcompareTo = (d->fcompareTo != NULL) ? d->fcompareTo : knh_fcompareTo__default;
		ctx->tStruct[sid].fhashCode = (d->fhashCode != NULL) ? d->fhashCode : knh_fhashCode__default;
		ctx->tStruct[sid].fnewClass = (d->fnewClass != NULL) ? d->fnewClass : knh_fnewClass__default;
		d++;
	}
	KNH_UNLOCK(ctx, ctx->tableLock);
}

/* ------------------------------------------------------------------------ */

static
void konoha_loadClassData(Ctx *ctx, knh_ClassData_t *data)
{
	while(data->name != NULL) {
		knh_class_t cid = data->cid;
		if(!(cid < ctx->share->tClassSize)) {
			ctx->share->tClassSize = cid + 1;
		}

		konoha_setClassName(ctx, cid, new_String__T(ctx, data->name));

		ctx->tClass[cid].cflag  = data->flag;
		ctx->tClass[cid].oflag  = KNH_FLAG_CF2OF(data->flag);

		ctx->tClass[cid].bcid   = data->bcid;
		ctx->tClass[cid].supcid = data->supcid;
		if(data->bcid == CLASS_Object && data->supcid != CLASS_Object) {
			ctx->tClass[cid].offset = ctx->tClass[data->supcid].bsize;
		}else {
			ctx->tClass[cid].offset = 0;
		}

		if(data->bcid == CLASS_Object) {
			ctx->tClass[cid].sid = STRUCT_FIELD(data->field_size);
			ctx->tClass[cid].bsize = (ctx->tClass[cid].offset + data->field_size);

		}else if(data->bcid == CLASS_Script ) {
			ctx->tClass[cid].sid = STRUCT_Script;
			ctx->tClass[cid].bsize = data->field_size;
			ctx->tClass[cid].size = ctx->tClass[cid].bsize * sizeof(knh_Object_t*);
		}
		else {
			KNH_ASSERT(ctx->tClass[cid].offset == 0);
			ctx->tClass[cid].sid    = (knh_struct_t)data->bcid;
			ctx->tClass[cid].size =   ctx->tStruct[data->bcid].size;
			ctx->tClass[cid].bsize  = ctx->tClass[cid].size / sizeof(knh_Object_t*);
		}

		KNH_ASSERT(ctx->tClass[cid].cstruct == NULL);
		if(data->method_size > 0) {
			KNH_INITv(ctx->tClass[cid].cstruct, new_ClassStruct0(ctx, data->field_size, data->method_size));
		}
		else {
			KNH_INITv(ctx->tClass[cid].cstruct, ctx->tClass[data->supcid].cstruct);
		}
		KNH_ASSERT(ctx->tClass[cid].cmap == NULL);
		ClassMap *cm = ctx->tClass[CLASS_Any].cmap;
		if((data->flag & KNH_FLAG_CF_PRIVATE) == 0 || cid == CLASS_Any) {
			cm = new_ClassMap0(ctx, data->mapper_size);
		}
		KNH_INITv(ctx->tClass[cid].cmap, cm);
		konoha_setClassDefaultValue(ctx, cid, KNH_NULL, data->fdefault);
		data++;
	}
}

/* ------------------------------------------------------------------------ */

static
void konoha_loadGenericsData(Ctx *ctx, knh_GenericsData_t *data)
{
	while(data->name != NULL) {
		konoha_addGenericsClass(ctx, data->cid, new_String__T(ctx, data->name), data->bcid, data->p1, data->p2);
		data++;
	}
}

/* ------------------------------------------------------------------------ */

static
void konoha_loadClosureData(Ctx *ctx, knh_ClosureData_t *data)
{
	while(data->name != NULL) {
		konoha_addClosureClass(ctx, data->cid, new_String__T(ctx, data->name), data->r0, data->p1, data->p2, data->p3);
		data++;
	}
}

/* ------------------------------------------------------------------------ */

static
void konoha_loadExptData(Ctx *ctx, knh_ExptData_t *data)
{
	while(data->name != NULL) {
		konoha_addException(ctx, data->flag, data->eid, new_String__T(ctx, data->name), data->parent);
		data++;
	}
}

/* ------------------------------------------------------------------------ */

static
void konoha_loadFuncData(Ctx *ctx, knh_FuncData_t *data)
{
	while(data->name != NULL) {
		knh_DictSet_set(ctx, DP(ctx->sys)->funcDictSet, new_String__T(ctx, data->name), (knh_uint_t)data->func);
		data++;
	}
}

/* ------------------------------------------------------------------------ */

static
void konoha_loadFieldNameData(Ctx *ctx, knh_FieldNameData_t *data)
{
	while(data->name != NULL) {
		String *name = new_String__T(ctx, data->name);
#if defined(KNH_DBGMODE2)
		knh_fieldn_t fn = (knh_fieldn_t)knh_DictIdx_add__fast(ctx, DP(ctx->sys)->tfieldnDictIdx, name);
			//DEBUG("'%s' %d %d", knh_String_tochar(name), fn, fn2);
		KNH_ASSERT(fn == data->fn);
#else
		knh_DictIdx_add__fast(ctx, DP(ctx->sys)->tfieldnDictIdx, name);
#endif
		data++;
	}
}

/* ------------------------------------------------------------------------ */

static
void konoha_loadMethodFieldData(Ctx *ctx, knh_MethodFieldData_t *data)
{
	while(data->mfindex != -1) {
		MethodField *mf = new_MethodField(ctx, 1 + data->psize);
		mf->params[0].type = data->rtype;
		mf->params[0].fn = FIELDN_return;
		if(data->psize >= 1) {
			mf->params[1].type = data->t1;
			mf->params[1].fn = data->f1;
		}
		if(data->psize >= 2) {
			mf->params[2].type = data->t2;
			mf->params[2].fn = data->f2;
		}
		if(data->psize >= 3) {
			mf->params[3].type = data->t3;
			mf->params[3].fn = data->f3;
		}
		if(data->psize >= 4) {
			mf->params[4].type = data->t4;
			mf->params[4].fn = data->f4;
		}
		knh_tMethodField_add(ctx, mf);
		DBG2_ASSERT(knh_tMethodField(ctx, data->mfindex) == mf);
		data++;
	}
}

/* ------------------------------------------------------------------------ */

static
void konoha_loadMethodData(Ctx *ctx, knh_MethodData_t *data)
{
	while(data->func != NULL) {
		KNH_ASSERT_cid(data->cid);
		Method *mtd = new_Method(ctx, data->flag, data->cid, data->mn, data->func);
		DP(mtd)->code  = data->optfunc;
		DP(mtd)->delta = data->delta;
		MethodField *mf = knh_tMethodField(ctx, data->mfindex);
		KNH_ASSERT(IS_MethodField(mf));
		KNH_SETv(ctx, DP(mtd)->mf, mf);

		ClassStruct *cs = ctx->tClass[data->cid].cstruct;
		if(knh_class_isSingleton(data->cid)) {
			DP(mtd)->flag = DP(mtd)->flag | KNH_FLAG_MF_STATIC;
		}
		knh_Array_add(ctx, cs->methods, UP(mtd));
		data++;
	}
}

/* ------------------------------------------------------------------------ */

static
void konoha_loadMapperData(Ctx *ctx, knh_MapperData_t *data)
{
	while(data->func != NULL) {
		konoha_addMapper(ctx, data->flag, data->scid, data->tcid, data->func, KNH_NULL);
		data++;
	}
}

/* ------------------------------------------------------------------------ */

static void konoha_loadIntConstData(Ctx *ctx, knh_IntConstData_t *data)
{
	DictMap *tcmap = DP(ctx->sys)->tConstDictMap;
	while(data->name != NULL) {
		String *n = new_String__T(ctx, data->name);
		knh_DictMap_append(ctx, tcmap, n, UP(new_Int(ctx, data->ivalue)));
		data++;
	}
}

/* ------------------------------------------------------------------------ */

static void konoha_loadFloatConstData(Ctx *ctx, knh_FloatConstData_t *data)
{
	DictMap *tcmap = DP(ctx->sys)->tConstDictMap;
	while(data->name != NULL) {
		String *n = new_String__T(ctx, data->name);
		knh_DictMap_append(ctx, tcmap, n, UP(new_Float(ctx, data->fvalue)));
		data++;
	}
}

/* ------------------------------------------------------------------------ */

static void konoha_loadStringConstData(Ctx *ctx, knh_StringConstData_t *data)
{
	DictMap *tcmap = DP(ctx->sys)->tConstDictMap;
	while(data->name != NULL) {
		String *n = new_String__T(ctx, data->name);
		knh_DictMap_append(ctx, tcmap, n, UP(new_String__T(ctx, data->value)));
		data++;
	}
}

/* ------------------------------------------------------------------------ */

static void konoha_loadStringPropertyData(Ctx *ctx, knh_StringConstData_t *data)
{
	DictMap *pmap = DP(ctx->sys)->props;
	while(data->name != NULL) {
		String *n = new_String__T(ctx, data->name);
		knh_DictMap_append(ctx, pmap, n, UP(new_String__T(ctx, data->value)));
		data++;
	}
}

/* ------------------------------------------------------------------------ */
/* @data */

static knh_IntConstData_t IntConstData[] = {
	{"Int.MAX", KNH_INTEGER_MAX},
	{"Int.MIN", KNH_INTEGER_MIN},
	{NULL, 0}
};

static knh_FloatConstData_t FloatConstData[] = {
	{"Float.MAX", KNH_FLOAT_MAX},
	{"Float.MIN", KNH_FLOAT_MIN},
	{NULL, 0.0}
};

static knh_StringConstData_t StringConstData[] = {
	{NULL, NULL}
};

static knh_StringConstData_t StringPropertyData[] = {
	{"konoha.version", KONOHA_VERSION},
	{NULL, NULL}
};

/* ------------------------------------------------------------------------ */

static void konoha_loadClassProperties(Ctx *ctx)
{
	KNH_SETv(ctx, ctx->tClass[CLASS_Int].cspec,
		new_IntUnit(ctx, 0, CLASS_Int, TS_EMPTY, TS_EMPTY, KNH_INTEGER_MIN, KNH_INTEGER_MAX, 0));
	KNH_SETv(ctx, ctx->tClass[CLASS_Float].cspec,
			new_FloatUnit(ctx, 0, CLASS_Float, TS_EMPTY, KNH_FLOAT_MIN, KNH_FLOAT_MAX, 0.0, 0.0));
	KNH_SETv(ctx, ctx->tClass[CLASS_String].cspec,
			new_StringUnit(ctx, 0, CLASS_String, TS_EMPTY));

	konoha_setClassParam(ctx, CLASS_Array, CLASS_Any, CLASS_Nue);
	konoha_setClassParam(ctx, CLASS_Iterator, CLASS_Any, CLASS_Nue);
	konoha_setClassParam(ctx, CLASS_DictMap, CLASS_Any, CLASS_Nue);
	konoha_setClassParam(ctx, CLASS_DictSet, CLASS_Any, CLASS_Nue);
	konoha_setClassParam(ctx, CLASS_HashMap, CLASS_Any, CLASS_Any);
	konoha_setClassParam(ctx, CLASS_HashSet, CLASS_Any, CLASS_Any);
	ctx->tClass[CLASS_Closure].r0 = CLASS_Any;
	ctx->tClass[CLASS_Closure].p1 = CLASS_Any;
	ctx->tClass[CLASS_Closure].p2 = CLASS_Any;
	ctx->tClass[CLASS_Closure].p3 = CLASS_Any;

	konoha_loadIntConstData(ctx, IntConstData);
	konoha_loadFloatConstData(ctx, FloatConstData);
	konoha_loadStringConstData(ctx, StringConstData);
	konoha_loadStringPropertyData(ctx, StringPropertyData);
}

/* ------------------------------------------------------------------------ */

void konoha_loadSystemData(Ctx *ctx)
{
	konoha_loadStructData(ctx, StructData);
	konoha_initContext(ctx, (Context*)ctx, ctx->stacksize);
	konoha_loadClassData(ctx, ClassData);
	konoha_loadMethodFieldData(ctx, MethodFieldData);

	konoha_loadClassProperties(ctx);
	konoha_loadGenericsData(ctx, GenericsData);
	konoha_loadClosureData(ctx, ClosureData);
	konoha_loadExptData(ctx, ExptData);
	konoha_loadFuncData(ctx, FuncData);

	konoha_loadFieldNameData(ctx, FieldNameData);
	konoha_loadMethodData(ctx, MethodData);
	konoha_loadMapperData(ctx, MapperData);

	knh_finit_ *f = InitFuncData;
	while(f[0] != NULL) {
		f[0](ctx);
		f++;
	}
}

/* ------------------------------------------------------------------------ */

#ifdef __cplusplus
}
#endif
