/****************************************************************************
 * KONOHA COPYRIGHT, LICENSE NOTICE, AND DISCRIMER
 *
 * Copyright (c) 2005-2008, Kimio Kuramitsu <kimio at ynu.ac.jp>
 *           (c) 2008-      Konoha Software Foundation
 * All rights reserved.
 *
 * You may choose one of the following two licenses when you use konoha.
 * See www.konohaware.org/license.html for further information.
 *
 * (1) GNU General Public License 2.0      (with    KONOHA_UNDER_GPL2)
 * (2) Konoha Software Foundation License 1.0
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

/* ************************************************************************ */

#include"commons.h"

/* ************************************************************************ */

#ifdef __cplusplus
extern "C" {
#endif

/* ======================================================================== */
/* [new] */

/* ------------------------------------------------------------------------ */
/* @method This! Array.new(Int init) */

METHOD knh__Array_new(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	size_t init = IS_NULL(sfp[1].o) ? KNH_ARRAY_INITSIZE: p_size(sfp[1]);
	if(init > 0) {
		knh_Array_grow(ctx, o, init, KNH_NULL);
	}
	o->size = 0;
	KNH_RETURN(ctx, sfp, o);
}

/* ------------------------------------------------------------------------ */
/* @method This! Array.new:array(Int n) */

METHOD knh__Array_new__array(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	size_t init = IS_NULL(sfp[1].o) ? KNH_ARRAY_INITSIZE : p_size(sfp[1]);
	knh_class_t p1 = ctx->tClass[knh_Object_cid(o)].p1;
	Object *v = (p1 == CLASS_Any) ? KNH_NULL : konoha_getClassDefaultValue(ctx, p1);
	if(init > 0) {
		knh_Array_grow(ctx, o, init, v);
		o->size = init;
	}
	KNH_RETURN(ctx, sfp, o);
}

/* ------------------------------------------------------------------------ */
/* @method[VARARGS] This! Array.new:init(Any1 value) @VARARGS */

METHOD knh__Array_new__init(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	knh_sfp_t *v = sfp + 1;
	int i, ac = knh_sfp_argc(ctx, v);
	if(ac > 0) {
		knh_Array_grow(ctx, o, ac, KNH_NULL);
	}
	for(i = 0; i < ac; i++) {
		knh_sfp_boxing(ctx, v + i);
		knh_Array_add(ctx, o, v[i].o);
	}
	KNH_RETURN(ctx, sfp, o);
}

/* ======================================================================== */
/* [method] */

/* ------------------------------------------------------------------------ */
/* @method void Array.add(Any1 value) */

METHOD knh__Array_add(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		if(o->size == o->capacity) {
			knh_Array_grow(ctx, o, knh_array_newsize(o->capacity * 2, sizeof(Object*)), KNH_NULL);
		}
		knh_sfp_boxing(ctx, sfp + 1);
		KNH_SETv(ctx, o->list[o->size], sfp[1].o);
		o->size++;
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ------------------------------------------------------------------------ */
/* @method[VARARGS] void Array.opLshift(Any1 v) */

METHOD knh__Array_opLshift(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		knh_sfp_t *v = sfp + 1;
		int i, ac = knh_sfp_argc(ctx, v);
		for(i = 0; i < ac; i++) {
			knh_sfp_boxing(ctx, v + i);
			knh_Array_add(ctx, o, v[i].o);
		}
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ======================================================================== */
/* @method Int! Array.opSize() */

METHOD knh__Array_opSize(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	KNH_RETURN_Int(ctx, sfp, o->size);
}

/* ------------------------------------------------------------------------ */
/* @method Any1 Array.get(Int! n) */

METHOD knh__Array_get(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	size_t n2 = knh_array_index(ctx, p_int(sfp[1]), o->size);
	KNH_RETURN(ctx, sfp, o->list[n2]);
}

/* ------------------------------------------------------------------------ */
/* @method void Array.set(Int! n, Any1 v) */

METHOD knh__Array_set(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		size_t n2 = knh_array_index(ctx, p_int(sfp[1]), o->size);
		knh_sfp_boxing(ctx, sfp + 2);
		KNH_SETv(ctx, o->list[n2], sfp[2].o);
	}
	KNH_RETURN_void(ctx, sfp);
}
/* ------------------------------------------------------------------------ */
/* @method void Array.setAll(Any1 v) */

METHOD knh__Array_setAll(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		size_t i;
		for(i = 0; i < o->size; i++) {
			knh_sfp_boxing(ctx, sfp + 1);
			KNH_SETv(ctx, o->list[i], sfp[1].o);
		}
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ------------------------------------------------------------------------ */

static
Array *new_Array__subset(Ctx *ctx, Array *a, size_t s, size_t e)
{
	Array *newa = (Array*)new_Object_init(ctx, a->h.flag, a->h.cid, 0);
	if(e < s) {
		size_t t = s; s = e; e = t;
	}
	if(s < e) {
		KNH_ASSERT(e <= a->size);
		size_t i, newsize = e - s;
		newa->list = (Object**)KNH_MALLOC(ctx, newsize * sizeof(Object*));
		knh_memcpy(newa->list, &a->list[s], newsize * sizeof(Object*));
#ifdef KNH_USING_RCGC
		for(i = 0; i < newsize; i++) {
			knh_Object_RCinc(newa->list[i]);
		}
#endif
		newa->capacity = newsize;
		newa->size = newsize;
	}
	return newa;
}

/* ------------------------------------------------------------------------ */
/* @method This! Array.opSubset(Int s, Int e) @Debug */

METHOD knh__Array_opSubset(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	size_t s = IS_NULL(sfp[1].o) ? 0 : knh_array_index(ctx, p_int(sfp[1]), o->size);
	size_t e = IS_NULL(sfp[2].o) ? (o->size) : knh_array_index(ctx, p_int(sfp[2]), o->size);
	KNH_RETURN(ctx, sfp, new_Array__subset(ctx, o, s, e));
}

/* ------------------------------------------------------------------------ */
/* @method This! Array.opSubsete(Int s, Int e) @Debug */

METHOD knh__Array_opSubsete(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	size_t s = IS_NULL(sfp[1].o) ? 0 : knh_array_index(ctx, p_int(sfp[1]), o->size);
	size_t e = IS_NULL(sfp[2].o) ? (o->size) : knh_array_index(ctx, p_int(sfp[2]) + 1, o->size) ;
	KNH_RETURN(ctx, sfp, new_Array__subset(ctx, o, s, e));
}

/* ------------------------------------------------------------------------ */
/* @method This! Array.opOffset(Int s, Int offset) @Debug */

METHOD knh__Array_opOffset(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	size_t s = IS_NULL(sfp[1].o) ? 0 : knh_array_index(ctx, p_int(sfp[1]), o->size);
	knh_int_t e = IS_NULL(sfp[1].o) ? o->size : s + p_int(sfp[2]);
	if(e > o->size) e = o->size; else if(e < 0) e = 0;
	KNH_RETURN(ctx, sfp, new_Array__subset(ctx, o, s, e));
}

/* ======================================================================== */

/* ------------------------------------------------------------------------ */
/* @method void Array.insert(Int! n, Any1 v) */

METHOD knh__Array_insert(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		size_t i, n = knh_array_index(ctx, p_int(sfp[1]), o->size);
		if(o->size == o->capacity) {
			knh_Array_grow(ctx, o, knh_array_newsize(o->capacity * 2, sizeof(Object*)), KNH_NULL);
		}
		Object *temp = o->list[o->size];
		o->size++;
		for(i = o->size - 1; i > n ; i--) {
			o->list[i] = o->list[i-1];
		}
		o->list[n] = temp;
		knh_sfp_boxing(ctx, sfp + 2);
		KNH_SETv(ctx, o->list[n], sfp[2].o);
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ------------------------------------------------------------------------ */
/* @method void Array.remove(Int! n) */

METHOD knh__Array_remove(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		size_t n = knh_array_index(ctx, p_int(sfp[1]), o->size);
		knh_Array_remove(ctx, o, n);
	}
	KNH_RETURN_void(ctx,sfp);
}

/* ------------------------------------------------------------------------ */
/* @method Any1 Array.pop() */

METHOD knh__Array_pop(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o) && o->size > 0) {
		o->size--;
		KNH_RETURN(ctx, sfp, o->list[o->size]);
	}else {
		KNH_RETURN(ctx,sfp, KNH_NULL);
	}
}

/* ------------------------------------------------------------------------ */
/* @method void Array.clear() */

void knh_Array_clear(Ctx *ctx, Array *o)
{
	if(!knh_Object_isImmutable(o)) {
		size_t i;
		for(i = 0; i < o->size; i++) {
			KNH_SETv(ctx, o->list[i], KNH_NULL);
		}
		o->size = 0;
	}
}

/* ------------------------------------------------------------------------ */
/* @method Int! Array.indexOf(Any1 v) */

METHOD knh__Array_indexOf(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	knh_sfp_boxing(ctx, sfp + 1);
	size_t i;
	for(i = 0; i < o->size; i++) {
		if(knh_Object_compareTo(ctx, o->list[i], sfp[1].o) == 0) {
			KNH_RETURN_Int(ctx, sfp, i);
		}
	}
	KNH_RETURN_Int(ctx, sfp, -1); // Not Found
}

/* ------------------------------------------------------------------------ */
/* @method Int! Array.lastIndexOf(Any1 v) */

METHOD knh__Array_lastIndexOf(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	knh_sfp_boxing(ctx, sfp + 1);
	size_t i;
	for(i = o->size - 1; i >= 0; i--) {
		if(knh_Object_compareTo(ctx, o->list[i], sfp[1].o) == 0) {
			KNH_RETURN_Int(ctx, sfp, i);
		}
	}
	KNH_RETURN_Int(ctx, sfp, -1); // Not Found
}

/* ------------------------------------------------------------------------ */
/* @method Boolean! Array.opHas(Any1 v) */

METHOD knh__Array_opHas(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	knh_sfp_boxing(ctx, sfp + 1);
	size_t i;
	for(i = 0; i < o->size; i++) {
		if(knh_Object_compareTo(ctx, o->list[i], sfp[1].o) == 0) {
			KNH_RETURN_Boolean(ctx, sfp, 1);
		}
	}
	KNH_RETURN_Boolean(ctx, sfp, 0); // Not Found
}

/* ======================================================================== */
/* [Collections] */

/* ------------------------------------------------------------------------ */
/* @method void Array.sort() */

METHOD knh__Array_sort(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		knh_qsort(o->list, o->size, sizeof(Object*), (int (*)(const void*, const void*))knh_Object_compareTo);
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ------------------------------------------------------------------------ */
/* @method void Array.reverse() */

METHOD knh__Array_reverse(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		size_t i;
		for(i = 0; i < o->size / 2; i++) {
			size_t n = o->size - i;
			Object *temp = o->list[i];
			o->list[i] = o->list[n];
			o->list[n] = temp;
		}
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ------------------------------------------------------------------------ */
/* @method void Array.swap(Int! m, Int! n) */

METHOD knh__Array_swap(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		size_t m = knh_array_index(ctx, p_int(sfp[1]), o->size);
		size_t n = knh_array_index(ctx, p_int(sfp[2]), o->size);
		Object *temp = o->list[n];
		o->list[n] = o->list[m];
		o->list[m] = temp;
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ------------------------------------------------------------------------ */
/* @method void Array.shuffle() */

METHOD knh__Array_shuffle(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		size_t i;
		for(i = 0; i < o->size; i++) {
			size_t m = knh_rand() % o->size;
			size_t n = knh_rand() % o->size;
			Object *temp = o->list[n];
			o->list[n] = o->list[m];
			o->list[m] = temp;
		}
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ======================================================================== */
/* [movabletext] */

/* ------------------------------------------------------------------------ */
/* @method void Array.%k(OutputStream w, String m) */

void knh_Array__k(Ctx *ctx, Array *o, OutputStream *w, String *m)
{
	knh_putc(ctx, w, '[');
	size_t c;
	for(c = 0; c < o->size; c++) {
		if(c > 0) {
			knh_write_delim(ctx,w);
		}
		knh_format(ctx, w, METHODN__k, o->list[c], KNH_NULL);
	}
	knh_putc(ctx, w, ']');
}

/* ======================================================================== */
/* [mapping] */

static
ITRNEXT knh_Array_var_next(Ctx *ctx, knh_sfp_t *sfp, int n)
{
	Array *o = (Array*)DP(sfp[0].it)->source;
	KNH_ASSERT(IS_bArray(o));
	size_t pos = DP(sfp[0].it)->pos;
	while(pos < o->size) {
		if(IS_NOTNULL(o->list[pos])) {
			DP(sfp[0].it)->pos = pos+1;
			KNH_ITRNEXT(ctx, sfp, n, o->list[pos]);
		}
		pos++;
	}
	KNH_ITREND(ctx, sfp, n);
}

/* ------------------------------------------------------------------------ */
/* @map Array Iterator! */

MAPPER knh_Array_Iterator(Ctx *ctx, knh_sfp_t *sfp)
{
	KNH_MAPPED(ctx, sfp,
		new_Iterator(ctx, ctx->tClass[(sfp[0].o)->h.cid].p1, sfp[0].o, knh_Array_var_next));
}

/* ------------------------------------------------------------------------ */
/* @method Any1.. Array.opItr() */

METHOD knh__Array_opItr(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	KNH_RETURN(ctx, sfp, new_Iterator(ctx, ctx->tClass[o->h.cid].p1, UP(o), knh_Array_var_next));
}

/* ------------------------------------------------------------------------ */
/* @map Iterator Array! */

MAPPER knh_Iterator_Array(Ctx *ctx, knh_sfp_t *sfp)
{
	Iterator *it = sfp[0].it;
	Array *a = new_Array(ctx, ctx->tClass[it->h.cid].p1, 0);
	it->fnext_1(ctx, sfp, 1);
	while(HAS_ITRNEXT(sfp[1].o)) {
		knh_Array_add(ctx, a, sfp[1].o);
		it->fnext_1(ctx, sfp, 1);
	}
	KNH_MAPPED(ctx, sfp, a);
}

/* ------------------------------------------------------------------------ */

Mapper *knh_tMapper_newIteratorArray(Ctx *ctx, knh_class_t icid, knh_class_t acid)
{
	Mapper *mpr = new_Mapper(ctx, KNH_FLAG_MMF_TOTAL, acid, icid, knh_Array_Iterator, KNH_NULL);
	knh_ClassMap_add(ctx, ctx->tClass[acid].cmap, mpr);
	mpr = new_Mapper(ctx, KNH_FLAG_MMF_TOTAL, icid, acid, knh_Iterator_Array, KNH_NULL);
	knh_ClassMap_add(ctx, ctx->tClass[icid].cmap, mpr);
	return mpr;
}

/* ------------------------------------------------------------------------ */

Mapper *knh_tMapper_newArrayIterator(Ctx *ctx, knh_class_t acid, knh_class_t icid)
{
	Mapper *mpr = new_Mapper(ctx, KNH_FLAG_MMF_TOTAL, icid, acid, knh_Iterator_Array, KNH_NULL);
	knh_ClassMap_add(ctx, ctx->tClass[icid].cmap, mpr);
	mpr = new_Mapper(ctx, KNH_FLAG_MMF_TOTAL, acid, icid, knh_Array_Iterator, KNH_NULL);
	knh_ClassMap_add(ctx, ctx->tClass[acid].cmap, mpr);
	return mpr;
}

/* ======================================================================== */
/* [2d,3d] */
/* @data */

typedef struct {
	size_t x;
	size_t y;
	size_t z;
	size_t w;   /* unused */
} knh_darray_t;

/* ------------------------------------------------------------------------ */

static
void *knh_array_dmalloc(Ctx *ctx, size_t x, size_t y, size_t z, size_t usize)
{
	size_t capacity = x * y * z;
	knh_darray_t *d = (knh_darray_t*)KNH_MALLOC(ctx, (capacity * usize) + sizeof(knh_darray_t));
	d->x = x; d->y = y; d->z = z; d->w = d->x * d->y;
	return (void*)(d+1);
}

/* ------------------------------------------------------------------------ */

void knh_array_dfree(Ctx *ctx, void *ptr, size_t capacity, size_t usize)
{
	knh_darray_t *d = ((knh_darray_t*)ptr) - 1;
	KNH_FREE(ctx, d, (capacity * usize) + sizeof(knh_darray_t));
}

/* ------------------------------------------------------------------------ */

static
void knh_ArrayDim_init(Ctx *ctx, Array *a, size_t x, size_t y, size_t z)
{
	knh_Array_setDim(a, 1);
	a->list = (Object**)knh_array_dmalloc(ctx, x, y, z, sizeof(Object*));
	a->capacity = x * y * z;
	knh_class_t p1 = ctx->tClass[knh_Object_cid(a)].p1;
	Object *v = (p1 == CLASS_Any) ? KNH_NULL : konoha_getClassDefaultValue(ctx, p1);
	int i = 0;
	for(i = 0; i < a->capacity; i++) {
		KNH_INITv(a->list[i], v);
	}
	a->size = a->capacity;
}

/* ------------------------------------------------------------------------ */
/* @method This! Array.new:array2D(Int! x, Int! y) @Private */

METHOD knh__Array_new__array2D(Ctx *ctx, knh_sfp_t *sfp)
{
	knh_ArrayDim_init(ctx, (Array*)sfp[0].o, p_size(sfp[1]), p_size(sfp[2]), 1);
	KNH_RETURN(ctx, sfp, sfp[0].o);
}

/* ------------------------------------------------------------------------ */
/* @method This! Array.new:array3D(Int! x, Int! y, Int! z) @Private */

METHOD knh__Array_new__array3D(Ctx *ctx, knh_sfp_t *sfp)
{
	knh_ArrayDim_init(ctx, (Array*)sfp[0].o, p_size(sfp[1]), p_size(sfp[2]), p_size(sfp[3]));
	KNH_RETURN(ctx, sfp, sfp[0].o);
}

/* ------------------------------------------------------------------------ */
/* @method Any1 Array.get2D(Int! x, Int! y) */

METHOD knh__Array_get2D(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	knh_darray_t *d = ((knh_darray_t*)o->list) - 1;
	size_t x = knh_array_index(ctx, p_int(sfp[1]), d->x);
	size_t y = knh_array_index(ctx, p_int(sfp[2]), d->y);
	KNH_RETURN(ctx, sfp, o->list[x + y * d->x]);
}

/* ------------------------------------------------------------------------ */
/* @method Any1 Array.get3D(Int! x, Int! y, Int! z) */

METHOD knh__Array_get3D(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	knh_darray_t *d = ((knh_darray_t*)o->list) - 1;
	size_t x = knh_array_index(ctx, p_int(sfp[1]), d->x);
	size_t y = knh_array_index(ctx, p_int(sfp[2]), d->y);
	size_t z = knh_array_index(ctx, p_int(sfp[3]), d->z);
	KNH_RETURN(ctx, sfp, o->list[x + y * d->x + z * d->w]);
}

/* ------------------------------------------------------------------------ */
/* @method void Array.set2D(Int! x, Int! y, Any1 v) */

METHOD knh__Array_set2D(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		knh_darray_t *d = ((knh_darray_t*)o->list) - 1;
		size_t x = knh_array_index(ctx, p_int(sfp[1]), d->x);
		size_t y = knh_array_index(ctx, p_int(sfp[2]), d->y);
		knh_sfp_boxing(ctx, sfp + 3);
		KNH_SETv(ctx, o->list[x + y * d->x], sfp[3].o);
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ------------------------------------------------------------------------ */
/* @method void Array.set3D(Int! x, Int! y, Int! z, Any1 v) */

METHOD knh__Array_set3D(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		knh_darray_t *d = ((knh_darray_t*)o->list) - 1;
		size_t x = knh_array_index(ctx, p_int(sfp[1]), d->x);
		size_t y = knh_array_index(ctx, p_int(sfp[2]), d->y);
		size_t z = knh_array_index(ctx, p_int(sfp[3]), d->z);
		knh_sfp_boxing(ctx, sfp + 4);
		KNH_SETv(ctx, o->list[x + y * d->x + z * d->w], sfp[4].o);
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ------------------------------------------------------------------------ */

#ifdef __cplusplus
}
#endif
