

/****************************************************************************
 * KONOHA COPYRIGHT, LICENSE NOTICE, AND DISCRIMER
 *
 * Copyright (c) 2005-2008, Kimio Kuramitsu <kimio at ynu.ac.jp>
 *           (c) 2008-      Konoha Software Foundation
 * All rights reserved.
 *
 * You may choose one of the following two licenses when you use konoha.
 * See www.konohaware.org/license.html for further information.
 *
 * (1) GNU General Public License 2.0      (with    KONOHA_UNDER_GPL2)
 * (2) Konoha Software Foundation License 1.0
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

/* ************************************************************************ */

#include<konoha.h>

#define KNH_PKGNAME "unix"

#ifdef KNH_USING_POSIX
#include <unistd.h>
#include <signal.h>
#include <dirent.h>
#include <sys/stat.h>
#endif


#ifdef __cplusplus
extern "C" {
#endif

#ifdef KNH_USING_POSIX

/* ======================================================================== */
/* [info] */

/* ------------------------------------------------------------------------ */
/* @method String! System.gethostname() */

static
METHOD knh__System_gethostname(Ctx *ctx, knh_sfp_t *sfp)
{
	char buf[256];
	if(gethostname(buf, sizeof(buf)) == -1) {
		KNH_PERRNO(ctx, "OS!!", "gethostname");
	}
	KNH_RETURN(ctx, sfp, new_String(ctx, B(buf), NULL));
}

/* ------------------------------------------------------------------------ */
/* @method Int! System.getPid() */

static
METHOD knh__System_getpid(Ctx *ctx, knh_sfp_t *sfp)
{
	KNH_RETURN_Int(ctx, sfp, getpid());
}

/* ------------------------------------------------------------------------ */
/* @method Int! System.getPPid() */

static
METHOD knh__System_getppid(Ctx *ctx, knh_sfp_t *sfp)
{
	KNH_RETURN_Int(ctx, sfp, getppid());
}

/* ------------------------------------------------------------------------ */
/* @method void System.kill(Int! pid, Int! sig) */

static
METHOD knh__System_kill(Ctx *ctx, knh_sfp_t *sfp)
{
	KNH_SECURE(ctx);
	if(kill(p_int(sfp[1]), p_int(sfp[2])) == -1) {
		KNH_PERRNO(ctx, "OS!!", "kill");
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ------------------------------------------------------------------------ */
/* @method void System.system(String! cmd) */

static
METHOD knh__System_system(Ctx *ctx, knh_sfp_t *sfp)
{
	KNH_SECURE(ctx);
	if(system(p_char(sfp[1])) == -1) {
		KNH_PERRNO(ctx, "OS!!", "system");
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ======================================================================== */
/* [PIPE] */

static
knh_io_t knh_iodrv_open__PIPE(Ctx *ctx, InputStream *in, OutputStream *out, knh_bytes_t file, char *mode)
{
	char *cmd = (char*)knh_bytes_skipscheme(file).buf;
	FILE *fp = NULL;
	KNH_WARNING(ctx, "opening pipe '%s'", cmd);
	KNH_SECURE(ctx);

	if(in != NULL) {
		KNH_ASSERT(IS_InputStream(in));
		fp = popen(cmd, "r");
	}
	else {
		KNH_ASSERT(IS_OutputStream(out));
		fp = popen(cmd, "w");
	}
	if(fp == NULL) {
		KNH_PERRNO(ctx, "IO!!", "popen");
		return (knh_io_t)-1;
	}
	return (knh_io_t)fp;
}

/* ------------------------------------------------------------------------ */

static
knh_int_t knh_iodrv_read__PIPE(Ctx *ctx, knh_io_t fd, char *buf, size_t bufsiz)
{
	FILE *fp = (FILE*)fd;
	size_t ssize = fread(buf, 1, bufsiz, fp);
	return ssize;
}

/* ------------------------------------------------------------------------ */

static
knh_int_t knh_iodrv_write__PIPE(Ctx *ctx, knh_io_t fd, char *buf, size_t bufsiz)
{
	FILE *fp = (FILE*)fd;
	size_t ssize = fwrite(buf, 1, bufsiz, fp);
	fflush(fp);
	return ssize;
}

/* ------------------------------------------------------------------------ */

static
void knh_iodrv_close__PIPE(Ctx *ctx, knh_io_t fd)
{
	FILE *fp = (FILE*)fd;
	pclose(fp);
}

/* ------------------------------------------------------------------------ */
/* @data */

static knh_iodrv_t IO__PIPE = {
	KNH_DRVAPI_TYPE__IO, "pipe",
	4096,
	knh_iodrv_open__PIPE,
	knh_iodrv_read__PIPE,
	knh_iodrv_write__PIPE,
	knh_iodrv_close__PIPE
};

/* ======================================================================== */
/* [FILESYSTEM] */

/* ------------------------------------------------------------------------ */
/* @method String[] System.listdir(String dirname) */

static
METHOD knh__System_listdir(Ctx *ctx, knh_sfp_t *sfp)
{
	DIR *dirptr;
	struct dirent *direntp;
	char dirname[FILENAME_BUFSIZ];
	knh_format_ospath(ctx, dirname, sizeof(dirname), knh_StringNULL_tobytes(sfp[1].s, STEXT(".")));
	Array *a = new_Array(ctx, CLASS_String, 0);

	if ((dirptr = opendir(dirname)) == NULL) {
		KNH_PERRNO(ctx, "OS!!", "opendir");
	} else {
		while ((direntp = readdir(dirptr)) != NULL) {
			char *p = direntp->d_name;
			if(p[0] == '.' && (p[1] == 0 || p[1] == '.')) continue;
			knh_Array_add(ctx, a, UP(new_String(ctx, B(p), NULL)));
		}
		closedir(dirptr);
	}
	KNH_RETURN(ctx, sfp, a);
}

/* ------------------------------------------------------------------------ */
/* @method String! System.getcwd() */

static
METHOD knh__System_getcwd(Ctx *ctx, knh_sfp_t *sfp)
{
	// TODO: when filename is longer than 1024
	// to escape this, if we use getcwd(NULL, 0), system malloc buf for filename.
	char *res = NULL;
	char tmpbuf[FILENAME_BUFSIZ];
	res = getcwd(tmpbuf, sizeof(tmpbuf));
	if (res == NULL) {
		KNH_PERRNO(ctx, "OS!!", "getcwd");
	}
	String *s = new_String(ctx, B(tmpbuf), NULL);
	KNH_RETURN(ctx, sfp, s);
}

/* ------------------------------------------------------------------------ */
/* @method void System.chdir(String dirname) */

static
METHOD knh__System_chdir(Ctx *ctx, knh_sfp_t *sfp)
{
	char dirname[FILENAME_BUFSIZ];
	knh_format_ospath(ctx, dirname, sizeof(dirname), knh_StringNULL_tobytes(sfp[1].s, STEXT(".")));
	if(chdir(dirname) == -1) {
		KNH_PERRNO(ctx, "OS!!", "chdir");
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ======================================================================== */

#endif/*KNH_USING_POSIX*/

/* ======================================================================== */
/* [KNHAPI] */

int knhunix_init(Ctx *ctx)
{
#ifdef KNH_USING_POSIX
	KNH_NOTICE(ctx, "loading unix..");
	konoha_addMethod(ctx, "String! System.getHostName()", knh__System_gethostname);
	konoha_addMethod(ctx, "Int! System.getPid()", knh__System_getpid);
	konoha_addMethod(ctx, "Int! System.getPPid()", knh__System_getppid);
	konoha_addMethod(ctx, "void System.kill(Int! pid, Int! sig)", knh__System_kill);
	// PLEASE ADD MAJOR SIGNAL !!
	konoha_addIntConst(ctx, "SIG_KILL", SIGKILL);

	konoha_addMethod(ctx, "void System.shell(String! cmd)", knh__System_system);
	konoha_addMethod(ctx, "void System.system(String! cmd)", knh__System_system);
	konoha_addIODriver(ctx, NULL, &IO__PIPE);
	konoha_addIODriver(ctx, "sh", &IO__PIPE);
	konoha_addIODriver(ctx, "cmd", &IO__PIPE);

	konoha_addMethod(ctx, "String![] System.listDir(String dirname)", knh__System_listdir);
	konoha_addMethod(ctx, "String! System.getcwd()", knh__System_getcwd);
	konoha_addMethod(ctx, "void System.chdir(String dirname)", knh__System_chdir);
#else
	KNH_WARNING(ctx, "unsupported unix package..");
#endif
	return 1;
}

/* ------------------------------------------------------------------------ */

void knhunix_traverse(Ctx *ctx, knh_ftraverse ftr)
{
	if(IS_SWEEP(ftr)) {

	}
}

/* ------------------------------------------------------------------------ */

static knh_pkgmeta_t PKGDATA = {
	KONOHA_BUILDID, /* Don't change this */
	KNH_PKGNAME,    /* package name (not NULL) */
	"1.0",  /* package version (not NULL) */
	NULL,   /* additional information */
	KONOHA_URLBASE "package.unix", /* URL */
	knhunix_init,
	knhunix_traverse,
	NULL
};

KNHAPI(knh_pkgmeta_t*) knhunix_load(Ctx *ctx)
{
	return &(PKGDATA);
}

/* ------------------------------------------------------------------------ */

#ifdef __cplusplus
}
#endif
