/****************************************************************************
 * KONOHA COPYRIGHT, LICENSE NOTICE, AND DISCRIMER  
 * 
 * Copyright (c) 2005-2008, Kimio Kuramitsu <kimio at ynu.ac.jp>
 *           (c) 2008-      Konoha Software Foundation  
 * All rights reserved.
 * 
 * You may choose one of the following two licenses when you use konoha. 
 * See www.konohaware.org/license.html for further information.
 * 
 * (1) GNU General Public License 2.0      (with    KONOHA_UNDER_GPL2)
 * (2) Konoha Software Foundation License 1.0
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER 
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *  
 ****************************************************************************/

#include<konoha/konoha_dev.h>


#ifdef __cplusplus 
extern "C" {
#endif

/* ======================================================================== */

/* ======================================================================== */
/* [token] */

static char* knh_token_array_tochar[] = {
    "namespace",  /* namespace */ 
    "import",  /* import */ 
    "pragma",  /* pragma */ 
    "using",  /* using */ 
    "with",  /* with */ 
    "class",  /* class */ 
    "extends",  /* extends */ 
    "implements",  /* implements */ 
    "method",  /* method */ 
    "format",  /* format */ 
    "return",  /* return */ 
    "mapmap",  /* mapmap */ 
    "weave",  /* weave */ 
    "aspect",  /* aspect */ 
    "block",  /* block */ 
    "if",  /* if */ 
    "else",  /* else */ 
    "switch",  /* switch */ 
    "case",  /* case */ 
    "while",  /* while */ 
    "do",  /* do */ 
    "for",  /* for */ 
    "foreach",  /* foreach */ 
    "from",  /* from */ 
    "where",  /* where */ 
    "break",  /* break */ 
    "continue",  /* continue */ 
    "try",  /* try */ 
    "catch",  /* catch */ 
    "finally",  /* finally */ 
    "throw",  /* throw */ 
    "local",  /* local */ 
    "print",  /* print */ 
    "assert",  /* assert */ 
    "{",  /* BRACE */ 
    "(",  /* PARENTHESIS */ 
    "[",  /* BRANCET */ 
    "...",  /* DOTS */ 
    ";",  /* SEMICOLON */ 
    ",",  /* COMMA */ 
    "=",  /* LET */ 
    "<<=",  /* lshifte */ 
    ">>=",  /* rshifte */ 
    "+=",  /* adde */ 
    "-=",  /* sube */ 
    "*=",  /* mule */ 
    "/=",  /* dive */ 
    "%=",  /* mode */ 
    "|=",  /* bitore */ 
    "&=",  /* bitande */ 
    "?\?=",  /* ALTLET */ 
    "?",  /* QUESTION */ 
    "..",  /* Itr */ 
    "..<",  /* SUBSET */ 
    "..+",  /* OFFSET */ 
    ":",  /* COLON */ 
    "?\?",  /* ALT */ 
    "||",  /* OR */ 
    "&&",  /* AND */ 
    "!",  /* Not */ 
    "instanceof",  /* Instanceof */ 
    "isa?",  /* Isa */ 
    "as",  /* As */ 
    "in?",  /* Has */ 
    "===",  /* Is */ 
    "to?",  /* To */ 
    "==",  /* Eq */ 
    "!=",  /* Neq */ 
    "<",  /* Lt */ 
    "<=",  /* Lte */ 
    ">",  /* Gt */ 
    ">=",  /* Gte */ 
    "=~",  /* Match */ 
    "|",  /* Lor */ 
    "%%",  /* Fmt */ 
    "<<",  /* Lshift */ 
    ">>",  /* Rshift */ 
    "+",  /* Add */ 
    "-",  /* Sub */ 
    "/",  /* Div */ 
    "%",  /* Mod */ 
    "*",  /* Mul */ 
    "&",  /* Land */ 
    "~",  /* Lnot */ 
    "^",  /* Xor */ 
    "++",  /* Next */ 
    "--",  /* Prev */ 
    "---",  /* Neg */ 
    "NUM",  /* NUM */ 
    "STR",  /* STR */ 
    "TSTR",  /* TSTR */ 
    "EVAL",  /* EVAL */ 
    "LABEL",  /* LABEL */ 
    "METAN",  /* METAN */ 
    "PROPN",  /* PROPN */ 
    "URN",  /* URN */ 
    "MT",  /* MT */ 
    "CONSTN",  /* CONSTN */ 
    "TYPEN",  /* TYPEN */ 
    "CMETHODN",  /* CMETHODN */ 
    "NAME",  /* NAME */ 
    "ERR",  /* ERR */ 
    "MN",  /* MN */ 
    "FN",  /* FN */ 
    "CID",  /* CID */ 
    "MPR",  /* MPR */ 
    "EOT",  /* EOT */ 
    "CONST",  /* CONST */ 
    "DEFVAL",  /* DEFVAL */ 
    "SYSVAL",  /* SYSVAL */ 
    "STACK",  /* STACK */ 
    "FIELD",  /* FIELD */ 
    "GLOBAL",  /* GLOBAL */ 
    "MEMBER",  /* MEMBER */ 
    "CLASSID",  /* CLASSID */ 
    "_",  /* ASIS */ 
    "NOP",  /* NOP */ 
    //(char*)0
};

char* knh_token_tochar(knh_token_t t)
{
    KNH_ASSERT(t < KNH_TOKEN_MAXSIZ);
    return knh_token_array_tochar[t];
}

/* ------------------------------------------------------------------------ */

static knh_methodn_t knh_token_array_tomethodn[] = {
    METHODN_NONAME,  /* namespace */ 
    METHODN_NONAME,  /* import */ 
    METHODN_NONAME,  /* pragma */ 
    METHODN_NONAME,  /* using */ 
    METHODN_NONAME,  /* with */ 
    METHODN_NONAME,  /* class */ 
    METHODN_NONAME,  /* extends */ 
    METHODN_NONAME,  /* implements */ 
    METHODN_NONAME,  /* method */ 
    METHODN_NONAME,  /* format */ 
    METHODN_NONAME,  /* return */ 
    METHODN_NONAME,  /* mapmap */ 
    METHODN_NONAME,  /* weave */ 
    METHODN_NONAME,  /* aspect */ 
    METHODN_NONAME,  /* block */ 
    METHODN_NONAME,  /* if */ 
    METHODN_NONAME,  /* else */ 
    METHODN_NONAME,  /* switch */ 
    METHODN_NONAME,  /* case */ 
    METHODN_NONAME,  /* while */ 
    METHODN_NONAME,  /* do */ 
    METHODN_NONAME,  /* for */ 
    METHODN_NONAME,  /* foreach */ 
    METHODN_NONAME,  /* from */ 
    METHODN_NONAME,  /* where */ 
    METHODN_NONAME,  /* break */ 
    METHODN_NONAME,  /* continue */ 
    METHODN_NONAME,  /* try */ 
    METHODN_NONAME,  /* catch */ 
    METHODN_NONAME,  /* finally */ 
    METHODN_NONAME,  /* throw */ 
    METHODN_NONAME,  /* local */ 
    METHODN_NONAME,  /* print */ 
    METHODN_NONAME,  /* assert */ 
    METHODN_NONAME,  /* BRACE */ 
    METHODN_NONAME,  /* PARENTHESIS */ 
    METHODN_NONAME,  /* BRANCET */ 
    METHODN_NONAME,  /* DOTS */ 
    METHODN_NONAME,  /* SEMICOLON */ 
    METHODN_NONAME,  /* COMMA */ 
    METHODN_NONAME,  /* LET */ 
    METHODN_NONAME,  /* lshifte */ 
    METHODN_NONAME,  /* rshifte */ 
    METHODN_NONAME,  /* adde */ 
    METHODN_NONAME,  /* sube */ 
    METHODN_NONAME,  /* mule */ 
    METHODN_NONAME,  /* dive */ 
    METHODN_NONAME,  /* mode */ 
    METHODN_NONAME,  /* bitore */ 
    METHODN_NONAME,  /* bitande */ 
    METHODN_NONAME,  /* ALTLET */ 
    METHODN_NONAME,  /* QUESTION */ 
    METHODN_opItr,  /* Itr */ 
    METHODN_NONAME,  /* SUBSET */ 
    METHODN_NONAME,  /* OFFSET */ 
    METHODN_NONAME,  /* COLON */ 
    METHODN_opALT,  /* ALT */ 
    METHODN_NONAME,  /* OR */ 
    METHODN_NONAME,  /* AND */ 
    METHODN_opNot,  /* Not */ 
    METHODN_opInstanceof,  /* Instanceof */ 
    METHODN_opIsa,  /* Isa */ 
    METHODN_opAs,  /* As */ 
    METHODN_opHas,  /* Has */ 
    METHODN_opIs,  /* Is */ 
    METHODN_opTo,  /* To */ 
    METHODN_opEq,  /* Eq */ 
    METHODN_opNeq,  /* Neq */ 
    METHODN_opLt,  /* Lt */ 
    METHODN_opLte,  /* Lte */ 
    METHODN_opGt,  /* Gt */ 
    METHODN_opGte,  /* Gte */ 
    METHODN_opMatch,  /* Match */ 
    METHODN_opLor,  /* Lor */ 
    METHODN_opFmt,  /* Fmt */ 
    METHODN_opLshift,  /* Lshift */ 
    METHODN_opRshift,  /* Rshift */ 
    METHODN_opAdd,  /* Add */ 
    METHODN_opSub,  /* Sub */ 
    METHODN_opDiv,  /* Div */ 
    METHODN_opMod,  /* Mod */ 
    METHODN_opMul,  /* Mul */ 
    METHODN_opLand,  /* Land */ 
    METHODN_opLnot,  /* Lnot */ 
    METHODN_opXor,  /* Xor */ 
    METHODN_opNext,  /* Next */ 
    METHODN_opPrev,  /* Prev */ 
    METHODN_opNeg,  /* Neg */ 
    METHODN_NONAME,  /* NUM */ 
    METHODN_NONAME,  /* STR */ 
    METHODN_NONAME,  /* TSTR */ 
    METHODN_NONAME,  /* EVAL */ 
    METHODN_NONAME,  /* LABEL */ 
    METHODN_NONAME,  /* METAN */ 
    METHODN_NONAME,  /* PROPN */ 
    METHODN_NONAME,  /* URN */ 
    METHODN_NONAME,  /* MT */ 
    METHODN_NONAME,  /* CONSTN */ 
    METHODN_NONAME,  /* TYPEN */ 
    METHODN_NONAME,  /* CMETHODN */ 
    METHODN_NONAME,  /* NAME */ 
    METHODN_NONAME,  /* ERR */ 
    METHODN_NONAME,  /* MN */ 
    METHODN_NONAME,  /* FN */ 
    METHODN_NONAME,  /* CID */ 
    METHODN_NONAME,  /* MPR */ 
    METHODN_NONAME,  /* EOT */ 
    METHODN_NONAME,  /* CONST */ 
    METHODN_NONAME,  /* DEFVAL */ 
    METHODN_NONAME,  /* SYSVAL */ 
    METHODN_NONAME,  /* STACK */ 
    METHODN_NONAME,  /* FIELD */ 
    METHODN_NONAME,  /* GLOBAL */ 
    METHODN_NONAME,  /* MEMBER */ 
    METHODN_NONAME,  /* CLASSID */ 
    METHODN_NONAME,  /* ASIS */ 
    METHODN_NONAME,  /* NOP */ 
    //(knh_methodn_t)0
};

knh_methodn_t knh_token_tomethodn(knh_token_t t)
{
    KNH_ASSERT(t < KNH_TOKEN_MAXSIZ);
    return knh_token_array_tomethodn[t];
}

/* ------------------------------------------------------------------------ */

static int knh_token_array_getOpPriority[] = {
    0,  /* namespace */ 
    0,  /* import */ 
    0,  /* pragma */ 
    0,  /* using */ 
    0,  /* with */ 
    0,  /* class */ 
    0,  /* extends */ 
    0,  /* implements */ 
    0,  /* method */ 
    0,  /* format */ 
    0,  /* return */ 
    0,  /* mapmap */ 
    0,  /* weave */ 
    0,  /* aspect */ 
    0,  /* block */ 
    0,  /* if */ 
    0,  /* else */ 
    0,  /* switch */ 
    0,  /* case */ 
    0,  /* while */ 
    0,  /* do */ 
    0,  /* for */ 
    0,  /* foreach */ 
    0,  /* from */ 
    0,  /* where */ 
    0,  /* break */ 
    0,  /* continue */ 
    0,  /* try */ 
    0,  /* catch */ 
    0,  /* finally */ 
    0,  /* throw */ 
    0,  /* local */ 
    0,  /* print */ 
    0,  /* assert */ 
    99,  /* BRACE */ 
    99,  /* PARENTHESIS */ 
    99,  /* BRANCET */ 
    0,  /* DOTS */ 
    0,  /* SEMICOLON */ 
    0,  /* COMMA */ 
    0,  /* LET */ 
    0,  /* lshifte */ 
    0,  /* rshifte */ 
    0,  /* adde */ 
    0,  /* sube */ 
    0,  /* mule */ 
    0,  /* dive */ 
    0,  /* mode */ 
    0,  /* bitore */ 
    0,  /* bitande */ 
    0,  /* ALTLET */ 
    2,  /* QUESTION */ 
    2,  /* Itr */ 
    99,  /* SUBSET */ 
    99,  /* OFFSET */ 
    99,  /* COLON */ 
    2,  /* ALT */ 
    5,  /* OR */ 
    6,  /* AND */ 
    7,  /* Not */ 
    9,  /* Instanceof */ 
    9,  /* Isa */ 
    9,  /* As */ 
    9,  /* Has */ 
    9,  /* Is */ 
    9,  /* To */ 
    9,  /* Eq */ 
    9,  /* Neq */ 
    9,  /* Lt */ 
    9,  /* Lte */ 
    9,  /* Gt */ 
    9,  /* Gte */ 
    9,  /* Match */ 
    11,  /* Lor */ 
    12,  /* Fmt */ 
    15,  /* Lshift */ 
    15,  /* Rshift */ 
    20,  /* Add */ 
    20,  /* Sub */ 
    24,  /* Div */ 
    24,  /* Mod */ 
    24,  /* Mul */ 
    27,  /* Land */ 
    27,  /* Lnot */ 
    27,  /* Xor */ 
    28,  /* Next */ 
    28,  /* Prev */ 
    28,  /* Neg */ 
    99,  /* NUM */ 
    99,  /* STR */ 
    99,  /* TSTR */ 
    99,  /* EVAL */ 
    0,  /* LABEL */ 
    0,  /* METAN */ 
    99,  /* PROPN */ 
    99,  /* URN */ 
    99,  /* MT */ 
    99,  /* CONSTN */ 
    99,  /* TYPEN */ 
    99,  /* CMETHODN */ 
    99,  /* NAME */ 
    0,  /* ERR */ 
    99,  /* MN */ 
    99,  /* FN */ 
    99,  /* CID */ 
    99,  /* MPR */ 
    0,  /* EOT */ 
    99,  /* CONST */ 
    99,  /* DEFVAL */ 
    99,  /* SYSVAL */ 
    99,  /* STACK */ 
    99,  /* FIELD */ 
    99,  /* GLOBAL */ 
    99,  /* MEMBER */ 
    99,  /* CLASSID */ 
    99,  /* ASIS */ 
    99,  /* NOP */ 
    //(int)0
};

int knh_token_getOpPriority(knh_token_t t)
{
    KNH_ASSERT(t < KNH_TOKEN_MAXSIZ);
    return knh_token_array_getOpPriority[t];
}

/* ------------------------------------------------------------------------ */

static int knh_token_array_getOpSize[] = {
    0,  /* namespace */ 
    0,  /* import */ 
    0,  /* pragma */ 
    0,  /* using */ 
    0,  /* with */ 
    0,  /* class */ 
    0,  /* extends */ 
    0,  /* implements */ 
    0,  /* method */ 
    0,  /* format */ 
    0,  /* return */ 
    0,  /* mapmap */ 
    0,  /* weave */ 
    0,  /* aspect */ 
    0,  /* block */ 
    0,  /* if */ 
    0,  /* else */ 
    0,  /* switch */ 
    0,  /* case */ 
    0,  /* while */ 
    0,  /* do */ 
    0,  /* for */ 
    0,  /* foreach */ 
    0,  /* from */ 
    0,  /* where */ 
    0,  /* break */ 
    0,  /* continue */ 
    0,  /* try */ 
    0,  /* catch */ 
    0,  /* finally */ 
    0,  /* throw */ 
    0,  /* local */ 
    0,  /* print */ 
    0,  /* assert */ 
    0,  /* BRACE */ 
    0,  /* PARENTHESIS */ 
    0,  /* BRANCET */ 
    0,  /* DOTS */ 
    0,  /* SEMICOLON */ 
    0,  /* COMMA */ 
    0,  /* LET */ 
    0,  /* lshifte */ 
    0,  /* rshifte */ 
    0,  /* adde */ 
    0,  /* sube */ 
    0,  /* mule */ 
    0,  /* dive */ 
    0,  /* mode */ 
    0,  /* bitore */ 
    0,  /* bitande */ 
    0,  /* ALTLET */ 
    0,  /* QUESTION */ 
    1,  /* Itr */ 
    0,  /* SUBSET */ 
    0,  /* OFFSET */ 
    0,  /* COLON */ 
    0,  /* ALT */ 
    0,  /* OR */ 
    0,  /* AND */ 
    1,  /* Not */ 
    2,  /* Instanceof */ 
    2,  /* Isa */ 
    2,  /* As */ 
    2,  /* Has */ 
    2,  /* Is */ 
    2,  /* To */ 
    2,  /* Eq */ 
    2,  /* Neq */ 
    2,  /* Lt */ 
    2,  /* Lte */ 
    2,  /* Gt */ 
    2,  /* Gte */ 
    2,  /* Match */ 
    0,  /* Lor */ 
    0,  /* Fmt */ 
    0,  /* Lshift */ 
    0,  /* Rshift */ 
    0,  /* Add */ 
    2,  /* Sub */ 
    2,  /* Div */ 
    2,  /* Mod */ 
    2,  /* Mul */ 
    0,  /* Land */ 
    1,  /* Lnot */ 
    2,  /* Xor */ 
    1,  /* Next */ 
    1,  /* Prev */ 
    1,  /* Neg */ 
    0,  /* NUM */ 
    0,  /* STR */ 
    0,  /* TSTR */ 
    0,  /* EVAL */ 
    0,  /* LABEL */ 
    0,  /* METAN */ 
    0,  /* PROPN */ 
    0,  /* URN */ 
    0,  /* MT */ 
    0,  /* CONSTN */ 
    0,  /* TYPEN */ 
    0,  /* CMETHODN */ 
    0,  /* NAME */ 
    0,  /* ERR */ 
    0,  /* MN */ 
    0,  /* FN */ 
    0,  /* CID */ 
    0,  /* MPR */ 
    0,  /* EOT */ 
    0,  /* CONST */ 
    0,  /* DEFVAL */ 
    0,  /* SYSVAL */ 
    0,  /* STACK */ 
    0,  /* FIELD */ 
    0,  /* GLOBAL */ 
    0,  /* MEMBER */ 
    0,  /* CLASSID */ 
    0,  /* ASIS */ 
    0,  /* NOP */ 
    //(int)0
};

int knh_token_getOpSize(knh_token_t t)
{
    KNH_ASSERT(t < KNH_TOKEN_MAXSIZ);
    return knh_token_array_getOpSize[t];
}

/* ------------------------------------------------------------------------ */

static knh_bool_t knh_token_array_isEndOfStmt[] = {
    1,  /* namespace */ 
    1,  /* import */ 
    1,  /* pragma */ 
    1,  /* using */ 
    1,  /* with */ 
    1,  /* class */ 
    1,  /* extends */ 
    1,  /* implements */ 
    1,  /* method */ 
    1,  /* format */ 
    1,  /* return */ 
    1,  /* mapmap */ 
    1,  /* weave */ 
    1,  /* aspect */ 
    1,  /* block */ 
    1,  /* if */ 
    1,  /* else */ 
    1,  /* switch */ 
    1,  /* case */ 
    1,  /* while */ 
    1,  /* do */ 
    1,  /* for */ 
    1,  /* foreach */ 
    1,  /* from */ 
    1,  /* where */ 
    1,  /* break */ 
    1,  /* continue */ 
    1,  /* try */ 
    1,  /* catch */ 
    1,  /* finally */ 
    1,  /* throw */ 
    1,  /* local */ 
    1,  /* print */ 
    1,  /* assert */ 
    0,  /* BRACE */ 
    0,  /* PARENTHESIS */ 
    0,  /* BRANCET */ 
    0,  /* DOTS */ 
    1,  /* SEMICOLON */ 
    0,  /* COMMA */ 
    0,  /* LET */ 
    0,  /* lshifte */ 
    0,  /* rshifte */ 
    0,  /* adde */ 
    0,  /* sube */ 
    0,  /* mule */ 
    0,  /* dive */ 
    0,  /* mode */ 
    0,  /* bitore */ 
    0,  /* bitande */ 
    0,  /* ALTLET */ 
    0,  /* QUESTION */ 
    0,  /* Itr */ 
    0,  /* SUBSET */ 
    0,  /* OFFSET */ 
    0,  /* COLON */ 
    0,  /* ALT */ 
    0,  /* OR */ 
    0,  /* AND */ 
    0,  /* Not */ 
    0,  /* Instanceof */ 
    0,  /* Isa */ 
    0,  /* As */ 
    0,  /* Has */ 
    0,  /* Is */ 
    0,  /* To */ 
    0,  /* Eq */ 
    0,  /* Neq */ 
    0,  /* Lt */ 
    0,  /* Lte */ 
    0,  /* Gt */ 
    0,  /* Gte */ 
    0,  /* Match */ 
    0,  /* Lor */ 
    0,  /* Fmt */ 
    0,  /* Lshift */ 
    0,  /* Rshift */ 
    0,  /* Add */ 
    0,  /* Sub */ 
    0,  /* Div */ 
    0,  /* Mod */ 
    0,  /* Mul */ 
    0,  /* Land */ 
    0,  /* Lnot */ 
    0,  /* Xor */ 
    0,  /* Next */ 
    0,  /* Prev */ 
    0,  /* Neg */ 
    0,  /* NUM */ 
    0,  /* STR */ 
    0,  /* TSTR */ 
    0,  /* EVAL */ 
    0,  /* LABEL */ 
    1,  /* METAN */ 
    0,  /* PROPN */ 
    0,  /* URN */ 
    0,  /* MT */ 
    0,  /* CONSTN */ 
    0,  /* TYPEN */ 
    0,  /* CMETHODN */ 
    0,  /* NAME */ 
    0,  /* ERR */ 
    0,  /* MN */ 
    0,  /* FN */ 
    0,  /* CID */ 
    0,  /* MPR */ 
    1,  /* EOT */ 
    0,  /* CONST */ 
    0,  /* DEFVAL */ 
    0,  /* SYSVAL */ 
    0,  /* STACK */ 
    0,  /* FIELD */ 
    0,  /* GLOBAL */ 
    0,  /* MEMBER */ 
    0,  /* CLASSID */ 
    0,  /* ASIS */ 
    0,  /* NOP */ 
    //(knh_bool_t)0
};

knh_bool_t knh_token_isEndOfStmt(knh_token_t t)
{
    KNH_ASSERT(t < KNH_TOKEN_MAXSIZ);
    return knh_token_array_isEndOfStmt[t];
}

/* ------------------------------------------------------------------------ */

/* ======================================================================== */
/* [token parser] */

PUBLIC
Token *new_Token__parse(Ctx *ctx, knh_flag_t flag, knh_fileid_t fileid, knh_sline_t line, knh_bytes_t token, BytesConv *bconv)
{
    knh_bytes_t t = token;
    knh_token_t tt = ((knh_token_t)-1);
    L_TAIL:;
    //if(t.buf[0] == '.' && islower(t.buf[1])) {   /* .name   => name */
        //flag |= KNH_FLAG_TKF_TOPDOT;
        //t.buf++; t.len--;
    //}
    if(isalpha(t.buf[0])) {
        if(t.buf[t.len-2]=='.' && t.buf[t.len-1]=='.') {
            flag |= KNH_FLAG_TKF_ITERATORTYPE;
            t.len -= 2;
        }
        if(t.buf[t.len-2]=='.' && t.buf[t.len-1]=='*') {
            flag |= KNH_FLAG_TKF_TAILWILDCARD;
            t.len -= 2;
        }
        if(isupper(t.buf[0]) && t.len > 9 && knh_bytes_endsWith(t, STEXT("Exception"))) {
            flag |= (KNH_FLAG_TKF_EXCEPTIONTYPE|KNH_FLAG_TKF_NOTNULLTYPE);
            t.len -= 9;
        }
        else if(t.buf[t.len-2]=='!' && t.buf[t.len-1]=='!') {
            flag |= (KNH_FLAG_TKF_EXCEPTIONTYPE|KNH_FLAG_TKF_NOTNULLTYPE);
            t.len -= 2;
        }
        else if(t.buf[t.len-1]=='!') {
            flag |= KNH_FLAG_TKF_NOTNULLTYPE;
            t.len -= 1;
        }
    }

    switch(t.buf[0]) {
    case '!':
        if(ISB(t, "!")) { tt = TT_NOT; break; }
        if(ISB(t, "!=")) { tt = TT_NEQ; break; }
    break;
    case '%':
        if(ISB(t, "%=")) { tt = TT_MODE; break; }
        if(ISB(t, "%%")) { tt = TT_FMT; break; }
        if(ISB(t, "%")) { tt = TT_MOD; break; }
    break;
    case '&':
        if(ISB(t, "&=")) { tt = TT_BITANDE; break; }
        if(ISB(t, "&&")) { tt = TT_AND; break; }
        if(ISB(t, "&")) { tt = TT_LAND; break; }
    break;
    case '(':
        if(ISB(t, "(")) { tt = TT_PARENTHESIS; break; }
    break;
    case '*':
        if(ISB(t, "*=")) { tt = TT_MULE; break; }
        if(ISB(t, "*")) { tt = TT_MUL; break; }
    break;
    case '+':
        if(ISB(t, "+=")) { tt = TT_ADDE; break; }
        if(ISB(t, "+")) { tt = TT_ADD; break; }
        if(ISB(t, "++")) { tt = TT_NEXT; break; }
    break;
    case ',':
        if(ISB(t, ",")) { tt = TT_COMMA; break; }
    break;
    case '-':
        if(ISB(t, "-=")) { tt = TT_SUBE; break; }
        if(ISB(t, "-->")) { tt = TT_TO; break; }
        if(ISB(t, "-")) { tt = TT_SUB; break; }
        if(ISB(t, "--")) { tt = TT_PREV; break; }
        if(ISB(t, "---")) { tt = TT_NEG; break; }
    break;
    case '.':
        if(ISB(t, "...")) { tt = TT_DOTS; break; }
        if(ISB(t, "..")) { tt = TT_ITR; break; }
        if(ISB(t, "..<")) { tt = TT_SUBSET; break; }
        if(ISB(t, "..+")) { tt = TT_OFFSET; break; }
    break;
    case '/':
        if(ISB(t, "/=")) { tt = TT_DIVE; break; }
        if(ISB(t, "/")) { tt = TT_DIV; break; }
    break;
    case ':':
        if(ISB(t, ":")) { tt = TT_COLON; break; }
    break;
    case ';':
        if(ISB(t, ";")) { tt = TT_SEMICOLON; break; }
    break;
    case '<':
        if(ISB(t, "<<=")) { tt = TT_LSHIFTE; break; }
        if(ISB(t, "<")) { tt = TT_LT; break; }
        if(ISB(t, "<=")) { tt = TT_LTE; break; }
        if(ISB(t, "<<")) { tt = TT_LSHIFT; break; }
    break;
    case '=':
        if(ISB(t, "=")) { tt = TT_LET; break; }
        if(ISB(t, "===")) { tt = TT_IS; break; }
        if(ISB(t, "==")) { tt = TT_EQ; break; }
        if(ISB(t, "=~")) { tt = TT_MATCH; break; }
    break;
    case '>':
        if(ISB(t, ">>=")) { tt = TT_RSHIFTE; break; }
        if(ISB(t, ">")) { tt = TT_GT; break; }
        if(ISB(t, ">=")) { tt = TT_GTE; break; }
        if(ISB(t, ">>")) { tt = TT_RSHIFT; break; }
    break;
    case '?':
        if(ISB(t, "?\?=")) { tt = TT_ALTLET; break; }
        if(ISB(t, "?")) { tt = TT_QUESTION; break; }
        if(ISB(t, "?\?")) { tt = TT_ALT; break; }
    break;
    case 'A':
        if(ISB(t, "ArrayList")) { t = STEXT("Array"); goto L_TAIL; }
    break;
    case 'F':
        if(ISB(t, "FALSE")) { t = STEXT("false"); goto L_TAIL; }
        if(ISB(t, "False")) { t = STEXT("false"); goto L_TAIL; }
    break;
    case 'I':
        if(ISB(t, "Integer")) { t = STEXT("Int"); goto L_TAIL; }
    break;
    case 'N':
        if(ISB(t, "NULL")) { t = STEXT("null"); goto L_TAIL; }
    break;
    case 'T':
        if(ISB(t, "TRUE")) { t = STEXT("true"); goto L_TAIL; }
        if(ISB(t, "True")) { t = STEXT("true"); goto L_TAIL; }
    break;
    case '[':
        if(ISB(t, "[")) { tt = TT_BRANCET; break; }
    break;
    case '^':
        if(ISB(t, "^")) { tt = TT_XOR; break; }
    break;
    case 'a':
        if(ISB(t, "aspect")) { tt = TT_ASPECT; break; }
        if(ISB(t, "assert")) { tt = TT_ASSERT; break; }
        if(ISB(t, "and")) { tt = TT_AND; break; }
        if(ISB(t, "as")) { tt = TT_AS; break; }
    break;
    case 'b':
        if(ISB(t, "block")) { tt = TT_BLOCK; break; }
        if(ISB(t, "break")) { tt = TT_BREAK; break; }
        if(ISB(t, "boolean")) { t = STEXT("Boolean!"); goto L_TAIL; }
    break;
    case 'c':
        if(ISB(t, "class")) { tt = TT_CLASS; break; }
        if(ISB(t, "case")) { tt = TT_CASE; break; }
        if(ISB(t, "continue")) { tt = TT_CONTINUE; break; }
        if(ISB(t, "catch")) { tt = TT_CATCH; break; }
        if(ISB(t, "const")) { tt = TT_METAN; break; }
        if(ISB(t, "char")) { t = STEXT("Int!"); goto L_TAIL; }
    break;
    case 'd':
        if(ISB(t, "do")) { tt = TT_DO; break; }
        if(ISB(t, "double")) { t = STEXT("Float!"); goto L_TAIL; }
        if(ISB(t, "def")) { t = STEXT("Any"); goto L_TAIL; }
    break;
    case 'e':
        if(ISB(t, "extends")) { tt = TT_EXTENDS; break; }
        if(ISB(t, "else")) { tt = TT_ELSE; break; }
    break;
    case 'f':
        if(ISB(t, "format")) { tt = TT_FORMAT; break; }
        if(ISB(t, "for")) { tt = TT_FOR; break; }
        if(ISB(t, "foreach")) { tt = TT_FOREACH; break; }
        if(ISB(t, "from")) { tt = TT_FROM; break; }
        if(ISB(t, "finally")) { tt = TT_FINALLY; break; }
        if(ISB(t, "final")) { tt = TT_METAN; break; }
        if(ISB(t, "float")) { t = STEXT("Float!"); goto L_TAIL; }
    break;
    case 'i':
        if(ISB(t, "import")) { tt = TT_IMPORT; break; }
        if(ISB(t, "implements")) { tt = TT_IMPLEMENTS; break; }
        if(ISB(t, "if")) { tt = TT_IF; break; }
        if(ISB(t, "instanceof")) { tt = TT_INSTANCEOF; break; }
        if(ISB(t, "isa?")) { tt = TT_ISA; break; }
        if(ISB(t, "isa")) { tt = TT_ISA; break; }
        if(ISB(t, "in?")) { tt = TT_HAS; break; }
        if(ISB(t, "is?")) { tt = TT_IS; break; }
        if(ISB(t, "int")) { t = STEXT("Int!"); goto L_TAIL; }
    break;
    case 'l':
        if(ISB(t, "local")) { tt = TT_LOCAL; break; }
        if(ISB(t, "long")) { t = STEXT("Int!"); goto L_TAIL; }
    break;
    case 'm':
        if(ISB(t, "method")) { tt = TT_METHOD; break; }
        if(ISB(t, "mapmap")) { tt = TT_MAPMAP; break; }
        if(ISB(t, "mod")) { tt = TT_MOD; break; }
    break;
    case 'n':
        if(ISB(t, "namespace")) { tt = TT_NAMESPACE; break; }
        if(ISB(t, "not")) { tt = TT_NOT; break; }
    break;
    case 'o':
        if(ISB(t, "or")) { tt = TT_OR; break; }
    break;
    case 'p':
        if(ISB(t, "pragma")) { tt = TT_PRAGMA; break; }
        if(ISB(t, "print")) { tt = TT_PRINT; break; }
        if(ISB(t, "public")) { tt = TT_METAN; break; }
        if(ISB(t, "private")) { tt = TT_METAN; break; }
    break;
    case 'r':
        if(ISB(t, "return")) { tt = TT_RETURN; break; }
    break;
    case 's':
        if(ISB(t, "switch")) { tt = TT_SWITCH; break; }
        if(ISB(t, "static")) { tt = TT_METAN; break; }
        if(ISB(t, "short")) { t = STEXT("Int!"); goto L_TAIL; }
        if(ISB(t, "string")) { t = STEXT("String!"); goto L_TAIL; }
    break;
    case 't':
        if(ISB(t, "try")) { tt = TT_TRY; break; }
        if(ISB(t, "throw")) { tt = TT_THROW; break; }
        if(ISB(t, "to?")) { tt = TT_TO; break; }
    break;
    case 'u':
        if(ISB(t, "using")) { tt = TT_USING; break; }
    break;
    case 'v':
        if(ISB(t, "virtual")) { tt = TT_METAN; break; }
    break;
    case 'w':
        if(ISB(t, "with")) { tt = TT_WITH; break; }
        if(ISB(t, "weave")) { tt = TT_WEAVE; break; }
        if(ISB(t, "while")) { tt = TT_WHILE; break; }
        if(ISB(t, "where")) { tt = TT_WHERE; break; }
    break;
    case '{':
        if(ISB(t, "{")) { tt = TT_BRACE; break; }
    break;
    case '|':
        if(ISB(t, "|=")) { tt = TT_BITORE; break; }
        if(ISB(t, "||")) { tt = TT_OR; break; }
        if(ISB(t, "|")) { tt = TT_LOR; break; }
    break;
    case '~':
        if(ISB(t, "~")) { tt = TT_LNOT; break; }
    break;
    /* built-in */
    }
    if(tt == ((knh_token_t)-1)) {
        return new_Token__name(ctx, flag, fileid, line, t);
    }
    else {
        Token *tk = new_Token(ctx, flag, fileid, line, tt);
        if(tt == TT_METAN) {
            KNH_SETv(ctx, DP(tk)->data, new_String(ctx, t, NULL));
        }
        return  tk;
    }
}

/* ======================================================================== */
/* [token checker] */

int knh_bytes_istoken(knh_bytes_t t, int ch)
{
    switch(ch) {

    case '%':
        if(ISB(t, "%")) { return 1; }
        return 0; 
    case '&':
        if(ISB(t, "&")) { return 1; }
        return 0; 
    case '+':
        if(ISB(t, "..")) { return 1; }
        if(ISB(t, "+")) { return 1; }
        return 0; 
    case '-':
        if(ISB(t, "-")) { return 1; }
        if(ISB(t, "--")) { return 1; }
        return 0; 
    case '.':
        if(ISB(t, "..")) { return 1; }
        if(ISB(t, ".")) { return 1; }
        return 0; 
    case '<':
        if(ISB(t, "..")) { return 1; }
        if(ISB(t, "<")) { return 1; }
        return 0; 
    case '=':
        if(ISB(t, "<<")) { return 1; }
        if(ISB(t, ">>")) { return 1; }
        if(ISB(t, "+")) { return 1; }
        if(ISB(t, "-")) { return 1; }
        if(ISB(t, "*")) { return 1; }
        if(ISB(t, "/")) { return 1; }
        if(ISB(t, "%")) { return 1; }
        if(ISB(t, "|")) { return 1; }
        if(ISB(t, "&")) { return 1; }
        if(ISB(t, "??")) { return 1; }
        if(ISB(t, "==")) { return 1; }
        if(ISB(t, "=")) { return 1; }
        if(ISB(t, "!")) { return 1; }
        if(ISB(t, "<")) { return 1; }
        if(ISB(t, ">")) { return 1; }
        return 0; 
    case '>':
        if(ISB(t, "--")) { return 1; }
        if(ISB(t, ">")) { return 1; }
        return 0; 
    case '?':
        if(ISB(t, "?")) { return 1; }
        if(ISB(t, "isa")) { return 1; }
        if(ISB(t, "in")) { return 1; }
        if(ISB(t, "is")) { return 1; }
        if(ISB(t, "to")) { return 1; }
        return 0; 
    case '|':
        if(ISB(t, "|")) { return 1; }
        return 0; 
    case '~':
        if(ISB(t, "=")) { return 1; }
        return 0; 
    /* built-in */
    }
    return 0;
}

/* ======================================================================== */
/* [stmt] */

static char* knh_stmt_array_tochar[] = {
    "done",  /* done */ 
    "namespace",  /* namespace */ 
    "import",  /* import */ 
    "pragma",  /* pragma */ 
    "using",  /* using */ 
    "using import",  /* using import */ 
    "using alias",  /* using alias */ 
    "using unit",  /* using unit */ 
    "using enum",  /* using enum */ 
    "using vocab",  /* using vocab */ 
    "with",  /* with */ 
    "using func",  /* using func */ 
    "using mapmap",  /* using mapmap */ 
    "class",  /* class */ 
    "extends",  /* extends */ 
    "implements",  /* implements */ 
    "method",  /* method */ 
    "closure",  /* closure */ 
    "format",  /* format */ 
    "decl",  /* decl */ 
    "return",  /* return */ 
    "mapmap",  /* mapmap */ 
    "weave",  /* weave */ 
    "aspect",  /* aspect */ 
    "block",  /* block */ 
    "if",  /* if */ 
    "else",  /* else */ 
    "switch",  /* switch */ 
    "case",  /* case */ 
    "while",  /* while */ 
    "do",  /* do */ 
    "for",  /* for */ 
    "foreach",  /* foreach */ 
    "from",  /* from */ 
    "where",  /* where */ 
    "break",  /* break */ 
    "continue",  /* continue */ 
    "try",  /* try */ 
    "catch",  /* catch */ 
    "finally",  /* finally */ 
    "throw",  /* throw */ 
    "err",  /* err */ 
    "let",  /* let */ 
    "local",  /* local */ 
    "letmulti",  /* letmulti */ 
    "new",  /* new */ 
    "mapcast",  /* mapcast */ 
    "call",  /* call */ 
    "call1",  /* call1 */ 
    "pcall",  /* pcall */ 
    "mt",  /* mt */ 
    "op",  /* op */ 
    "tri",  /* tri */ 
    "alt",  /* alt */ 
    "and",  /* and */ 
    "or",  /* or */ 
    "print",  /* print */ 
    "assert",  /* assert */ 
    //(char*)0
};

char* knh_stmt_tochar(knh_stmt_t t)
{
    KNH_ASSERT(t < KNH_STMT_MAXSIZ);
    return knh_stmt_array_tochar[t];
}

/* ------------------------------------------------------------------------ */

static size_t knh_stmt_array_size[] = {
    0,  /* done */ 
    1,  /* namespace */ 
    1,  /* import */ 
    2,  /* pragma */ 
    0,  /* using */ 
    1,  /* using import */ 
    2,  /* using alias */ 
    2,  /* using unit */ 
    2,  /* using enum */ 
    2,  /* using vocab */ 
    0,  /* with */ 
    1,  /* using func */ 
    3,  /* using mapmap */ 
    4,  /* class */ 
    0,  /* extends */ 
    5,  /* implements */ 
    5,  /* method */ 
    3,  /* closure */ 
    7,  /* format */ 
    3,  /* decl */ 
    5,  /* return */ 
    5,  /* mapmap */ 
    2,  /* weave */ 
    5,  /* aspect */ 
    1,  /* block */ 
    3,  /* if */ 
    0,  /* else */ 
    2,  /* switch */ 
    0,  /* case */ 
    2,  /* while */ 
    2,  /* do */ 
    4,  /* for */ 
    5,  /* foreach */ 
    0,  /* from */ 
    0,  /* where */ 
    1,  /* break */ 
    1,  /* continue */ 
    3,  /* try */ 
    3,  /* catch */ 
    0,  /* finally */ 
    1,  /* throw */ 
    1,  /* err */ 
    2,  /* let */ 
    5,  /* local */ 
    5,  /* letmulti */ 
    7,  /* new */ 
    3,  /* mapcast */ 
    7,  /* call */ 
    1,  /* call1 */ 
    7,  /* pcall */ 
    3,  /* mt */ 
    7,  /* op */ 
    3,  /* tri */ 
    5,  /* alt */ 
    5,  /* and */ 
    5,  /* or */ 
    5,  /* print */ 
    1,  /* assert */ 
    //(size_t)0
};

size_t knh_stmt_size(knh_stmt_t t)
{
    KNH_ASSERT(t < KNH_STMT_MAXSIZ);
    return knh_stmt_array_size[t];
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [EXPTN] */

void knh_Stmt_add_EXPTN(Ctx *ctx, Stmt *o, knh_tokens_t *tc)
{
	if(SP(o)->stt == STT_ERR) return;
	if(tc->c < tc->e && knh_Token_isEXPTN(tc->ts[tc->c])) {
		knh_Stmt_add(ctx, o, UP(tc->ts[tc->c]));
		tc->c += 1;
	}
	else {
		knh_Stmt_tokens_perror(ctx, o, tc, KMSG_TEXPTN);
	}
}

/* ------------------------------------------------------------------------ */
/* [NSNAME] */

void knh_Stmt_add_NSNAME(Ctx *ctx, Stmt *o, knh_tokens_t *tc)
{
	if(SP(o)->stt == STT_ERR) return;
	if(tc->c < tc->e && knh_Token_isNSNAME(tc->ts[tc->c])) {
		knh_Stmt_add(ctx, o, UP(tc->ts[tc->c]));
		tc->c += 1;
	}
	else {
		knh_Stmt_tokens_perror(ctx, o, tc, KMSG_TNSNAME);
	}
}

/* ------------------------------------------------------------------------ */
/* [CURN] */

void knh_Stmt_add_CURN(Ctx *ctx, Stmt *o, knh_tokens_t *tc)
{
	if(SP(o)->stt == STT_ERR) return;
	if(tc->c < tc->e && knh_Token_isCURN(tc->ts[tc->c])) {
		knh_Stmt_add(ctx, o, UP(tc->ts[tc->c]));
		tc->c += 1;
	}
	else {
		knh_Stmt_tokens_perror(ctx, o, tc, KMSG_TCURN);
	}
}

/* ------------------------------------------------------------------------ */
/* [NSCLASSN] */

void knh_Stmt_add_NSCLASSN(Ctx *ctx, Stmt *o, knh_tokens_t *tc)
{
	if(SP(o)->stt == STT_ERR) return;
	if(tc->c < tc->e && knh_Token_isNSCLASSN(tc->ts[tc->c])) {
		knh_Stmt_add(ctx, o, UP(tc->ts[tc->c]));
		tc->c += 1;
	}
	else {
		knh_Stmt_tokens_perror(ctx, o, tc, KMSG_TNSCLASSN);
	}
}

/* ------------------------------------------------------------------------ */
/* [TYPEN] */

void knh_Stmt_add_TYPEN(Ctx *ctx, Stmt *o, knh_tokens_t *tc)
{
	if(SP(o)->stt == STT_ERR) return;
	if(tc->c < tc->e && knh_Token_isTYPEN(tc->ts[tc->c])) {
		knh_Stmt_add(ctx, o, UP(tc->ts[tc->c]));
		tc->c += 1;
	}
	else {
		knh_Stmt_tokens_perror(ctx, o, tc, KMSG_TTYPEN);
	}
}

/* ------------------------------------------------------------------------ */
/* [METHODN] */

void knh_Stmt_add_METHODN(Ctx *ctx, Stmt *o, knh_tokens_t *tc)
{
	if(SP(o)->stt == STT_ERR) return;
	if(tc->c < tc->e && knh_Token_isMETHODN(tc->ts[tc->c])) {
		knh_Stmt_add(ctx, o, UP(tc->ts[tc->c]));
		tc->c += 1;
	}
	else {
		knh_Stmt_tokens_perror(ctx, o, tc, KMSG_TMETHODN);
	}
}

/* ------------------------------------------------------------------------ */
/* [FURN] */

void knh_Stmt_add_FURN(Ctx *ctx, Stmt *o, knh_tokens_t *tc)
{
	if(SP(o)->stt == STT_ERR) return;
	if(tc->c < tc->e && knh_Token_isFURN(tc->ts[tc->c])) {
		knh_Stmt_add(ctx, o, UP(tc->ts[tc->c]));
		tc->c += 1;
	}
	else {
		knh_Stmt_tokens_perror(ctx, o, tc, KMSG_TFURN);
	}
}

/* ------------------------------------------------------------------------ */
/* [CLASSTN] */

void knh_Stmt_add_CLASSTN(Ctx *ctx, Stmt *o, knh_tokens_t *tc)
{
	if(SP(o)->stt == STT_ERR) return;
	if(tc->c < tc->e && knh_Token_isCLASSTN(tc->ts[tc->c])) {
		knh_Stmt_add(ctx, o, UP(tc->ts[tc->c]));
		tc->c += 1;
	}
	else {
		knh_Stmt_tokens_perror(ctx, o, tc, KMSG_TCLASSTN);
	}
}

/* ------------------------------------------------------------------------ */
/* [STR] */

void knh_Stmt_add_STR(Ctx *ctx, Stmt *o, knh_tokens_t *tc)
{
	if(SP(o)->stt == STT_ERR) return;
	if(tc->c < tc->e && knh_Token_isSTR(tc->ts[tc->c])) {
		knh_Stmt_add(ctx, o, UP(tc->ts[tc->c]));
		tc->c += 1;
	}
	else {
		knh_Stmt_tokens_perror(ctx, o, tc, KMSG_TSTR);
	}
}

/* ------------------------------------------------------------------------ */
/* [VARN] */

void knh_Stmt_add_VARN(Ctx *ctx, Stmt *o, knh_tokens_t *tc)
{
	if(SP(o)->stt == STT_ERR) return;
	if(tc->c < tc->e && knh_Token_isVARN(tc->ts[tc->c])) {
		knh_Stmt_add(ctx, o, UP(tc->ts[tc->c]));
		tc->c += 1;
	}
	else {
		knh_Stmt_tokens_perror(ctx, o, tc, KMSG_TVARN);
	}
}

/* ------------------------------------------------------------------------ */
/* [CLASSN] */

void knh_Stmt_add_CLASSN(Ctx *ctx, Stmt *o, knh_tokens_t *tc)
{
	if(SP(o)->stt == STT_ERR) return;
	if(tc->c < tc->e && knh_Token_isCLASSN(tc->ts[tc->c])) {
		knh_Stmt_add(ctx, o, UP(tc->ts[tc->c]));
		tc->c += 1;
	}
	else {
		knh_Stmt_tokens_perror(ctx, o, tc, KMSG_TCLASSN);
	}
}

/* ------------------------------------------------------------------------ */
/* [CMETHODN] */

void knh_Stmt_add_CMETHODN(Ctx *ctx, Stmt *o, knh_tokens_t *tc)
{
	if(SP(o)->stt == STT_ERR) return;
	if(tc->c < tc->e && knh_Token_isCMETHODN(tc->ts[tc->c])) {
		knh_Stmt_add(ctx, o, UP(tc->ts[tc->c]));
		tc->c += 1;
	}
	else {
		knh_Stmt_tokens_perror(ctx, o, tc, KMSG_TCMETHODN);
	}
}

/* ------------------------------------------------------------------------ */
/* [MT] */

void knh_Stmt_add_MT(Ctx *ctx, Stmt *o, knh_tokens_t *tc)
{
	if(SP(o)->stt == STT_ERR) return;
	if(tc->c < tc->e && knh_Token_isMT(tc->ts[tc->c])) {
		knh_Stmt_add(ctx, o, UP(tc->ts[tc->c]));
		tc->c += 1;
	}
	else {
		knh_Stmt_tokens_perror(ctx, o, tc, KMSG_TMT);
	}
}

/* ------------------------------------------------------------------------ */
/* [ANY] */

void knh_Stmt_add_ANY(Ctx *ctx, Stmt *o, knh_tokens_t *tc)
{
	if(SP(o)->stt == STT_ERR) return;
	if(tc->c < tc->e && knh_Token_isANY(tc->ts[tc->c])) {
		knh_Stmt_add(ctx, o, UP(tc->ts[tc->c]));
		tc->c += 1;
	}
	else {
		knh_Stmt_tokens_perror(ctx, o, tc, KMSG_TANY);
	}
}

/* ======================================================================== */
/* [stmt parser] */

/* ------------------------------------------------------------------------ */
/* [namespace] */

static Stmt *new_StmtNAMESPACE(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_NAMESPACE);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_NSNAME(ctx, o, tc); /* NSNAME */
	knh_Stmt_add_SEMICOLON(ctx, o, tc); /* ; */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [import] */

static Stmt *new_StmtIMPORT(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_IMPORT);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_FURN(ctx, o, tc); /* FURN */
	knh_Stmt_add_SEMICOLON(ctx, o, tc); /* ; */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [pragma] */

static Stmt *new_StmtPRAGMA(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_PRAGMA);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_VARN(ctx, o, tc); /* VARN */
	knh_Stmt_add_PRAGMA(ctx, o, tc); /* pragma */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [class] */

static Stmt *new_StmtCLASS(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_CLASS);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_CLASSN(ctx, o, tc); /* CLASSN */
	knh_Stmt_add_EXTENDS(ctx, o, tc); /* extends */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [format] */

static Stmt *new_StmtFORMAT(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_FORMAT);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_MT(ctx, o, tc); /* MT */
	knh_Stmt_add_PARAMs(ctx, o, tc); /* PARAM* */
	knh_Stmt_add_STR(ctx, o, tc); /* STR */
	knh_Stmt_add_SEMICOLON(ctx, o, tc); /* ; */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [return] */

static Stmt *new_StmtRETURN(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_RETURN);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_EXPRs(ctx, o, tc); /* EXPR* */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [mapmap] */

static Stmt *new_StmtMAPMAP(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_MAPMAP);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_CLASSTNs(ctx, o, tc); /* CLASSTN* */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [weave] */

static Stmt *new_StmtWEAVE(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_WEAVE);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_ANY(ctx, o, tc); /* ANY */
	knh_Stmt_add_CMETHODN(ctx, o, tc); /* CMETHODN */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [aspect] */

static Stmt *new_StmtASPECT(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_ASPECT);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_TYPEN(ctx, o, tc); /* TYPEN */
	knh_Stmt_add_CMETHOD(ctx, o, tc); /* cmethod */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [block] */

static Stmt *new_StmtBLOCK(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_BLOCK);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_STMT1(ctx, o, tc); /* { */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [if] */

static Stmt *new_StmtIF(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_IF);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_PEXPR(ctx, o, tc); /* pexpr */
	knh_Stmt_add_STMT1(ctx, o, tc); /* { */
	knh_Stmt_add_ELSE(ctx, o, tc); /* else */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [switch] */

static Stmt *new_StmtSWITCH(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_SWITCH);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_PEXPR(ctx, o, tc); /* pexpr */
	knh_Stmt_add_STMT1(ctx, o, tc); /* { */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [while] */

static Stmt *new_StmtWHILE(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_WHILE);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_PEXPR(ctx, o, tc); /* pexpr */
	knh_Stmt_add_STMT1(ctx, o, tc); /* { */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [do] */

static Stmt *new_StmtDO(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_DO);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_STMT1(ctx, o, tc); /* { */
	knh_Stmt_add_PEXPR(ctx, o, tc); /* pexpr */
	knh_Stmt_add_SEMICOLON(ctx, o, tc); /* ; */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [for] */

static Stmt *new_StmtFOR(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_FOR);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_PSTMT3(ctx, o, tc); /* pstmt3 */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [foreach] */

static Stmt *new_StmtFOREACH(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_FOREACH);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_PEACH(ctx, o, tc); /* peach */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [break] */

static Stmt *new_StmtBREAK(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_BREAK);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_ANY_(ctx, o, tc); /* any_ */
	knh_Stmt_add_SEMICOLON(ctx, o, tc); /* ; */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [continue] */

static Stmt *new_StmtCONTINUE(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_CONTINUE);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_ANY_(ctx, o, tc); /* any_ */
	knh_Stmt_add_SEMICOLON(ctx, o, tc); /* ; */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [try] */

static Stmt *new_StmtTRY(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_TRY);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_STMT1(ctx, o, tc); /* { */
	knh_Stmt_add_CATCH(ctx, o, tc); /* catch */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [throw] */

static Stmt *new_StmtTHROW(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_THROW);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_EXPR(ctx, o, tc); /* expr */
	knh_Stmt_add_SEMICOLON(ctx, o, tc); /* ; */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [local] */

static Stmt *new_StmtLOCAL(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_LOCAL);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_EXPRs(ctx, o, tc); /* expr* */
	knh_Stmt_add_SEMICOLON(ctx, o, tc); /* ; */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [print] */

static Stmt *new_StmtPRINT(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_PRINT);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_EXPRs(ctx, o, tc); /* expr* */
	knh_Stmt_add_SEMICOLON(ctx, o, tc); /* ; */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [assert] */

static Stmt *new_StmtASSERT(Ctx *ctx, knh_tokens_t *tc)
{
	Stmt *o = new_Stmt(ctx, 0, STT_ASSERT);
	knh_StmtMETA_add_prestmt(ctx, o, tc, -2);
	knh_Stmt_add_PEXPR(ctx, o, tc); /* pexpr */
	knh_Stmt_add_SEMICOLON(ctx, o, tc); /* ; */
	return o;
}

/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* [stmt1] */

Stmt *
new_StmtSTMT1(Ctx *ctx, knh_tokens_t *tc)
{
    Token *tkc = NULL;
    L_TAIL:;
    if(!(tc->c < tc->e)) {
        DBG2_P("c = %d, e = %d", tc->c, tc->e);
        return new_StmtDONE(ctx);
    }
    tkc = tc->ts[tc->c];
    tc->c += 1;
    switch(SP(tkc)->tt) {
 
    case TT_NAMESPACE: 
        return new_StmtNAMESPACE(ctx, tc);
    case TT_IMPORT: 
        return new_StmtIMPORT(ctx, tc);
    case TT_PRAGMA: 
        return new_StmtPRAGMA(ctx, tc);
    case TT_USING: 
        return new_StmtUSING(ctx, tc);
    case TT_WITH:  /* with */
        break;
    case TT_CLASS: 
        return new_StmtCLASS(ctx, tc);
    case TT_EXTENDS:  /* extends */
        break;
    case TT_IMPLEMENTS:  /* implements */
        break;
    case TT_METHOD: 
        return new_StmtMETHOD(ctx, tc);
    case TT_FORMAT: 
        return new_StmtFORMAT(ctx, tc);
    case TT_RETURN: 
        return new_StmtRETURN(ctx, tc);
    case TT_MAPMAP: 
        return new_StmtMAPMAP(ctx, tc);
    case TT_WEAVE: 
        return new_StmtWEAVE(ctx, tc);
    case TT_ASPECT: 
        return new_StmtASPECT(ctx, tc);
    case TT_BLOCK: 
        return new_StmtBLOCK(ctx, tc);
    case TT_IF: 
        return new_StmtIF(ctx, tc);
    case TT_ELSE:  /* else */
        break;
    case TT_SWITCH: 
        return new_StmtSWITCH(ctx, tc);
    case TT_CASE:  /* case */
        break;
    case TT_WHILE: 
        return new_StmtWHILE(ctx, tc);
    case TT_DO: 
        return new_StmtDO(ctx, tc);
    case TT_FOR: 
        return new_StmtFOR(ctx, tc);
    case TT_FOREACH: 
        return new_StmtFOREACH(ctx, tc);
    case TT_FROM:  /* from */
        break;
    case TT_WHERE:  /* where */
        break;
    case TT_BREAK: 
        return new_StmtBREAK(ctx, tc);
    case TT_CONTINUE: 
        return new_StmtCONTINUE(ctx, tc);
    case TT_TRY: 
        return new_StmtTRY(ctx, tc);
    case TT_CATCH:  /* catch */
        break;
    case TT_FINALLY:  /* finally */
        break;
    case TT_THROW: 
        return new_StmtTHROW(ctx, tc);
    case TT_LOCAL: 
        return new_StmtLOCAL(ctx, tc);
    case TT_PRINT: 
        return new_StmtPRINT(ctx, tc);
    case TT_ASSERT: 
        return new_StmtASSERT(ctx, tc);
    case TT_BRACE:  /* BRACE */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_PARENTHESIS:  /* PARENTHESIS */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_BRANCET:  /* BRANCET */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_DOTS:  /* DOTS */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_SEMICOLON:
        return new_StmtDONE(ctx);
    case TT_COMMA:  /* COMMA */
        break;
    case TT_LET:  /* LET */
        break;
    case TT_LSHIFTE:  /* lshifte */
        break;
    case TT_RSHIFTE:  /* rshifte */
        break;
    case TT_ADDE:  /* adde */
        break;
    case TT_SUBE:  /* sube */
        break;
    case TT_MULE:  /* mule */
        break;
    case TT_DIVE:  /* dive */
        break;
    case TT_MODE:  /* mode */
        break;
    case TT_BITORE:  /* bitore */
        break;
    case TT_BITANDE:  /* bitande */
        break;
    case TT_ALTLET:  /* ALTLET */
        break;
    case TT_QUESTION:  /* QUESTION */
        break;
    case TT_ITR:  /* Itr */
        break;
    case TT_SUBSET:  /* SUBSET */
        break;
    case TT_OFFSET:  /* OFFSET */
        break;
    case TT_COLON:  /* COLON */
        break;
    case TT_ALT:  /* ALT */
        break;
    case TT_OR:  /* OR */
        break;
    case TT_AND:  /* AND */
        break;
    case TT_NOT:  /* Not */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_INSTANCEOF:  /* Instanceof */
        break;
    case TT_ISA:  /* Isa */
        break;
    case TT_AS:  /* As */
        break;
    case TT_HAS:  /* Has */
        break;
    case TT_IS:  /* Is */
        break;
    case TT_TO:  /* To */
        break;
    case TT_EQ:  /* Eq */
        break;
    case TT_NEQ:  /* Neq */
        break;
    case TT_LT:  /* Lt */
        break;
    case TT_LTE:  /* Lte */
        break;
    case TT_GT:  /* Gt */
        break;
    case TT_GTE:  /* Gte */
        break;
    case TT_MATCH:  /* Match */
        break;
    case TT_LOR:  /* Lor */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_FMT:  /* Fmt */
        break;
    case TT_LSHIFT:  /* Lshift */
        break;
    case TT_RSHIFT:  /* Rshift */
        break;
    case TT_ADD:  /* Add */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_SUB:  /* Sub */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_DIV:  /* Div */
        break;
    case TT_MOD:  /* Mod */
        break;
    case TT_MUL:  /* Mul */
        break;
    case TT_LAND:  /* Land */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_LNOT:  /* Lnot */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_XOR:  /* Xor */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_NEXT:  /* Next */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_PREV:  /* Prev */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_NEG:  /* Neg */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_NUM:  /* NUM */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_STR:  /* STR */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_TSTR:  /* TSTR */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_EVAL:  /* EVAL */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_LABEL: 
        goto L_TAIL;
    case TT_METAN: 
        if(DP(tkc)->tt_next == TT_PARENTHESIS) {
            tc->c += 1;
        };
        goto L_TAIL;
    case TT_PROPN:  /* PROPN */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_URN:  /* URN */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_MT:  /* MT */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_CONSTN:  /* CONSTN */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_TYPEN:  /* TYPEN */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_CMETHODN:  /* CMETHODN */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_NAME:  /* NAME */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_ERR:
        /* no error message */
        return new_StmtERR(ctx, tc);
    case TT_MN:  /* MN */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_FN:  /* FN */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_CID:  /* CID */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_MPR:  /* MPR */
        tc->c -= 1;
        return new_StmtFUNCEXPR(ctx, tc);
    case TT_EOT:  /* EOT */
        break;
    case TT_CONST:  /* CONST */
        break;
    case TT_DEFVAL:  /* DEFVAL */
        break;
    case TT_SYSVAL:  /* SYSVAL */
        break;
    case TT_STACK:  /* STACK */
        break;
    case TT_FIELD:  /* FIELD */
        break;
    case TT_GLOBAL:  /* GLOBAL */
        break;
    case TT_MEMBER:  /* MEMBER */
        break;
    case TT_CLASSID:  /* CLASSID */
        break;
    case TT_ASIS:  /* ASIS */
        break;
    case TT_NOP:  /* NOP */
        break;
    }
    knh_Token_perror(ctx, tkc, KMSG_TSTMT1);
    return new_StmtERR(ctx, tc);
}


static int knh_stmt_array_isExpr[] = {
    0,  /* done */ 
    0,  /* namespace */ 
    0,  /* import */ 
    0,  /* pragma */ 
    0,  /* using */ 
    0,  /* using import */ 
    0,  /* using alias */ 
    0,  /* using unit */ 
    0,  /* using enum */ 
    0,  /* using vocab */ 
    0,  /* with */ 
    0,  /* using func */ 
    0,  /* using mapmap */ 
    0,  /* class */ 
    0,  /* extends */ 
    0,  /* implements */ 
    0,  /* method */ 
    0,  /* closure */ 
    0,  /* format */ 
    0,  /* decl */ 
    0,  /* return */ 
    0,  /* mapmap */ 
    0,  /* weave */ 
    0,  /* aspect */ 
    0,  /* block */ 
    0,  /* if */ 
    0,  /* else */ 
    0,  /* switch */ 
    0,  /* case */ 
    0,  /* while */ 
    0,  /* do */ 
    0,  /* for */ 
    0,  /* foreach */ 
    0,  /* from */ 
    0,  /* where */ 
    0,  /* break */ 
    0,  /* continue */ 
    0,  /* try */ 
    0,  /* catch */ 
    0,  /* finally */ 
    0,  /* throw */ 
    0,  /* err */ 
    1,  /* let */ 
    0,  /* local */ 
    0,  /* letmulti */ 
    1,  /* new */ 
    1,  /* mapcast */ 
    1,  /* call */ 
    1,  /* call1 */ 
    1,  /* pcall */ 
    1,  /* mt */ 
    1,  /* op */ 
    1,  /* tri */ 
    1,  /* alt */ 
    1,  /* and */ 
    1,  /* or */ 
    0,  /* print */ 
    0,  /* assert */ 
    //(int)0
};

int knh_stmt_isExpr(knh_stmt_t t)
{
    KNH_ASSERT(t < KNH_STMT_MAXSIZ);
    return knh_stmt_array_isExpr[t];
}

/* ------------------------------------------------------------------------ */

#ifdef __cplusplus
}
#endif
